"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualService = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
/**
 * (experimental) VirtualService represents a service inside an AppMesh.
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 * @experimental
 */
class VirtualService extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        if (props.virtualNode && props.virtualRouter) {
            throw new Error('Must provide only one of virtualNode or virtualRouter for the provider');
        }
        this.mesh = props.mesh;
        // Check which provider to use node or router (or neither)
        if (props.virtualRouter) {
            this.virtualServiceProvider = this.addVirtualRouter(props.virtualRouter.virtualRouterName);
        }
        if (props.virtualNode) {
            this.virtualServiceProvider = this.addVirtualNode(props.virtualNode.virtualNodeName);
        }
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: this.virtualServiceProvider,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualService given an ARN.
     *
     * @experimental
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new class extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.virtualServiceArn = virtualServiceArn;
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(virtualServiceArn).resourceName);
                this.virtualServiceName = cdk.Fn.select(2, this.parsedArn);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
            }
        }(scope, id);
    }
    /**
     * (experimental) Import an existing VirtualService given its attributes.
     *
     * @experimental
     */
    static fromVirtualServiceAttributes(scope, id, attrs) {
        return new class extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.virtualServiceName = attrs.virtualServiceName;
                this.mesh = attrs.mesh;
                this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualService`,
                    resourceName: this.virtualServiceName,
                });
            }
        }(scope, id);
    }
    addVirtualRouter(name) {
        return {
            virtualRouter: {
                virtualRouterName: name,
            },
        };
    }
    addVirtualNode(name) {
        return {
            virtualNode: {
                virtualNodeName: name,
            },
        };
    }
}
exports.VirtualService = VirtualService;
//# sourceMappingURL=data:application/json;base64,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