"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualGateway = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const gateway_route_1 = require("./gateway-route");
const mesh_1 = require("./mesh");
const virtual_gateway_listener_1 = require("./virtual-gateway-listener");
class VirtualGatewayBase extends cdk.Resource {
    /**
     * (experimental) Utility method to add a new GatewayRoute to the VirtualGateway.
     *
     * @experimental
     */
    addGatewayRoute(id, props) {
        return new gateway_route_1.GatewayRoute(this, id, {
            ...props,
            virtualGateway: this,
        });
    }
}
/**
 * (experimental) VirtualGateway represents a newly defined App Mesh Virtual Gateway.
 *
 * A virtual gateway allows resources that are outside of your mesh to communicate to resources that
 * are inside of your mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
 * @experimental
 */
class VirtualGateway extends VirtualGatewayBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.virtualGatewayName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        /**
         * @experimental
         */
        this.listeners = new Array();
        this.mesh = props.mesh;
        if (!props.listeners) {
            // Use listener default of http listener port 8080 if no listener is defined
            this.listeners.push(virtual_gateway_listener_1.VirtualGatewayListener.http().bind(this));
        }
        else {
            props.listeners.forEach(listener => this.listeners.push(listener.bind(this)));
        }
        const accessLogging = (_a = props.accessLog) === null || _a === void 0 ? void 0 : _a.bind(this);
        const node = new appmesh_generated_1.CfnVirtualGateway(this, 'Resource', {
            virtualGatewayName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners.map(listener => listener.listener),
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualGatewayAccessLog,
                } : undefined,
            },
        });
        this.virtualGatewayName = this.getResourceNameAttribute(node.attrVirtualGatewayName);
        this.virtualGatewayArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualGateway`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualGateway given an ARN.
     *
     * @experimental
     */
    static fromVirtualGatewayArn(scope, id, virtualGatewayArn) {
        return new class extends VirtualGatewayBase {
            constructor() {
                super(...arguments);
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(virtualGatewayArn).resourceName);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
                this.virtualGatewayArn = virtualGatewayArn;
                this.virtualGatewayName = cdk.Fn.select(2, this.parsedArn);
            }
        }(scope, id);
    }
    /**
     * (experimental) Import an existing VirtualGateway given its attributes.
     *
     * @experimental
     */
    static fromVirtualGatewayAttributes(scope, id, attrs) {
        return new class extends VirtualGatewayBase {
            constructor() {
                super(...arguments);
                this.mesh = attrs.mesh;
                this.virtualGatewayName = attrs.virtualGatewayName;
                this.virtualGatewayArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualGateway`,
                    resourceName: this.virtualGatewayName,
                });
            }
        }(scope, id);
    }
}
exports.VirtualGateway = VirtualGateway;
//# sourceMappingURL=data:application/json;base64,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