'''
# SSM Document CDK

This library provides a code-based utility for implementing SSM Documents. The SSM Document objects can be used to print YAML/JSON documents and to mimic document processing locally.

This library abstracts SSM Documents at a high level, with each step as well as the document itself being objects. The properties needed to build these objects correlate to the settings that apply to them, making them simple to make.
This library can be used to test your document locally before deploying it to SSM.

Since the library is written in JSII, it can be exported to other languages that support JSII (Java, Python).

This is what you'd use if you wanted to:

1. The ability to test without deploying resources or executing an actual SSM on AWS.
2. Reusability of steps between documents by reusing existing items
3. Create logical higher-level groupings of reusable groups of steps ("Patterns")
4. Simple to use interface for writing documents
5. Import existing documents from a file (or string) and mimic them locally to test them.

## Usage

### Document Creation

Typescript usage (Execute AWS API Step)...
The below creates the AutomationDocument in an AWS CDK stack.

```python
import { AutomationDocument } from './automation-document';

export class HelloWorld extends Stack {
  constructor(app: Construct, id: string) {
    super(app, id);

    // Create AutomationDocument
    const myDoc = new AutomationDocument(this, "MyDoc", {
      documentFormat: DocumentFormat.JSON,
      documentName: "MyDoc",
      docInputs: [Input.ofTypeString('MyInput', { defaultValue: 'a' })],
    });

    // Define your steps...
    myDoc.addStep(new PauseStep(this, "MyPauseStep", {
      name: "MyPauseStep",
      explicitNextStep: StepRef.fromName("step1") // Optional (will default to next added step)
    }));

    myDoc.addStep(new ExecuteScriptStep(this, "MyExecuteStep", {
      name: "step1",
      language: ScriptLanguage.python(PythonVersion.VERSION_3_6, 'my_func'),
      code: ScriptCode.fromFile(resolve("test/test_file.py")),
      // OR ScriptCode.inline("def my_func(args, context):\n  return {'MyReturn': args['MyInput'] + '-suffix'}\n"),
      outputs: [{
        outputType: DataTypeEnum.STRING,
        name: "MyFuncOut",
        selector: "$.Payload.MyReturn"
      }],
      onFailure: OnFailure.abort(),
      inputPayload: { MyInput: StringVariable.of('MyInput') },
    }));
  }
}
```

### Document JSON/YAML Export as YAML/JSON

You can deploy the above document using CDK.
To print the above document object as a JSON (or YAML), do the following:

```python
const myDocJson = myDoc.print(); // Print YAML by setting the documentFormat to YAML
```

### Document Simulation

To run the document object in simulation mode, use the below. Simulation mode does NOT hit the SSM API, rather it mimics the execution that will happen in an SSM execution. The run happens locally and allows you to mock the calls to external services (AWS APIs for example) or to invoke those services using your local credentials.

```python
import { Simulation } from './simulation';

const myDocJson = Simulation.ofAutomation(myDoc, {}).simulate({ MyInput: "FooBar" });
```

### Command Documents

Below is an example of how to use the library to create Command documents.
Simulation for command documents is not yet supported for all command plugins.
You can use a Docker image/container as a playground for testing the Command document execution for the supported plugins.

In this example there is a complete CDK stack. Notice that the `CommandDocument` is saved as a field so that it can be tested from the test code.

```python
export class HelloCdkStack extends Stack {
  readonly myCommandDoc: CommandDocument;
  constructor(scope: Construct, id: string, props?: StackProps) {
    super(scope, id, props);
    this.myCommandDoc = new CommandDocument(this, "MyCommandDoc", {
      docInputs: [Input.ofTypeString('FirstCommand', { defaultValue: 'a' })],
    })
    const runScriptStep = new RunShellScriptStep(this, "MyShellScript", {
      runCommand: [
        StringVariable.of("FirstCommand"),
        HardCodedString.of("mkdir asdf"),
      ],
    });
    this.myCommandDoc.addStep(runScriptStep);
  }
}
```

Below is an example of how you would run a simulation against the above `CommandDocument`.

Currently, `bash` must be available in the container or the executions against the docker will not succeed.

```python
test('Test command doc', () => {
  const app = new cdk.App();
  const stack = new HelloCdk.HelloCdkStack(app, 'MyTestStack');
  // 1. $ docker pull amazonlinux
  // 2. $ docker run -di amazonlinux
  const simulation = Simulation.ofCommand(stack.myCommandDoc, {
    simulationPlatform: Platform.LINUX,
    environment: DockerEnvironment.fromContainer('MY_CONTAINER_ID')
  });
  simulation.simulate({FirstCommand: 'mkdir foobar'})
  // 3. The document should run the first command (create 'foobar') and create file 'asdf'
  // 4. $ docker exec -it <container name> bash
  // 5. Ensure that 'asdf' and 'foobar' were written to /tmp
});
```

## Patterns (High-Level Constructs)

In typical CDK style, you can assemble often used groups of steps into higher level Constructs.

Consider if you typically create AutomationDocuments that start with logging the time and end with logging the total time taken. You can create a high-level Automation Document and extend that when you implement an Automation.

See the `TimedDocument` class to see such implementation.

Or consider the case of multiple steps that are always run together such as rebooting and instance and waiting for it to be active.

The below example is copied from the `RebootInstanceAndWait` class:

```python
export class RebootInstanceAndWait extends CompositeAutomationStep {

  readonly reboot: AwsApiStep;
  readonly describe: WaitForResourceStep;

  constructor(scope: Construct, id: string, instanceId: IStringVariable) {
    super(scope, id);
    this.reboot = new AwsApiStep(this, 'RebootInstances', {
      service: AwsService.EC2,
      pascalCaseApi: 'RebootInstances',
      apiParams: { InstanceIds: [instanceId] },
      outputs: [],
    });
    this.describe = new WaitForResourceStep(this, 'DescribeInstances', {
      service: AwsService.EC2,
      pascalCaseApi: 'DescribeInstances',
      apiParams: { InstanceIds: [instanceId] },
      selector: '$.Reservations[0].Instances[0].State.Name',
      desiredValues: ['running'],
    });
  }

  addToDocument(doc: AutomationDocumentBuilder): void {
    doc.addStep(this.reboot);
    doc.addStep(this.describe);
  }
}
```

Now, you can use `RebootInstanceAndWait` as a step in a document and the child steps will be included.

## Existing Documents

Do you have an existing document that you want to convert to code and/or test locally using the simulation?

### Import Existing Document

Here is an example of how you can import an existing document and then simulate it locally with mocked AWS resources:

```python
// Initialize Mocks
const sleeper = new MockSleep();
const awsInvoker = new MockAwsInvoker();
awsInvoker.whenThen(
    // when invoked with...
    {awsApi: 'listBuckets', awsParams: {}, service: AwsService.S3},
    // then response with...
    {Owner: {ID: "BUCKET_ID"}})

// ======> Create document from file <=======
const stack: Stack = new Stack();
const myAutomationDoc = StringDocument.fromFile(stack, "MyAutomationDoc", 'test/myAutomation.json', {
                                                                        // ======================
});

// Execute simulation
const simOutput = Simulation.ofAutomation(myAutomationDoc, {
  sleepHook: sleeper,
  awsInvoker: awsInvoker
}).simulate({});

// Assert simulation result
assert.deepEqual(awsInvoker.previousInvocations, [
    { awsApi: 'listBuckets', awsParams: {}, service: AwsService.S3 }]);
assert.deepEqual(sleeper.sleepMilliInvocations, [3000]);
assert.deepEqual(simOutput.outputs['simulationSteps'], ['MySleep', 'GetBucketId']);
```

### Import Existing Steps

You can also grab a string step (or steps) and import them as CDK step constructs.
This can be used to convert existing documents into CDK with each step defined separately.
Doing so will allow you do modify steps and reuse them in other documents.

Here's a simple example of a sleep step copy and pasted from its original yaml:

```python
StringStep.fromYaml(this, `
    name: sleep
    action: aws:sleep
    inputs:
      Duration: PT0M
`, {});
```

The above will return the CDK construct SleepStep.

## Incident Manager

This library provides L2 constructs for IncidentResponse as follows:

```python
new IncidentResponse(this, "MyIncidentResponsePlan", {
      incidentTemplate: IncidentTemplate.critical('EC2 Instance Utilization Impacted', {
        summary: 'EC2 Instance Impacted'
      }),
      actions: [
        IncidentResponseAction.ssmAutomation(myAutomationDoc, ec2CwAlarmRole, {
          parameters: {
            IncidentRecordArn: StringVariable.of('INCIDENT_RECORD_ARN'),
            InvolvedResources: StringVariable.of('INVOLVED_RESOURCES'),
            AutomationAssumeRole: HardCodedString.of(ec2CwAlarmRole.roleArn),
          }
        })
      ]
});
```

Notice how the `myAutomationDoc` is specified which is a reference to an AutomationDocument created using this library.

## What is Planned?

This library currently contains AutomationDocument and CommandDocument steps.
Simulation for AutomationDocuments is fully supported. Simulation for CommandDocuments is limited.

Stay tuned!

## Related Projects

* https://github.com/udondan/cdk-ssm-document

## Security

See [CONTRIBUTING](CONTRIBUTING.md#security-issue-notifications) for more information.

## License

This project is licensed under the Apache-2.0 License.
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from ._jsii import *

import aws_cdk
import aws_cdk.aws_iam
import aws_cdk.aws_ssm
import aws_cdk.aws_ssmincidents
import aws_cdk.cx_api
import constructs


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ApiExecuteAutomationProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker", "sleep_hook": "sleepHook"},
)
class ApiExecuteAutomationProps:
    def __init__(self, *, aws_invoker: "IAwsInvoker", sleep_hook: "ISleepHook") -> None:
        '''
        :param aws_invoker: 
        :param sleep_hook: 
        '''
        if __debug__:
            def stub(*, aws_invoker: "IAwsInvoker", sleep_hook: "ISleepHook") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def aws_invoker(self) -> "IAwsInvoker":
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast("IAwsInvoker", result)

    @builtins.property
    def sleep_hook(self) -> "ISleepHook":
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast("ISleepHook", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApiExecuteAutomationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ApiRunCommandProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker", "sleep_hook": "sleepHook"},
)
class ApiRunCommandProps:
    def __init__(self, *, aws_invoker: "IAwsInvoker", sleep_hook: "ISleepHook") -> None:
        '''
        :param aws_invoker: 
        :param sleep_hook: 
        '''
        if __debug__:
            def stub(*, aws_invoker: "IAwsInvoker", sleep_hook: "ISleepHook") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def aws_invoker(self) -> "IAwsInvoker":
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast("IAwsInvoker", result)

    @builtins.property
    def sleep_hook(self) -> "ISleepHook":
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast("ISleepHook", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApiRunCommandProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ApproveSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"approve_hook": "approveHook"},
)
class ApproveSimulationProps:
    def __init__(self, *, approve_hook: "IApproveHook") -> None:
        '''Properties for ApproveStep.

        :param approve_hook: (Optional) Approve hook to be called to pause the execution. To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class. Default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        '''
        if __debug__:
            def stub(*, approve_hook: "IApproveHook") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument approve_hook", value=approve_hook, expected_type=type_hints["approve_hook"])
        self._values: typing.Dict[str, typing.Any] = {
            "approve_hook": approve_hook,
        }

    @builtins.property
    def approve_hook(self) -> "IApproveHook":
        '''(Optional) Approve hook to be called to pause the execution.

        To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class.

        :default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        '''
        result = self._values.get("approve_hook")
        assert result is not None, "Required property 'approve_hook' is missing"
        return typing.cast("IApproveHook", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApproveSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuthMethod(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.AuthMethod",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="ofBasic")
    @builtins.classmethod
    def of_basic(
        cls,
        user_name: "SecureVariable",
        password: "SecureVariable",
    ) -> "AuthMethod":
        '''
        :param user_name: -
        :param password: -
        '''
        if __debug__:
            def stub(user_name: "SecureVariable", password: "SecureVariable") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument user_name", value=user_name, expected_type=type_hints["user_name"])
            check_type(argname="argument password", value=password, expected_type=type_hints["password"])
        return typing.cast("AuthMethod", jsii.sinvoke(cls, "ofBasic", [user_name, password]))

    @jsii.member(jsii_name="ofDigest")
    @builtins.classmethod
    def of_digest(
        cls,
        user_name: "SecureVariable",
        password: "SecureVariable",
    ) -> "AuthMethod":
        '''
        :param user_name: -
        :param password: -
        '''
        if __debug__:
            def stub(user_name: "SecureVariable", password: "SecureVariable") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument user_name", value=user_name, expected_type=type_hints["user_name"])
            check_type(argname="argument password", value=password, expected_type=type_hints["password"])
        return typing.cast("AuthMethod", jsii.sinvoke(cls, "ofDigest", [user_name, password]))

    @jsii.member(jsii_name="ofNone")
    @builtins.classmethod
    def of_none(cls) -> "AuthMethod":
        return typing.cast("AuthMethod", jsii.sinvoke(cls, "ofNone", []))

    @jsii.member(jsii_name="requiredInputs")
    @abc.abstractmethod
    def required_inputs(self) -> typing.List[builtins.str]:
        ...

    @jsii.member(jsii_name="toEntry")
    @abc.abstractmethod
    def to_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        ...


class _AuthMethodProxy(AuthMethod):
    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="toEntry")
    def to_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toEntry", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AuthMethod).__jsii_proxy_class__ = lambda : _AuthMethodProxy


class AutomationDocumentBuilder(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationDocumentBuilder",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="addStep")
    def add_step(self, step: "AutomationStep") -> None:
        '''
        :param step: -
        '''
        if __debug__:
            def stub(step: "AutomationStep") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        return typing.cast(None, jsii.invoke(self, "addStep", [step]))

    @builtins.property
    @jsii.member(jsii_name="steps")
    def steps(self) -> typing.List["AutomationStep"]:
        return typing.cast(typing.List["AutomationStep"], jsii.get(self, "steps"))


class AutomationSimulation(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationSimulation",
):
    def __init__(
        self,
        automation_document: "AutomationDocument",
        *,
        approve_hook: typing.Optional["IApproveHook"] = None,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        environment: typing.Optional["IEnvironment"] = None,
        input_observer: typing.Optional["IObserver"] = None,
        output_observer: typing.Optional["IObserver"] = None,
        parameter_resolver: typing.Optional["IParameterResolver"] = None,
        pause_hook: typing.Optional["IPauseHook"] = None,
        run_command_hook: typing.Optional["IRunCommandHook"] = None,
        simulation_platform: typing.Optional["Platform"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
        webhook: typing.Optional["IWebhook"] = None,
    ) -> None:
        '''
        :param automation_document: -
        :param approve_hook: 
        :param aws_invoker: 
        :param environment: 
        :param input_observer: 
        :param output_observer: 
        :param parameter_resolver: 
        :param pause_hook: 
        :param run_command_hook: 
        :param simulation_platform: 
        :param sleep_hook: 
        :param webhook: 
        '''
        if __debug__:
            def stub(
                automation_document: "AutomationDocument",
                *,
                approve_hook: typing.Optional["IApproveHook"] = None,
                aws_invoker: typing.Optional["IAwsInvoker"] = None,
                environment: typing.Optional["IEnvironment"] = None,
                input_observer: typing.Optional["IObserver"] = None,
                output_observer: typing.Optional["IObserver"] = None,
                parameter_resolver: typing.Optional["IParameterResolver"] = None,
                pause_hook: typing.Optional["IPauseHook"] = None,
                run_command_hook: typing.Optional["IRunCommandHook"] = None,
                simulation_platform: typing.Optional["Platform"] = None,
                sleep_hook: typing.Optional["ISleepHook"] = None,
                webhook: typing.Optional["IWebhook"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument automation_document", value=automation_document, expected_type=type_hints["automation_document"])
        props = SimulationProps(
            approve_hook=approve_hook,
            aws_invoker=aws_invoker,
            environment=environment,
            input_observer=input_observer,
            output_observer=output_observer,
            parameter_resolver=parameter_resolver,
            pause_hook=pause_hook,
            run_command_hook=run_command_hook,
            simulation_platform=simulation_platform,
            sleep_hook=sleep_hook,
            webhook=webhook,
        )

        jsii.create(self.__class__, self, [automation_document, props])

    @jsii.member(jsii_name="start")
    def start(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "SimulationResult":
        '''Starts the execution of the steps by invoking the first step.

        The subsequent steps will be invoked by the steps themselves.

        :param inputs: all of the inputs necessary for the document to execute.

        :return: the outputs specified by all the steps.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast("SimulationResult", jsii.invoke(self, "start", [inputs]))


class AutomationSimulationBase(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationSimulationBase",
):
    def __init__(self, step: "AutomationStep") -> None:
        '''
        :param step: -
        '''
        if __debug__:
            def stub(step: "AutomationStep") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        jsii.create(self.__class__, self, [step])

    @jsii.member(jsii_name="executeStep")
    @abc.abstractmethod
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        ...

    @jsii.member(jsii_name="nextStep")
    def next_step(
        self,
        _inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Optional["AutomationStep"]:
        '''
        :param _inputs: -
        '''
        if __debug__:
            def stub(_inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument _inputs", value=_inputs, expected_type=type_hints["_inputs"])
        return typing.cast(typing.Optional["AutomationStep"], jsii.invoke(self, "nextStep", [_inputs]))


class _AutomationSimulationBaseProxy(AutomationSimulationBase):
    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AutomationSimulationBase).__jsii_proxy_class__ = lambda : _AutomationSimulationBaseProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationSimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "approve_hook": "approveHook",
        "aws_invoker": "awsInvoker",
        "execute_automation_hook": "executeAutomationHook",
        "input_observer": "inputObserver",
        "output_observer": "outputObserver",
        "parameter_resolver": "parameterResolver",
        "pause_hook": "pauseHook",
        "run_command_hook": "runCommandHook",
        "sleep_hook": "sleepHook",
        "webhook": "webhook",
    },
)
class AutomationSimulationProps:
    def __init__(
        self,
        *,
        approve_hook: typing.Optional["IApproveHook"] = None,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        execute_automation_hook: typing.Optional["IExecuteAutomationHook"] = None,
        input_observer: typing.Optional["IObserver"] = None,
        output_observer: typing.Optional["IObserver"] = None,
        parameter_resolver: typing.Optional["IParameterResolver"] = None,
        pause_hook: typing.Optional["IPauseHook"] = None,
        run_command_hook: typing.Optional["IRunCommandHook"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
        webhook: typing.Optional["IWebhook"] = None,
    ) -> None:
        '''
        :param approve_hook: (Optional) Approve hook to be called to pause the execution. To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class. Default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param execute_automation_hook: Hook for simulating aws:executeAutomation. Default: - Uses AWS API to execute the document remotely.
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param parameter_resolver: (Optional) Resolver for secure strings in parameters. Required to simulate if using tokens in parameters input. Default: - Treats parameters as literal
        :param pause_hook: (Optional) Pause hook to be called to pause the execution. To mock this implemenation either inject an instance of IPauseHook or use the provided MockPause class. Default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        :param run_command_hook: Hook for simulating aws:runCommand. Default: - Uses AWS API to execute the document remotely.
        :param sleep_hook: (Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations). Default: - really perform sleep using SleeperImpl class.
        :param webhook: (Optional) Hook for simulating aws:invokeWebhook. Default: - Returns 204 with an empty response
        '''
        if __debug__:
            def stub(
                *,
                approve_hook: typing.Optional["IApproveHook"] = None,
                aws_invoker: typing.Optional["IAwsInvoker"] = None,
                execute_automation_hook: typing.Optional["IExecuteAutomationHook"] = None,
                input_observer: typing.Optional["IObserver"] = None,
                output_observer: typing.Optional["IObserver"] = None,
                parameter_resolver: typing.Optional["IParameterResolver"] = None,
                pause_hook: typing.Optional["IPauseHook"] = None,
                run_command_hook: typing.Optional["IRunCommandHook"] = None,
                sleep_hook: typing.Optional["ISleepHook"] = None,
                webhook: typing.Optional["IWebhook"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument approve_hook", value=approve_hook, expected_type=type_hints["approve_hook"])
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument execute_automation_hook", value=execute_automation_hook, expected_type=type_hints["execute_automation_hook"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument parameter_resolver", value=parameter_resolver, expected_type=type_hints["parameter_resolver"])
            check_type(argname="argument pause_hook", value=pause_hook, expected_type=type_hints["pause_hook"])
            check_type(argname="argument run_command_hook", value=run_command_hook, expected_type=type_hints["run_command_hook"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
            check_type(argname="argument webhook", value=webhook, expected_type=type_hints["webhook"])
        self._values: typing.Dict[str, typing.Any] = {}
        if approve_hook is not None:
            self._values["approve_hook"] = approve_hook
        if aws_invoker is not None:
            self._values["aws_invoker"] = aws_invoker
        if execute_automation_hook is not None:
            self._values["execute_automation_hook"] = execute_automation_hook
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if parameter_resolver is not None:
            self._values["parameter_resolver"] = parameter_resolver
        if pause_hook is not None:
            self._values["pause_hook"] = pause_hook
        if run_command_hook is not None:
            self._values["run_command_hook"] = run_command_hook
        if sleep_hook is not None:
            self._values["sleep_hook"] = sleep_hook
        if webhook is not None:
            self._values["webhook"] = webhook

    @builtins.property
    def approve_hook(self) -> typing.Optional["IApproveHook"]:
        '''(Optional) Approve hook to be called to pause the execution.

        To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class.

        :default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        '''
        result = self._values.get("approve_hook")
        return typing.cast(typing.Optional["IApproveHook"], result)

    @builtins.property
    def aws_invoker(self) -> typing.Optional["IAwsInvoker"]:
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        return typing.cast(typing.Optional["IAwsInvoker"], result)

    @builtins.property
    def execute_automation_hook(self) -> typing.Optional["IExecuteAutomationHook"]:
        '''Hook for simulating aws:executeAutomation.

        :default: - Uses AWS API to execute the document remotely.
        '''
        result = self._values.get("execute_automation_hook")
        return typing.cast(typing.Optional["IExecuteAutomationHook"], result)

    @builtins.property
    def input_observer(self) -> typing.Optional["IObserver"]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional["IObserver"], result)

    @builtins.property
    def output_observer(self) -> typing.Optional["IObserver"]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional["IObserver"], result)

    @builtins.property
    def parameter_resolver(self) -> typing.Optional["IParameterResolver"]:
        '''(Optional) Resolver for secure strings in parameters.

        Required to simulate if using tokens in parameters input.

        :default: - Treats parameters as literal
        '''
        result = self._values.get("parameter_resolver")
        return typing.cast(typing.Optional["IParameterResolver"], result)

    @builtins.property
    def pause_hook(self) -> typing.Optional["IPauseHook"]:
        '''(Optional) Pause hook to be called to pause the execution.

        To mock this implemenation either inject an instance of IPauseHook or use the provided MockPause class.

        :default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        '''
        result = self._values.get("pause_hook")
        return typing.cast(typing.Optional["IPauseHook"], result)

    @builtins.property
    def run_command_hook(self) -> typing.Optional["IRunCommandHook"]:
        '''Hook for simulating aws:runCommand.

        :default: - Uses AWS API to execute the document remotely.
        '''
        result = self._values.get("run_command_hook")
        return typing.cast(typing.Optional["IRunCommandHook"], result)

    @builtins.property
    def sleep_hook(self) -> typing.Optional["ISleepHook"]:
        '''(Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations).

        :default: - really perform sleep using SleeperImpl class.
        '''
        result = self._values.get("sleep_hook")
        return typing.cast(typing.Optional["ISleepHook"], result)

    @builtins.property
    def webhook(self) -> typing.Optional["IWebhook"]:
        '''(Optional) Hook for simulating aws:invokeWebhook.

        :default: - Returns 204 with an empty response
        '''
        result = self._values.get("webhook")
        return typing.cast(typing.Optional["IWebhook"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutomationSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AutomationStepSimulation(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationStepSimulation",
):
    def __init__(
        self,
        step: "AutomationStep",
        *,
        approve_hook: typing.Optional["IApproveHook"] = None,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        execute_automation_hook: typing.Optional["IExecuteAutomationHook"] = None,
        input_observer: typing.Optional["IObserver"] = None,
        output_observer: typing.Optional["IObserver"] = None,
        parameter_resolver: typing.Optional["IParameterResolver"] = None,
        pause_hook: typing.Optional["IPauseHook"] = None,
        run_command_hook: typing.Optional["IRunCommandHook"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
        webhook: typing.Optional["IWebhook"] = None,
    ) -> None:
        '''
        :param step: -
        :param approve_hook: (Optional) Approve hook to be called to pause the execution. To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class. Default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param execute_automation_hook: Hook for simulating aws:executeAutomation. Default: - Uses AWS API to execute the document remotely.
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param parameter_resolver: (Optional) Resolver for secure strings in parameters. Required to simulate if using tokens in parameters input. Default: - Treats parameters as literal
        :param pause_hook: (Optional) Pause hook to be called to pause the execution. To mock this implemenation either inject an instance of IPauseHook or use the provided MockPause class. Default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        :param run_command_hook: Hook for simulating aws:runCommand. Default: - Uses AWS API to execute the document remotely.
        :param sleep_hook: (Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations). Default: - really perform sleep using SleeperImpl class.
        :param webhook: (Optional) Hook for simulating aws:invokeWebhook. Default: - Returns 204 with an empty response
        '''
        if __debug__:
            def stub(
                step: "AutomationStep",
                *,
                approve_hook: typing.Optional["IApproveHook"] = None,
                aws_invoker: typing.Optional["IAwsInvoker"] = None,
                execute_automation_hook: typing.Optional["IExecuteAutomationHook"] = None,
                input_observer: typing.Optional["IObserver"] = None,
                output_observer: typing.Optional["IObserver"] = None,
                parameter_resolver: typing.Optional["IParameterResolver"] = None,
                pause_hook: typing.Optional["IPauseHook"] = None,
                run_command_hook: typing.Optional["IRunCommandHook"] = None,
                sleep_hook: typing.Optional["ISleepHook"] = None,
                webhook: typing.Optional["IWebhook"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = AutomationSimulationProps(
            approve_hook=approve_hook,
            aws_invoker=aws_invoker,
            execute_automation_hook=execute_automation_hook,
            input_observer=input_observer,
            output_observer=output_observer,
            parameter_resolver=parameter_resolver,
            pause_hook=pause_hook,
            run_command_hook=run_command_hook,
            sleep_hook=sleep_hook,
            webhook=webhook,
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "SimulationResult":
        '''Invokes the current step on the input and will return a SimulationResult.

        :param inputs: must contain all of the inputs declared by the current step.

        :return: a SimulationResult with the step outputs in the case of success or stacktrace in the case of failure.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast("SimulationResult", jsii.invoke(self, "invoke", [inputs]))

    @jsii.member(jsii_name="prependSelf")
    def _prepend_self(
        self,
        subsequent_steps: typing.Sequence[builtins.str],
    ) -> typing.List[builtins.str]:
        '''Adds this step name to the list of executed steps.

        Since the steps are invoked as a chain, the current step is prepended as invocation stack is popped.

        :param subsequent_steps: -
        '''
        if __debug__:
            def stub(subsequent_steps: typing.Sequence[builtins.str]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument subsequent_steps", value=subsequent_steps, expected_type=type_hints["subsequent_steps"])
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "prependSelf", [subsequent_steps]))

    @builtins.property
    @jsii.member(jsii_name="props")
    def props(self) -> "RequiredAutomationSimulationProps":
        return typing.cast("RequiredAutomationSimulationProps", jsii.get(self, "props"))

    @builtins.property
    @jsii.member(jsii_name="step")
    def step(self) -> "AutomationStep":
        return typing.cast("AutomationStep", jsii.get(self, "step"))


class AwsApiSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AwsApiSimulation",
):
    '''AutomationStep implementation of aws:executeAwsApi.'''

    def __init__(self, step: "AwsApiStep", *, aws_invoker: "IAwsInvoker") -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(step: "AwsApiStep", *, aws_invoker: "IAwsInvoker") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Invokes the specified service (param) with the specified api (param) with the specified apiParams (param).

        This call will be invoked synchronously.
        The input variables in apiParams (param) specified using "{{ INPUT }}" syntax will be replaced with the inputs.

        :param inputs: -

        :return: the AWS api response. The Output selection will take place outside of this function.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="awsApiStep")
    def aws_api_step(self) -> "AwsApiStep":
        return typing.cast("AwsApiStep", jsii.get(self, "awsApiStep"))

    @builtins.property
    @jsii.member(jsii_name="awsInvoker")
    def aws_invoker(self) -> "IAwsInvoker":
        return typing.cast("IAwsInvoker", jsii.get(self, "awsInvoker"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AwsInvocationSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker"},
)
class AwsInvocationSimulationProps:
    def __init__(self, *, aws_invoker: "IAwsInvoker") -> None:
        '''
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(*, aws_invoker: "IAwsInvoker") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
        }

    @builtins.property
    def aws_invoker(self) -> "IAwsInvoker":
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast("IAwsInvoker", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsInvocationSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AwsService(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AwsService",
):
    '''Represents an AWS service.

    The namespace which is meant to be used for the service declaration in SSM may be difficult to guess.
    The name provided for simulation may at times be different.
    This class provides many ready-to-use implementations of AWS services.
    If a service is not on the list, you can fall back to constructing this object on your own.
    In other words, you can reference S3 using ``AwsService.S3``
    For a NewService, you can reference it using ``new AwsService('new-service', 'NewService')``.
    It is risky to auto convert between namespace and javaScriptName because they are not consistent in all cases.
    '''

    def __init__(self, namespace: builtins.str, java_script_name: builtins.str) -> None:
        '''
        :param namespace: -
        :param java_script_name: -
        '''
        if __debug__:
            def stub(namespace: builtins.str, java_script_name: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
            check_type(argname="argument java_script_name", value=java_script_name, expected_type=type_hints["java_script_name"])
        jsii.create(self.__class__, self, [namespace, java_script_name])

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "toJSON", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ACCESS_ANALYZER")
    def ACCESS_ANALYZER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ACCESS_ANALYZER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ACCOUNT")
    def ACCOUNT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ACCOUNT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ACM")
    def ACM(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ACM"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ACMPCA")
    def ACMPCA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ACMPCA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ALEXA_FOR_BUSINESS")
    def ALEXA_FOR_BUSINESS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ALEXA_FOR_BUSINESS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMP")
    def AMP(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "AMP"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMPLIFY")
    def AMPLIFY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "AMPLIFY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMPLIFY_BACKEND")
    def AMPLIFY_BACKEND(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "AMPLIFY_BACKEND"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMPLIFY_UI_BUILDER")
    def AMPLIFY_UI_BUILDER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "AMPLIFY_UI_BUILDER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="API_GATEWAY_MANAGEMENT_API")
    def API_GATEWAY_MANAGEMENT_API(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "API_GATEWAY_MANAGEMENT_API"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="API_GATEWAY_V2")
    def API_GATEWAY_V2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "API_GATEWAY_V2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APIGATEWAY")
    def APIGATEWAY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APIGATEWAY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APP_CONFIG")
    def APP_CONFIG(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APP_CONFIG"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APP_CONFIG_DATA")
    def APP_CONFIG_DATA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APP_CONFIG_DATA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APP_INTEGRATIONS")
    def APP_INTEGRATIONS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APP_INTEGRATIONS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APP_MESH")
    def APP_MESH(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APP_MESH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APP_RUNNER")
    def APP_RUNNER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APP_RUNNER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APP_STREAM")
    def APP_STREAM(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APP_STREAM"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APP_SYNC")
    def APP_SYNC(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APP_SYNC"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APPFLOW")
    def APPFLOW(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APPFLOW"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APPLICATION_AUTO_SCALING")
    def APPLICATION_AUTO_SCALING(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APPLICATION_AUTO_SCALING"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APPLICATION_COST_PROFILER")
    def APPLICATION_COST_PROFILER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APPLICATION_COST_PROFILER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APPLICATION_INSIGHTS")
    def APPLICATION_INSIGHTS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "APPLICATION_INSIGHTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ATHENA")
    def ATHENA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ATHENA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AUDIT_MANAGER")
    def AUDIT_MANAGER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "AUDIT_MANAGER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AUTO_SCALING")
    def AUTO_SCALING(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "AUTO_SCALING"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AUTO_SCALING_PLANS")
    def AUTO_SCALING_PLANS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "AUTO_SCALING_PLANS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="BACKUP")
    def BACKUP(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "BACKUP"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="BACKUP_GATEWAY")
    def BACKUP_GATEWAY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "BACKUP_GATEWAY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="BACKUP_STORAGE")
    def BACKUP_STORAGE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "BACKUP_STORAGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="BATCH")
    def BATCH(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "BATCH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="BILLINGCONDUCTOR")
    def BILLINGCONDUCTOR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "BILLINGCONDUCTOR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="BRAKET")
    def BRAKET(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "BRAKET"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="BUDGETS")
    def BUDGETS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "BUDGETS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CHIME")
    def CHIME(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CHIME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CHIME_SDK_IDENTITY")
    def CHIME_SDK_IDENTITY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CHIME_SDK_IDENTITY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CHIME_SDK_MEDIA_PIPELINES")
    def CHIME_SDK_MEDIA_PIPELINES(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CHIME_SDK_MEDIA_PIPELINES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CHIME_SDK_MEETINGS")
    def CHIME_SDK_MEETINGS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CHIME_SDK_MEETINGS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CHIME_SDK_MESSAGING")
    def CHIME_SDK_MESSAGING(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CHIME_SDK_MESSAGING"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD_CONTROL")
    def CLOUD_CONTROL(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD_CONTROL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD_DIRECTORY")
    def CLOUD_DIRECTORY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD_DIRECTORY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD_FORMATION")
    def CLOUD_FORMATION(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD_FORMATION"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD_FRONT")
    def CLOUD_FRONT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD_FRONT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD_HS_M")
    def CLOUD_HS_M(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD_HS_M"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD_HSM_V2")
    def CLOUD_HSM_V2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD_HSM_V2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD_SEARCH")
    def CLOUD_SEARCH(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD_SEARCH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD_SEARCH_DOMAIN")
    def CLOUD_SEARCH_DOMAIN(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD_SEARCH_DOMAIN"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD_TRAIL")
    def CLOUD_TRAIL(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD_TRAIL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD_WATCH")
    def CLOUD_WATCH(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD_WATCH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CLOUD9")
    def CLOUD9(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CLOUD9"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_ARTIFACT")
    def CODE_ARTIFACT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CODE_ARTIFACT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_BUILD")
    def CODE_BUILD(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CODE_BUILD"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_COMMIT")
    def CODE_COMMIT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CODE_COMMIT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_DEPLOY")
    def CODE_DEPLOY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CODE_DEPLOY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_GURU_PROFILER")
    def CODE_GURU_PROFILER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CODE_GURU_PROFILER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_GURU_REVIEWER")
    def CODE_GURU_REVIEWER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CODE_GURU_REVIEWER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_PIPELINE")
    def CODE_PIPELINE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CODE_PIPELINE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_STAR")
    def CODE_STAR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CODE_STAR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_STAR_NOTIFICATIONS")
    def CODE_STAR_NOTIFICATIONS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CODE_STAR_NOTIFICATIONS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CODE_STARCONNECTIONS")
    def CODE_STARCONNECTIONS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CODE_STARCONNECTIONS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="COGNITO_IDENTITY")
    def COGNITO_IDENTITY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "COGNITO_IDENTITY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="COGNITO_SYNC")
    def COGNITO_SYNC(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "COGNITO_SYNC"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="COMPREHEND")
    def COMPREHEND(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "COMPREHEND"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="COMPREHEND_MEDICAL")
    def COMPREHEND_MEDICAL(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "COMPREHEND_MEDICAL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="COMPUTE_OPTIMIZER")
    def COMPUTE_OPTIMIZER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "COMPUTE_OPTIMIZER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CONNECT")
    def CONNECT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CONNECT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CONNECT_CAMPAIGNS")
    def CONNECT_CAMPAIGNS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CONNECT_CAMPAIGNS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CONNECT_CASES")
    def CONNECT_CASES(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CONNECT_CASES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CONNECT_CONTACT_LENS")
    def CONNECT_CONTACT_LENS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CONNECT_CONTACT_LENS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CONNECT_PARTICIPANT")
    def CONNECT_PARTICIPANT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CONNECT_PARTICIPANT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CONTROL_TOWER")
    def CONTROL_TOWER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CONTROL_TOWER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CUR")
    def CUR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CUR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CUSTOMER_PROFILES")
    def CUSTOMER_PROFILES(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "CUSTOMER_PROFILES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DATA_BREW")
    def DATA_BREW(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DATA_BREW"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DATA_EXCHANGE")
    def DATA_EXCHANGE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DATA_EXCHANGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DATA_PIPELINE")
    def DATA_PIPELINE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DATA_PIPELINE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DATA_SYNC")
    def DATA_SYNC(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DATA_SYNC"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DAX")
    def DAX(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DAX"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DETECTIVE")
    def DETECTIVE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DETECTIVE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DEV_OPS_GURU")
    def DEV_OPS_GURU(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DEV_OPS_GURU"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DEVICE_FARM")
    def DEVICE_FARM(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DEVICE_FARM"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DIRECT_CONNECT")
    def DIRECT_CONNECT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DIRECT_CONNECT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DISCOVERY")
    def DISCOVERY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DISCOVERY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DLM")
    def DLM(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DLM"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DMS")
    def DMS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DMS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DOC_DB")
    def DOC_DB(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DOC_DB"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DRS")
    def DRS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DRS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DYNAMO_DB")
    def DYNAMO_DB(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DYNAMO_DB"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DYNAMO_DB_STREAMS")
    def DYNAMO_DB_STREAMS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "DYNAMO_DB_STREAMS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EBS")
    def EBS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "EBS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EC2")
    def EC2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "EC2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EC2_INSTANCE_CONNECT")
    def EC2_INSTANCE_CONNECT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "EC2_INSTANCE_CONNECT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ECR")
    def ECR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ECR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ECRPUBLIC")
    def ECRPUBLIC(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ECRPUBLIC"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ECS")
    def ECS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ECS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EFS")
    def EFS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "EFS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EKS")
    def EKS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "EKS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ELASTI_CACHE")
    def ELASTI_CACHE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ELASTI_CACHE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ELASTIC_BEANSTALK")
    def ELASTIC_BEANSTALK(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ELASTIC_BEANSTALK"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ELASTIC_INFERENCE")
    def ELASTIC_INFERENCE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ELASTIC_INFERENCE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ELASTIC_TRANSCODER")
    def ELASTIC_TRANSCODER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ELASTIC_TRANSCODER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ELB")
    def ELB(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ELB"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ELBV2")
    def ELBV2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ELBV2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EMR")
    def EMR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "EMR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EMRCONTAINERS")
    def EMRCONTAINERS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "EMRCONTAINERS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EMRSERVERLESS")
    def EMRSERVERLESS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "EMRSERVERLESS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ES")
    def ES(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EVIDENTLY")
    def EVIDENTLY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "EVIDENTLY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FINSPACE")
    def FINSPACE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "FINSPACE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FINSPACEDATA")
    def FINSPACEDATA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "FINSPACEDATA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FIREHOSE")
    def FIREHOSE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "FIREHOSE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FIS")
    def FIS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "FIS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FMS")
    def FMS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "FMS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FRAUD_DETECTOR")
    def FRAUD_DETECTOR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "FRAUD_DETECTOR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FSX")
    def FSX(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "FSX"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GAME_LIFT")
    def GAME_LIFT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "GAME_LIFT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GAME_SPARKS")
    def GAME_SPARKS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "GAME_SPARKS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GLACIER")
    def GLACIER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "GLACIER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GLOBAL_ACCELERATOR")
    def GLOBAL_ACCELERATOR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "GLOBAL_ACCELERATOR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GLUE")
    def GLUE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "GLUE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GRAFANA")
    def GRAFANA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "GRAFANA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GREENGRASS")
    def GREENGRASS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "GREENGRASS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GREENGRASS_V2")
    def GREENGRASS_V2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "GREENGRASS_V2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GROUND_STATION")
    def GROUND_STATION(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "GROUND_STATION"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GUARD_DUTY")
    def GUARD_DUTY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "GUARD_DUTY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HEALTH")
    def HEALTH(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "HEALTH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HEALTH_LAKE")
    def HEALTH_LAKE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "HEALTH_LAKE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="HONEYCODE")
    def HONEYCODE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "HONEYCODE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IAM")
    def IAM(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IAM"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IDENTITY_STORE")
    def IDENTITY_STORE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IDENTITY_STORE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IMAGEBUILDER")
    def IMAGEBUILDER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IMAGEBUILDER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IMPORT_EXPORT")
    def IMPORT_EXPORT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IMPORT_EXPORT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="INSPECTOR")
    def INSPECTOR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "INSPECTOR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="INSPECTOR2")
    def INSPECTOR2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "INSPECTOR2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IO_T_ANALYTICS")
    def IO_T_ANALYTICS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IO_T_ANALYTICS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IO_T1_CLICK_PROJECTS")
    def IO_T1_CLICK_PROJECTS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IO_T1_CLICK_PROJECTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT")
    def IOT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_DATA")
    def IOT_DATA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_DATA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_DEVICE_ADVISOR")
    def IOT_DEVICE_ADVISOR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_DEVICE_ADVISOR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_EVENTS")
    def IOT_EVENTS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_EVENTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_EVENTS_DATA")
    def IOT_EVENTS_DATA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_EVENTS_DATA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_FLEET_HUB")
    def IOT_FLEET_HUB(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_FLEET_HUB"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_FLEET_WISE")
    def IOT_FLEET_WISE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_FLEET_WISE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_SECURE_TUNNELING")
    def IOT_SECURE_TUNNELING(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_SECURE_TUNNELING"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_SITE_WISE")
    def IOT_SITE_WISE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_SITE_WISE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_THINGS_GRAPH")
    def IOT_THINGS_GRAPH(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_THINGS_GRAPH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_TWIN_MAKER")
    def IOT_TWIN_MAKER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_TWIN_MAKER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IOT_WIRELESS")
    def IOT_WIRELESS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IOT_WIRELESS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IVS")
    def IVS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IVS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="IVSCHAT")
    def IVSCHAT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "IVSCHAT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KAFKA")
    def KAFKA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KAFKA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KAFKA_CONNECT")
    def KAFKA_CONNECT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KAFKA_CONNECT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KENDRA")
    def KENDRA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KENDRA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KEYSPACES")
    def KEYSPACES(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KEYSPACES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KINESIS")
    def KINESIS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KINESIS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KINESIS_ANALYTICS")
    def KINESIS_ANALYTICS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KINESIS_ANALYTICS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KINESIS_ANALYTICS_V2")
    def KINESIS_ANALYTICS_V2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KINESIS_ANALYTICS_V2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KINESIS_VIDEO")
    def KINESIS_VIDEO(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KINESIS_VIDEO"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KINESIS_VIDEO_ARCHIVED_MEDIA")
    def KINESIS_VIDEO_ARCHIVED_MEDIA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KINESIS_VIDEO_ARCHIVED_MEDIA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KINESIS_VIDEO_MEDIA")
    def KINESIS_VIDEO_MEDIA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KINESIS_VIDEO_MEDIA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KMS")
    def KMS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "KMS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LAKE_FORMATION")
    def LAKE_FORMATION(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "LAKE_FORMATION"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LAMBDA")
    def LAMBDA_(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "LAMBDA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LEX_RUNTIME")
    def LEX_RUNTIME(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "LEX_RUNTIME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LICENSE_MANAGER")
    def LICENSE_MANAGER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "LICENSE_MANAGER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LICENSE_MANAGER_USER_SUBSCRIPTIONS")
    def LICENSE_MANAGER_USER_SUBSCRIPTIONS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "LICENSE_MANAGER_USER_SUBSCRIPTIONS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LIGHTSAIL")
    def LIGHTSAIL(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "LIGHTSAIL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LOCATION")
    def LOCATION(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "LOCATION"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LOOKOUT_EQUIPMENT")
    def LOOKOUT_EQUIPMENT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "LOOKOUT_EQUIPMENT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LOOKOUT_METRICS")
    def LOOKOUT_METRICS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "LOOKOUT_METRICS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LOOKOUT_VISION")
    def LOOKOUT_VISION(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "LOOKOUT_VISION"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="M2")
    def M2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "M2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MACHINE_LEARNING")
    def MACHINE_LEARNING(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MACHINE_LEARNING"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MACIE")
    def MACIE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MACIE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MACIE2")
    def MACIE2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MACIE2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MANAGED_BLOCKCHAIN")
    def MANAGED_BLOCKCHAIN(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MANAGED_BLOCKCHAIN"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MARKETPLACE_CATALOG")
    def MARKETPLACE_CATALOG(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MARKETPLACE_CATALOG"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MARKETPLACE_COMMERCE_ANALYTICS")
    def MARKETPLACE_COMMERCE_ANALYTICS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MARKETPLACE_COMMERCE_ANALYTICS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MEDIA_CONNECT")
    def MEDIA_CONNECT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MEDIA_CONNECT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MEDIA_CONVERT")
    def MEDIA_CONVERT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MEDIA_CONVERT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MEDIA_LIVE")
    def MEDIA_LIVE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MEDIA_LIVE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MEDIA_PACKAGE")
    def MEDIA_PACKAGE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MEDIA_PACKAGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MEDIA_PACKAGE_VOD")
    def MEDIA_PACKAGE_VOD(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MEDIA_PACKAGE_VOD"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MEDIA_STORE")
    def MEDIA_STORE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MEDIA_STORE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MEDIA_STORE_DATA")
    def MEDIA_STORE_DATA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MEDIA_STORE_DATA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MEDIA_TAILOR")
    def MEDIA_TAILOR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MEDIA_TAILOR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MEMORY_DB")
    def MEMORY_DB(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MEMORY_DB"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MGN")
    def MGN(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MGN"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MIGRATION_HUB_CONFIG")
    def MIGRATION_HUB_CONFIG(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MIGRATION_HUB_CONFIG"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MIGRATION_HUB_ORCHESTRATOR")
    def MIGRATION_HUB_ORCHESTRATOR(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MIGRATION_HUB_ORCHESTRATOR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MIGRATION_HUB_REFACTOR_SPACES")
    def MIGRATION_HUB_REFACTOR_SPACES(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MIGRATION_HUB_REFACTOR_SPACES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MIGRATION_HUB_STRATEGY")
    def MIGRATION_HUB_STRATEGY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MIGRATION_HUB_STRATEGY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MOBILE")
    def MOBILE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MOBILE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MQ")
    def MQ(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MQ"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MTURK")
    def MTURK(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MTURK"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MWAA")
    def MWAA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "MWAA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NEPTUNE")
    def NEPTUNE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "NEPTUNE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NETWORK_FIREWALL")
    def NETWORK_FIREWALL(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "NETWORK_FIREWALL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NETWORK_MANAGER")
    def NETWORK_MANAGER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "NETWORK_MANAGER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NIMBLE")
    def NIMBLE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "NIMBLE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="OPEN_SEARCH")
    def OPEN_SEARCH(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "OPEN_SEARCH"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="OPS_WORKS")
    def OPS_WORKS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "OPS_WORKS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="OPS_WORKS_C_M")
    def OPS_WORKS_C_M(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "OPS_WORKS_C_M"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ORGANIZATIONS")
    def ORGANIZATIONS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ORGANIZATIONS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="OUTPOSTS")
    def OUTPOSTS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "OUTPOSTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PANORAMA")
    def PANORAMA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PANORAMA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PERSONALIZE")
    def PERSONALIZE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PERSONALIZE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PERSONALIZE_EVENTS")
    def PERSONALIZE_EVENTS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PERSONALIZE_EVENTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PERSONALIZE_RUNTIME")
    def PERSONALIZE_RUNTIME(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PERSONALIZE_RUNTIME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PI")
    def PI(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PI"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PINPOINT")
    def PINPOINT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PINPOINT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PINPOINT_EMAIL")
    def PINPOINT_EMAIL(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PINPOINT_EMAIL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PINPOINT_SMS_VOICE")
    def PINPOINT_SMS_VOICE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PINPOINT_SMS_VOICE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PINPOINT_SMS_VOICE_V2")
    def PINPOINT_SMS_VOICE_V2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PINPOINT_SMS_VOICE_V2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="POLLY")
    def POLLY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "POLLY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PRICING")
    def PRICING(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PRICING"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PRIVATE_NETWORKS")
    def PRIVATE_NETWORKS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PRIVATE_NETWORKS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PROTON")
    def PROTON(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "PROTON"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="QLDB")
    def QLDB(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "QLDB"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="QLDB_SESSION")
    def QLDB_SESSION(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "QLDB_SESSION"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="QUICK_SIGHT")
    def QUICK_SIGHT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "QUICK_SIGHT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RAM")
    def RAM(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "RAM"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RBIN")
    def RBIN(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "RBIN"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RDS")
    def RDS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "RDS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="REDSHIFT")
    def REDSHIFT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "REDSHIFT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="REDSHIFT_DATA")
    def REDSHIFT_DATA(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "REDSHIFT_DATA"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="REDSHIFT_SERVERLESS")
    def REDSHIFT_SERVERLESS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "REDSHIFT_SERVERLESS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="REKOGNITION")
    def REKOGNITION(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "REKOGNITION"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RESILIENCEHUB")
    def RESILIENCEHUB(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "RESILIENCEHUB"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RESOURCE_EXPLORER2")
    def RESOURCE_EXPLORER2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "RESOURCE_EXPLORER2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RESOURCE_GROUPS")
    def RESOURCE_GROUPS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "RESOURCE_GROUPS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RESOURCE_GROUPS_TAGGING_AP_I")
    def RESOURCE_GROUPS_TAGGING_AP_I(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "RESOURCE_GROUPS_TAGGING_AP_I"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ROBO_MAKER")
    def ROBO_MAKER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ROBO_MAKER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ROLES_ANYWHERE")
    def ROLES_ANYWHERE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ROLES_ANYWHERE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ROUTE53")
    def ROUTE53(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ROUTE53"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ROUTE53_DOMAINS")
    def ROUTE53_DOMAINS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ROUTE53_DOMAINS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ROUTE53_RECOVERY_CLUSTER")
    def ROUTE53_RECOVERY_CLUSTER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ROUTE53_RECOVERY_CLUSTER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ROUTE53_RECOVERY_CONTROL_CONFIG")
    def ROUTE53_RECOVERY_CONTROL_CONFIG(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ROUTE53_RECOVERY_CONTROL_CONFIG"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ROUTE53_RECOVERY_READINESS")
    def ROUTE53_RECOVERY_READINESS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ROUTE53_RECOVERY_READINESS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ROUTE53_RESOLVER")
    def ROUTE53_RESOLVER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "ROUTE53_RESOLVER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="RUM")
    def RUM(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "RUM"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="S3")
    def S3(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "S3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="S3_CONTROL")
    def S3_CONTROL(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "S3_CONTROL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="S3_OUTPOSTS")
    def S3_OUTPOSTS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "S3_OUTPOSTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SAGE_MAKER")
    def SAGE_MAKER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SAGE_MAKER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SAGE_MAKER_FEATURE_STORE_RUNTIME")
    def SAGE_MAKER_FEATURE_STORE_RUNTIME(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SAGE_MAKER_FEATURE_STORE_RUNTIME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SAGE_MAKER_RUNTIME")
    def SAGE_MAKER_RUNTIME(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SAGE_MAKER_RUNTIME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SAGEMAKER_EDGE")
    def SAGEMAKER_EDGE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SAGEMAKER_EDGE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SAVINGS_PLANS")
    def SAVINGS_PLANS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SAVINGS_PLANS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SCHEDULER")
    def SCHEDULER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SCHEDULER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SCHEMAS")
    def SCHEMAS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SCHEMAS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SECRETS_MANAGER")
    def SECRETS_MANAGER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SECRETS_MANAGER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SECURITY_HUB")
    def SECURITY_HUB(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SECURITY_HUB"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SERVICE_CATALOG")
    def SERVICE_CATALOG(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SERVICE_CATALOG"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SERVICE_CATALOG_APP_REGISTRY")
    def SERVICE_CATALOG_APP_REGISTRY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SERVICE_CATALOG_APP_REGISTRY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SERVICE_DISCOVERY")
    def SERVICE_DISCOVERY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SERVICE_DISCOVERY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SERVICE_QUOTAS")
    def SERVICE_QUOTAS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SERVICE_QUOTAS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SES")
    def SES(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SES_V2")
    def SES_V2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SES_V2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SHIELD")
    def SHIELD(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SHIELD"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SIGNER")
    def SIGNER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SIGNER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SMS")
    def SMS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SMS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SNOW_DEVICE_MANAGEMENT")
    def SNOW_DEVICE_MANAGEMENT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SNOW_DEVICE_MANAGEMENT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SNOWBALL")
    def SNOWBALL(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SNOWBALL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SNS")
    def SNS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SNS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SQS")
    def SQS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SQS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SSM")
    def SSM(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SSM"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SSM_CONTACTS")
    def SSM_CONTACTS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SSM_CONTACTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SSM_INCIDENTS")
    def SSM_INCIDENTS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SSM_INCIDENTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SSO")
    def SSO(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SSO"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SSO_ADMIN")
    def SSO_ADMIN(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SSO_ADMIN"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SSO_OIDC")
    def SSO_OIDC(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SSO_OIDC"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="STEP_FUNCTIONS")
    def STEP_FUNCTIONS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "STEP_FUNCTIONS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="STORAGE_GATEWAY")
    def STORAGE_GATEWAY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "STORAGE_GATEWAY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="STS")
    def STS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "STS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SUPPORT")
    def SUPPORT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SUPPORT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SUPPORT_APP")
    def SUPPORT_APP(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SUPPORT_APP"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SWF")
    def SWF(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SWF"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SYNTHETICS")
    def SYNTHETICS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "SYNTHETICS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TEXTRACT")
    def TEXTRACT(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "TEXTRACT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TIMESTREAM_QUERY")
    def TIMESTREAM_QUERY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "TIMESTREAM_QUERY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TIMESTREAM_WRITE")
    def TIMESTREAM_WRITE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "TIMESTREAM_WRITE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TRANSFER")
    def TRANSFER(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "TRANSFER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="TRANSLATE")
    def TRANSLATE(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "TRANSLATE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="VOICE_I_D")
    def VOICE_I_D(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "VOICE_I_D"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WAF")
    def WAF(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WAF"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WAF_REGIONAL")
    def WAF_REGIONAL(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WAF_REGIONAL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WAF_V2")
    def WAF_V2(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WAF_V2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WELL_ARCHITECTED")
    def WELL_ARCHITECTED(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WELL_ARCHITECTED"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WISDOM")
    def WISDOM(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WISDOM"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WORK_DOCS")
    def WORK_DOCS(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WORK_DOCS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WORK_LINK")
    def WORK_LINK(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WORK_LINK"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WORK_MAIL")
    def WORK_MAIL(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WORK_MAIL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WORK_MAIL_MESSAGE_FLOW")
    def WORK_MAIL_MESSAGE_FLOW(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WORK_MAIL_MESSAGE_FLOW"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WORK_SPACES")
    def WORK_SPACES(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WORK_SPACES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="WORK_SPACES_WEB")
    def WORK_SPACES_WEB(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "WORK_SPACES_WEB"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="XRAY")
    def XRAY(cls) -> "AwsService":
        return typing.cast("AwsService", jsii.sget(cls, "XRAY"))

    @builtins.property
    @jsii.member(jsii_name="javaScriptName")
    def java_script_name(self) -> builtins.str:
        '''Used for simulation of service requests using the simulated execution.'''
        return typing.cast(builtins.str, jsii.get(self, "javaScriptName"))

    @builtins.property
    @jsii.member(jsii_name="namespace")
    def namespace(self) -> builtins.str:
        '''Used for declaration in SSM Automation document.'''
        return typing.cast(builtins.str, jsii.get(self, "namespace"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.BodyOrUrlProp",
    jsii_struct_bases=[],
    name_mapping={"prop_type": "propType", "value": "value"},
)
class BodyOrUrlProp:
    def __init__(self, *, prop_type: "BodyOrUrlType", value: "IStringVariable") -> None:
        '''Allow passing in a body or URL version of the property value.

        :param prop_type: Whether the body or URL was provided.
        :param value: Body or URL string.
        '''
        if __debug__:
            def stub(*, prop_type: "BodyOrUrlType", value: "IStringVariable") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument prop_type", value=prop_type, expected_type=type_hints["prop_type"])
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        self._values: typing.Dict[str, typing.Any] = {
            "prop_type": prop_type,
            "value": value,
        }

    @builtins.property
    def prop_type(self) -> "BodyOrUrlType":
        '''Whether the body or URL was provided.'''
        result = self._values.get("prop_type")
        assert result is not None, "Required property 'prop_type' is missing"
        return typing.cast("BodyOrUrlType", result)

    @builtins.property
    def value(self) -> "IStringVariable":
        '''Body or URL string.'''
        result = self._values.get("value")
        assert result is not None, "Required property 'value' is missing"
        return typing.cast("IStringVariable", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BodyOrUrlProp(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.BodyOrUrlType")
class BodyOrUrlType(enum.Enum):
    BODY = "BODY"
    URL = "URL"


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.BooleanInputProps",
    jsii_struct_bases=[],
    name_mapping={"default_value": "defaultValue", "description": "description"},
)
class BooleanInputProps:
    def __init__(
        self,
        *,
        default_value: typing.Optional[builtins.bool] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        '''
        if __debug__:
            def stub(
                *,
                default_value: typing.Optional[builtins.bool] = None,
                description: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument default_value", value=default_value, expected_type=type_hints["default_value"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[str, typing.Any] = {}
        if default_value is not None:
            self._values["default_value"] = default_value
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def default_value(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Default value to use for this input if not specified when invoking the document.

        :default: undefined
        '''
        result = self._values.get("default_value")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) The description of the input.

        :default: name
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BooleanInputProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BranchSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.BranchSimulation",
):
    '''AutomationStep implementation of aws:branch https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-branch.html.'''

    def __init__(self, step: "BranchStep") -> None:
        '''
        :param step: -
        '''
        if __debug__:
            def stub(step: "BranchStep") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        jsii.create(self.__class__, self, [step])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        _inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''noop.

        The logic performed in the branch branchStep happens in the invoke() function.

        :param _inputs: -
        '''
        if __debug__:
            def stub(_inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument _inputs", value=_inputs, expected_type=type_hints["_inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [_inputs]))

    @jsii.member(jsii_name="nextStep")
    def next_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Optional["AutomationStep"]:
        '''Overrides invoke because control flow of execution is different than standard steps.

        Will traverse the choices until one evaluated to true; will skip to that choice.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Optional["AutomationStep"], jsii.invoke(self, "nextStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class ChangeInstanceStateSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ChangeInstanceStateSimulation",
):
    '''AutomationStep implemenation for aws:changeInstanceState https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-changestate.html.'''

    def __init__(
        self,
        step: "ChangeInstanceStateStep",
        *,
        aws_invoker: "IAwsInvoker",
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(
                step: "ChangeInstanceStateStep",
                *,
                aws_invoker: "IAwsInvoker",
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class Choice(metaclass=jsii.JSIIMeta, jsii_type="@cdklabs/cdk-ssm-documents.Choice"):
    '''Choices are supplied to the BranchStep to determine under which conditions to jump to which steps.'''

    def __init__(
        self,
        *,
        constant: typing.Any,
        jump_to_step_name: builtins.str,
        operation: "Operation",
        variable: typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"],
    ) -> None:
        '''
        :param constant: (Required) the constant to test against the inputToTest.
        :param jump_to_step_name: A step to jump to if this choice is evaluated to true. Must reference another step in the currently executing AutomationDocument.
        :param operation: (Required) The operation used to compare the inputToTest with the constant.
        :param variable: (Required) the input used to test using the operation with the constant.
        '''
        props = ChoiceProps(
            constant=constant,
            jump_to_step_name=jump_to_step_name,
            operation=operation,
            variable=variable,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="asSsmEntry")
    def as_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :return: an object that can be used to print this choice into yaml/json format.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "asSsmEntry", []))

    @jsii.member(jsii_name="evaluate")
    def evaluate(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.bool:
        '''Evaluates this choice against the provided inputs.

        The value keyed on the inputToTest will be tested against the declared constant using the Operation specified.

        :param inputs: -

        :return: true if the evaluation is true. False otherwise.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(builtins.bool, jsii.invoke(self, "evaluate", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="constant")
    def constant(self) -> typing.Any:
        return typing.cast(typing.Any, jsii.get(self, "constant"))

    @builtins.property
    @jsii.member(jsii_name="jumpToStepName")
    def jump_to_step_name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "jumpToStepName"))

    @builtins.property
    @jsii.member(jsii_name="operation")
    def operation(self) -> "Operation":
        return typing.cast("Operation", jsii.get(self, "operation"))

    @builtins.property
    @jsii.member(jsii_name="variable")
    def variable(
        self,
    ) -> typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"]:
        return typing.cast(typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"], jsii.get(self, "variable"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ChoiceProps",
    jsii_struct_bases=[],
    name_mapping={
        "constant": "constant",
        "jump_to_step_name": "jumpToStepName",
        "operation": "operation",
        "variable": "variable",
    },
)
class ChoiceProps:
    def __init__(
        self,
        *,
        constant: typing.Any,
        jump_to_step_name: builtins.str,
        operation: "Operation",
        variable: typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"],
    ) -> None:
        '''Properties for a Choice used by the BranchStep.

        :param constant: (Required) the constant to test against the inputToTest.
        :param jump_to_step_name: A step to jump to if this choice is evaluated to true. Must reference another step in the currently executing AutomationDocument.
        :param operation: (Required) The operation used to compare the inputToTest with the constant.
        :param variable: (Required) the input used to test using the operation with the constant.
        '''
        if __debug__:
            def stub(
                *,
                constant: typing.Any,
                jump_to_step_name: builtins.str,
                operation: "Operation",
                variable: typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"],
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument constant", value=constant, expected_type=type_hints["constant"])
            check_type(argname="argument jump_to_step_name", value=jump_to_step_name, expected_type=type_hints["jump_to_step_name"])
            check_type(argname="argument operation", value=operation, expected_type=type_hints["operation"])
            check_type(argname="argument variable", value=variable, expected_type=type_hints["variable"])
        self._values: typing.Dict[str, typing.Any] = {
            "constant": constant,
            "jump_to_step_name": jump_to_step_name,
            "operation": operation,
            "variable": variable,
        }

    @builtins.property
    def constant(self) -> typing.Any:
        '''(Required) the constant to test against the inputToTest.'''
        result = self._values.get("constant")
        assert result is not None, "Required property 'constant' is missing"
        return typing.cast(typing.Any, result)

    @builtins.property
    def jump_to_step_name(self) -> builtins.str:
        '''A step to jump to if this choice is evaluated to true.

        Must reference another step in the currently executing AutomationDocument.
        '''
        result = self._values.get("jump_to_step_name")
        assert result is not None, "Required property 'jump_to_step_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def operation(self) -> "Operation":
        '''(Required) The operation used to compare the inputToTest with the constant.'''
        result = self._values.get("operation")
        assert result is not None, "Required property 'operation' is missing"
        return typing.cast("Operation", result)

    @builtins.property
    def variable(
        self,
    ) -> typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"]:
        '''(Required) the input used to test using the operation with the constant.'''
        result = self._values.get("variable")
        assert result is not None, "Required property 'variable' is missing"
        return typing.cast(typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ChoiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CommandDocumentBuilder(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandDocumentBuilder",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="addStep")
    def add_step(self, step: "CommandStep") -> None:
        '''
        :param step: -
        '''
        if __debug__:
            def stub(step: "CommandStep") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        return typing.cast(None, jsii.invoke(self, "addStep", [step]))

    @builtins.property
    @jsii.member(jsii_name="steps")
    def steps(self) -> typing.List["CommandStep"]:
        return typing.cast(typing.List["CommandStep"], jsii.get(self, "steps"))


class CommandSimulation(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandSimulation",
):
    def __init__(
        self,
        command_document: "CommandDocument",
        *,
        approve_hook: typing.Optional["IApproveHook"] = None,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        environment: typing.Optional["IEnvironment"] = None,
        input_observer: typing.Optional["IObserver"] = None,
        output_observer: typing.Optional["IObserver"] = None,
        parameter_resolver: typing.Optional["IParameterResolver"] = None,
        pause_hook: typing.Optional["IPauseHook"] = None,
        run_command_hook: typing.Optional["IRunCommandHook"] = None,
        simulation_platform: typing.Optional["Platform"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
        webhook: typing.Optional["IWebhook"] = None,
    ) -> None:
        '''
        :param command_document: -
        :param approve_hook: 
        :param aws_invoker: 
        :param environment: 
        :param input_observer: 
        :param output_observer: 
        :param parameter_resolver: 
        :param pause_hook: 
        :param run_command_hook: 
        :param simulation_platform: 
        :param sleep_hook: 
        :param webhook: 
        '''
        if __debug__:
            def stub(
                command_document: "CommandDocument",
                *,
                approve_hook: typing.Optional["IApproveHook"] = None,
                aws_invoker: typing.Optional["IAwsInvoker"] = None,
                environment: typing.Optional["IEnvironment"] = None,
                input_observer: typing.Optional["IObserver"] = None,
                output_observer: typing.Optional["IObserver"] = None,
                parameter_resolver: typing.Optional["IParameterResolver"] = None,
                pause_hook: typing.Optional["IPauseHook"] = None,
                run_command_hook: typing.Optional["IRunCommandHook"] = None,
                simulation_platform: typing.Optional["Platform"] = None,
                sleep_hook: typing.Optional["ISleepHook"] = None,
                webhook: typing.Optional["IWebhook"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument command_document", value=command_document, expected_type=type_hints["command_document"])
        props = SimulationProps(
            approve_hook=approve_hook,
            aws_invoker=aws_invoker,
            environment=environment,
            input_observer=input_observer,
            output_observer=output_observer,
            parameter_resolver=parameter_resolver,
            pause_hook=pause_hook,
            run_command_hook=run_command_hook,
            simulation_platform=simulation_platform,
            sleep_hook=sleep_hook,
            webhook=webhook,
        )

        jsii.create(self.__class__, self, [command_document, props])

    @jsii.member(jsii_name="start")
    def start(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "SimulationResult":
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast("SimulationResult", jsii.invoke(self, "start", [inputs]))


class CommandSimulationBase(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandSimulationBase",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="executeStep")
    @abc.abstractmethod
    def execute_step(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param inputs: -
        '''
        ...


class _CommandSimulationBaseProxy(CommandSimulationBase):
    @jsii.member(jsii_name="executeStep")
    def execute_step(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(None, jsii.invoke(self, "executeStep", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CommandSimulationBase).__jsii_proxy_class__ = lambda : _CommandSimulationBaseProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CommandSimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "simulation_platform": "simulationPlatform",
        "environment": "environment",
    },
)
class CommandSimulationProps:
    def __init__(
        self,
        *,
        simulation_platform: "Platform",
        environment: typing.Optional["IEnvironment"] = None,
    ) -> None:
        '''
        :param simulation_platform: The Platform used in executing the command step.
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        if __debug__:
            def stub(
                *,
                simulation_platform: "Platform",
                environment: typing.Optional["IEnvironment"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument simulation_platform", value=simulation_platform, expected_type=type_hints["simulation_platform"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
        self._values: typing.Dict[str, typing.Any] = {
            "simulation_platform": simulation_platform,
        }
        if environment is not None:
            self._values["environment"] = environment

    @builtins.property
    def simulation_platform(self) -> "Platform":
        '''The Platform used in executing the command step.'''
        result = self._values.get("simulation_platform")
        assert result is not None, "Required property 'simulation_platform' is missing"
        return typing.cast("Platform", result)

    @builtins.property
    def environment(self) -> typing.Optional["IEnvironment"]:
        '''(Optional) Specify here the environment in which to execute the scripts.

        Use the DockerEnvironment to execute the commands inside the docker.
        You can alternatively use the LoggingEnvironment which simply logs the commands
        or MockEnvironment which saves them for validation.

        :default: LoggingEnvironment
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional["IEnvironment"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommandSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CommandStepSimulation(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandStepSimulation",
):
    def __init__(
        self,
        step: "CommandStep",
        *,
        simulation_platform: "Platform",
        environment: typing.Optional["IEnvironment"] = None,
    ) -> None:
        '''
        :param step: -
        :param simulation_platform: The Platform used in executing the command step.
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        if __debug__:
            def stub(
                step: "CommandStep",
                *,
                simulation_platform: "Platform",
                environment: typing.Optional["IEnvironment"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = CommandSimulationProps(
            simulation_platform=simulation_platform, environment=environment
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "SimulationResult":
        '''Invokes the current step on the input and will return a SimulationResult.

        :param inputs: must contain all of the inputs declared by the current step.

        :return: a SimulationResult with the step outputs in the case of success or stacktrace in the case of failure.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast("SimulationResult", jsii.invoke(self, "invoke", [inputs]))

    @jsii.member(jsii_name="prependSelf")
    def _prepend_self(
        self,
        future_steps: typing.Sequence[builtins.str],
    ) -> typing.List[builtins.str]:
        '''Adds this step name to the list of executed steps.

        Since the steps are invoked as a chain, the current step is prepended as invocation stack is popped.

        :param future_steps: -
        '''
        if __debug__:
            def stub(future_steps: typing.Sequence[builtins.str]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument future_steps", value=future_steps, expected_type=type_hints["future_steps"])
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "prependSelf", [future_steps]))

    @builtins.property
    @jsii.member(jsii_name="props")
    def props(self) -> "RequiredCommandSimulationProps":
        return typing.cast("RequiredCommandSimulationProps", jsii.get(self, "props"))

    @builtins.property
    @jsii.member(jsii_name="step")
    def step(self) -> "CommandStep":
        return typing.cast("CommandStep", jsii.get(self, "step"))


class CopyImageSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CopyImageSimulation",
):
    '''AutomationStep implemenation for aws:copyImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-copyimage.html.'''

    def __init__(self, step: "CopyImageStep", *, aws_invoker: "IAwsInvoker") -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(step: "CopyImageStep", *, aws_invoker: "IAwsInvoker") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class CreateImageSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateImageSimulation",
):
    '''AutomationStep implemenation for aws:createImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-create.html.'''

    def __init__(self, step: "CreateImageStep", *, aws_invoker: "IAwsInvoker") -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(step: "CreateImageStep", *, aws_invoker: "IAwsInvoker") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class CreateStackSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateStackSimulation",
):
    '''AutomationStep implementation for aws:createStack https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createstack.html.'''

    def __init__(
        self,
        step: "CreateStackStep",
        *,
        aws_invoker: "IAwsInvoker",
        parameter_resolver: "IParameterResolver",
        sleep_hook: "ISleepHook",
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param parameter_resolver: (Optional) Resolver for secure strings in parameters. Required to simulate if using tokens in parameters input. Default: - Treats parameters as literal
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        if __debug__:
            def stub(
                step: "CreateStackStep",
                *,
                aws_invoker: "IAwsInvoker",
                parameter_resolver: "IParameterResolver",
                sleep_hook: "ISleepHook",
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = CreateStackSimulationProps(
            aws_invoker=aws_invoker,
            parameter_resolver=parameter_resolver,
            sleep_hook=sleep_hook,
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CreateStackSimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "aws_invoker": "awsInvoker",
        "parameter_resolver": "parameterResolver",
        "sleep_hook": "sleepHook",
    },
)
class CreateStackSimulationProps:
    def __init__(
        self,
        *,
        aws_invoker: "IAwsInvoker",
        parameter_resolver: "IParameterResolver",
        sleep_hook: "ISleepHook",
    ) -> None:
        '''Properties for CreateStackStep.

        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param parameter_resolver: (Optional) Resolver for secure strings in parameters. Required to simulate if using tokens in parameters input. Default: - Treats parameters as literal
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        if __debug__:
            def stub(
                *,
                aws_invoker: "IAwsInvoker",
                parameter_resolver: "IParameterResolver",
                sleep_hook: "ISleepHook",
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument parameter_resolver", value=parameter_resolver, expected_type=type_hints["parameter_resolver"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
            "parameter_resolver": parameter_resolver,
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def aws_invoker(self) -> "IAwsInvoker":
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast("IAwsInvoker", result)

    @builtins.property
    def parameter_resolver(self) -> "IParameterResolver":
        '''(Optional) Resolver for secure strings in parameters.

        Required to simulate if using tokens in parameters input.

        :default: - Treats parameters as literal
        '''
        result = self._values.get("parameter_resolver")
        assert result is not None, "Required property 'parameter_resolver' is missing"
        return typing.cast("IParameterResolver", result)

    @builtins.property
    def sleep_hook(self) -> "ISleepHook":
        '''(Optional) Whether to really perform a pause of the runtime.

        To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class.

        :default: SleeperImpl
        '''
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast("ISleepHook", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateStackSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CreateTagsSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateTagsSimulation",
):
    '''AutomationStep implemenation for aws:createTags https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createtag.html.'''

    def __init__(
        self,
        step: "CreateTagsStep",
        *,
        aws_invoker: "IAwsInvoker",
        sleep_hook: "ISleepHook",
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        if __debug__:
            def stub(
                step: "CreateTagsStep",
                *,
                aws_invoker: "IAwsInvoker",
                sleep_hook: "ISleepHook",
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = CreateTagsSimulationProps(
            aws_invoker=aws_invoker, sleep_hook=sleep_hook
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CreateTagsSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker", "sleep_hook": "sleepHook"},
)
class CreateTagsSimulationProps:
    def __init__(self, *, aws_invoker: "IAwsInvoker", sleep_hook: "ISleepHook") -> None:
        '''Properties for CreateTagStep.

        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        if __debug__:
            def stub(*, aws_invoker: "IAwsInvoker", sleep_hook: "ISleepHook") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def aws_invoker(self) -> "IAwsInvoker":
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast("IAwsInvoker", result)

    @builtins.property
    def sleep_hook(self) -> "ISleepHook":
        '''(Optional) Whether to really perform a pause of the runtime.

        To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class.

        :default: SleeperImpl
        '''
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast("ISleepHook", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateTagsSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DataType(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DataType",
):
    '''JSII does not allow functions or constants declared in an enum class directly.

    Therefore, interaction with DataTypeEnum happens through this class.
    '''

    def __init__(self, data_type_enum: "DataTypeEnum") -> None:
        '''
        :param data_type_enum: -
        '''
        if __debug__:
            def stub(data_type_enum: "DataTypeEnum") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument data_type_enum", value=data_type_enum, expected_type=type_hints["data_type_enum"])
        jsii.create(self.__class__, self, [data_type_enum])

    @jsii.member(jsii_name="fromDataType")
    @builtins.classmethod
    def from_data_type(cls, ssm_data_type: builtins.str) -> "DataType":
        '''
        :param ssm_data_type: -
        '''
        if __debug__:
            def stub(ssm_data_type: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument ssm_data_type", value=ssm_data_type, expected_type=type_hints["ssm_data_type"])
        return typing.cast("DataType", jsii.sinvoke(cls, "fromDataType", [ssm_data_type]))

    @jsii.member(jsii_name="toSsmString")
    def to_ssm_string(self) -> builtins.str:
        '''
        :return: the SSM string representation of this DataType
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toSsmString", []))

    @jsii.member(jsii_name="validateType")
    def validate_type(self, val: typing.Any) -> builtins.bool:
        '''
        :param val: -

        :return: true if the input variable matches the associated dataTypeEnum
        '''
        if __debug__:
            def stub(val: typing.Any) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument val", value=val, expected_type=type_hints["val"])
        return typing.cast(builtins.bool, jsii.invoke(self, "validateType", [val]))

    @builtins.property
    @jsii.member(jsii_name="dataTypeEnum")
    def data_type_enum(self) -> "DataTypeEnum":
        return typing.cast("DataTypeEnum", jsii.get(self, "dataTypeEnum"))


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.DataTypeEnum")
class DataTypeEnum(enum.Enum):
    '''DataTypes as supported by SSM Documents.'''

    STRING = "STRING"
    INTEGER = "INTEGER"
    BOOLEAN = "BOOLEAN"
    STRING_LIST = "STRING_LIST"
    STRING_MAP = "STRING_MAP"
    MAP_LIST = "MAP_LIST"


class DeleteImageSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteImageSimulation",
):
    '''AutomationStep implementation for aws:deleteImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-delete.html.'''

    def __init__(
        self,
        step: "DeleteImageStep",
        *,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        if __debug__:
            def stub(
                step: "DeleteImageStep",
                *,
                aws_invoker: typing.Optional["IAwsInvoker"] = None,
                sleep_hook: typing.Optional["ISleepHook"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = DeleteImageSimulationProps(
            aws_invoker=aws_invoker, sleep_hook=sleep_hook
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteImageSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker", "sleep_hook": "sleepHook"},
)
class DeleteImageSimulationProps:
    def __init__(
        self,
        *,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
    ) -> None:
        '''Properties for DeleteImageStep.

        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        if __debug__:
            def stub(
                *,
                aws_invoker: typing.Optional["IAwsInvoker"] = None,
                sleep_hook: typing.Optional["ISleepHook"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
        self._values: typing.Dict[str, typing.Any] = {}
        if aws_invoker is not None:
            self._values["aws_invoker"] = aws_invoker
        if sleep_hook is not None:
            self._values["sleep_hook"] = sleep_hook

    @builtins.property
    def aws_invoker(self) -> typing.Optional["IAwsInvoker"]:
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        return typing.cast(typing.Optional["IAwsInvoker"], result)

    @builtins.property
    def sleep_hook(self) -> typing.Optional["ISleepHook"]:
        '''(Optional) Whether to really perform a pause of the runtime.

        To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class.

        :default: SleeperImpl
        '''
        result = self._values.get("sleep_hook")
        return typing.cast(typing.Optional["ISleepHook"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DeleteImageSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DeleteStackSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteStackSimulation",
):
    '''AutomationStep implemenation for aws:deleteStack https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-deletestack.html.'''

    def __init__(self, step: "DeleteStackStep", *, aws_invoker: "IAwsInvoker") -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(step: "DeleteStackStep", *, aws_invoker: "IAwsInvoker") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.DocumentFormat")
class DocumentFormat(enum.Enum):
    JSON = "JSON"
    YAML = "YAML"


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DocumentOutput",
    jsii_struct_bases=[],
    name_mapping={"name": "name", "output_type": "outputType"},
)
class DocumentOutput:
    def __init__(self, *, name: builtins.str, output_type: DataTypeEnum) -> None:
        '''Outputs from a document via one of the steps.

        The outputs are available for use in other documents calling this document via aws:executeAutomation.
        The outputs returned from an automation document are returned as a String list named "ouptuts".

        :param name: The step and name (in STEP.NAME format) to identify an output expected from one of the document steps. The name must therefore identify a step and an output or will fail validation.
        :param output_type: The DataType expected by this output. This will be validated in simulation mode and will also be used when printing to yaml/json.
        '''
        if __debug__:
            def stub(*, name: builtins.str, output_type: DataTypeEnum) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_type", value=output_type, expected_type=type_hints["output_type"])
        self._values: typing.Dict[str, typing.Any] = {
            "name": name,
            "output_type": output_type,
        }

    @builtins.property
    def name(self) -> builtins.str:
        '''The step and name (in STEP.NAME format) to identify an output expected from one of the document steps. The name must therefore identify a step and an output or will fail validation.

        Example::

            MyStep.MyOutput
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def output_type(self) -> DataTypeEnum:
        '''The DataType expected by this output.

        This will be validated in simulation mode and will also be used when printing to yaml/json.
        '''
        result = self._values.get("output_type")
        assert result is not None, "Required property 'output_type' is missing"
        return typing.cast(DataTypeEnum, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DocumentOutput(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DocumentSource(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.DocumentSource",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromArn")
    @builtins.classmethod
    def from_arn(cls, arn: "IStringVariable") -> "DocumentSource":
        '''
        :param arn: -
        '''
        if __debug__:
            def stub(arn: "IStringVariable") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
        return typing.cast("DocumentSource", jsii.sinvoke(cls, "fromArn", [arn]))

    @jsii.member(jsii_name="fromName")
    @builtins.classmethod
    def from_name(
        cls,
        name: "IStringVariable",
        version: typing.Optional["IStringVariable"] = None,
    ) -> "DocumentSource":
        '''
        :param name: -
        :param version: -
        '''
        if __debug__:
            def stub(
                name: "IStringVariable",
                version: typing.Optional["IStringVariable"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        return typing.cast("DocumentSource", jsii.sinvoke(cls, "fromName", [name, version]))

    @jsii.member(jsii_name="formatRequest")
    @abc.abstractmethod
    def format_request(self) -> typing.Mapping[builtins.str, typing.Any]:
        ...

    @jsii.member(jsii_name="requiredInputs")
    @abc.abstractmethod
    def required_inputs(self) -> typing.List[builtins.str]:
        ...


class _DocumentSourceProxy(DocumentSource):
    @jsii.member(jsii_name="formatRequest")
    def format_request(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatRequest", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, DocumentSource).__jsii_proxy_class__ = lambda : _DocumentSourceProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.EnvironmentProps",
    jsii_struct_bases=[],
    name_mapping={"environment": "environment"},
)
class EnvironmentProps:
    def __init__(self, *, environment: "IEnvironment") -> None:
        '''
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        if __debug__:
            def stub(*, environment: "IEnvironment") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
        self._values: typing.Dict[str, typing.Any] = {
            "environment": environment,
        }

    @builtins.property
    def environment(self) -> "IEnvironment":
        '''(Optional) Specify here the environment in which to execute the scripts.

        Use the DockerEnvironment to execute the commands inside the docker.
        You can alternatively use the LoggingEnvironment which simply logs the commands
        or MockEnvironment which saves them for validation.

        :default: LoggingEnvironment
        '''
        result = self._values.get("environment")
        assert result is not None, "Required property 'environment' is missing"
        return typing.cast("IEnvironment", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EnvironmentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteAutomationOutputs",
    jsii_struct_bases=[],
    name_mapping={
        "execution_id": "executionId",
        "status": "status",
        "output": "output",
    },
)
class ExecuteAutomationOutputs:
    def __init__(
        self,
        *,
        execution_id: builtins.str,
        status: builtins.str,
        output: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Outputs for aws:executeAutomation.

        :param execution_id: The ID of the secondary automation.
        :param status: The status of the secondary automation.
        :param output: The output generated by the secondary automation.
        '''
        if __debug__:
            def stub(
                *,
                execution_id: builtins.str,
                status: builtins.str,
                output: typing.Optional[typing.Sequence[builtins.str]] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument execution_id", value=execution_id, expected_type=type_hints["execution_id"])
            check_type(argname="argument status", value=status, expected_type=type_hints["status"])
            check_type(argname="argument output", value=output, expected_type=type_hints["output"])
        self._values: typing.Dict[str, typing.Any] = {
            "execution_id": execution_id,
            "status": status,
        }
        if output is not None:
            self._values["output"] = output

    @builtins.property
    def execution_id(self) -> builtins.str:
        '''The ID of the secondary automation.'''
        result = self._values.get("execution_id")
        assert result is not None, "Required property 'execution_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def status(self) -> builtins.str:
        '''The status of the secondary automation.'''
        result = self._values.get("status")
        assert result is not None, "Required property 'status' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def output(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The output generated by the secondary automation.'''
        result = self._values.get("output")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExecuteAutomationOutputs(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteAutomationProps",
    jsii_struct_bases=[],
    name_mapping={
        "document_name": "documentName",
        "document_version": "documentVersion",
        "max_concurrency": "maxConcurrency",
        "max_errors": "maxErrors",
        "runtime_parameters": "runtimeParameters",
        "tags": "tags",
        "target_locations": "targetLocations",
        "target_maps": "targetMaps",
        "target_parameter_name": "targetParameterName",
        "targets": "targets",
    },
)
class ExecuteAutomationProps:
    def __init__(
        self,
        *,
        document_name: builtins.str,
        document_version: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[builtins.str] = None,
        max_errors: typing.Optional[builtins.str] = None,
        runtime_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        tags: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_locations: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_maps: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_parameter_name: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Inputs for aws:executeAutomation.

        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        '''
        if __debug__:
            def stub(
                *,
                document_name: builtins.str,
                document_version: typing.Optional[builtins.str] = None,
                max_concurrency: typing.Optional[builtins.str] = None,
                max_errors: typing.Optional[builtins.str] = None,
                runtime_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
                tags: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
                target_locations: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
                target_maps: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
                target_parameter_name: typing.Optional[builtins.str] = None,
                targets: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument document_name", value=document_name, expected_type=type_hints["document_name"])
            check_type(argname="argument document_version", value=document_version, expected_type=type_hints["document_version"])
            check_type(argname="argument max_concurrency", value=max_concurrency, expected_type=type_hints["max_concurrency"])
            check_type(argname="argument max_errors", value=max_errors, expected_type=type_hints["max_errors"])
            check_type(argname="argument runtime_parameters", value=runtime_parameters, expected_type=type_hints["runtime_parameters"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument target_locations", value=target_locations, expected_type=type_hints["target_locations"])
            check_type(argname="argument target_maps", value=target_maps, expected_type=type_hints["target_maps"])
            check_type(argname="argument target_parameter_name", value=target_parameter_name, expected_type=type_hints["target_parameter_name"])
            check_type(argname="argument targets", value=targets, expected_type=type_hints["targets"])
        self._values: typing.Dict[str, typing.Any] = {
            "document_name": document_name,
        }
        if document_version is not None:
            self._values["document_version"] = document_version
        if max_concurrency is not None:
            self._values["max_concurrency"] = max_concurrency
        if max_errors is not None:
            self._values["max_errors"] = max_errors
        if runtime_parameters is not None:
            self._values["runtime_parameters"] = runtime_parameters
        if tags is not None:
            self._values["tags"] = tags
        if target_locations is not None:
            self._values["target_locations"] = target_locations
        if target_maps is not None:
            self._values["target_maps"] = target_maps
        if target_parameter_name is not None:
            self._values["target_parameter_name"] = target_parameter_name
        if targets is not None:
            self._values["targets"] = targets

    @builtins.property
    def document_name(self) -> builtins.str:
        result = self._values.get("document_name")
        assert result is not None, "Required property 'document_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def document_version(self) -> typing.Optional[builtins.str]:
        result = self._values.get("document_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_concurrency(self) -> typing.Optional[builtins.str]:
        result = self._values.get("max_concurrency")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_errors(self) -> typing.Optional[builtins.str]:
        result = self._values.get("max_errors")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def runtime_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        result = self._values.get("runtime_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def tags(
        self,
    ) -> typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]]:
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]], result)

    @builtins.property
    def target_locations(
        self,
    ) -> typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]]:
        result = self._values.get("target_locations")
        return typing.cast(typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]], result)

    @builtins.property
    def target_maps(
        self,
    ) -> typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]]:
        result = self._values.get("target_maps")
        return typing.cast(typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]], result)

    @builtins.property
    def target_parameter_name(self) -> typing.Optional[builtins.str]:
        result = self._values.get("target_parameter_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def targets(
        self,
    ) -> typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]]:
        result = self._values.get("targets")
        return typing.cast(typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExecuteAutomationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ExecuteScriptSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteScriptSimulation",
):
    '''AutomationStep implementation for aws:executeScript https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html.'''

    def __init__(self, step: "ExecuteScriptStep") -> None:
        '''
        :param step: -
        '''
        if __debug__:
            def stub(step: "ExecuteScriptStep") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        jsii.create(self.__class__, self, [step])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Runs the simulation.

        Nests returned object into a "Payload" key to mimic SSM behavior.
        Switch by language and execute code based on specified language.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="executeScriptStep")
    def execute_script_step(self) -> "ExecuteScriptStep":
        return typing.cast("ExecuteScriptStep", jsii.get(self, "executeScriptStep"))


class ExecuteStateMachineSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteStateMachineSimulation",
):
    '''AutomationStep implementation of `aws:executeStateMachine <https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeStateMachine.html>`_.'''

    def __init__(
        self,
        step: "ExecuteStateMachineStep",
        *,
        aws_invoker: "IAwsInvoker",
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(
                step: "ExecuteStateMachineStep",
                *,
                aws_invoker: "IAwsInvoker",
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="executeStateMachineStep")
    def execute_state_machine_step(self) -> "ExecuteStateMachineStep":
        return typing.cast("ExecuteStateMachineStep", jsii.get(self, "executeStateMachineStep"))

    @builtins.property
    @jsii.member(jsii_name="props")
    def props(self) -> AwsInvocationSimulationProps:
        return typing.cast(AwsInvocationSimulationProps, jsii.get(self, "props"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.GitContentProps",
    jsii_struct_bases=[],
    name_mapping={
        "repository": "repository",
        "branch": "branch",
        "commit_id": "commitId",
        "password": "password",
        "private_ssh_key": "privateSshKey",
        "skip_host_key_checking": "skipHostKeyChecking",
        "user_name": "userName",
    },
)
class GitContentProps:
    def __init__(
        self,
        *,
        repository: "IStringVariable",
        branch: typing.Optional[builtins.str] = None,
        commit_id: typing.Optional[builtins.str] = None,
        password: typing.Optional["SecureVariable"] = None,
        private_ssh_key: typing.Optional["SecureVariable"] = None,
        skip_host_key_checking: typing.Optional["IBooleanVariable"] = None,
        user_name: typing.Optional["SecureVariable"] = None,
    ) -> None:
        '''Properties for sourceType Git.

        :param repository: The Git repository URL to the file or directory you want to download.
        :param branch: The default is master. branch parameter is required only if your SSM document is stored in a branch other than master. Supply either commitId or branch (or neither).
        :param commit_id: The default is head. To use the version of your SSM document in a commit other than the latest, specify the full commit ID. For example: "bbc1ddb94...b76d3bEXAMPLE". Supply either commitId or branch (or neither).
        :param password: The password to use when connecting to the repository you specify using HTTP.
        :param private_ssh_key: The SSH key to use when connecting to the repository you specify.
        :param skip_host_key_checking: Determines the value of the StrictHostKeyChecking option when connecting to the repository you specify. The default value is false.
        :param user_name: The username to use when connecting to the repository you specify using HTTP.
        '''
        if __debug__:
            def stub(
                *,
                repository: "IStringVariable",
                branch: typing.Optional[builtins.str] = None,
                commit_id: typing.Optional[builtins.str] = None,
                password: typing.Optional["SecureVariable"] = None,
                private_ssh_key: typing.Optional["SecureVariable"] = None,
                skip_host_key_checking: typing.Optional["IBooleanVariable"] = None,
                user_name: typing.Optional["SecureVariable"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
            check_type(argname="argument branch", value=branch, expected_type=type_hints["branch"])
            check_type(argname="argument commit_id", value=commit_id, expected_type=type_hints["commit_id"])
            check_type(argname="argument password", value=password, expected_type=type_hints["password"])
            check_type(argname="argument private_ssh_key", value=private_ssh_key, expected_type=type_hints["private_ssh_key"])
            check_type(argname="argument skip_host_key_checking", value=skip_host_key_checking, expected_type=type_hints["skip_host_key_checking"])
            check_type(argname="argument user_name", value=user_name, expected_type=type_hints["user_name"])
        self._values: typing.Dict[str, typing.Any] = {
            "repository": repository,
        }
        if branch is not None:
            self._values["branch"] = branch
        if commit_id is not None:
            self._values["commit_id"] = commit_id
        if password is not None:
            self._values["password"] = password
        if private_ssh_key is not None:
            self._values["private_ssh_key"] = private_ssh_key
        if skip_host_key_checking is not None:
            self._values["skip_host_key_checking"] = skip_host_key_checking
        if user_name is not None:
            self._values["user_name"] = user_name

    @builtins.property
    def repository(self) -> "IStringVariable":
        '''The Git repository URL to the file or directory you want to download.'''
        result = self._values.get("repository")
        assert result is not None, "Required property 'repository' is missing"
        return typing.cast("IStringVariable", result)

    @builtins.property
    def branch(self) -> typing.Optional[builtins.str]:
        '''The default is master.

        branch parameter is required only if your SSM document is stored in a branch other than master.
        Supply either commitId or branch (or neither).

        Example::

            main
        '''
        result = self._values.get("branch")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def commit_id(self) -> typing.Optional[builtins.str]:
        '''The default is head.

        To use the version of your SSM document in a commit other than the latest, specify the full commit ID.
        For example: "bbc1ddb94...b76d3bEXAMPLE".
        Supply either commitId or branch (or neither).
        '''
        result = self._values.get("commit_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def password(self) -> typing.Optional["SecureVariable"]:
        '''The password to use when connecting to the repository you specify using HTTP.'''
        result = self._values.get("password")
        return typing.cast(typing.Optional["SecureVariable"], result)

    @builtins.property
    def private_ssh_key(self) -> typing.Optional["SecureVariable"]:
        '''The SSH key to use when connecting to the repository you specify.'''
        result = self._values.get("private_ssh_key")
        return typing.cast(typing.Optional["SecureVariable"], result)

    @builtins.property
    def skip_host_key_checking(self) -> typing.Optional["IBooleanVariable"]:
        '''Determines the value of the StrictHostKeyChecking option when connecting to the repository you specify.

        The default value is false.
        '''
        result = self._values.get("skip_host_key_checking")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    @builtins.property
    def user_name(self) -> typing.Optional["SecureVariable"]:
        '''The username to use when connecting to the repository you specify using HTTP.'''
        result = self._values.get("user_name")
        return typing.cast(typing.Optional["SecureVariable"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GitContentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.GitHubContentProps",
    jsii_struct_bases=[],
    name_mapping={
        "owner": "owner",
        "path": "path",
        "repository": "repository",
        "token_info": "tokenInfo",
        "branch": "branch",
        "commit_id": "commitId",
    },
)
class GitHubContentProps:
    def __init__(
        self,
        *,
        owner: "IStringVariable",
        path: "IStringVariable",
        repository: "IStringVariable",
        token_info: "SecureVariable",
        branch: typing.Optional[builtins.str] = None,
        commit_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for sourceType GitHub.

        :param owner: The repository owner.
        :param path: The path to the file or directory you want to download.
        :param repository: The name of the repository.
        :param token_info: The Systems Manager parameter (a SecureString parameter) where you store your GitHub access token information.
        :param branch: The default is master. branch parameter is required only if your SSM document is stored in a branch other than master. Supply either commitId or branch (or neither).
        :param commit_id: The default is head. To use the version of your SSM document in a commit other than the latest, specify the full commit ID. For example: "bbc1ddb94...b76d3bEXAMPLE". Supply either commitId or branch (or neither).
        '''
        if __debug__:
            def stub(
                *,
                owner: "IStringVariable",
                path: "IStringVariable",
                repository: "IStringVariable",
                token_info: "SecureVariable",
                branch: typing.Optional[builtins.str] = None,
                commit_id: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument owner", value=owner, expected_type=type_hints["owner"])
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
            check_type(argname="argument token_info", value=token_info, expected_type=type_hints["token_info"])
            check_type(argname="argument branch", value=branch, expected_type=type_hints["branch"])
            check_type(argname="argument commit_id", value=commit_id, expected_type=type_hints["commit_id"])
        self._values: typing.Dict[str, typing.Any] = {
            "owner": owner,
            "path": path,
            "repository": repository,
            "token_info": token_info,
        }
        if branch is not None:
            self._values["branch"] = branch
        if commit_id is not None:
            self._values["commit_id"] = commit_id

    @builtins.property
    def owner(self) -> "IStringVariable":
        '''The repository owner.'''
        result = self._values.get("owner")
        assert result is not None, "Required property 'owner' is missing"
        return typing.cast("IStringVariable", result)

    @builtins.property
    def path(self) -> "IStringVariable":
        '''The path to the file or directory you want to download.'''
        result = self._values.get("path")
        assert result is not None, "Required property 'path' is missing"
        return typing.cast("IStringVariable", result)

    @builtins.property
    def repository(self) -> "IStringVariable":
        '''The name of the repository.'''
        result = self._values.get("repository")
        assert result is not None, "Required property 'repository' is missing"
        return typing.cast("IStringVariable", result)

    @builtins.property
    def token_info(self) -> "SecureVariable":
        '''The Systems Manager parameter (a SecureString parameter) where you store your GitHub access token information.'''
        result = self._values.get("token_info")
        assert result is not None, "Required property 'token_info' is missing"
        return typing.cast("SecureVariable", result)

    @builtins.property
    def branch(self) -> typing.Optional[builtins.str]:
        '''The default is master.

        branch parameter is required only if your SSM document is stored in a branch other than master.
        Supply either commitId or branch (or neither).

        Example::

            main
        '''
        result = self._values.get("branch")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def commit_id(self) -> typing.Optional[builtins.str]:
        '''The default is head.

        To use the version of your SSM document in a commit other than the latest, specify the full commit ID.
        For example: "bbc1ddb94...b76d3bEXAMPLE".
        Supply either commitId or branch (or neither).
        '''
        result = self._values.get("commit_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GitHubContentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class HelloWorld(
    aws_cdk.Stack,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HelloWorld",
):
    def __init__(self, app: constructs.Construct, id: builtins.str) -> None:
        '''
        :param app: -
        :param id: -
        '''
        if __debug__:
            def stub(app: constructs.Construct, id: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument app", value=app, expected_type=type_hints["app"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        jsii.create(self.__class__, self, [app, id])


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.HttpContentProps",
    jsii_struct_bases=[],
    name_mapping={
        "url": "url",
        "allow_insecure_download": "allowInsecureDownload",
        "auth_method": "authMethod",
    },
)
class HttpContentProps:
    def __init__(
        self,
        *,
        url: "IStringVariable",
        allow_insecure_download: typing.Optional["IBooleanVariable"] = None,
        auth_method: typing.Optional[AuthMethod] = None,
    ) -> None:
        '''Properties for sourceType HTTP.

        :param url: The URL to the file or directory you want to download.
        :param allow_insecure_download: Determines whether a download can be performed over a connection that isn't encrypted with Secure Socket Layer (SSL) or Transport Layer Security (TLS). The default value is false. We don't recommend performing downloads without encryption. If you choose to do so, you assume all associated risks. Security is a shared responsibility between AWS and you. This is described as the shared responsibility model. To learn more, see the shared responsibility model.
        :param auth_method: Determines whether a username and password are used for authentication when connecting to the url you specify. If you specify Basic or Digest, you must provide values for the username and password parameters. To use the Digest method, SSM Agent version 3.0.1181.0 or later must be installed on your instance. The Digest method supports MD5 and SHA256 encryption.
        '''
        if __debug__:
            def stub(
                *,
                url: "IStringVariable",
                allow_insecure_download: typing.Optional["IBooleanVariable"] = None,
                auth_method: typing.Optional[AuthMethod] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument url", value=url, expected_type=type_hints["url"])
            check_type(argname="argument allow_insecure_download", value=allow_insecure_download, expected_type=type_hints["allow_insecure_download"])
            check_type(argname="argument auth_method", value=auth_method, expected_type=type_hints["auth_method"])
        self._values: typing.Dict[str, typing.Any] = {
            "url": url,
        }
        if allow_insecure_download is not None:
            self._values["allow_insecure_download"] = allow_insecure_download
        if auth_method is not None:
            self._values["auth_method"] = auth_method

    @builtins.property
    def url(self) -> "IStringVariable":
        '''The URL to the file or directory you want to download.'''
        result = self._values.get("url")
        assert result is not None, "Required property 'url' is missing"
        return typing.cast("IStringVariable", result)

    @builtins.property
    def allow_insecure_download(self) -> typing.Optional["IBooleanVariable"]:
        '''Determines whether a download can be performed over a connection that isn't encrypted with Secure Socket Layer (SSL) or Transport Layer Security (TLS).

        The default value is false. We don't recommend performing downloads without encryption. If you choose to do so, you assume all associated risks.
        Security is a shared responsibility between AWS and you. This is described as the shared responsibility model. To learn more, see the shared responsibility model.
        '''
        result = self._values.get("allow_insecure_download")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    @builtins.property
    def auth_method(self) -> typing.Optional[AuthMethod]:
        '''Determines whether a username and password are used for authentication when connecting to the url you specify.

        If you specify Basic or Digest, you must provide values for the username and password parameters.
        To use the Digest method, SSM Agent version 3.0.1181.0 or later must be installed on your instance. The Digest method supports MD5 and SHA256 encryption.
        '''
        result = self._values.get("auth_method")
        return typing.cast(typing.Optional[AuthMethod], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpContentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IApproveHook")
class IApproveHook(typing_extensions.Protocol):
    '''This can be used to provide a hook for approval implementation for ApproveStep.'''

    @jsii.member(jsii_name="ask")
    def ask(self, approver: builtins.str) -> builtins.bool:
        '''Ask for approval.

        :param approver: -
        '''
        ...


class _IApproveHookProxy:
    '''This can be used to provide a hook for approval implementation for ApproveStep.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IApproveHook"

    @jsii.member(jsii_name="ask")
    def ask(self, approver: builtins.str) -> builtins.bool:
        '''Ask for approval.

        :param approver: -
        '''
        if __debug__:
            def stub(approver: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument approver", value=approver, expected_type=type_hints["approver"])
        return typing.cast(builtins.bool, jsii.invoke(self, "ask", [approver]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IApproveHook).__jsii_proxy_class__ = lambda : _IApproveHookProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IAutomationComponent")
class IAutomationComponent(typing_extensions.Protocol):
    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        ...


class _IAutomationComponentProxy:
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IAutomationComponent"

    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        if __debug__:
            def stub(doc: AutomationDocumentBuilder) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument doc", value=doc, expected_type=type_hints["doc"])
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAutomationComponent).__jsii_proxy_class__ = lambda : _IAutomationComponentProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IAwsInvoker")
class IAwsInvoker(typing_extensions.Protocol):
    '''Aws invoker interface.

    This can be used to provide a hook for AWS invocation of steps so as to mock AWS behavior for simulation runs.
    '''

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        aws_api: builtins.str,
        aws_params: typing.Mapping[builtins.str, typing.Any],
        service: AwsService,
    ) -> typing.Any:
        '''Invoke AWS with the provided invocation request.

        :param aws_api: (Required) AWS api to invoke; should be referenced using lowerCamelCase.
        :param aws_params: (Required )AWS params.
        :param service: (Required) AWS service to invoke.

        :return: the AWS response object

        :see: Invocation
        '''
        ...


class _IAwsInvokerProxy:
    '''Aws invoker interface.

    This can be used to provide a hook for AWS invocation of steps so as to mock AWS behavior for simulation runs.
    '''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IAwsInvoker"

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        aws_api: builtins.str,
        aws_params: typing.Mapping[builtins.str, typing.Any],
        service: AwsService,
    ) -> typing.Any:
        '''Invoke AWS with the provided invocation request.

        :param aws_api: (Required) AWS api to invoke; should be referenced using lowerCamelCase.
        :param aws_params: (Required )AWS params.
        :param service: (Required) AWS service to invoke.

        :return: the AWS response object

        :see: Invocation
        '''
        invocation = Invocation(
            aws_api=aws_api, aws_params=aws_params, service=service
        )

        return typing.cast(typing.Any, jsii.invoke(self, "invoke", [invocation]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAwsInvoker).__jsii_proxy_class__ = lambda : _IAwsInvokerProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.ICommandComponent")
class ICommandComponent(typing_extensions.Protocol):
    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: CommandDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        ...


class _ICommandComponentProxy:
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.ICommandComponent"

    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: CommandDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        if __debug__:
            def stub(doc: CommandDocumentBuilder) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument doc", value=doc, expected_type=type_hints["doc"])
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ICommandComponent).__jsii_proxy_class__ = lambda : _ICommandComponentProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IDownloadableContent")
class IDownloadableContent(typing_extensions.Protocol):
    @builtins.property
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        ...

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        ...

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        ...


class _IDownloadableContentProxy:
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IDownloadableContent"

    @builtins.property
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IDownloadableContent).__jsii_proxy_class__ = lambda : _IDownloadableContentProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IEnvironment")
class IEnvironment(typing_extensions.Protocol):
    '''IEnvironment is an abstraction for EC2 instances.

    The implementations of this interface provides customers with
    alternatives to test their commands rather than a real instance.
    '''

    @jsii.member(jsii_name="run")
    def run(self, command: builtins.str) -> builtins.str:
        '''
        :param command: -
        '''
        ...


class _IEnvironmentProxy:
    '''IEnvironment is an abstraction for EC2 instances.

    The implementations of this interface provides customers with
    alternatives to test their commands rather than a real instance.
    '''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IEnvironment"

    @jsii.member(jsii_name="run")
    def run(self, command: builtins.str) -> builtins.str:
        '''
        :param command: -
        '''
        if __debug__:
            def stub(command: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
        return typing.cast(builtins.str, jsii.invoke(self, "run", [command]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEnvironment).__jsii_proxy_class__ = lambda : _IEnvironmentProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IExecuteAutomationHook")
class IExecuteAutomationHook(typing_extensions.Protocol):
    '''Interface for simulating aws:executeAutomation.'''

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        document_version: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[builtins.str] = None,
        max_errors: typing.Optional[builtins.str] = None,
        runtime_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        tags: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_locations: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_maps: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_parameter_name: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
    ) -> ExecuteAutomationOutputs:
        '''Simulate the aws:executeAutomation.

        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        '''
        ...


class _IExecuteAutomationHookProxy:
    '''Interface for simulating aws:executeAutomation.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IExecuteAutomationHook"

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        document_version: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[builtins.str] = None,
        max_errors: typing.Optional[builtins.str] = None,
        runtime_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        tags: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_locations: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_maps: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_parameter_name: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
    ) -> ExecuteAutomationOutputs:
        '''Simulate the aws:executeAutomation.

        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        '''
        props = ExecuteAutomationProps(
            document_name=document_name,
            document_version=document_version,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            runtime_parameters=runtime_parameters,
            tags=tags,
            target_locations=target_locations,
            target_maps=target_maps,
            target_parameter_name=target_parameter_name,
            targets=targets,
        )

        return typing.cast(ExecuteAutomationOutputs, jsii.invoke(self, "execute", [props]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IExecuteAutomationHook).__jsii_proxy_class__ = lambda : _IExecuteAutomationHookProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IGenericVariable")
class IGenericVariable(typing_extensions.Protocol):
    '''Variables hold references to a value.

    There are two implementations: HardCodedValue and ExportedVariable.
    This interface allows you to reference variable outputs from previous steps via the exported step.variable().
    If you do not have a reference to a variable, you may generate your own variable using the ExportedVariable ctor.
    In the event that a hardcoded value is passed into a step, you can reference the value with a HardCodedValue.
    '''

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{ Variable }} or the value.

        Example::

            {{ MyVariable }}
        '''
        ...

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        ...

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        ...


class _IGenericVariableProxy:
    '''Variables hold references to a value.

    There are two implementations: HardCodedValue and ExportedVariable.
    This interface allows you to reference variable outputs from previous steps via the exported step.variable().
    If you do not have a reference to a variable, you may generate your own variable using the ExportedVariable ctor.
    In the event that a hardcoded value is passed into a step, you can reference the value with a HardCodedValue.
    '''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IGenericVariable"

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{ Variable }} or the value.

        Example::

            {{ MyVariable }}
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IGenericVariable).__jsii_proxy_class__ = lambda : _IGenericVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IMapListVariable")
class IMapListVariable(IGenericVariable, typing_extensions.Protocol):
    '''A map list variable.'''

    @jsii.member(jsii_name="resolveToMapList")
    def resolve_to_map_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[typing.Mapping[builtins.str, typing.Any]]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _IMapListVariableProxy(
    jsii.proxy_for(IGenericVariable), # type: ignore[misc]
):
    '''A map list variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IMapListVariable"

    @jsii.member(jsii_name="resolveToMapList")
    def resolve_to_map_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[typing.Mapping[builtins.str, typing.Any]]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.List[typing.Mapping[builtins.str, typing.Any]], jsii.invoke(self, "resolveToMapList", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IMapListVariable).__jsii_proxy_class__ = lambda : _IMapListVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.INumberVariable")
class INumberVariable(IGenericVariable, typing_extensions.Protocol):
    '''A number variable.'''

    @jsii.member(jsii_name="resolveToNumber")
    def resolve_to_number(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> jsii.Number:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _INumberVariableProxy(
    jsii.proxy_for(IGenericVariable), # type: ignore[misc]
):
    '''A number variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.INumberVariable"

    @jsii.member(jsii_name="resolveToNumber")
    def resolve_to_number(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> jsii.Number:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(jsii.Number, jsii.invoke(self, "resolveToNumber", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, INumberVariable).__jsii_proxy_class__ = lambda : _INumberVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IObserver")
class IObserver(typing_extensions.Protocol):
    @jsii.member(jsii_name="accept")
    def accept(self, value: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param value: -
        '''
        ...


class _IObserverProxy:
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IObserver"

    @jsii.member(jsii_name="accept")
    def accept(self, value: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param value: -
        '''
        if __debug__:
            def stub(value: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast(None, jsii.invoke(self, "accept", [value]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IObserver).__jsii_proxy_class__ = lambda : _IObserverProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IParameterResolver")
class IParameterResolver(typing_extensions.Protocol):
    '''Resolver for secure strings in Parameters.'''

    @jsii.member(jsii_name="resolve")
    def resolve(self, input: builtins.str) -> builtins.str:
        '''Resolve the token to its value.

        :param input: -
        '''
        ...


class _IParameterResolverProxy:
    '''Resolver for secure strings in Parameters.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IParameterResolver"

    @jsii.member(jsii_name="resolve")
    def resolve(self, input: builtins.str) -> builtins.str:
        '''Resolve the token to its value.

        :param input: -
        '''
        if __debug__:
            def stub(input: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument input", value=input, expected_type=type_hints["input"])
        return typing.cast(builtins.str, jsii.invoke(self, "resolve", [input]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IParameterResolver).__jsii_proxy_class__ = lambda : _IParameterResolverProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IPauseHook")
class IPauseHook(typing_extensions.Protocol):
    '''This can be used to provide a hook for pausing implementation for PauseStep.'''

    @jsii.member(jsii_name="pause")
    def pause(self) -> None:
        ...


class _IPauseHookProxy:
    '''This can be used to provide a hook for pausing implementation for PauseStep.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IPauseHook"

    @jsii.member(jsii_name="pause")
    def pause(self) -> None:
        return typing.cast(None, jsii.invoke(self, "pause", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IPauseHook).__jsii_proxy_class__ = lambda : _IPauseHookProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IRunCommandHook")
class IRunCommandHook(typing_extensions.Protocol):
    '''Interface for simulating aws:runCommand.'''

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        targets: typing.Sequence[builtins.str],
        cloud_watch_output_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        document_hash: typing.Optional[builtins.str] = None,
        document_hash_type: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_errors: typing.Optional[jsii.Number] = None,
        notification_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        output_s3_bucket_name: typing.Optional[builtins.str] = None,
        output_s3_key_prefix: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        service_role_arn: typing.Optional[builtins.str] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
    ) -> "RunCommandOutputs":
        '''Simulate the aws:runCommand.

        :param document_name: If the Command type document is owned by you or AWS, specify the name of the document. If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        :param targets: The instance IDs where you want the command to run. You can specify a maximum of 50 IDs. You can also use the pseudo parameter {{ RESOURCE_ID }} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_. Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        :param cloud_watch_output_config: (Optional) Configuration options for sending command output to Amazon CloudWatch Logs. For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        :param comment: (Optional) User-defined information about the command.
        :param document_hash: (Optional) The hash for the document.
        :param document_hash_type: (Optional) The type of the hash.
        :param max_concurrency: (Optional) The maximum concurrency.
        :param max_errors: (Optional) The maximum errors.
        :param notification_config: (Optional) The configurations for sending notifications.
        :param output_s3_bucket_name: (Optional) The name of the S3 bucket for command output responses.
        :param output_s3_key_prefix: (Optional) The prefix.
        :param parameters: (Optional) The required and optional parameters specified in the document.
        :param service_role_arn: (Optional) The ARN of the AWS Identity and Access Management (IAM) role.
        :param timeout_seconds: (Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance. If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        ...


class _IRunCommandHookProxy:
    '''Interface for simulating aws:runCommand.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IRunCommandHook"

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        targets: typing.Sequence[builtins.str],
        cloud_watch_output_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        document_hash: typing.Optional[builtins.str] = None,
        document_hash_type: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_errors: typing.Optional[jsii.Number] = None,
        notification_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        output_s3_bucket_name: typing.Optional[builtins.str] = None,
        output_s3_key_prefix: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        service_role_arn: typing.Optional[builtins.str] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
    ) -> "RunCommandOutputs":
        '''Simulate the aws:runCommand.

        :param document_name: If the Command type document is owned by you or AWS, specify the name of the document. If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        :param targets: The instance IDs where you want the command to run. You can specify a maximum of 50 IDs. You can also use the pseudo parameter {{ RESOURCE_ID }} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_. Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        :param cloud_watch_output_config: (Optional) Configuration options for sending command output to Amazon CloudWatch Logs. For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        :param comment: (Optional) User-defined information about the command.
        :param document_hash: (Optional) The hash for the document.
        :param document_hash_type: (Optional) The type of the hash.
        :param max_concurrency: (Optional) The maximum concurrency.
        :param max_errors: (Optional) The maximum errors.
        :param notification_config: (Optional) The configurations for sending notifications.
        :param output_s3_bucket_name: (Optional) The name of the S3 bucket for command output responses.
        :param output_s3_key_prefix: (Optional) The prefix.
        :param parameters: (Optional) The required and optional parameters specified in the document.
        :param service_role_arn: (Optional) The ARN of the AWS Identity and Access Management (IAM) role.
        :param timeout_seconds: (Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance. If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        props = RunCommandProps(
            document_name=document_name,
            targets=targets,
            cloud_watch_output_config=cloud_watch_output_config,
            comment=comment,
            document_hash=document_hash,
            document_hash_type=document_hash_type,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            notification_config=notification_config,
            output_s3_bucket_name=output_s3_bucket_name,
            output_s3_key_prefix=output_s3_key_prefix,
            parameters=parameters,
            service_role_arn=service_role_arn,
            timeout_seconds=timeout_seconds,
        )

        return typing.cast("RunCommandOutputs", jsii.invoke(self, "execute", [props]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IRunCommandHook).__jsii_proxy_class__ = lambda : _IRunCommandHookProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IRunDocumentLocation")
class IRunDocumentLocation(typing_extensions.Protocol):
    @builtins.property
    @jsii.member(jsii_name="location")
    def location(self) -> "IStringVariable":
        ...

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        ...


class _IRunDocumentLocationProxy:
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IRunDocumentLocation"

    @builtins.property
    @jsii.member(jsii_name="location")
    def location(self) -> "IStringVariable":
        return typing.cast("IStringVariable", jsii.get(self, "location"))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "type"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IRunDocumentLocation).__jsii_proxy_class__ = lambda : _IRunDocumentLocationProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.ISleepHook")
class ISleepHook(typing_extensions.Protocol):
    '''This can be used to provide a hook for sleeping for SleepStep (and other places where sleep is required).'''

    @jsii.member(jsii_name="sleep")
    def sleep(self, time_millis: jsii.Number) -> None:
        '''
        :param time_millis: -
        '''
        ...


class _ISleepHookProxy:
    '''This can be used to provide a hook for sleeping for SleepStep (and other places where sleep is required).'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.ISleepHook"

    @jsii.member(jsii_name="sleep")
    def sleep(self, time_millis: jsii.Number) -> None:
        '''
        :param time_millis: -
        '''
        if __debug__:
            def stub(time_millis: jsii.Number) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument time_millis", value=time_millis, expected_type=type_hints["time_millis"])
        return typing.cast(None, jsii.invoke(self, "sleep", [time_millis]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ISleepHook).__jsii_proxy_class__ = lambda : _ISleepHookProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IStringListVariable")
class IStringListVariable(IGenericVariable, typing_extensions.Protocol):
    '''A string list variable.'''

    @jsii.member(jsii_name="resolveToStringList")
    def resolve_to_string_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[builtins.str]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _IStringListVariableProxy(
    jsii.proxy_for(IGenericVariable), # type: ignore[misc]
):
    '''A string list variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IStringListVariable"

    @jsii.member(jsii_name="resolveToStringList")
    def resolve_to_string_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[builtins.str]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "resolveToStringList", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStringListVariable).__jsii_proxy_class__ = lambda : _IStringListVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IStringMapVariable")
class IStringMapVariable(IGenericVariable, typing_extensions.Protocol):
    '''A string map variable.'''

    @jsii.member(jsii_name="resolveToStringMap")
    def resolve_to_string_map(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _IStringMapVariableProxy(
    jsii.proxy_for(IGenericVariable), # type: ignore[misc]
):
    '''A string map variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IStringMapVariable"

    @jsii.member(jsii_name="resolveToStringMap")
    def resolve_to_string_map(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "resolveToStringMap", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStringMapVariable).__jsii_proxy_class__ = lambda : _IStringMapVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IStringVariable")
class IStringVariable(IGenericVariable, typing_extensions.Protocol):
    '''A string variable.'''

    @jsii.member(jsii_name="resolveToString")
    def resolve_to_string(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.str:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _IStringVariableProxy(
    jsii.proxy_for(IGenericVariable), # type: ignore[misc]
):
    '''A string variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IStringVariable"

    @jsii.member(jsii_name="resolveToString")
    def resolve_to_string(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.str:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(builtins.str, jsii.invoke(self, "resolveToString", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStringVariable).__jsii_proxy_class__ = lambda : _IStringVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IWebhook")
class IWebhook(typing_extensions.Protocol):
    '''Hook for simulating aws:invokeWebhook.'''

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        integration_name: builtins.str,
        body: typing.Optional[builtins.str] = None,
    ) -> "InvokeWebhookResult":
        '''Invoke the web hook.

        :param integration_name: The name of the Automation integration. For example, exampleIntegration. The integration you specify must already exist.
        :param body: (Optional) The payload you want to send when your webhook integration is invoked.
        '''
        ...


class _IWebhookProxy:
    '''Hook for simulating aws:invokeWebhook.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IWebhook"

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        integration_name: builtins.str,
        body: typing.Optional[builtins.str] = None,
    ) -> "InvokeWebhookResult":
        '''Invoke the web hook.

        :param integration_name: The name of the Automation integration. For example, exampleIntegration. The integration you specify must already exist.
        :param body: (Optional) The payload you want to send when your webhook integration is invoked.
        '''
        props = InvokeWebhookProps(integration_name=integration_name, body=body)

        return typing.cast("InvokeWebhookResult", jsii.invoke(self, "invoke", [props]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IWebhook).__jsii_proxy_class__ = lambda : _IWebhookProxy


class IncidentResponse(
    constructs.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.IncidentResponse",
):
    '''Provides L2 construct for https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssmincidents-responseplan.html#aws-resource-ssmincidents-responseplan-properties.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        actions: typing.Sequence["IncidentResponseAction"],
        incident_template: "IncidentTemplate",
        chat_channel: typing.Optional[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.ChatChannelProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]] = None,
        display_name: typing.Optional[builtins.str] = None,
        engagements: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param actions: The actions that the response plan starts at the beginning of an incident.
        :param incident_template: Details used to create an incident when using this response plan.
        :param chat_channel: The AWS Chatbot chat channel used for collaboration during an incident.
        :param display_name: The human readable name of the response plan.
        :param engagements: The contacts and escalation plans that the response plan engages during an incident.
        :param tags: An array of key-value pairs to apply to this resource. For more information, see `Tag <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-resource-tags.html>`_ .
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                actions: typing.Sequence["IncidentResponseAction"],
                incident_template: "IncidentTemplate",
                chat_channel: typing.Optional[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.ChatChannelProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]] = None,
                display_name: typing.Optional[builtins.str] = None,
                engagements: typing.Optional[typing.Sequence[builtins.str]] = None,
                tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = IncidentResponseProps(
            actions=actions,
            incident_template=incident_template,
            chat_channel=chat_channel,
            display_name=display_name,
            engagements=engagements,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="cfnResponsePlan")
    def cfn_response_plan(self) -> aws_cdk.aws_ssmincidents.CfnResponsePlan:
        return typing.cast(aws_cdk.aws_ssmincidents.CfnResponsePlan, jsii.get(self, "cfnResponsePlan"))


class IncidentResponseAction(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.IncidentResponseAction",
):
    '''The Action property type specifies the configuration to launch.

    :link: : https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ssmincidents-responseplan-action.html
    '''

    def __init__(
        self,
        *,
        ssm_automation: typing.Optional[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.SsmAutomationProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]] = None,
    ) -> None:
        '''
        :param ssm_automation: Details about the Systems Manager automation document that will be used as a runbook during an incident.
        '''
        cfn_entry = aws_cdk.aws_ssmincidents.CfnResponsePlan.ActionProperty(
            ssm_automation=ssm_automation
        )

        jsii.create(self.__class__, self, [cfn_entry])

    @jsii.member(jsii_name="ssmAutomation")
    @builtins.classmethod
    def ssm_automation(
        cls,
        automation_document: "AutomationDocument",
        role: aws_cdk.aws_iam.Role,
        *,
        parameters: typing.Mapping[builtins.str, IGenericVariable],
        target_account: typing.Optional[builtins.str] = None,
    ) -> "IncidentResponseAction":
        '''Specify the AutomationDocument to use for the action property.

        :param automation_document: -
        :param role: -
        :param parameters: Specify either StringVariables or HardCodedValues.
        :param target_account: The account that the automation document will be run in. This can be in either the management account or an application account.
        '''
        if __debug__:
            def stub(
                automation_document: "AutomationDocument",
                role: aws_cdk.aws_iam.Role,
                *,
                parameters: typing.Mapping[builtins.str, IGenericVariable],
                target_account: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument automation_document", value=automation_document, expected_type=type_hints["automation_document"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
        props = SsmAutomationProps(
            parameters=parameters, target_account=target_account
        )

        return typing.cast("IncidentResponseAction", jsii.sinvoke(cls, "ssmAutomation", [automation_document, role, props]))

    @jsii.member(jsii_name="ssmAutomationEscapeHatch")
    @builtins.classmethod
    def ssm_automation_escape_hatch(
        cls,
        *,
        document_name: builtins.str,
        role_arn: builtins.str,
        document_version: typing.Optional[builtins.str] = None,
        dynamic_parameters: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.DynamicSsmParameterProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        parameters: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.SsmParameterProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        target_account: typing.Optional[builtins.str] = None,
    ) -> "IncidentResponseAction":
        '''
        :param document_name: The automation document's name.
        :param role_arn: The Amazon Resource Name (ARN) of the role that the automation document will assume when running commands.
        :param document_version: The automation document's version to use when running.
        :param dynamic_parameters: ``CfnResponsePlan.SsmAutomationProperty.DynamicParameters``.
        :param parameters: The key-value pair parameters to use when running the automation document.
        :param target_account: The account that the automation document will be run in. This can be in either the management account or an application account.
        '''
        ssm_automation_property = aws_cdk.aws_ssmincidents.CfnResponsePlan.SsmAutomationProperty(
            document_name=document_name,
            role_arn=role_arn,
            document_version=document_version,
            dynamic_parameters=dynamic_parameters,
            parameters=parameters,
            target_account=target_account,
        )

        return typing.cast("IncidentResponseAction", jsii.sinvoke(cls, "ssmAutomationEscapeHatch", [ssm_automation_property]))

    @builtins.property
    @jsii.member(jsii_name="cfnEntry")
    def cfn_entry(self) -> aws_cdk.aws_ssmincidents.CfnResponsePlan.ActionProperty:
        return typing.cast(aws_cdk.aws_ssmincidents.CfnResponsePlan.ActionProperty, jsii.get(self, "cfnEntry"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.IncidentResponseProps",
    jsii_struct_bases=[],
    name_mapping={
        "actions": "actions",
        "incident_template": "incidentTemplate",
        "chat_channel": "chatChannel",
        "display_name": "displayName",
        "engagements": "engagements",
        "tags": "tags",
    },
)
class IncidentResponseProps:
    def __init__(
        self,
        *,
        actions: typing.Sequence[IncidentResponseAction],
        incident_template: "IncidentTemplate",
        chat_channel: typing.Optional[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.ChatChannelProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]] = None,
        display_name: typing.Optional[builtins.str] = None,
        engagements: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
    ) -> None:
        '''Provides props IncidentResponse.

        :param actions: The actions that the response plan starts at the beginning of an incident.
        :param incident_template: Details used to create an incident when using this response plan.
        :param chat_channel: The AWS Chatbot chat channel used for collaboration during an incident.
        :param display_name: The human readable name of the response plan.
        :param engagements: The contacts and escalation plans that the response plan engages during an incident.
        :param tags: An array of key-value pairs to apply to this resource. For more information, see `Tag <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-resource-tags.html>`_ .
        '''
        if __debug__:
            def stub(
                *,
                actions: typing.Sequence[IncidentResponseAction],
                incident_template: "IncidentTemplate",
                chat_channel: typing.Optional[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.ChatChannelProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]] = None,
                display_name: typing.Optional[builtins.str] = None,
                engagements: typing.Optional[typing.Sequence[builtins.str]] = None,
                tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument actions", value=actions, expected_type=type_hints["actions"])
            check_type(argname="argument incident_template", value=incident_template, expected_type=type_hints["incident_template"])
            check_type(argname="argument chat_channel", value=chat_channel, expected_type=type_hints["chat_channel"])
            check_type(argname="argument display_name", value=display_name, expected_type=type_hints["display_name"])
            check_type(argname="argument engagements", value=engagements, expected_type=type_hints["engagements"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[str, typing.Any] = {
            "actions": actions,
            "incident_template": incident_template,
        }
        if chat_channel is not None:
            self._values["chat_channel"] = chat_channel
        if display_name is not None:
            self._values["display_name"] = display_name
        if engagements is not None:
            self._values["engagements"] = engagements
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def actions(self) -> typing.List[IncidentResponseAction]:
        '''The actions that the response plan starts at the beginning of an incident.'''
        result = self._values.get("actions")
        assert result is not None, "Required property 'actions' is missing"
        return typing.cast(typing.List[IncidentResponseAction], result)

    @builtins.property
    def incident_template(self) -> "IncidentTemplate":
        '''Details used to create an incident when using this response plan.'''
        result = self._values.get("incident_template")
        assert result is not None, "Required property 'incident_template' is missing"
        return typing.cast("IncidentTemplate", result)

    @builtins.property
    def chat_channel(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.ChatChannelProperty, aws_cdk.IResolvable]]:
        '''The AWS Chatbot chat channel used for collaboration during an incident.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssmincidents-responseplan.html#cfn-ssmincidents-responseplan-chatchannel
        '''
        result = self._values.get("chat_channel")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.ChatChannelProperty, aws_cdk.IResolvable]], result)

    @builtins.property
    def display_name(self) -> typing.Optional[builtins.str]:
        '''The human readable name of the response plan.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssmincidents-responseplan.html#cfn-ssmincidents-responseplan-displayname
        '''
        result = self._values.get("display_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def engagements(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The contacts and escalation plans that the response plan engages during an incident.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssmincidents-responseplan.html#cfn-ssmincidents-responseplan-engagements
        '''
        result = self._values.get("engagements")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.CfnTag]]:
        '''An array of key-value pairs to apply to this resource.

        For more information, see `Tag <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-resource-tags.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssmincidents-responseplan.html#cfn-ssmincidents-responseplan-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "IncidentResponseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class IncidentTemplate(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.IncidentTemplate",
):
    '''Provides L2 construct for https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ssmincidents-responseplan-incidenttemplate.html.'''

    def __init__(
        self,
        impact: jsii.Number,
        title: builtins.str,
        *,
        dedupe_string: typing.Optional[builtins.str] = None,
        incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
        notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        summary: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param impact: -
        :param title: -
        :param dedupe_string: Used to create only one incident record for an incident.
        :param incident_tags: ``CfnResponsePlan.IncidentTemplateProperty.IncidentTags``.
        :param notification_targets: The SNS targets that AWS Chatbot uses to notify the chat channel of updates to an incident. You can also make updates to the incident through the chat channel using the SNS topics.
        :param summary: The summary describes what has happened during the incident.
        '''
        if __debug__:
            def stub(
                impact: jsii.Number,
                title: builtins.str,
                *,
                dedupe_string: typing.Optional[builtins.str] = None,
                incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
                notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                summary: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument impact", value=impact, expected_type=type_hints["impact"])
            check_type(argname="argument title", value=title, expected_type=type_hints["title"])
        props = IncidentTemplateProps(
            dedupe_string=dedupe_string,
            incident_tags=incident_tags,
            notification_targets=notification_targets,
            summary=summary,
        )

        jsii.create(self.__class__, self, [impact, title, props])

    @jsii.member(jsii_name="critical")
    @builtins.classmethod
    def critical(
        cls,
        title: builtins.str,
        *,
        dedupe_string: typing.Optional[builtins.str] = None,
        incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
        notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        summary: typing.Optional[builtins.str] = None,
    ) -> "IncidentTemplate":
        '''Critical impact typically relates to full application failure that impacts many to all customers.

        :param title: -
        :param dedupe_string: Used to create only one incident record for an incident.
        :param incident_tags: ``CfnResponsePlan.IncidentTemplateProperty.IncidentTags``.
        :param notification_targets: The SNS targets that AWS Chatbot uses to notify the chat channel of updates to an incident. You can also make updates to the incident through the chat channel using the SNS topics.
        :param summary: The summary describes what has happened during the incident.
        '''
        if __debug__:
            def stub(
                title: builtins.str,
                *,
                dedupe_string: typing.Optional[builtins.str] = None,
                incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
                notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                summary: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument title", value=title, expected_type=type_hints["title"])
        props = IncidentTemplateProps(
            dedupe_string=dedupe_string,
            incident_tags=incident_tags,
            notification_targets=notification_targets,
            summary=summary,
        )

        return typing.cast("IncidentTemplate", jsii.sinvoke(cls, "critical", [title, props]))

    @jsii.member(jsii_name="high")
    @builtins.classmethod
    def high(
        cls,
        title: builtins.str,
        *,
        dedupe_string: typing.Optional[builtins.str] = None,
        incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
        notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        summary: typing.Optional[builtins.str] = None,
    ) -> "IncidentTemplate":
        '''High impact denotes partial application failure with impact to many customers.

        :param title: -
        :param dedupe_string: Used to create only one incident record for an incident.
        :param incident_tags: ``CfnResponsePlan.IncidentTemplateProperty.IncidentTags``.
        :param notification_targets: The SNS targets that AWS Chatbot uses to notify the chat channel of updates to an incident. You can also make updates to the incident through the chat channel using the SNS topics.
        :param summary: The summary describes what has happened during the incident.
        '''
        if __debug__:
            def stub(
                title: builtins.str,
                *,
                dedupe_string: typing.Optional[builtins.str] = None,
                incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
                notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                summary: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument title", value=title, expected_type=type_hints["title"])
        props = IncidentTemplateProps(
            dedupe_string=dedupe_string,
            incident_tags=incident_tags,
            notification_targets=notification_targets,
            summary=summary,
        )

        return typing.cast("IncidentTemplate", jsii.sinvoke(cls, "high", [title, props]))

    @jsii.member(jsii_name="low")
    @builtins.classmethod
    def low(
        cls,
        title: builtins.str,
        *,
        dedupe_string: typing.Optional[builtins.str] = None,
        incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
        notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        summary: typing.Optional[builtins.str] = None,
    ) -> "IncidentTemplate":
        '''Low impact denotes that customers may not be impacted by the problem yet.

        :param title: -
        :param dedupe_string: Used to create only one incident record for an incident.
        :param incident_tags: ``CfnResponsePlan.IncidentTemplateProperty.IncidentTags``.
        :param notification_targets: The SNS targets that AWS Chatbot uses to notify the chat channel of updates to an incident. You can also make updates to the incident through the chat channel using the SNS topics.
        :param summary: The summary describes what has happened during the incident.
        '''
        if __debug__:
            def stub(
                title: builtins.str,
                *,
                dedupe_string: typing.Optional[builtins.str] = None,
                incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
                notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                summary: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument title", value=title, expected_type=type_hints["title"])
        props = IncidentTemplateProps(
            dedupe_string=dedupe_string,
            incident_tags=incident_tags,
            notification_targets=notification_targets,
            summary=summary,
        )

        return typing.cast("IncidentTemplate", jsii.sinvoke(cls, "low", [title, props]))

    @jsii.member(jsii_name="medium")
    @builtins.classmethod
    def medium(
        cls,
        title: builtins.str,
        *,
        dedupe_string: typing.Optional[builtins.str] = None,
        incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
        notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        summary: typing.Optional[builtins.str] = None,
    ) -> "IncidentTemplate":
        '''Medium impact denotes that the application is providing reduced service to customers.

        :param title: -
        :param dedupe_string: Used to create only one incident record for an incident.
        :param incident_tags: ``CfnResponsePlan.IncidentTemplateProperty.IncidentTags``.
        :param notification_targets: The SNS targets that AWS Chatbot uses to notify the chat channel of updates to an incident. You can also make updates to the incident through the chat channel using the SNS topics.
        :param summary: The summary describes what has happened during the incident.
        '''
        if __debug__:
            def stub(
                title: builtins.str,
                *,
                dedupe_string: typing.Optional[builtins.str] = None,
                incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
                notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                summary: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument title", value=title, expected_type=type_hints["title"])
        props = IncidentTemplateProps(
            dedupe_string=dedupe_string,
            incident_tags=incident_tags,
            notification_targets=notification_targets,
            summary=summary,
        )

        return typing.cast("IncidentTemplate", jsii.sinvoke(cls, "medium", [title, props]))

    @jsii.member(jsii_name="noImpact")
    @builtins.classmethod
    def no_impact(
        cls,
        title: builtins.str,
        *,
        dedupe_string: typing.Optional[builtins.str] = None,
        incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
        notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        summary: typing.Optional[builtins.str] = None,
    ) -> "IncidentTemplate":
        '''No impact denotes that customers aren't currently impacted but urgent action is needed to avoid impact.

        :param title: -
        :param dedupe_string: Used to create only one incident record for an incident.
        :param incident_tags: ``CfnResponsePlan.IncidentTemplateProperty.IncidentTags``.
        :param notification_targets: The SNS targets that AWS Chatbot uses to notify the chat channel of updates to an incident. You can also make updates to the incident through the chat channel using the SNS topics.
        :param summary: The summary describes what has happened during the incident.
        '''
        if __debug__:
            def stub(
                title: builtins.str,
                *,
                dedupe_string: typing.Optional[builtins.str] = None,
                incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
                notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                summary: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument title", value=title, expected_type=type_hints["title"])
        props = IncidentTemplateProps(
            dedupe_string=dedupe_string,
            incident_tags=incident_tags,
            notification_targets=notification_targets,
            summary=summary,
        )

        return typing.cast("IncidentTemplate", jsii.sinvoke(cls, "noImpact", [title, props]))

    @builtins.property
    @jsii.member(jsii_name="cfnEntry")
    def cfn_entry(
        self,
    ) -> aws_cdk.aws_ssmincidents.CfnResponsePlan.IncidentTemplateProperty:
        return typing.cast(aws_cdk.aws_ssmincidents.CfnResponsePlan.IncidentTemplateProperty, jsii.get(self, "cfnEntry"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.IncidentTemplateProps",
    jsii_struct_bases=[],
    name_mapping={
        "dedupe_string": "dedupeString",
        "incident_tags": "incidentTags",
        "notification_targets": "notificationTargets",
        "summary": "summary",
    },
)
class IncidentTemplateProps:
    def __init__(
        self,
        *,
        dedupe_string: typing.Optional[builtins.str] = None,
        incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
        notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        summary: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Provides L2 construct for https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ssmincidents-responseplan-incidenttemplate.html.

        :param dedupe_string: Used to create only one incident record for an incident.
        :param incident_tags: ``CfnResponsePlan.IncidentTemplateProperty.IncidentTags``.
        :param notification_targets: The SNS targets that AWS Chatbot uses to notify the chat channel of updates to an incident. You can also make updates to the incident through the chat channel using the SNS topics.
        :param summary: The summary describes what has happened during the incident.
        '''
        if __debug__:
            def stub(
                *,
                dedupe_string: typing.Optional[builtins.str] = None,
                incident_tags: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.IResolvable, typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]]]] = None,
                notification_targets: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                summary: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument dedupe_string", value=dedupe_string, expected_type=type_hints["dedupe_string"])
            check_type(argname="argument incident_tags", value=incident_tags, expected_type=type_hints["incident_tags"])
            check_type(argname="argument notification_targets", value=notification_targets, expected_type=type_hints["notification_targets"])
            check_type(argname="argument summary", value=summary, expected_type=type_hints["summary"])
        self._values: typing.Dict[str, typing.Any] = {}
        if dedupe_string is not None:
            self._values["dedupe_string"] = dedupe_string
        if incident_tags is not None:
            self._values["incident_tags"] = incident_tags
        if notification_targets is not None:
            self._values["notification_targets"] = notification_targets
        if summary is not None:
            self._values["summary"] = summary

    @builtins.property
    def dedupe_string(self) -> typing.Optional[builtins.str]:
        '''Used to create only one incident record for an incident.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ssmincidents-responseplan-incidenttemplate.html#cfn-ssmincidents-responseplan-incidenttemplate-dedupestring
        '''
        result = self._values.get("dedupe_string")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def incident_tags(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.IResolvable, aws_cdk.CfnTag]]]]:
        '''``CfnResponsePlan.IncidentTemplateProperty.IncidentTags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ssmincidents-responseplan-incidenttemplate.html#cfn-ssmincidents-responseplan-incidenttemplate-incidenttags
        '''
        result = self._values.get("incident_tags")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.IResolvable, aws_cdk.CfnTag]]]], result)

    @builtins.property
    def notification_targets(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, aws_cdk.IResolvable]]]]:
        '''The SNS targets that AWS Chatbot uses to notify the chat channel of updates to an incident.

        You can also make updates to the incident through the chat channel using the SNS topics.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ssmincidents-responseplan-incidenttemplate.html#cfn-ssmincidents-responseplan-incidenttemplate-notificationtargets
        '''
        result = self._values.get("notification_targets")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssmincidents.CfnResponsePlan.NotificationTargetItemProperty, aws_cdk.IResolvable]]]], result)

    @builtins.property
    def summary(self) -> typing.Optional[builtins.str]:
        '''The summary describes what has happened during the incident.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ssmincidents-responseplan-incidenttemplate.html#cfn-ssmincidents-responseplan-incidenttemplate-summary
        '''
        result = self._values.get("summary")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "IncidentTemplateProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Input(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.Input",
):
    def __init__(
        self,
        *,
        input_type: DataTypeEnum,
        name: builtins.str,
        allowed_pattern: typing.Optional[builtins.str] = None,
        allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
        default_value: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_chars: typing.Optional[jsii.Number] = None,
        max_items: typing.Optional[jsii.Number] = None,
        min_chars: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param input_type: (Required) The DataTypeEnum of the input.
        :param name: (Required) The name of the input by which to be referenced by steps in the document.
        :param allowed_pattern: (Optional) Pattern that this input value must match. Default: undefined
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_chars: (Optional) Maximum number of chars that this input value (string) must contain. Default: undefined
        :param max_items: (Optional) Maximum number of items that this input value (list) must contain. Default: undefined
        :param min_chars: (Optional) Minimum number of chars that this input value (string) must contain. Default: undefined
        :param min_items: (Optional) Minimum number of items that this input value (list) must contain. Default: undefined
        '''
        props = InputProps(
            input_type=input_type,
            name=name,
            allowed_pattern=allowed_pattern,
            allowed_values=allowed_values,
            default_value=default_value,
            description=description,
            max_chars=max_chars,
            max_items=max_items,
            min_chars=min_chars,
            min_items=min_items,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="ofSpecifiedType")
    @builtins.classmethod
    def of_specified_type(
        cls,
        type: DataTypeEnum,
        input_name: builtins.str,
        *,
        input_type: DataTypeEnum,
        name: builtins.str,
        allowed_pattern: typing.Optional[builtins.str] = None,
        allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
        default_value: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_chars: typing.Optional[jsii.Number] = None,
        max_items: typing.Optional[jsii.Number] = None,
        min_chars: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
    ) -> "Input":
        '''
        :param type: -
        :param input_name: -
        :param input_type: (Required) The DataTypeEnum of the input.
        :param name: (Required) The name of the input by which to be referenced by steps in the document.
        :param allowed_pattern: (Optional) Pattern that this input value must match. Default: undefined
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_chars: (Optional) Maximum number of chars that this input value (string) must contain. Default: undefined
        :param max_items: (Optional) Maximum number of items that this input value (list) must contain. Default: undefined
        :param min_chars: (Optional) Minimum number of chars that this input value (string) must contain. Default: undefined
        :param min_items: (Optional) Minimum number of items that this input value (list) must contain. Default: undefined
        '''
        if __debug__:
            def stub(
                type: DataTypeEnum,
                input_name: builtins.str,
                *,
                input_type: DataTypeEnum,
                name: builtins.str,
                allowed_pattern: typing.Optional[builtins.str] = None,
                allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
                default_value: typing.Any = None,
                description: typing.Optional[builtins.str] = None,
                max_chars: typing.Optional[jsii.Number] = None,
                max_items: typing.Optional[jsii.Number] = None,
                min_chars: typing.Optional[jsii.Number] = None,
                min_items: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument input_name", value=input_name, expected_type=type_hints["input_name"])
        props = InputProps(
            input_type=input_type,
            name=name,
            allowed_pattern=allowed_pattern,
            allowed_values=allowed_values,
            default_value=default_value,
            description=description,
            max_chars=max_chars,
            max_items=max_items,
            min_chars=min_chars,
            min_items=min_items,
        )

        return typing.cast("Input", jsii.sinvoke(cls, "ofSpecifiedType", [type, input_name, props]))

    @jsii.member(jsii_name="ofTypeBoolean")
    @builtins.classmethod
    def of_type_boolean(
        cls,
        name: builtins.str,
        *,
        default_value: typing.Optional[builtins.bool] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> "Input":
        '''
        :param name: -
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        '''
        if __debug__:
            def stub(
                name: builtins.str,
                *,
                default_value: typing.Optional[builtins.bool] = None,
                description: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        props = BooleanInputProps(default_value=default_value, description=description)

        return typing.cast("Input", jsii.sinvoke(cls, "ofTypeBoolean", [name, props]))

    @jsii.member(jsii_name="ofTypeInteger")
    @builtins.classmethod
    def of_type_integer(
        cls,
        name: builtins.str,
        *,
        allowed_values: typing.Optional[typing.Sequence[jsii.Number]] = None,
        default_value: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> "Input":
        '''
        :param name: -
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        '''
        if __debug__:
            def stub(
                name: builtins.str,
                *,
                allowed_values: typing.Optional[typing.Sequence[jsii.Number]] = None,
                default_value: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        props = IntegerInputProps(
            allowed_values=allowed_values,
            default_value=default_value,
            description=description,
        )

        return typing.cast("Input", jsii.sinvoke(cls, "ofTypeInteger", [name, props]))

    @jsii.member(jsii_name="ofTypeMapList")
    @builtins.classmethod
    def of_type_map_list(
        cls,
        name: builtins.str,
        *,
        allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
        default_value: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_items: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
    ) -> "Input":
        '''
        :param name: -
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_items: (Optional) Maximum number of items that this input value (list) must contain. Default: undefined
        :param min_items: (Optional) Minimum number of items that this input value (list) must contain. Default: undefined
        '''
        if __debug__:
            def stub(
                name: builtins.str,
                *,
                allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
                default_value: typing.Any = None,
                description: typing.Optional[builtins.str] = None,
                max_items: typing.Optional[jsii.Number] = None,
                min_items: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        props = MapListInputProps(
            allowed_values=allowed_values,
            default_value=default_value,
            description=description,
            max_items=max_items,
            min_items=min_items,
        )

        return typing.cast("Input", jsii.sinvoke(cls, "ofTypeMapList", [name, props]))

    @jsii.member(jsii_name="ofTypeString")
    @builtins.classmethod
    def of_type_string(
        cls,
        name: builtins.str,
        *,
        allowed_pattern: typing.Optional[builtins.str] = None,
        allowed_values: typing.Optional[typing.Sequence[builtins.str]] = None,
        default_value: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        max_chars: typing.Optional[jsii.Number] = None,
        min_chars: typing.Optional[jsii.Number] = None,
    ) -> "Input":
        '''
        :param name: -
        :param allowed_pattern: (Optional) Pattern that this input value must match. Default: undefined
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_chars: (Optional) Maximum number of chars that this input value (string) must contain. Default: undefined
        :param min_chars: (Optional) Minimum number of chars that this input value (string) must contain. Default: undefined
        '''
        if __debug__:
            def stub(
                name: builtins.str,
                *,
                allowed_pattern: typing.Optional[builtins.str] = None,
                allowed_values: typing.Optional[typing.Sequence[builtins.str]] = None,
                default_value: typing.Optional[builtins.str] = None,
                description: typing.Optional[builtins.str] = None,
                max_chars: typing.Optional[jsii.Number] = None,
                min_chars: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        props = StringInputProps(
            allowed_pattern=allowed_pattern,
            allowed_values=allowed_values,
            default_value=default_value,
            description=description,
            max_chars=max_chars,
            min_chars=min_chars,
        )

        return typing.cast("Input", jsii.sinvoke(cls, "ofTypeString", [name, props]))

    @jsii.member(jsii_name="ofTypeStringList")
    @builtins.classmethod
    def of_type_string_list(
        cls,
        name: builtins.str,
        *,
        allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
        default_value: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_items: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
    ) -> "Input":
        '''
        :param name: -
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_items: (Optional) Maximum number of items that this input value (list) must contain. Default: undefined
        :param min_items: (Optional) Minimum number of items that this input value (list) must contain. Default: undefined
        '''
        if __debug__:
            def stub(
                name: builtins.str,
                *,
                allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
                default_value: typing.Any = None,
                description: typing.Optional[builtins.str] = None,
                max_items: typing.Optional[jsii.Number] = None,
                min_items: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        props = StringListInputProps(
            allowed_values=allowed_values,
            default_value=default_value,
            description=description,
            max_items=max_items,
            min_items=min_items,
        )

        return typing.cast("Input", jsii.sinvoke(cls, "ofTypeStringList", [name, props]))

    @jsii.member(jsii_name="ofTypeStringMap")
    @builtins.classmethod
    def of_type_string_map(
        cls,
        name: builtins.str,
        *,
        allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
        default_value: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_items: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
    ) -> "Input":
        '''
        :param name: -
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_items: (Optional) Maximum number of items that this input value (list) must contain. Default: undefined
        :param min_items: (Optional) Minimum number of items that this input value (list) must contain. Default: undefined
        '''
        if __debug__:
            def stub(
                name: builtins.str,
                *,
                allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
                default_value: typing.Any = None,
                description: typing.Optional[builtins.str] = None,
                max_items: typing.Optional[jsii.Number] = None,
                min_items: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        props = StringMapInputProps(
            allowed_values=allowed_values,
            default_value=default_value,
            description=description,
            max_items=max_items,
            min_items=min_items,
        )

        return typing.cast("Input", jsii.sinvoke(cls, "ofTypeStringMap", [name, props]))

    @jsii.member(jsii_name="toSsm")
    def to_ssm(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsm", []))

    @jsii.member(jsii_name="validate")
    def validate(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        if __debug__:
            def stub(value: typing.Any) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast(None, jsii.invoke(self, "validate", [value]))

    @builtins.property
    @jsii.member(jsii_name="inputType")
    def input_type(self) -> DataTypeEnum:
        return typing.cast(DataTypeEnum, jsii.get(self, "inputType"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="allowedPattern")
    def allowed_pattern(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "allowedPattern"))

    @builtins.property
    @jsii.member(jsii_name="allowedValues")
    def allowed_values(self) -> typing.Optional[typing.List[builtins.str]]:
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "allowedValues"))

    @builtins.property
    @jsii.member(jsii_name="defaultValue")
    def default_value(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "defaultValue"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @builtins.property
    @jsii.member(jsii_name="maxChars")
    def max_chars(self) -> typing.Optional[jsii.Number]:
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxChars"))

    @builtins.property
    @jsii.member(jsii_name="maxItems")
    def max_items(self) -> typing.Optional[jsii.Number]:
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxItems"))

    @builtins.property
    @jsii.member(jsii_name="minChars")
    def min_chars(self) -> typing.Optional[jsii.Number]:
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "minChars"))

    @builtins.property
    @jsii.member(jsii_name="minItems")
    def min_items(self) -> typing.Optional[jsii.Number]:
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "minItems"))


class _InputProxy(Input):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Input).__jsii_proxy_class__ = lambda : _InputProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.InputProps",
    jsii_struct_bases=[],
    name_mapping={
        "input_type": "inputType",
        "name": "name",
        "allowed_pattern": "allowedPattern",
        "allowed_values": "allowedValues",
        "default_value": "defaultValue",
        "description": "description",
        "max_chars": "maxChars",
        "max_items": "maxItems",
        "min_chars": "minChars",
        "min_items": "minItems",
    },
)
class InputProps:
    def __init__(
        self,
        *,
        input_type: DataTypeEnum,
        name: builtins.str,
        allowed_pattern: typing.Optional[builtins.str] = None,
        allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
        default_value: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_chars: typing.Optional[jsii.Number] = None,
        max_items: typing.Optional[jsii.Number] = None,
        min_chars: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties of inputs supported by SSM documents.

        These are NOT used for declaring step inputs, rather only for document inputs.
        See https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-doc-syntax.html

        :param input_type: (Required) The DataTypeEnum of the input.
        :param name: (Required) The name of the input by which to be referenced by steps in the document.
        :param allowed_pattern: (Optional) Pattern that this input value must match. Default: undefined
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_chars: (Optional) Maximum number of chars that this input value (string) must contain. Default: undefined
        :param max_items: (Optional) Maximum number of items that this input value (list) must contain. Default: undefined
        :param min_chars: (Optional) Minimum number of chars that this input value (string) must contain. Default: undefined
        :param min_items: (Optional) Minimum number of items that this input value (list) must contain. Default: undefined
        '''
        if __debug__:
            def stub(
                *,
                input_type: DataTypeEnum,
                name: builtins.str,
                allowed_pattern: typing.Optional[builtins.str] = None,
                allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
                default_value: typing.Any = None,
                description: typing.Optional[builtins.str] = None,
                max_chars: typing.Optional[jsii.Number] = None,
                max_items: typing.Optional[jsii.Number] = None,
                min_chars: typing.Optional[jsii.Number] = None,
                min_items: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument input_type", value=input_type, expected_type=type_hints["input_type"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument allowed_pattern", value=allowed_pattern, expected_type=type_hints["allowed_pattern"])
            check_type(argname="argument allowed_values", value=allowed_values, expected_type=type_hints["allowed_values"])
            check_type(argname="argument default_value", value=default_value, expected_type=type_hints["default_value"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument max_chars", value=max_chars, expected_type=type_hints["max_chars"])
            check_type(argname="argument max_items", value=max_items, expected_type=type_hints["max_items"])
            check_type(argname="argument min_chars", value=min_chars, expected_type=type_hints["min_chars"])
            check_type(argname="argument min_items", value=min_items, expected_type=type_hints["min_items"])
        self._values: typing.Dict[str, typing.Any] = {
            "input_type": input_type,
            "name": name,
        }
        if allowed_pattern is not None:
            self._values["allowed_pattern"] = allowed_pattern
        if allowed_values is not None:
            self._values["allowed_values"] = allowed_values
        if default_value is not None:
            self._values["default_value"] = default_value
        if description is not None:
            self._values["description"] = description
        if max_chars is not None:
            self._values["max_chars"] = max_chars
        if max_items is not None:
            self._values["max_items"] = max_items
        if min_chars is not None:
            self._values["min_chars"] = min_chars
        if min_items is not None:
            self._values["min_items"] = min_items

    @builtins.property
    def input_type(self) -> DataTypeEnum:
        '''(Required) The DataTypeEnum of the input.'''
        result = self._values.get("input_type")
        assert result is not None, "Required property 'input_type' is missing"
        return typing.cast(DataTypeEnum, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''(Required) The name of the input by which to be referenced by steps in the document.'''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def allowed_pattern(self) -> typing.Optional[builtins.str]:
        '''(Optional) Pattern that this input value must match.

        :default: undefined
        '''
        result = self._values.get("allowed_pattern")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def allowed_values(self) -> typing.Optional[typing.List[typing.Any]]:
        '''(Optional) List of allowed values that this input may be.

        :default: undefined
        '''
        result = self._values.get("allowed_values")
        return typing.cast(typing.Optional[typing.List[typing.Any]], result)

    @builtins.property
    def default_value(self) -> typing.Any:
        '''(Optional) Default value to use for this input if not specified when invoking the document.

        :default: undefined
        '''
        result = self._values.get("default_value")
        return typing.cast(typing.Any, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) The description of the input.

        :default: name
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_chars(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Maximum number of chars that this input value (string) must contain.

        :default: undefined
        '''
        result = self._values.get("max_chars")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_items(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Maximum number of items that this input value (list) must contain.

        :default: undefined
        '''
        result = self._values.get("max_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_chars(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Minimum number of chars that this input value (string) must contain.

        :default: undefined
        '''
        result = self._values.get("min_chars")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_items(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Minimum number of items that this input value (list) must contain.

        :default: undefined
        '''
        result = self._values.get("min_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InputProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.IntegerInputProps",
    jsii_struct_bases=[],
    name_mapping={
        "allowed_values": "allowedValues",
        "default_value": "defaultValue",
        "description": "description",
    },
)
class IntegerInputProps:
    def __init__(
        self,
        *,
        allowed_values: typing.Optional[typing.Sequence[jsii.Number]] = None,
        default_value: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        '''
        if __debug__:
            def stub(
                *,
                allowed_values: typing.Optional[typing.Sequence[jsii.Number]] = None,
                default_value: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument allowed_values", value=allowed_values, expected_type=type_hints["allowed_values"])
            check_type(argname="argument default_value", value=default_value, expected_type=type_hints["default_value"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[str, typing.Any] = {}
        if allowed_values is not None:
            self._values["allowed_values"] = allowed_values
        if default_value is not None:
            self._values["default_value"] = default_value
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def allowed_values(self) -> typing.Optional[typing.List[jsii.Number]]:
        '''(Optional) List of allowed values that this input may be.

        :default: undefined
        '''
        result = self._values.get("allowed_values")
        return typing.cast(typing.Optional[typing.List[jsii.Number]], result)

    @builtins.property
    def default_value(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Default value to use for this input if not specified when invoking the document.

        :default: undefined
        '''
        result = self._values.get("default_value")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) The description of the input.

        :default: name
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "IntegerInputProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.Invocation",
    jsii_struct_bases=[],
    name_mapping={
        "aws_api": "awsApi",
        "aws_params": "awsParams",
        "service": "service",
    },
)
class Invocation:
    def __init__(
        self,
        *,
        aws_api: builtins.str,
        aws_params: typing.Mapping[builtins.str, typing.Any],
        service: AwsService,
    ) -> None:
        '''
        :param aws_api: (Required) AWS api to invoke; should be referenced using lowerCamelCase.
        :param aws_params: (Required )AWS params.
        :param service: (Required) AWS service to invoke.
        '''
        if __debug__:
            def stub(
                *,
                aws_api: builtins.str,
                aws_params: typing.Mapping[builtins.str, typing.Any],
                service: AwsService,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_api", value=aws_api, expected_type=type_hints["aws_api"])
            check_type(argname="argument aws_params", value=aws_params, expected_type=type_hints["aws_params"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
        self._values: typing.Dict[str, typing.Any] = {
            "aws_api": aws_api,
            "aws_params": aws_params,
            "service": service,
        }

    @builtins.property
    def aws_api(self) -> builtins.str:
        '''(Required) AWS api to invoke;

        should be referenced using lowerCamelCase.

        Example::

            describeInstance
        '''
        result = self._values.get("aws_api")
        assert result is not None, "Required property 'aws_api' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def aws_params(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''(Required )AWS params.

        Example::

            { 'Filters': [{'Name': 'instance-id', 'Values': ['{{ InstanceId }}'] }] }
        '''
        result = self._values.get("aws_params")
        assert result is not None, "Required property 'aws_params' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    @builtins.property
    def service(self) -> AwsService:
        '''(Required) AWS service to invoke.

        Example::

            AwsService.EC2
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(AwsService, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Invocation(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class InvokeLambdaFunctionSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeLambdaFunctionSimulation",
):
    '''AutomationStep implemenation for aws:invokeLambdaFunction https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-lamb.html.'''

    def __init__(
        self,
        step: "InvokeLambdaFunctionStep",
        *,
        aws_invoker: IAwsInvoker,
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(
                step: "InvokeLambdaFunctionStep",
                *,
                aws_invoker: IAwsInvoker,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeWebhookProps",
    jsii_struct_bases=[],
    name_mapping={"integration_name": "integrationName", "body": "body"},
)
class InvokeWebhookProps:
    def __init__(
        self,
        *,
        integration_name: builtins.str,
        body: typing.Optional[builtins.str] = None,
    ) -> None:
        '''The properties for IWebhook.Invoke.

        :param integration_name: The name of the Automation integration. For example, exampleIntegration. The integration you specify must already exist.
        :param body: (Optional) The payload you want to send when your webhook integration is invoked.
        '''
        if __debug__:
            def stub(
                *,
                integration_name: builtins.str,
                body: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument integration_name", value=integration_name, expected_type=type_hints["integration_name"])
            check_type(argname="argument body", value=body, expected_type=type_hints["body"])
        self._values: typing.Dict[str, typing.Any] = {
            "integration_name": integration_name,
        }
        if body is not None:
            self._values["body"] = body

    @builtins.property
    def integration_name(self) -> builtins.str:
        '''The name of the Automation integration.

        For example, exampleIntegration. The integration you specify must already exist.
        '''
        result = self._values.get("integration_name")
        assert result is not None, "Required property 'integration_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def body(self) -> typing.Optional[builtins.str]:
        '''(Optional) The payload you want to send when your webhook integration is invoked.'''
        result = self._values.get("body")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InvokeWebhookProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeWebhookResult",
    jsii_struct_bases=[],
    name_mapping={"response": "response", "response_code": "responseCode"},
)
class InvokeWebhookResult:
    def __init__(self, *, response: builtins.str, response_code: jsii.Number) -> None:
        '''Response from IWebhook.Invoke.

        :param response: The text received from the webhook provider response.
        :param response_code: The HTTP status code received from the webhook provider response.
        '''
        if __debug__:
            def stub(*, response: builtins.str, response_code: jsii.Number) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument response", value=response, expected_type=type_hints["response"])
            check_type(argname="argument response_code", value=response_code, expected_type=type_hints["response_code"])
        self._values: typing.Dict[str, typing.Any] = {
            "response": response,
            "response_code": response_code,
        }

    @builtins.property
    def response(self) -> builtins.str:
        '''The text received from the webhook provider response.'''
        result = self._values.get("response")
        assert result is not None, "Required property 'response' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def response_code(self) -> jsii.Number:
        '''The HTTP status code received from the webhook provider response.'''
        result = self._values.get("response_code")
        assert result is not None, "Required property 'response_code' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InvokeWebhookResult(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class InvokeWebhookSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeWebhookSimulation",
):
    '''AutomationStep implementation for `aws:invokeWebhook <https://docs.aws.amazon.com/systems-manager/latest/userguide/invoke-webhook.html>`_.'''

    def __init__(self, step: "InvokeWebhookStep", *, webhook: IWebhook) -> None:
        '''
        :param step: -
        :param webhook: (Optional) Hook for simulating aws:invokeWebhook. Default: - Returns 204 with an empty response
        '''
        if __debug__:
            def stub(step: "InvokeWebhookStep", *, webhook: IWebhook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = InvokeWebhookSimulationProps(webhook=webhook)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeWebhookSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"webhook": "webhook"},
)
class InvokeWebhookSimulationProps:
    def __init__(self, *, webhook: IWebhook) -> None:
        '''Properties for InvokeWebhookStep.

        :param webhook: (Optional) Hook for simulating aws:invokeWebhook. Default: - Returns 204 with an empty response
        '''
        if __debug__:
            def stub(*, webhook: IWebhook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument webhook", value=webhook, expected_type=type_hints["webhook"])
        self._values: typing.Dict[str, typing.Any] = {
            "webhook": webhook,
        }

    @builtins.property
    def webhook(self) -> IWebhook:
        '''(Optional) Hook for simulating aws:invokeWebhook.

        :default: - Returns 204 with an empty response
        '''
        result = self._values.get("webhook")
        assert result is not None, "Required property 'webhook' is missing"
        return typing.cast(IWebhook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InvokeWebhookSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IRunDocumentLocation)
class LocalRunDocument(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.LocalRunDocument",
):
    def __init__(self, document_path: IStringVariable) -> None:
        '''Specify the path to the document on the local share.

        :param document_path: -
        '''
        if __debug__:
            def stub(document_path: IStringVariable) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument document_path", value=document_path, expected_type=type_hints["document_path"])
        jsii.create(self.__class__, self, [document_path])

    @builtins.property
    @jsii.member(jsii_name="location")
    def location(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "location"))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "type"))


@jsii.implements(IEnvironment)
class LoggingEnvironment(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.LoggingEnvironment",
):
    '''Environment that simply logs the commands that it receives and displays them on the console.'''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="run")
    def run(self, command: builtins.str) -> builtins.str:
        '''
        :param command: -
        '''
        if __debug__:
            def stub(command: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
        return typing.cast(builtins.str, jsii.invoke(self, "run", [command]))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.MapListInputProps",
    jsii_struct_bases=[],
    name_mapping={
        "allowed_values": "allowedValues",
        "default_value": "defaultValue",
        "description": "description",
        "max_items": "maxItems",
        "min_items": "minItems",
    },
)
class MapListInputProps:
    def __init__(
        self,
        *,
        allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
        default_value: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_items: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_items: (Optional) Maximum number of items that this input value (list) must contain. Default: undefined
        :param min_items: (Optional) Minimum number of items that this input value (list) must contain. Default: undefined
        '''
        if __debug__:
            def stub(
                *,
                allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
                default_value: typing.Any = None,
                description: typing.Optional[builtins.str] = None,
                max_items: typing.Optional[jsii.Number] = None,
                min_items: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument allowed_values", value=allowed_values, expected_type=type_hints["allowed_values"])
            check_type(argname="argument default_value", value=default_value, expected_type=type_hints["default_value"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument max_items", value=max_items, expected_type=type_hints["max_items"])
            check_type(argname="argument min_items", value=min_items, expected_type=type_hints["min_items"])
        self._values: typing.Dict[str, typing.Any] = {}
        if allowed_values is not None:
            self._values["allowed_values"] = allowed_values
        if default_value is not None:
            self._values["default_value"] = default_value
        if description is not None:
            self._values["description"] = description
        if max_items is not None:
            self._values["max_items"] = max_items
        if min_items is not None:
            self._values["min_items"] = min_items

    @builtins.property
    def allowed_values(self) -> typing.Optional[typing.List[typing.Any]]:
        '''(Optional) List of allowed values that this input may be.

        :default: undefined
        '''
        result = self._values.get("allowed_values")
        return typing.cast(typing.Optional[typing.List[typing.Any]], result)

    @builtins.property
    def default_value(self) -> typing.Any:
        '''(Optional) Default value to use for this input if not specified when invoking the document.

        :default: undefined
        '''
        result = self._values.get("default_value")
        return typing.cast(typing.Any, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) The description of the input.

        :default: name
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_items(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Maximum number of items that this input value (list) must contain.

        :default: undefined
        '''
        result = self._values.get("max_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_items(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Minimum number of items that this input value (list) must contain.

        :default: undefined
        '''
        result = self._values.get("min_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "MapListInputProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IApproveHook)
class MockApprove(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MockApprove",
):
    '''Mock implementation of IApproveHook.

    Does not simulate an approval request.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="ask")
    def ask(self, _approver: builtins.str) -> builtins.bool:
        '''Ask for approval.

        :param _approver: -
        '''
        if __debug__:
            def stub(_approver: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument _approver", value=_approver, expected_type=type_hints["_approver"])
        return typing.cast(builtins.bool, jsii.invoke(self, "ask", [_approver]))

    @builtins.property
    @jsii.member(jsii_name="timesInvoked")
    def times_invoked(self) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.get(self, "timesInvoked"))

    @times_invoked.setter
    def times_invoked(self, value: jsii.Number) -> None:
        if __debug__:
            def stub(value: jsii.Number) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "timesInvoked", value)


@jsii.implements(IAwsInvoker)
class MockAwsInvoker(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MockAwsInvoker",
):
    '''Mock implementation of IAwsInvoker.

    This class can be reused for testing in exported JSII languages.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        aws_api: builtins.str,
        aws_params: typing.Mapping[builtins.str, typing.Any],
        service: AwsService,
    ) -> typing.Any:
        '''Saves the invocation to be retrieved using getInvocations().

        :param aws_api: (Required) AWS api to invoke; should be referenced using lowerCamelCase.
        :param aws_params: (Required )AWS params.
        :param service: (Required) AWS service to invoke.

        :return: the next result as set by the setReturn function
        '''
        invocation = Invocation(
            aws_api=aws_api, aws_params=aws_params, service=service
        )

        return typing.cast(typing.Any, jsii.invoke(self, "invoke", [invocation]))

    @jsii.member(jsii_name="nextReturn")
    def next_return(self, aws_result: typing.Any) -> "MockAwsInvoker":
        '''Set the next return value.

        This function can be chained to return subsequent return values.
        Values are read in order they were inserted.
        The last value is used as a default if there are no other values retrieved.
        In that way this function behaves the same way as Mockito .thenReturn(val).

        :param aws_result: -
        '''
        if __debug__:
            def stub(aws_result: typing.Any) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_result", value=aws_result, expected_type=type_hints["aws_result"])
        return typing.cast("MockAwsInvoker", jsii.invoke(self, "nextReturn", [aws_result]))

    @jsii.member(jsii_name="whenThen")
    def when_then(
        self,
        when: typing.Union[Invocation, typing.Dict[str, typing.Any]],
        then: typing.Mapping[builtins.str, typing.Any],
    ) -> None:
        '''Allows developers to mock out responses from the AwsInvoker depending on the input that it receives.

        :param when: defines the invocation to match and return the then.
        :param then: is the value that should be returned if the above when Invocation is matched.
        '''
        if __debug__:
            def stub(
                when: typing.Union[Invocation, typing.Dict[str, typing.Any]],
                then: typing.Mapping[builtins.str, typing.Any],
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument when", value=when, expected_type=type_hints["when"])
            check_type(argname="argument then", value=then, expected_type=type_hints["then"])
        return typing.cast(None, jsii.invoke(self, "whenThen", [when, then]))

    @builtins.property
    @jsii.member(jsii_name="previousInvocations")
    def previous_invocations(self) -> typing.List[Invocation]:
        '''All of the invocations that have been submitted to this invoker until present.'''
        return typing.cast(typing.List[Invocation], jsii.get(self, "previousInvocations"))


@jsii.implements(IEnvironment)
class MockEnvironment(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MockEnvironment",
):
    '''Environment that simply saves commands into a previousCommands variable.

    This is useful if you want to unit test the commands that would be sent to a real environment.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="run")
    def run(self, command: builtins.str) -> builtins.str:
        '''
        :param command: -
        '''
        if __debug__:
            def stub(command: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
        return typing.cast(builtins.str, jsii.invoke(self, "run", [command]))

    @builtins.property
    @jsii.member(jsii_name="previousCommands")
    def previous_commands(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "previousCommands"))

    @previous_commands.setter
    def previous_commands(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            def stub(value: typing.List[builtins.str]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "previousCommands", value)


@jsii.implements(IPauseHook)
class MockPause(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MockPause",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="pause")
    def pause(self) -> None:
        return typing.cast(None, jsii.invoke(self, "pause", []))

    @builtins.property
    @jsii.member(jsii_name="timesInvoked")
    def times_invoked(self) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.get(self, "timesInvoked"))

    @times_invoked.setter
    def times_invoked(self, value: jsii.Number) -> None:
        if __debug__:
            def stub(value: jsii.Number) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "timesInvoked", value)


@jsii.implements(ISleepHook)
class MockSleep(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MockSleep",
):
    '''Mock ISleeper implementation.

    Simply logs that it is sleeping and returns immediately.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="sleep")
    def sleep(self, time_millis: jsii.Number) -> None:
        '''
        :param time_millis: -
        '''
        if __debug__:
            def stub(time_millis: jsii.Number) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument time_millis", value=time_millis, expected_type=type_hints["time_millis"])
        return typing.cast(None, jsii.invoke(self, "sleep", [time_millis]))

    @builtins.property
    @jsii.member(jsii_name="sleepMilliInvocations")
    def sleep_milli_invocations(self) -> typing.List[jsii.Number]:
        return typing.cast(typing.List[jsii.Number], jsii.get(self, "sleepMilliInvocations"))


class NameDoc(
    DocumentSource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.NameDoc",
):
    def __init__(
        self,
        name: IStringVariable,
        version: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''
        :param name: -
        :param version: -
        '''
        if __debug__:
            def stub(
                name: IStringVariable,
                version: typing.Optional[IStringVariable] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        jsii.create(self.__class__, self, [name, version])

    @jsii.member(jsii_name="formatRequest")
    def format_request(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatRequest", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "version"))


class NoAuthMethod(
    AuthMethod,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.NoAuthMethod",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="toEntry")
    def to_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toEntry", []))


@jsii.implements(IObserver)
class NoopObserver(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.NoopObserver",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="accept")
    def accept(self, _value: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param _value: -
        '''
        if __debug__:
            def stub(_value: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument _value", value=_value, expected_type=type_hints["_value"])
        return typing.cast(None, jsii.invoke(self, "accept", [_value]))


class OnFailure(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.OnFailure",
):
    '''Steps can specify an action to take onFailure.

    See docs here: https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#failProp
    The supported actions are abort (default), continue, or invoking a specific step.
    This behavior can be adopted by using the static methods available on OnFailure.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="abort")
    @builtins.classmethod
    def abort(cls) -> "OnCancel":
        '''To abort execution if a failure occurs during execution of the current step.

        (This is the default behavior.)
        '''
        return typing.cast("OnCancel", jsii.sinvoke(cls, "abort", []))

    @jsii.member(jsii_name="continue")
    @builtins.classmethod
    def continue_(cls) -> "OnFailure":
        '''To continue execution of the subsequent step if a failure occurs during execution of the current step.'''
        return typing.cast("OnFailure", jsii.sinvoke(cls, "continue", []))

    @jsii.member(jsii_name="invokeStep")
    @builtins.classmethod
    def invoke_step(cls, step: "AutomationStep") -> "OnCancel":
        '''Invoke a specific step.

        Provide the step object to execute for the onFailure action.
        If you don't have a handle to the step object, use the invokeStepByName function.

        :param step: -
        '''
        if __debug__:
            def stub(step: "AutomationStep") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        return typing.cast("OnCancel", jsii.sinvoke(cls, "invokeStep", [step]))

    @jsii.member(jsii_name="invokeStepByName")
    @builtins.classmethod
    def invoke_step_by_name(cls, step_name: builtins.str) -> "OnCancel":
        '''Invoke a specific step by the step name for the OnFailure action.

        :param step_name: -
        '''
        if __debug__:
            def stub(step_name: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step_name", value=step_name, expected_type=type_hints["step_name"])
        return typing.cast("OnCancel", jsii.sinvoke(cls, "invokeStepByName", [step_name]))

    @jsii.member(jsii_name="stepToInvoke")
    @abc.abstractmethod
    def step_to_invoke(self, current_step: "AutomationStep") -> builtins.str:
        '''
        :param current_step: -
        '''
        ...

    @jsii.member(jsii_name="toSsmValue")
    @abc.abstractmethod
    def to_ssm_value(self) -> builtins.str:
        ...


class _OnFailureProxy(OnFailure):
    @jsii.member(jsii_name="stepToInvoke")
    def step_to_invoke(self, current_step: "AutomationStep") -> builtins.str:
        '''
        :param current_step: -
        '''
        if __debug__:
            def stub(current_step: "AutomationStep") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument current_step", value=current_step, expected_type=type_hints["current_step"])
        return typing.cast(builtins.str, jsii.invoke(self, "stepToInvoke", [current_step]))

    @jsii.member(jsii_name="toSsmValue")
    def to_ssm_value(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "toSsmValue", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, OnFailure).__jsii_proxy_class__ = lambda : _OnFailureProxy


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.Operation")
class Operation(enum.Enum):
    '''Operation to use for comparing a Choice's or Preconditions with provided value.

    See https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-branch.html
    or https://docs.aws.amazon.com/systems-manager/latest/userguide/document-schemas-features.html
    '''

    BOOLEAN_EQUALS = "BOOLEAN_EQUALS"
    CONTAINS = "CONTAINS"
    ENDS_WITH = "ENDS_WITH"
    STARTS_WITH = "STARTS_WITH"
    STRING_EQUALS = "STRING_EQUALS"
    EQUALS_IGNORE_CASE = "EQUALS_IGNORE_CASE"
    NUMERIC_EQUALS = "NUMERIC_EQUALS"
    NUMERIC_GREATER = "NUMERIC_GREATER"
    NUMERIC_GREATER_OR_EQUALS = "NUMERIC_GREATER_OR_EQUALS"
    NUMERIC_LESSER = "NUMERIC_LESSER"
    NUMERIC_LESSER_OR_EQUALS = "NUMERIC_LESSER_OR_EQUALS"


class OperationEvaluator(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.OperationEvaluator",
):
    def __init__(self, operation: Operation) -> None:
        '''
        :param operation: -
        '''
        if __debug__:
            def stub(operation: Operation) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument operation", value=operation, expected_type=type_hints["operation"])
        jsii.create(self.__class__, self, [operation])

    @jsii.member(jsii_name="fromOperationName")
    @builtins.classmethod
    def from_operation_name(cls, operation_name: builtins.str) -> Operation:
        '''Converts a string to OperationType.

        :param operation_name: an operation name to return its OperationType.

        :return: an OperationType

        :throws: if operationName is not supported
        '''
        if __debug__:
            def stub(operation_name: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument operation_name", value=operation_name, expected_type=type_hints["operation_name"])
        return typing.cast(Operation, jsii.sinvoke(cls, "fromOperationName", [operation_name]))

    @jsii.member(jsii_name="evaluate")
    def evaluate(self, value1: typing.Any, value2: typing.Any) -> builtins.bool:
        '''Evaluates this operation against the provided inputs.

        :param value1: -
        :param value2: -

        :return: true if the evaluation is true. False otherwise.
        '''
        if __debug__:
            def stub(value1: typing.Any, value2: typing.Any) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value1", value=value1, expected_type=type_hints["value1"])
            check_type(argname="argument value2", value=value2, expected_type=type_hints["value2"])
        return typing.cast(builtins.bool, jsii.invoke(self, "evaluate", [value1, value2]))

    @jsii.member(jsii_name="toOperationName")
    def to_operation_name(self) -> builtins.str:
        '''
        :return: the string representation used by SSM for the operation.
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toOperationName", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="STRING_TO_OPERATION")
    def STRING_TO_OPERATION(cls) -> typing.Mapping[builtins.str, Operation]:
        return typing.cast(typing.Mapping[builtins.str, Operation], jsii.sget(cls, "STRING_TO_OPERATION"))

    @builtins.property
    @jsii.member(jsii_name="operation")
    def operation(self) -> Operation:
        return typing.cast(Operation, jsii.get(self, "operation"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.Output",
    jsii_struct_bases=[],
    name_mapping={"name": "name", "output_type": "outputType", "selector": "selector"},
)
class Output:
    def __init__(
        self,
        *,
        name: builtins.str,
        output_type: DataTypeEnum,
        selector: builtins.str,
    ) -> None:
        '''Object used to specify step output.

        :param name: The name that can be used by subsequent steps to refernce the stored value. Note that Outputs will be PREPENDED with the step name.
        :param output_type: The DataType expected by this output. This will be validated in simulation mode and will also be used when printing to yaml/json.
        :param selector: Json selector for locating the value in the json step response.
        '''
        if __debug__:
            def stub(
                *,
                name: builtins.str,
                output_type: DataTypeEnum,
                selector: builtins.str,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_type", value=output_type, expected_type=type_hints["output_type"])
            check_type(argname="argument selector", value=selector, expected_type=type_hints["selector"])
        self._values: typing.Dict[str, typing.Any] = {
            "name": name,
            "output_type": output_type,
            "selector": selector,
        }

    @builtins.property
    def name(self) -> builtins.str:
        '''The name that can be used by subsequent steps to refernce the stored value.

        Note that Outputs will be PREPENDED with the step name.
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def output_type(self) -> DataTypeEnum:
        '''The DataType expected by this output.

        This will be validated in simulation mode and will also be used when printing to yaml/json.
        '''
        result = self._values.get("output_type")
        assert result is not None, "Required property 'output_type' is missing"
        return typing.cast(DataTypeEnum, result)

    @builtins.property
    def selector(self) -> builtins.str:
        '''Json selector for locating the value in the json step response.'''
        result = self._values.get("selector")
        assert result is not None, "Required property 'selector' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Output(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IPauseHook)
class PauseImpl(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.PauseImpl",
):
    '''This IPauseHook implementation provides a real pause and wait for user input of Enter.

    This implementation does not work well on all exported JSII languages.
    Users can provide their own impl using the IPauseHook interface.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="pause")
    def pause(self) -> None:
        return typing.cast(None, jsii.invoke(self, "pause", []))


class PauseSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.PauseSimulation",
):
    '''AutomationStep implementation for aws:pause https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-pause.html.'''

    def __init__(self, step: "PauseStep", *, pause_hook: IPauseHook) -> None:
        '''
        :param step: -
        :param pause_hook: (Optional) Pause hook to be called to pause the execution. To mock this implementation either inject an instance of IPauseHook or use the provided MockPause class. Default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        '''
        if __debug__:
            def stub(step: "PauseStep", *, pause_hook: IPauseHook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = PauseSimulationProps(pause_hook=pause_hook)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        _inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''May perform a real pause based on the params used during instance creation.

        :param _inputs: -
        '''
        if __debug__:
            def stub(_inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument _inputs", value=_inputs, expected_type=type_hints["_inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [_inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.PauseSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"pause_hook": "pauseHook"},
)
class PauseSimulationProps:
    def __init__(self, *, pause_hook: IPauseHook) -> None:
        '''Properties for PauseStep.

        :param pause_hook: (Optional) Pause hook to be called to pause the execution. To mock this implementation either inject an instance of IPauseHook or use the provided MockPause class. Default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        '''
        if __debug__:
            def stub(*, pause_hook: IPauseHook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument pause_hook", value=pause_hook, expected_type=type_hints["pause_hook"])
        self._values: typing.Dict[str, typing.Any] = {
            "pause_hook": pause_hook,
        }

    @builtins.property
    def pause_hook(self) -> IPauseHook:
        '''(Optional) Pause hook to be called to pause the execution.

        To mock this implementation either inject an instance of IPauseHook or use the provided MockPause class.

        :default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        '''
        result = self._values.get("pause_hook")
        assert result is not None, "Required property 'pause_hook' is missing"
        return typing.cast(IPauseHook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PauseSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.Platform")
class Platform(enum.Enum):
    '''Command steps are not all applicable to all platforms.

    Each command step declares which Platforms it supports.
    That allows customers to validate their CommandDocument against a given platform prior to execution.
    '''

    LINUX = "LINUX"
    WINDOWS = "WINDOWS"
    MAC_OS = "MAC_OS"


class Platforms(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.Platforms",
):
    @jsii.member(jsii_name="toPlatform")
    @builtins.classmethod
    def to_platform(cls, platform_string: builtins.str) -> Platform:
        '''Converts a string to Platform.

        :param platform_string: a platform name to return its Platform type.

        :return: a Platform

        :throws: if platofrmString is not supported
        '''
        if __debug__:
            def stub(platform_string: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument platform_string", value=platform_string, expected_type=type_hints["platform_string"])
        return typing.cast(Platform, jsii.sinvoke(cls, "toPlatform", [platform_string]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="STRING_TO_PLATFORM")
    def STRING_TO_PLATFORM(cls) -> typing.Mapping[builtins.str, Platform]:
        return typing.cast(typing.Mapping[builtins.str, Platform], jsii.sget(cls, "STRING_TO_PLATFORM"))


class Precondition(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.Precondition",
):
    @jsii.member(jsii_name="newPlatformPrecondition")
    @builtins.classmethod
    def new_platform_precondition(cls, platform: Platform) -> "Precondition":
        '''Returns a new Precondition.

        :param platform: The platform the preconditions tests against.

        :return: new Precondition with platformType EnvironmentVariable, operation and a constant
        '''
        if __debug__:
            def stub(platform: Platform) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
        return typing.cast("Precondition", jsii.sinvoke(cls, "newPlatformPrecondition", [platform]))

    @jsii.member(jsii_name="asSsmEntry")
    def as_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :return: an object that can be used to print this choice into yaml/json format.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "asSsmEntry", []))

    @jsii.member(jsii_name="evaluate")
    def evaluate(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.bool:
        '''Evaluates if the precondition is met, by comparing the variable with the constant using the operator.

        :param inputs: -

        :return: true if the evaluation is true. False otherwise.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(builtins.bool, jsii.invoke(self, "evaluate", [inputs]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="INJECTED_PLAYFORM_TYPE_KEY")
    def INJECTED_PLAYFORM_TYPE_KEY(cls) -> builtins.str:
        return typing.cast(builtins.str, jsii.sget(cls, "INJECTED_PLAYFORM_TYPE_KEY"))

    @builtins.property
    @jsii.member(jsii_name="operationType")
    def operation_type(self) -> Operation:
        return typing.cast(Operation, jsii.get(self, "operationType"))

    @builtins.property
    @jsii.member(jsii_name="variable1")
    def variable1(self) -> IGenericVariable:
        return typing.cast(IGenericVariable, jsii.get(self, "variable1"))

    @builtins.property
    @jsii.member(jsii_name="variable2")
    def variable2(self) -> IGenericVariable:
        return typing.cast(IGenericVariable, jsii.get(self, "variable2"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.PreconditionProps",
    jsii_struct_bases=[],
    name_mapping={
        "operation_type": "operationType",
        "variable1": "variable1",
        "variable2": "variable2",
    },
)
class PreconditionProps:
    def __init__(
        self,
        *,
        operation_type: Operation,
        variable1: IGenericVariable,
        variable2: IGenericVariable,
    ) -> None:
        '''The precondition parameter is used as a comparator of SSM documents inputs to determine whether a command step would be executed or not.

        See https://docs.aws.amazon.com/systems-manager/latest/userguide/document-schemas-features.html

        :param operation_type: (Required) the operation used to compare the parameter with the variable.
        :param variable1: (Required) the variable to compare against the constant.
        :param variable2: (Required) the being compared against the variable.
        '''
        if __debug__:
            def stub(
                *,
                operation_type: Operation,
                variable1: IGenericVariable,
                variable2: IGenericVariable,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument operation_type", value=operation_type, expected_type=type_hints["operation_type"])
            check_type(argname="argument variable1", value=variable1, expected_type=type_hints["variable1"])
            check_type(argname="argument variable2", value=variable2, expected_type=type_hints["variable2"])
        self._values: typing.Dict[str, typing.Any] = {
            "operation_type": operation_type,
            "variable1": variable1,
            "variable2": variable2,
        }

    @builtins.property
    def operation_type(self) -> Operation:
        '''(Required) the operation used to compare the parameter with the variable.'''
        result = self._values.get("operation_type")
        assert result is not None, "Required property 'operation_type' is missing"
        return typing.cast(Operation, result)

    @builtins.property
    def variable1(self) -> IGenericVariable:
        '''(Required) the variable to compare against the constant.'''
        result = self._values.get("variable1")
        assert result is not None, "Required property 'variable1' is missing"
        return typing.cast(IGenericVariable, result)

    @builtins.property
    def variable2(self) -> IGenericVariable:
        '''(Required) the being compared against the variable.'''
        result = self._values.get("variable2")
        assert result is not None, "Required property 'variable2' is missing"
        return typing.cast(IGenericVariable, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PreconditionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class PsModuleSimulation(
    CommandSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.PsModuleSimulation",
):
    def __init__(self, step: "PsModuleStep", *, environment: IEnvironment) -> None:
        '''
        :param step: -
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        if __debug__:
            def stub(step: "PsModuleStep", *, environment: IEnvironment) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = EnvironmentProps(environment=environment)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''Installs the module specified by source then runs the specified commands.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(None, jsii.invoke(self, "executeStep", [inputs]))


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.PythonVersion")
class PythonVersion(enum.Enum):
    '''Python runtime to use when writing SSM Document.

    Simulation will use local python version.
    '''

    VERSION_3_6 = "VERSION_3_6"
    VERSION_3_7 = "VERSION_3_7"
    VERSION_3_8 = "VERSION_3_8"


@jsii.implements(IAwsInvoker)
class ReflectiveAwsInvoker(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ReflectiveAwsInvoker",
):
    '''Implementation of IAwsInvoker that executes the AWS api for real.

    If using this implementation, be sure that AWS credentials are available to the execution.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        aws_api: builtins.str,
        aws_params: typing.Mapping[builtins.str, typing.Any],
        service: AwsService,
    ) -> typing.Any:
        '''Invoke AWS with the provided invocation request.

        :param aws_api: (Required) AWS api to invoke; should be referenced using lowerCamelCase.
        :param aws_params: (Required )AWS params.
        :param service: (Required) AWS service to invoke.
        '''
        invocation = Invocation(
            aws_api=aws_api, aws_params=aws_params, service=service
        )

        return typing.cast(typing.Any, jsii.invoke(self, "invoke", [invocation]))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RequiredAutomationSimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "approve_hook": "approveHook",
        "aws_invoker": "awsInvoker",
        "execute_automation_hook": "executeAutomationHook",
        "input_observer": "inputObserver",
        "output_observer": "outputObserver",
        "parameter_resolver": "parameterResolver",
        "pause_hook": "pauseHook",
        "run_command_hook": "runCommandHook",
        "sleep_hook": "sleepHook",
        "webhook": "webhook",
    },
)
class RequiredAutomationSimulationProps:
    def __init__(
        self,
        *,
        approve_hook: IApproveHook,
        aws_invoker: IAwsInvoker,
        execute_automation_hook: IExecuteAutomationHook,
        input_observer: IObserver,
        output_observer: IObserver,
        parameter_resolver: IParameterResolver,
        pause_hook: IPauseHook,
        run_command_hook: IRunCommandHook,
        sleep_hook: ISleepHook,
        webhook: IWebhook,
    ) -> None:
        '''The same interface as AutomationSimulationProps but all fields are required.

        :param approve_hook: 
        :param aws_invoker: 
        :param execute_automation_hook: 
        :param input_observer: 
        :param output_observer: 
        :param parameter_resolver: 
        :param pause_hook: 
        :param run_command_hook: 
        :param sleep_hook: 
        :param webhook: 
        '''
        if __debug__:
            def stub(
                *,
                approve_hook: IApproveHook,
                aws_invoker: IAwsInvoker,
                execute_automation_hook: IExecuteAutomationHook,
                input_observer: IObserver,
                output_observer: IObserver,
                parameter_resolver: IParameterResolver,
                pause_hook: IPauseHook,
                run_command_hook: IRunCommandHook,
                sleep_hook: ISleepHook,
                webhook: IWebhook,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument approve_hook", value=approve_hook, expected_type=type_hints["approve_hook"])
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument execute_automation_hook", value=execute_automation_hook, expected_type=type_hints["execute_automation_hook"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument parameter_resolver", value=parameter_resolver, expected_type=type_hints["parameter_resolver"])
            check_type(argname="argument pause_hook", value=pause_hook, expected_type=type_hints["pause_hook"])
            check_type(argname="argument run_command_hook", value=run_command_hook, expected_type=type_hints["run_command_hook"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
            check_type(argname="argument webhook", value=webhook, expected_type=type_hints["webhook"])
        self._values: typing.Dict[str, typing.Any] = {
            "approve_hook": approve_hook,
            "aws_invoker": aws_invoker,
            "execute_automation_hook": execute_automation_hook,
            "input_observer": input_observer,
            "output_observer": output_observer,
            "parameter_resolver": parameter_resolver,
            "pause_hook": pause_hook,
            "run_command_hook": run_command_hook,
            "sleep_hook": sleep_hook,
            "webhook": webhook,
        }

    @builtins.property
    def approve_hook(self) -> IApproveHook:
        result = self._values.get("approve_hook")
        assert result is not None, "Required property 'approve_hook' is missing"
        return typing.cast(IApproveHook, result)

    @builtins.property
    def aws_invoker(self) -> IAwsInvoker:
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast(IAwsInvoker, result)

    @builtins.property
    def execute_automation_hook(self) -> IExecuteAutomationHook:
        result = self._values.get("execute_automation_hook")
        assert result is not None, "Required property 'execute_automation_hook' is missing"
        return typing.cast(IExecuteAutomationHook, result)

    @builtins.property
    def input_observer(self) -> IObserver:
        result = self._values.get("input_observer")
        assert result is not None, "Required property 'input_observer' is missing"
        return typing.cast(IObserver, result)

    @builtins.property
    def output_observer(self) -> IObserver:
        result = self._values.get("output_observer")
        assert result is not None, "Required property 'output_observer' is missing"
        return typing.cast(IObserver, result)

    @builtins.property
    def parameter_resolver(self) -> IParameterResolver:
        result = self._values.get("parameter_resolver")
        assert result is not None, "Required property 'parameter_resolver' is missing"
        return typing.cast(IParameterResolver, result)

    @builtins.property
    def pause_hook(self) -> IPauseHook:
        result = self._values.get("pause_hook")
        assert result is not None, "Required property 'pause_hook' is missing"
        return typing.cast(IPauseHook, result)

    @builtins.property
    def run_command_hook(self) -> IRunCommandHook:
        result = self._values.get("run_command_hook")
        assert result is not None, "Required property 'run_command_hook' is missing"
        return typing.cast(IRunCommandHook, result)

    @builtins.property
    def sleep_hook(self) -> ISleepHook:
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast(ISleepHook, result)

    @builtins.property
    def webhook(self) -> IWebhook:
        result = self._values.get("webhook")
        assert result is not None, "Required property 'webhook' is missing"
        return typing.cast(IWebhook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RequiredAutomationSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RequiredCommandSimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "environment": "environment",
        "simulation_platform": "simulationPlatform",
    },
)
class RequiredCommandSimulationProps:
    def __init__(
        self,
        *,
        environment: IEnvironment,
        simulation_platform: Platform,
    ) -> None:
        '''The same interface as CommandSimulationProps but all fields are required.

        :param environment: 
        :param simulation_platform: 
        '''
        if __debug__:
            def stub(
                *,
                environment: IEnvironment,
                simulation_platform: Platform,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument simulation_platform", value=simulation_platform, expected_type=type_hints["simulation_platform"])
        self._values: typing.Dict[str, typing.Any] = {
            "environment": environment,
            "simulation_platform": simulation_platform,
        }

    @builtins.property
    def environment(self) -> IEnvironment:
        result = self._values.get("environment")
        assert result is not None, "Required property 'environment' is missing"
        return typing.cast(IEnvironment, result)

    @builtins.property
    def simulation_platform(self) -> Platform:
        result = self._values.get("simulation_platform")
        assert result is not None, "Required property 'simulation_platform' is missing"
        return typing.cast(Platform, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RequiredCommandSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.ResponseCode")
class ResponseCode(enum.Enum):
    '''Steps report their ResponseCode using this enum.

    A successful response will contain the outputs expected.
    A failed/canceled response will contain the stackTrace.
    '''

    SUCCESS = "SUCCESS"
    CANCELED = "CANCELED"
    FAILED = "FAILED"


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunCommandOutputs",
    jsii_struct_bases=[],
    name_mapping={
        "command_id": "commandId",
        "status": "status",
        "output": "output",
        "response_code": "responseCode",
    },
)
class RunCommandOutputs:
    def __init__(
        self,
        *,
        command_id: builtins.str,
        status: builtins.str,
        output: typing.Optional[builtins.str] = None,
        response_code: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Outputs for aws:runCommand.

        :param command_id: The ID of the command.
        :param status: The status of the command.
        :param output: The output of the command.
        :param response_code: The response code of the command.
        '''
        if __debug__:
            def stub(
                *,
                command_id: builtins.str,
                status: builtins.str,
                output: typing.Optional[builtins.str] = None,
                response_code: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument command_id", value=command_id, expected_type=type_hints["command_id"])
            check_type(argname="argument status", value=status, expected_type=type_hints["status"])
            check_type(argname="argument output", value=output, expected_type=type_hints["output"])
            check_type(argname="argument response_code", value=response_code, expected_type=type_hints["response_code"])
        self._values: typing.Dict[str, typing.Any] = {
            "command_id": command_id,
            "status": status,
        }
        if output is not None:
            self._values["output"] = output
        if response_code is not None:
            self._values["response_code"] = response_code

    @builtins.property
    def command_id(self) -> builtins.str:
        '''The ID of the command.'''
        result = self._values.get("command_id")
        assert result is not None, "Required property 'command_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def status(self) -> builtins.str:
        '''The status of the command.'''
        result = self._values.get("status")
        assert result is not None, "Required property 'status' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def output(self) -> typing.Optional[builtins.str]:
        '''The output of the command.'''
        result = self._values.get("output")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def response_code(self) -> typing.Optional[jsii.Number]:
        '''The response code of the command.'''
        result = self._values.get("response_code")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunCommandOutputs(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunCommandProps",
    jsii_struct_bases=[],
    name_mapping={
        "document_name": "documentName",
        "targets": "targets",
        "cloud_watch_output_config": "cloudWatchOutputConfig",
        "comment": "comment",
        "document_hash": "documentHash",
        "document_hash_type": "documentHashType",
        "max_concurrency": "maxConcurrency",
        "max_errors": "maxErrors",
        "notification_config": "notificationConfig",
        "output_s3_bucket_name": "outputS3BucketName",
        "output_s3_key_prefix": "outputS3KeyPrefix",
        "parameters": "parameters",
        "service_role_arn": "serviceRoleArn",
        "timeout_seconds": "timeoutSeconds",
    },
)
class RunCommandProps:
    def __init__(
        self,
        *,
        document_name: builtins.str,
        targets: typing.Sequence[builtins.str],
        cloud_watch_output_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        document_hash: typing.Optional[builtins.str] = None,
        document_hash_type: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_errors: typing.Optional[jsii.Number] = None,
        notification_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        output_s3_bucket_name: typing.Optional[builtins.str] = None,
        output_s3_key_prefix: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        service_role_arn: typing.Optional[builtins.str] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Inputs for aws:runCommand.

        :param document_name: If the Command type document is owned by you or AWS, specify the name of the document. If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        :param targets: The instance IDs where you want the command to run. You can specify a maximum of 50 IDs. You can also use the pseudo parameter {{ RESOURCE_ID }} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_. Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        :param cloud_watch_output_config: (Optional) Configuration options for sending command output to Amazon CloudWatch Logs. For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        :param comment: (Optional) User-defined information about the command.
        :param document_hash: (Optional) The hash for the document.
        :param document_hash_type: (Optional) The type of the hash.
        :param max_concurrency: (Optional) The maximum concurrency.
        :param max_errors: (Optional) The maximum errors.
        :param notification_config: (Optional) The configurations for sending notifications.
        :param output_s3_bucket_name: (Optional) The name of the S3 bucket for command output responses.
        :param output_s3_key_prefix: (Optional) The prefix.
        :param parameters: (Optional) The required and optional parameters specified in the document.
        :param service_role_arn: (Optional) The ARN of the AWS Identity and Access Management (IAM) role.
        :param timeout_seconds: (Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance. If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        if __debug__:
            def stub(
                *,
                document_name: builtins.str,
                targets: typing.Sequence[builtins.str],
                cloud_watch_output_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
                comment: typing.Optional[builtins.str] = None,
                document_hash: typing.Optional[builtins.str] = None,
                document_hash_type: typing.Optional[builtins.str] = None,
                max_concurrency: typing.Optional[jsii.Number] = None,
                max_errors: typing.Optional[jsii.Number] = None,
                notification_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
                output_s3_bucket_name: typing.Optional[builtins.str] = None,
                output_s3_key_prefix: typing.Optional[builtins.str] = None,
                parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
                service_role_arn: typing.Optional[builtins.str] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument document_name", value=document_name, expected_type=type_hints["document_name"])
            check_type(argname="argument targets", value=targets, expected_type=type_hints["targets"])
            check_type(argname="argument cloud_watch_output_config", value=cloud_watch_output_config, expected_type=type_hints["cloud_watch_output_config"])
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument document_hash", value=document_hash, expected_type=type_hints["document_hash"])
            check_type(argname="argument document_hash_type", value=document_hash_type, expected_type=type_hints["document_hash_type"])
            check_type(argname="argument max_concurrency", value=max_concurrency, expected_type=type_hints["max_concurrency"])
            check_type(argname="argument max_errors", value=max_errors, expected_type=type_hints["max_errors"])
            check_type(argname="argument notification_config", value=notification_config, expected_type=type_hints["notification_config"])
            check_type(argname="argument output_s3_bucket_name", value=output_s3_bucket_name, expected_type=type_hints["output_s3_bucket_name"])
            check_type(argname="argument output_s3_key_prefix", value=output_s3_key_prefix, expected_type=type_hints["output_s3_key_prefix"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument service_role_arn", value=service_role_arn, expected_type=type_hints["service_role_arn"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
        self._values: typing.Dict[str, typing.Any] = {
            "document_name": document_name,
            "targets": targets,
        }
        if cloud_watch_output_config is not None:
            self._values["cloud_watch_output_config"] = cloud_watch_output_config
        if comment is not None:
            self._values["comment"] = comment
        if document_hash is not None:
            self._values["document_hash"] = document_hash
        if document_hash_type is not None:
            self._values["document_hash_type"] = document_hash_type
        if max_concurrency is not None:
            self._values["max_concurrency"] = max_concurrency
        if max_errors is not None:
            self._values["max_errors"] = max_errors
        if notification_config is not None:
            self._values["notification_config"] = notification_config
        if output_s3_bucket_name is not None:
            self._values["output_s3_bucket_name"] = output_s3_bucket_name
        if output_s3_key_prefix is not None:
            self._values["output_s3_key_prefix"] = output_s3_key_prefix
        if parameters is not None:
            self._values["parameters"] = parameters
        if service_role_arn is not None:
            self._values["service_role_arn"] = service_role_arn
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds

    @builtins.property
    def document_name(self) -> builtins.str:
        '''If the Command type document is owned by you or AWS, specify the name of the document.

        If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        '''
        result = self._values.get("document_name")
        assert result is not None, "Required property 'document_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def targets(self) -> typing.List[builtins.str]:
        '''The instance IDs where you want the command to run. You can specify a maximum of 50 IDs.

        You can also use the pseudo parameter {{ RESOURCE_ID }} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_.

        Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        '''
        result = self._values.get("targets")
        assert result is not None, "Required property 'targets' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def cloud_watch_output_config(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(Optional) Configuration options for sending command output to Amazon CloudWatch Logs.

        For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        '''
        result = self._values.get("cloud_watch_output_config")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(Optional) User-defined information about the command.'''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def document_hash(self) -> typing.Optional[builtins.str]:
        '''(Optional) The hash for the document.'''
        result = self._values.get("document_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def document_hash_type(self) -> typing.Optional[builtins.str]:
        '''(Optional) The type of the hash.'''
        result = self._values.get("document_hash_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_concurrency(self) -> typing.Optional[jsii.Number]:
        '''(Optional) The maximum concurrency.'''
        result = self._values.get("max_concurrency")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_errors(self) -> typing.Optional[jsii.Number]:
        '''(Optional) The maximum errors.'''
        result = self._values.get("max_errors")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def notification_config(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(Optional) The configurations for sending notifications.'''
        result = self._values.get("notification_config")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def output_s3_bucket_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) The name of the S3 bucket for command output responses.'''
        result = self._values.get("output_s3_bucket_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_s3_key_prefix(self) -> typing.Optional[builtins.str]:
        '''(Optional) The prefix.'''
        result = self._values.get("output_s3_key_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(Optional) The required and optional parameters specified in the document.'''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def service_role_arn(self) -> typing.Optional[builtins.str]:
        '''(Optional) The ARN of the AWS Identity and Access Management (IAM) role.'''
        result = self._values.get("service_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance.

        If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunCommandProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RunCommandSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunCommandSimulation",
):
    '''AutomationStep implementation of `aws:runCommand <https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-runcommand.html>`_.'''

    def __init__(
        self,
        step: "RunCommandStep",
        *,
        run_command_hook: IRunCommandHook,
    ) -> None:
        '''
        :param step: -
        :param run_command_hook: Hook for simulating aws:runCommand. Default: - Uses AWS API to execute the document remotely.
        '''
        if __debug__:
            def stub(
                step: "RunCommandStep",
                *,
                run_command_hook: IRunCommandHook,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = RunCommandSimulationProps(run_command_hook=run_command_hook)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''May perform a real approval ask based on the params used during instance creation.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunCommandSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"run_command_hook": "runCommandHook"},
)
class RunCommandSimulationProps:
    def __init__(self, *, run_command_hook: IRunCommandHook) -> None:
        '''Properties for RunCommandStep.

        :param run_command_hook: Hook for simulating aws:runCommand. Default: - Uses AWS API to execute the document remotely.
        '''
        if __debug__:
            def stub(*, run_command_hook: IRunCommandHook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument run_command_hook", value=run_command_hook, expected_type=type_hints["run_command_hook"])
        self._values: typing.Dict[str, typing.Any] = {
            "run_command_hook": run_command_hook,
        }

    @builtins.property
    def run_command_hook(self) -> IRunCommandHook:
        '''Hook for simulating aws:runCommand.

        :default: - Uses AWS API to execute the document remotely.
        '''
        result = self._values.get("run_command_hook")
        assert result is not None, "Required property 'run_command_hook' is missing"
        return typing.cast(IRunCommandHook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunCommandSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RunInstanceSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunInstanceSimulation",
):
    '''AutomationStep implemenation for aws:runInstance https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-runinstance.html.'''

    def __init__(self, step: "RunInstanceStep", *, aws_invoker: IAwsInvoker) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(step: "RunInstanceStep", *, aws_invoker: IAwsInvoker) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class RunPowerShellScriptSimulation(
    CommandSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunPowerShellScriptSimulation",
):
    def __init__(
        self,
        step: "RunPowerShellScriptStep",
        *,
        environment: IEnvironment,
    ) -> None:
        '''
        :param step: -
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        if __debug__:
            def stub(
                step: "RunPowerShellScriptStep",
                *,
                environment: IEnvironment,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = EnvironmentProps(environment=environment)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''Executes the runCommands against the environment provided in the constructor.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(None, jsii.invoke(self, "executeStep", [inputs]))


class RunShellScriptSimulation(
    CommandSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunShellScriptSimulation",
):
    def __init__(
        self,
        step: "RunShellScriptStep",
        *,
        environment: IEnvironment,
    ) -> None:
        '''
        :param step: -
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        if __debug__:
            def stub(step: "RunShellScriptStep", *, environment: IEnvironment) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = EnvironmentProps(environment=environment)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''Executes the runCommands against the environment provided in the constructor.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(None, jsii.invoke(self, "executeStep", [inputs]))


@jsii.implements(IDownloadableContent)
class S3Content(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.S3Content",
):
    def __init__(self, *, path: IStringVariable) -> None:
        '''
        :param path: The URL to the file or directory you want to download.
        '''
        props = S3ContentProps(path=path)

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property
    @jsii.member(jsii_name="path")
    def path(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "path"))

    @builtins.property
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.S3ContentProps",
    jsii_struct_bases=[],
    name_mapping={"path": "path"},
)
class S3ContentProps:
    def __init__(self, *, path: IStringVariable) -> None:
        '''Properties.json for sourceType GitHub.

        :param path: The URL to the file or directory you want to download.
        '''
        if __debug__:
            def stub(*, path: IStringVariable) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        self._values: typing.Dict[str, typing.Any] = {
            "path": path,
        }

    @builtins.property
    def path(self) -> IStringVariable:
        '''The URL to the file or directory you want to download.'''
        result = self._values.get("path")
        assert result is not None, "Required property 'path' is missing"
        return typing.cast(IStringVariable, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "S3ContentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IDownloadableContent)
class SSMDocumentContent(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SSMDocumentContent",
):
    def __init__(self, *, document_source: DocumentSource) -> None:
        '''
        :param document_source: specify one of the following The name and version of the document in the following format: name:version. Version is optional. or The ARN for the document in the following format: arn:aws:ssm:region:account_id:document/document_name
        '''
        props = SsmDocumentContentProps(document_source=document_source)

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property
    @jsii.member(jsii_name="documentSource")
    def document_source(self) -> DocumentSource:
        return typing.cast(DocumentSource, jsii.get(self, "documentSource"))

    @builtins.property
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))


class ScriptCode(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.ScriptCode",
):
    '''The code to run for the execution.

    See "script" parameter here:
    https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html
    Attachments are not yet supported.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromFile")
    @builtins.classmethod
    def from_file(cls, full_path: builtins.str) -> "FileScriptCode":
        '''Full path to the code to execute.

        File is parsed to produce yaml/json.
        Simulation will execute this file using the language specified.
        (Attachments not yet supported)

        :param full_path: -
        '''
        if __debug__:
            def stub(full_path: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument full_path", value=full_path, expected_type=type_hints["full_path"])
        return typing.cast("FileScriptCode", jsii.sinvoke(cls, "fromFile", [full_path]))

    @jsii.member(jsii_name="inline")
    @builtins.classmethod
    def inline(cls, code: builtins.str) -> "InlineScriptCode":
        '''Inline code to be executed.

        String will be used to produce function in yaml/json.
        Simulation will execute the function in this string using the language specified.

        :param code: -
        '''
        if __debug__:
            def stub(code: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
        return typing.cast("InlineScriptCode", jsii.sinvoke(cls, "inline", [code]))

    @jsii.member(jsii_name="codeAsString")
    @abc.abstractmethod
    def code_as_string(self) -> builtins.str:
        '''
        :return: code as a string
        '''
        ...

    @jsii.member(jsii_name="createOrGetFile")
    @abc.abstractmethod
    def create_or_get_file(self, suffix: builtins.str) -> builtins.str:
        '''If there is a file for this code, return it.

        Otherwise, create a file with the specified suffix.

        :param suffix: of the file to create (such as ".py").
        '''
        ...


class _ScriptCodeProxy(ScriptCode):
    @jsii.member(jsii_name="codeAsString")
    def code_as_string(self) -> builtins.str:
        '''
        :return: code as a string
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "codeAsString", []))

    @jsii.member(jsii_name="createOrGetFile")
    def create_or_get_file(self, suffix: builtins.str) -> builtins.str:
        '''If there is a file for this code, return it.

        Otherwise, create a file with the specified suffix.

        :param suffix: of the file to create (such as ".py").
        '''
        if __debug__:
            def stub(suffix: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
        return typing.cast(builtins.str, jsii.invoke(self, "createOrGetFile", [suffix]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, ScriptCode).__jsii_proxy_class__ = lambda : _ScriptCodeProxy


class ScriptLanguage(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.ScriptLanguage",
):
    '''Specifies the script language as described in the "Runtime" argument here: https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html.'''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromRuntime")
    @builtins.classmethod
    def from_runtime(
        cls,
        runtime: builtins.str,
        handler_name: typing.Optional[builtins.str] = None,
    ) -> "ScriptLanguage":
        '''Creates a ScriptLanguage based on the provided runtime.

        Prefer one of the other static constructors if possible.

        :param runtime: is the runtime name (such as "python3.6").
        :param handler_name: to be provided for python executions.
        '''
        if __debug__:
            def stub(
                runtime: builtins.str,
                handler_name: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument runtime", value=runtime, expected_type=type_hints["runtime"])
            check_type(argname="argument handler_name", value=handler_name, expected_type=type_hints["handler_name"])
        return typing.cast("ScriptLanguage", jsii.sinvoke(cls, "fromRuntime", [runtime, handler_name]))

    @jsii.member(jsii_name="python")
    @builtins.classmethod
    def python(
        cls,
        version: PythonVersion,
        handler_name: builtins.str,
    ) -> "ScriptLanguage":
        '''Create a new ScriptLanguage for python execution.

        :param version: is the pythonVersion to use when writing the document (for simulation will not matter).
        :param handler_name: is the function name in code as entry point for script handler.
        '''
        if __debug__:
            def stub(version: PythonVersion, handler_name: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            check_type(argname="argument handler_name", value=handler_name, expected_type=type_hints["handler_name"])
        return typing.cast("ScriptLanguage", jsii.sinvoke(cls, "python", [version, handler_name]))

    @jsii.member(jsii_name="fileSuffix")
    @abc.abstractmethod
    def file_suffix(self) -> builtins.str:
        '''The suffix to apply to file names of this type of execution.'''
        ...

    @jsii.member(jsii_name="runtime")
    @abc.abstractmethod
    def runtime(self) -> builtins.str:
        '''The associated runtime of this ScriptLanguage.'''
        ...

    @jsii.member(jsii_name="simulate")
    @abc.abstractmethod
    def simulate(
        self,
        code: ScriptCode,
        inputs: typing.Mapping[builtins.str, builtins.str],
    ) -> typing.Mapping[builtins.str, builtins.str]:
        '''Simulate an execution of this ScriptLanguage.

        Provide the inputs after replaced with the actual values (not variables).

        :param code: -
        :param inputs: -
        '''
        ...

    @jsii.member(jsii_name="ssmInputs")
    @abc.abstractmethod
    def ssm_inputs(self) -> typing.Mapping[builtins.str, builtins.str]:
        '''Builds the ssm inputs.'''
        ...


class _ScriptLanguageProxy(ScriptLanguage):
    @jsii.member(jsii_name="fileSuffix")
    def file_suffix(self) -> builtins.str:
        '''The suffix to apply to file names of this type of execution.'''
        return typing.cast(builtins.str, jsii.invoke(self, "fileSuffix", []))

    @jsii.member(jsii_name="runtime")
    def runtime(self) -> builtins.str:
        '''The associated runtime of this ScriptLanguage.'''
        return typing.cast(builtins.str, jsii.invoke(self, "runtime", []))

    @jsii.member(jsii_name="simulate")
    def simulate(
        self,
        code: ScriptCode,
        inputs: typing.Mapping[builtins.str, builtins.str],
    ) -> typing.Mapping[builtins.str, builtins.str]:
        '''Simulate an execution of this ScriptLanguage.

        Provide the inputs after replaced with the actual values (not variables).

        :param code: -
        :param inputs: -
        '''
        if __debug__:
            def stub(
                code: ScriptCode,
                inputs: typing.Mapping[builtins.str, builtins.str],
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, builtins.str], jsii.invoke(self, "simulate", [code, inputs]))

    @jsii.member(jsii_name="ssmInputs")
    def ssm_inputs(self) -> typing.Mapping[builtins.str, builtins.str]:
        '''Builds the ssm inputs.'''
        return typing.cast(typing.Mapping[builtins.str, builtins.str], jsii.invoke(self, "ssmInputs", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, ScriptLanguage).__jsii_proxy_class__ = lambda : _ScriptLanguageProxy


@jsii.implements(IGenericVariable)
class SecureVariable(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.SecureVariable",
):
    '''A secure string variable.

    Only supported by Command documents (only supported in downloadContent plugin).
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="ofParameter")
    @builtins.classmethod
    def of_parameter(cls, reference: builtins.str) -> "NonSecureVariable":
        '''
        :param reference: -
        '''
        if __debug__:
            def stub(reference: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument reference", value=reference, expected_type=type_hints["reference"])
        return typing.cast("NonSecureVariable", jsii.sinvoke(cls, "ofParameter", [reference]))

    @jsii.member(jsii_name="ofSecureToken")
    @builtins.classmethod
    def of_secure_token(cls, secure_token: builtins.str) -> "SsmSecureVariable":
        '''
        :param secure_token: -
        '''
        if __debug__:
            def stub(secure_token: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument secure_token", value=secure_token, expected_type=type_hints["secure_token"])
        return typing.cast("SsmSecureVariable", jsii.sinvoke(cls, "ofSecureToken", [secure_token]))

    @jsii.member(jsii_name="ofValue")
    @builtins.classmethod
    def of_value(cls, value: builtins.str) -> "HardCodedSecureVariable":
        '''
        :param value: -
        '''
        if __debug__:
            def stub(value: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("HardCodedSecureVariable", jsii.sinvoke(cls, "ofValue", [value]))

    @jsii.member(jsii_name="print")
    @abc.abstractmethod
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{ Variable }} or the value.
        '''
        ...

    @jsii.member(jsii_name="requiredInputs")
    @abc.abstractmethod
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        ...

    @jsii.member(jsii_name="resolve")
    @abc.abstractmethod
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        ...

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))


class _SecureVariableProxy(SecureVariable):
    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{ Variable }} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, SecureVariable).__jsii_proxy_class__ = lambda : _SecureVariableProxy


class Simulation(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.Simulation",
):
    @jsii.member(jsii_name="ofAutomation")
    @builtins.classmethod
    def of_automation(
        cls,
        document: "AutomationDocument",
        *,
        approve_hook: typing.Optional[IApproveHook] = None,
        aws_invoker: typing.Optional[IAwsInvoker] = None,
        execute_automation_hook: typing.Optional[IExecuteAutomationHook] = None,
        input_observer: typing.Optional[IObserver] = None,
        output_observer: typing.Optional[IObserver] = None,
        parameter_resolver: typing.Optional[IParameterResolver] = None,
        pause_hook: typing.Optional[IPauseHook] = None,
        run_command_hook: typing.Optional[IRunCommandHook] = None,
        sleep_hook: typing.Optional[ISleepHook] = None,
        webhook: typing.Optional[IWebhook] = None,
    ) -> "Simulation":
        '''
        :param document: -
        :param approve_hook: (Optional) Approve hook to be called to pause the execution. To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class. Default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param execute_automation_hook: Hook for simulating aws:executeAutomation. Default: - Uses AWS API to execute the document remotely.
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param parameter_resolver: (Optional) Resolver for secure strings in parameters. Required to simulate if using tokens in parameters input. Default: - Treats parameters as literal
        :param pause_hook: (Optional) Pause hook to be called to pause the execution. To mock this implemenation either inject an instance of IPauseHook or use the provided MockPause class. Default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        :param run_command_hook: Hook for simulating aws:runCommand. Default: - Uses AWS API to execute the document remotely.
        :param sleep_hook: (Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations). Default: - really perform sleep using SleeperImpl class.
        :param webhook: (Optional) Hook for simulating aws:invokeWebhook. Default: - Returns 204 with an empty response
        '''
        if __debug__:
            def stub(
                document: "AutomationDocument",
                *,
                approve_hook: typing.Optional[IApproveHook] = None,
                aws_invoker: typing.Optional[IAwsInvoker] = None,
                execute_automation_hook: typing.Optional[IExecuteAutomationHook] = None,
                input_observer: typing.Optional[IObserver] = None,
                output_observer: typing.Optional[IObserver] = None,
                parameter_resolver: typing.Optional[IParameterResolver] = None,
                pause_hook: typing.Optional[IPauseHook] = None,
                run_command_hook: typing.Optional[IRunCommandHook] = None,
                sleep_hook: typing.Optional[ISleepHook] = None,
                webhook: typing.Optional[IWebhook] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument document", value=document, expected_type=type_hints["document"])
        props = AutomationSimulationProps(
            approve_hook=approve_hook,
            aws_invoker=aws_invoker,
            execute_automation_hook=execute_automation_hook,
            input_observer=input_observer,
            output_observer=output_observer,
            parameter_resolver=parameter_resolver,
            pause_hook=pause_hook,
            run_command_hook=run_command_hook,
            sleep_hook=sleep_hook,
            webhook=webhook,
        )

        return typing.cast("Simulation", jsii.sinvoke(cls, "ofAutomation", [document, props]))

    @jsii.member(jsii_name="ofCommand")
    @builtins.classmethod
    def of_command(
        cls,
        document: "CommandDocument",
        *,
        simulation_platform: Platform,
        environment: typing.Optional[IEnvironment] = None,
    ) -> "Simulation":
        '''
        :param document: -
        :param simulation_platform: The Platform used in executing the command step.
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        if __debug__:
            def stub(
                document: "CommandDocument",
                *,
                simulation_platform: Platform,
                environment: typing.Optional[IEnvironment] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument document", value=document, expected_type=type_hints["document"])
        props = CommandSimulationProps(
            simulation_platform=simulation_platform, environment=environment
        )

        return typing.cast("Simulation", jsii.sinvoke(cls, "ofCommand", [document, props]))

    @jsii.member(jsii_name="simulate")
    def simulate(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "DocumentResult":
        '''Synthesize before calling this function! You can use this to Synthesize: SynthUtils.synthesize(stack);

        Executes the SSM Document in simulation mode.
        This method DOES NOT result in invocation of SSM APIs.
        Rather, all steps are executed locally and mimic the behavior of SSM.
        If any inputs are not provided in this function, the specified defaults for the inputs will be used.

        :param inputs: the inputs to feed into the simulated execution.

        :return: the outputs of all the executed steps.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast("DocumentResult", jsii.invoke(self, "simulate", [inputs]))

    @jsii.member(jsii_name="start")
    def _start(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "SimulationResult":
        '''Delegates the execution of the Document to the subclass (Automation, etc).

        :param inputs: a merge of the defined inputs to the document and the default values if not supplied.

        :return: the outputs that were emitted from all of the steps.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast("SimulationResult", jsii.invoke(self, "start", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="document")
    def document(self) -> "SsmDocument":
        return typing.cast("SsmDocument", jsii.get(self, "document"))

    @builtins.property
    @jsii.member(jsii_name="props")
    def props(self) -> "SimulationProps":
        return typing.cast("SimulationProps", jsii.get(self, "props"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "approve_hook": "approveHook",
        "aws_invoker": "awsInvoker",
        "environment": "environment",
        "input_observer": "inputObserver",
        "output_observer": "outputObserver",
        "parameter_resolver": "parameterResolver",
        "pause_hook": "pauseHook",
        "run_command_hook": "runCommandHook",
        "simulation_platform": "simulationPlatform",
        "sleep_hook": "sleepHook",
        "webhook": "webhook",
    },
)
class SimulationProps:
    def __init__(
        self,
        *,
        approve_hook: typing.Optional[IApproveHook] = None,
        aws_invoker: typing.Optional[IAwsInvoker] = None,
        environment: typing.Optional[IEnvironment] = None,
        input_observer: typing.Optional[IObserver] = None,
        output_observer: typing.Optional[IObserver] = None,
        parameter_resolver: typing.Optional[IParameterResolver] = None,
        pause_hook: typing.Optional[IPauseHook] = None,
        run_command_hook: typing.Optional[IRunCommandHook] = None,
        simulation_platform: typing.Optional[Platform] = None,
        sleep_hook: typing.Optional[ISleepHook] = None,
        webhook: typing.Optional[IWebhook] = None,
    ) -> None:
        '''Universe of Automation and Command simulation props.

        :param approve_hook: 
        :param aws_invoker: 
        :param environment: 
        :param input_observer: 
        :param output_observer: 
        :param parameter_resolver: 
        :param pause_hook: 
        :param run_command_hook: 
        :param simulation_platform: 
        :param sleep_hook: 
        :param webhook: 
        '''
        if __debug__:
            def stub(
                *,
                approve_hook: typing.Optional[IApproveHook] = None,
                aws_invoker: typing.Optional[IAwsInvoker] = None,
                environment: typing.Optional[IEnvironment] = None,
                input_observer: typing.Optional[IObserver] = None,
                output_observer: typing.Optional[IObserver] = None,
                parameter_resolver: typing.Optional[IParameterResolver] = None,
                pause_hook: typing.Optional[IPauseHook] = None,
                run_command_hook: typing.Optional[IRunCommandHook] = None,
                simulation_platform: typing.Optional[Platform] = None,
                sleep_hook: typing.Optional[ISleepHook] = None,
                webhook: typing.Optional[IWebhook] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument approve_hook", value=approve_hook, expected_type=type_hints["approve_hook"])
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument parameter_resolver", value=parameter_resolver, expected_type=type_hints["parameter_resolver"])
            check_type(argname="argument pause_hook", value=pause_hook, expected_type=type_hints["pause_hook"])
            check_type(argname="argument run_command_hook", value=run_command_hook, expected_type=type_hints["run_command_hook"])
            check_type(argname="argument simulation_platform", value=simulation_platform, expected_type=type_hints["simulation_platform"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
            check_type(argname="argument webhook", value=webhook, expected_type=type_hints["webhook"])
        self._values: typing.Dict[str, typing.Any] = {}
        if approve_hook is not None:
            self._values["approve_hook"] = approve_hook
        if aws_invoker is not None:
            self._values["aws_invoker"] = aws_invoker
        if environment is not None:
            self._values["environment"] = environment
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if parameter_resolver is not None:
            self._values["parameter_resolver"] = parameter_resolver
        if pause_hook is not None:
            self._values["pause_hook"] = pause_hook
        if run_command_hook is not None:
            self._values["run_command_hook"] = run_command_hook
        if simulation_platform is not None:
            self._values["simulation_platform"] = simulation_platform
        if sleep_hook is not None:
            self._values["sleep_hook"] = sleep_hook
        if webhook is not None:
            self._values["webhook"] = webhook

    @builtins.property
    def approve_hook(self) -> typing.Optional[IApproveHook]:
        result = self._values.get("approve_hook")
        return typing.cast(typing.Optional[IApproveHook], result)

    @builtins.property
    def aws_invoker(self) -> typing.Optional[IAwsInvoker]:
        result = self._values.get("aws_invoker")
        return typing.cast(typing.Optional[IAwsInvoker], result)

    @builtins.property
    def environment(self) -> typing.Optional[IEnvironment]:
        result = self._values.get("environment")
        return typing.cast(typing.Optional[IEnvironment], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def parameter_resolver(self) -> typing.Optional[IParameterResolver]:
        result = self._values.get("parameter_resolver")
        return typing.cast(typing.Optional[IParameterResolver], result)

    @builtins.property
    def pause_hook(self) -> typing.Optional[IPauseHook]:
        result = self._values.get("pause_hook")
        return typing.cast(typing.Optional[IPauseHook], result)

    @builtins.property
    def run_command_hook(self) -> typing.Optional[IRunCommandHook]:
        result = self._values.get("run_command_hook")
        return typing.cast(typing.Optional[IRunCommandHook], result)

    @builtins.property
    def simulation_platform(self) -> typing.Optional[Platform]:
        result = self._values.get("simulation_platform")
        return typing.cast(typing.Optional[Platform], result)

    @builtins.property
    def sleep_hook(self) -> typing.Optional[ISleepHook]:
        result = self._values.get("sleep_hook")
        return typing.cast(typing.Optional[ISleepHook], result)

    @builtins.property
    def webhook(self) -> typing.Optional[IWebhook]:
        result = self._values.get("webhook")
        return typing.cast(typing.Optional[IWebhook], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SimulationResult",
    jsii_struct_bases=[],
    name_mapping={
        "executed_steps": "executedSteps",
        "response_code": "responseCode",
        "outputs": "outputs",
        "stack_trace": "stackTrace",
    },
)
class SimulationResult:
    def __init__(
        self,
        *,
        executed_steps: typing.Sequence[builtins.str],
        response_code: ResponseCode,
        outputs: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        stack_trace: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Response object returned from steps.

        A successful response will contain the outputs expected.
        A failed/canceled response will contain the stackTrace.
        A failure will propagate up the stack unless the step is marked to succeed on failure.

        :param executed_steps: All the steps that were executed in this Simulation.
        :param response_code: 
        :param outputs: May be empty if responseCode is FAILED/CANCELLED. There are no outputs provided for Command steps or documents.
        :param stack_trace: undefined if responseCode is SUCCESS.
        '''
        if __debug__:
            def stub(
                *,
                executed_steps: typing.Sequence[builtins.str],
                response_code: ResponseCode,
                outputs: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
                stack_trace: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument executed_steps", value=executed_steps, expected_type=type_hints["executed_steps"])
            check_type(argname="argument response_code", value=response_code, expected_type=type_hints["response_code"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument stack_trace", value=stack_trace, expected_type=type_hints["stack_trace"])
        self._values: typing.Dict[str, typing.Any] = {
            "executed_steps": executed_steps,
            "response_code": response_code,
        }
        if outputs is not None:
            self._values["outputs"] = outputs
        if stack_trace is not None:
            self._values["stack_trace"] = stack_trace

    @builtins.property
    def executed_steps(self) -> typing.List[builtins.str]:
        '''All the steps that were executed in this Simulation.'''
        result = self._values.get("executed_steps")
        assert result is not None, "Required property 'executed_steps' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def response_code(self) -> ResponseCode:
        result = self._values.get("response_code")
        assert result is not None, "Required property 'response_code' is missing"
        return typing.cast(ResponseCode, result)

    @builtins.property
    def outputs(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''May be empty if responseCode is FAILED/CANCELLED.

        There are no outputs provided for Command steps or documents.
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def stack_trace(self) -> typing.Optional[builtins.str]:
        '''undefined if responseCode is SUCCESS.'''
        result = self._values.get("stack_trace")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SimulationResult(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(ISleepHook)
class SleepImpl(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SleepImpl",
):
    '''Performs a real sleep.'''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="sleep")
    def sleep(self, time_millis: jsii.Number) -> None:
        '''Synchronously sleeps for duration specified in millis.

        :param time_millis: -
        '''
        if __debug__:
            def stub(time_millis: jsii.Number) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument time_millis", value=time_millis, expected_type=type_hints["time_millis"])
        return typing.cast(None, jsii.invoke(self, "sleep", [time_millis]))


class SleepSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SleepSimulation",
):
    '''AutomationStep implemenation for aws:sleep https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-sleep.html.'''

    def __init__(self, step: "SleepStep", *, sleep_hook: ISleepHook) -> None:
        '''
        :param step: -
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        if __debug__:
            def stub(step: "SleepStep", *, sleep_hook: ISleepHook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = SleepSimulationProps(sleep_hook=sleep_hook)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        _inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param _inputs: -
        '''
        if __debug__:
            def stub(_inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument _inputs", value=_inputs, expected_type=type_hints["_inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [_inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SleepSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"sleep_hook": "sleepHook"},
)
class SleepSimulationProps:
    def __init__(self, *, sleep_hook: ISleepHook) -> None:
        '''Properties for sleep step.

        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        if __debug__:
            def stub(*, sleep_hook: ISleepHook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
        self._values: typing.Dict[str, typing.Any] = {
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def sleep_hook(self) -> ISleepHook:
        '''(Optional) Whether to really perform a pause of the runtime.

        To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class.

        :default: SleeperImpl
        '''
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast(ISleepHook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SleepSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SsmAutomationProps",
    jsii_struct_bases=[],
    name_mapping={"parameters": "parameters", "target_account": "targetAccount"},
)
class SsmAutomationProps:
    def __init__(
        self,
        *,
        parameters: typing.Mapping[builtins.str, IGenericVariable],
        target_account: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param parameters: Specify either StringVariables or HardCodedValues.
        :param target_account: The account that the automation document will be run in. This can be in either the management account or an application account.
        '''
        if __debug__:
            def stub(
                *,
                parameters: typing.Mapping[builtins.str, IGenericVariable],
                target_account: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument target_account", value=target_account, expected_type=type_hints["target_account"])
        self._values: typing.Dict[str, typing.Any] = {
            "parameters": parameters,
        }
        if target_account is not None:
            self._values["target_account"] = target_account

    @builtins.property
    def parameters(self) -> typing.Mapping[builtins.str, IGenericVariable]:
        '''Specify either StringVariables or HardCodedValues.

        :link: : https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ssmincidents-responseplan-ssmparameter.html
        '''
        result = self._values.get("parameters")
        assert result is not None, "Required property 'parameters' is missing"
        return typing.cast(typing.Mapping[builtins.str, IGenericVariable], result)

    @builtins.property
    def target_account(self) -> typing.Optional[builtins.str]:
        '''The account that the automation document will be run in.

        This can be in either the management account or an application account.

        :link: : http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ssmincidents-responseplan-ssmautomation.html#cfn-ssmincidents-responseplan-ssmautomation-targetaccount
        '''
        result = self._values.get("target_account")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SsmAutomationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class SsmDocument(
    constructs.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.SsmDocument",
):
    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        assume_role: typing.Optional[IStringVariable] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
        target_type: typing.Optional[builtins.str] = None,
        update_method: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param update_method: If the document resource you specify in your template already exists, this parameter determines whether a new version of the existing document is created, or the existing document is replaced. ``Replace`` is the default method. If you specify ``NewVersion`` for the ``UpdateMethod`` parameter, and the ``Name`` of the document does not match an existing resource, a new document is created. When you specify ``NewVersion`` , the default version of the document is changed to the newly created version.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                assume_role: typing.Optional[IStringVariable] = None,
                description: typing.Optional[builtins.str] = None,
                doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
                doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
                document_format: typing.Optional[DocumentFormat] = None,
                document_name: typing.Optional[builtins.str] = None,
                header: typing.Optional[builtins.str] = None,
                requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
                target_type: typing.Optional[builtins.str] = None,
                update_method: typing.Optional[builtins.str] = None,
                version_name: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = SsmDocumentProps(
            assume_role=assume_role,
            description=description,
            doc_inputs=doc_inputs,
            doc_outputs=doc_outputs,
            document_format=document_format,
            document_name=document_name,
            header=header,
            requires=requires,
            tags=tags,
            target_type=target_type,
            update_method=update_method,
            version_name=version_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="buildSsmDocument")
    @abc.abstractmethod
    def _build_ssm_document(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Delegates building the SSM Document to be converted to a yaml/json to the subclass (Automation etc).'''
        ...

    @jsii.member(jsii_name="documentType")
    @abc.abstractmethod
    def document_type(self) -> builtins.str:
        ...

    @jsii.member(jsii_name="formatInputs")
    def _format_inputs(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''The SSM Document inputs optionally specify a number of parameters including allowedValues, minItems etc.

        This function builds an object containing the relevant (declared) input parameters.
        The return object will be used to build the yaml/json representation of the document.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatInputs", []))

    @jsii.member(jsii_name="print")
    def print(self) -> builtins.str:
        '''Synthesize before calling this function! You can use this to Synthesize: cdk.SynthUtils.synthesize(stack);

        Converts the objects define in the SSM Document (including all of the steps) to an SSM document string.
        The format is dependency on the documentFormat property provided to the class.
        The yaml can be used as is and will behave (or at least should behave) as was simulated in the runSimulation().

        :return: a string representation of this document as an SSM formatted yaml/json.
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "print", []))

    @builtins.property
    @jsii.member(jsii_name="cfnDocument")
    def cfn_document(self) -> aws_cdk.aws_ssm.CfnDocument:
        return typing.cast(aws_cdk.aws_ssm.CfnDocument, jsii.get(self, "cfnDocument"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "description"))

    @builtins.property
    @jsii.member(jsii_name="docInputs")
    def doc_inputs(self) -> typing.List[Input]:
        return typing.cast(typing.List[Input], jsii.get(self, "docInputs"))

    @builtins.property
    @jsii.member(jsii_name="docOutputs")
    def doc_outputs(self) -> typing.List[DocumentOutput]:
        return typing.cast(typing.List[DocumentOutput], jsii.get(self, "docOutputs"))

    @builtins.property
    @jsii.member(jsii_name="documentName")
    def document_name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "documentName"))

    @builtins.property
    @jsii.member(jsii_name="props")
    def props(self) -> "SsmDocumentProps":
        return typing.cast("SsmDocumentProps", jsii.get(self, "props"))

    @builtins.property
    @jsii.member(jsii_name="assumeRole")
    def assume_role(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "assumeRole"))

    @builtins.property
    @jsii.member(jsii_name="header")
    def header(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "header"))


class _SsmDocumentProxy(SsmDocument):
    @jsii.member(jsii_name="buildSsmDocument")
    def _build_ssm_document(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Delegates building the SSM Document to be converted to a yaml/json to the subclass (Automation etc).'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "buildSsmDocument", []))

    @jsii.member(jsii_name="documentType")
    def document_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "documentType", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, SsmDocument).__jsii_proxy_class__ = lambda : _SsmDocumentProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SsmDocumentContentProps",
    jsii_struct_bases=[],
    name_mapping={"document_source": "documentSource"},
)
class SsmDocumentContentProps:
    def __init__(self, *, document_source: DocumentSource) -> None:
        '''Properties.json for sourceType SsmDocument.

        :param document_source: specify one of the following The name and version of the document in the following format: name:version. Version is optional. or The ARN for the document in the following format: arn:aws:ssm:region:account_id:document/document_name
        '''
        if __debug__:
            def stub(*, document_source: DocumentSource) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument document_source", value=document_source, expected_type=type_hints["document_source"])
        self._values: typing.Dict[str, typing.Any] = {
            "document_source": document_source,
        }

    @builtins.property
    def document_source(self) -> DocumentSource:
        '''specify one of the following The name and version of the document in the following format: name:version.

        Version is optional.
        or The ARN for the document in the following format: arn:aws:ssm:region:account_id:document/document_name
        '''
        result = self._values.get("document_source")
        assert result is not None, "Required property 'document_source' is missing"
        return typing.cast(DocumentSource, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SsmDocumentContentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SsmDocumentProps",
    jsii_struct_bases=[],
    name_mapping={
        "assume_role": "assumeRole",
        "description": "description",
        "doc_inputs": "docInputs",
        "doc_outputs": "docOutputs",
        "document_format": "documentFormat",
        "document_name": "documentName",
        "header": "header",
        "requires": "requires",
        "tags": "tags",
        "target_type": "targetType",
        "update_method": "updateMethod",
        "version_name": "versionName",
    },
)
class SsmDocumentProps:
    def __init__(
        self,
        *,
        assume_role: typing.Optional[IStringVariable] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
        target_type: typing.Optional[builtins.str] = None,
        update_method: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param update_method: If the document resource you specify in your template already exists, this parameter determines whether a new version of the existing document is created, or the existing document is replaced. ``Replace`` is the default method. If you specify ``NewVersion`` for the ``UpdateMethod`` parameter, and the ``Name`` of the document does not match an existing resource, a new document is created. When you specify ``NewVersion`` , the default version of the document is changed to the newly created version.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        if __debug__:
            def stub(
                *,
                assume_role: typing.Optional[IStringVariable] = None,
                description: typing.Optional[builtins.str] = None,
                doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
                doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
                document_format: typing.Optional[DocumentFormat] = None,
                document_name: typing.Optional[builtins.str] = None,
                header: typing.Optional[builtins.str] = None,
                requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
                target_type: typing.Optional[builtins.str] = None,
                update_method: typing.Optional[builtins.str] = None,
                version_name: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument assume_role", value=assume_role, expected_type=type_hints["assume_role"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument doc_inputs", value=doc_inputs, expected_type=type_hints["doc_inputs"])
            check_type(argname="argument doc_outputs", value=doc_outputs, expected_type=type_hints["doc_outputs"])
            check_type(argname="argument document_format", value=document_format, expected_type=type_hints["document_format"])
            check_type(argname="argument document_name", value=document_name, expected_type=type_hints["document_name"])
            check_type(argname="argument header", value=header, expected_type=type_hints["header"])
            check_type(argname="argument requires", value=requires, expected_type=type_hints["requires"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument target_type", value=target_type, expected_type=type_hints["target_type"])
            check_type(argname="argument update_method", value=update_method, expected_type=type_hints["update_method"])
            check_type(argname="argument version_name", value=version_name, expected_type=type_hints["version_name"])
        self._values: typing.Dict[str, typing.Any] = {}
        if assume_role is not None:
            self._values["assume_role"] = assume_role
        if description is not None:
            self._values["description"] = description
        if doc_inputs is not None:
            self._values["doc_inputs"] = doc_inputs
        if doc_outputs is not None:
            self._values["doc_outputs"] = doc_outputs
        if document_format is not None:
            self._values["document_format"] = document_format
        if document_name is not None:
            self._values["document_name"] = document_name
        if header is not None:
            self._values["header"] = header
        if requires is not None:
            self._values["requires"] = requires
        if tags is not None:
            self._values["tags"] = tags
        if target_type is not None:
            self._values["target_type"] = target_type
        if update_method is not None:
            self._values["update_method"] = update_method
        if version_name is not None:
            self._values["version_name"] = version_name

    @builtins.property
    def assume_role(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Assume role to use for this document.

        If provided, this value MUST be included as one of the documentInput names.
        '''
        result = self._values.get("assume_role")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) Description of the document.

        Defaults to the document name.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def doc_inputs(self) -> typing.Optional[typing.List[Input]]:
        '''(Optional) Inputs required by the document.'''
        result = self._values.get("doc_inputs")
        return typing.cast(typing.Optional[typing.List[Input]], result)

    @builtins.property
    def doc_outputs(self) -> typing.Optional[typing.List[DocumentOutput]]:
        '''(Optional) Outputs to be emitted from the document.

        The outputs are placed in a StringSet called outputs (as is done in SSM).

        :default: []
        '''
        result = self._values.get("doc_outputs")
        return typing.cast(typing.Optional[typing.List[DocumentOutput]], result)

    @builtins.property
    def document_format(self) -> typing.Optional[DocumentFormat]:
        '''(Optional) Specifies whether this document should be written as YAML or JSON.

        :default: JSON
        '''
        result = self._values.get("document_format")
        return typing.cast(typing.Optional[DocumentFormat], result)

    @builtins.property
    def document_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the document.

        Will default to the id provided for the CDK node.
        '''
        result = self._values.get("document_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def header(self) -> typing.Optional[builtins.str]:
        '''(Optional) A Header/comment to include at the start of a YAML document.

        JSON documents do not support headers.
        '''
        result = self._values.get("header")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def requires(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]]:
        '''``AWS::SSM::Document.Requires``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-requires
        '''
        result = self._values.get("requires")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.CfnTag]]:
        '''``AWS::SSM::Document.Tags``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.CfnTag]], result)

    @builtins.property
    def target_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.TargetType``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-targettype
        '''
        result = self._values.get("target_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def update_method(self) -> typing.Optional[builtins.str]:
        '''If the document resource you specify in your template already exists, this parameter determines whether a new version of the existing document is created, or the existing document is replaced.

        ``Replace`` is the default method. If you specify ``NewVersion`` for the ``UpdateMethod`` parameter, and the ``Name`` of the document does not match an existing resource, a new document is created. When you specify ``NewVersion`` , the default version of the document is changed to the newly created version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-updatemethod
        '''
        result = self._values.get("update_method")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.VersionName``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-versionname
        '''
        result = self._values.get("version_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SsmDocumentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IRunDocumentLocation)
class SsmRunDocument(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SsmRunDocument",
):
    def __init__(self, document_name: IStringVariable) -> None:
        '''specify the name of the document.

        :param document_name: -
        '''
        if __debug__:
            def stub(document_name: IStringVariable) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument document_name", value=document_name, expected_type=type_hints["document_name"])
        jsii.create(self.__class__, self, [document_name])

    @builtins.property
    @jsii.member(jsii_name="location")
    def location(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "location"))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "type"))


class SsmSecureVariable(
    SecureVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SsmSecureVariable",
):
    def __init__(self, secure_token: builtins.str) -> None:
        '''
        :param secure_token: -
        '''
        if __debug__:
            def stub(secure_token: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument secure_token", value=secure_token, expected_type=type_hints["secure_token"])
        jsii.create(self.__class__, self, [secure_token])

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{ Variable }} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, _inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param _inputs: -
        '''
        if __debug__:
            def stub(_inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument _inputs", value=_inputs, expected_type=type_hints["_inputs"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [_inputs]))

    @builtins.property
    @jsii.member(jsii_name="secureToken")
    def secure_token(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "secureToken"))


class Step(
    constructs.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.Step",
):
    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = StepProps(
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    @abc.abstractmethod
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        ...

    @jsii.member(jsii_name="listOutputs")
    @abc.abstractmethod
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        ...

    @jsii.member(jsii_name="toSsmEntry")
    @abc.abstractmethod
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="action")
    @abc.abstractmethod
    def action(self) -> builtins.str:
        ...

    @builtins.property
    @jsii.member(jsii_name="inputObserver")
    def input_observer(self) -> IObserver:
        return typing.cast(IObserver, jsii.get(self, "inputObserver"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="outputObserver")
    def output_observer(self) -> IObserver:
        return typing.cast(IObserver, jsii.get(self, "outputObserver"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))


class _StepProxy(Step):
    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Step).__jsii_proxy_class__ = lambda : _StepProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.StepProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
    },
)
class StepProps:
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
        self._values: typing.Dict[str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "StepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class StepRef(metaclass=jsii.JSIIMeta, jsii_type="@cdklabs/cdk-ssm-documents.StepRef"):
    '''Class to reference AutomationSteps.

    The class allows steps to be referenced by the Step object or by the step name.
    '''

    def __init__(self, step_name: builtins.str) -> None:
        '''
        :param step_name: -
        '''
        if __debug__:
            def stub(step_name: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step_name", value=step_name, expected_type=type_hints["step_name"])
        jsii.create(self.__class__, self, [step_name])

    @jsii.member(jsii_name="fromName")
    @builtins.classmethod
    def from_name(cls, step_name: builtins.str) -> "StepRef":
        '''Static constructor for creating a reference to a step from a step name.

        :param step_name: -
        '''
        if __debug__:
            def stub(step_name: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step_name", value=step_name, expected_type=type_hints["step_name"])
        return typing.cast("StepRef", jsii.sinvoke(cls, "fromName", [step_name]))

    @jsii.member(jsii_name="fromObject")
    @builtins.classmethod
    def from_object(cls, step: "AutomationStep") -> "StepRef":
        '''Static constructor for creating a reference to a step from an AutomationStep object.

        :param step: -
        '''
        if __debug__:
            def stub(step: "AutomationStep") -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        return typing.cast("StepRef", jsii.sinvoke(cls, "fromObject", [step]))

    @jsii.member(jsii_name="resolve")
    def resolve(
        self,
        all_steps_in_execution: typing.Sequence["AutomationStep"],
    ) -> "AutomationStep":
        '''Resolve to an AutomationStep object.

        Provide all the steps in the execution to find the associated step.

        :param all_steps_in_execution: -
        '''
        if __debug__:
            def stub(all_steps_in_execution: typing.Sequence["AutomationStep"]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument all_steps_in_execution", value=all_steps_in_execution, expected_type=type_hints["all_steps_in_execution"])
        return typing.cast("AutomationStep", jsii.invoke(self, "resolve", [all_steps_in_execution]))

    @builtins.property
    @jsii.member(jsii_name="stepName")
    def step_name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "stepName"))


class StringDocument(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.StringDocument",
):
    '''This AutomationDocument supports declaring your document from an existing document (JSON/YAML String/File).

    Importing an existing file allows for benefiting from the simulated execution.
    The simulated execution will run locally in the same fashion that SSM Execution would run the document.
    You can supply mocks to the simulator and validate the calls and the flow of the document without running via SSM execution.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromFile")
    @builtins.classmethod
    def from_file(
        cls,
        stack: constructs.Construct,
        id: builtins.str,
        document_file_path: builtins.str,
    ) -> "AutomationDocument":
        '''Create an AutomationDocument from an existing AutomationDocument yaml or json file.

        Note: This function will deduce whether the file is written in yaml or json based on whether it has a .yaml or .yml extention.
        You can use the returned AutomationDocument to run simulations as you would other documents created using this library.

        :param stack: -
        :param id: -
        :param document_file_path: -
        '''
        if __debug__:
            def stub(
                stack: constructs.Construct,
                id: builtins.str,
                document_file_path: builtins.str,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument document_file_path", value=document_file_path, expected_type=type_hints["document_file_path"])
        return typing.cast("AutomationDocument", jsii.sinvoke(cls, "fromFile", [stack, id, document_file_path]))

    @jsii.member(jsii_name="fromJson")
    @builtins.classmethod
    def from_json(
        cls,
        stack: constructs.Construct,
        id: builtins.str,
        document_json: builtins.str,
    ) -> "AutomationDocument":
        '''Create an AutomationDocument from an existing json string.

        You can use the returned AutomationDocument to run simulations as you would other documents created using this library.

        :param stack: -
        :param id: -
        :param document_json: -
        '''
        if __debug__:
            def stub(
                stack: constructs.Construct,
                id: builtins.str,
                document_json: builtins.str,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument document_json", value=document_json, expected_type=type_hints["document_json"])
        return typing.cast("AutomationDocument", jsii.sinvoke(cls, "fromJson", [stack, id, document_json]))

    @jsii.member(jsii_name="fromYaml")
    @builtins.classmethod
    def from_yaml(
        cls,
        stack: constructs.Construct,
        id: builtins.str,
        document_yaml: builtins.str,
    ) -> "AutomationDocument":
        '''Create an AutomationDocument from an existing yaml string.

        You can use the returned AutomationDocument to run simulations as you would other documents created using this library.

        :param stack: -
        :param id: -
        :param document_yaml: -
        '''
        if __debug__:
            def stub(
                stack: constructs.Construct,
                id: builtins.str,
                document_yaml: builtins.str,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument document_yaml", value=document_yaml, expected_type=type_hints["document_yaml"])
        return typing.cast("AutomationDocument", jsii.sinvoke(cls, "fromYaml", [stack, id, document_yaml]))


@jsii.implements(IStringVariable)
class StringFormat(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.StringFormat",
):
    '''Replacement for strings using Java String format style "%s" replacements.

    Example: new StringFormat("This %s a replacement: %s", [new HardCodedValue("is"), new ExportedVariable("myInput")]);
    The supported variable strings are either:

    1. Implicit indices: "%s" where the first occurrence will match the first variable, the next will match the second...
    2. Explicit indices: Example: "%1$s"; where "%1$s" matches the first variable and "%1$s" matches the second.
       Do not combine usage of implicit and explicit indices. Choose one per StringFormat instance.
    '''

    def __init__(
        self,
        format: builtins.str,
        variables: typing.Optional[typing.Sequence[IGenericVariable]] = None,
    ) -> None:
        '''
        :param format: -
        :param variables: -
        '''
        if __debug__:
            def stub(
                format: builtins.str,
                variables: typing.Optional[typing.Sequence[IGenericVariable]] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            check_type(argname="argument variables", value=variables, expected_type=type_hints["variables"])
        jsii.create(self.__class__, self, [format, variables])

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{ Variable }} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

    @jsii.member(jsii_name="resolveToString")
    def resolve_to_string(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.str:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(builtins.str, jsii.invoke(self, "resolveToString", [inputs]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

    @builtins.property
    @jsii.member(jsii_name="format")
    def format(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "format"))

    @builtins.property
    @jsii.member(jsii_name="variables")
    def variables(self) -> typing.List[IGenericVariable]:
        return typing.cast(typing.List[IGenericVariable], jsii.get(self, "variables"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.StringInputProps",
    jsii_struct_bases=[],
    name_mapping={
        "allowed_pattern": "allowedPattern",
        "allowed_values": "allowedValues",
        "default_value": "defaultValue",
        "description": "description",
        "max_chars": "maxChars",
        "min_chars": "minChars",
    },
)
class StringInputProps:
    def __init__(
        self,
        *,
        allowed_pattern: typing.Optional[builtins.str] = None,
        allowed_values: typing.Optional[typing.Sequence[builtins.str]] = None,
        default_value: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        max_chars: typing.Optional[jsii.Number] = None,
        min_chars: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param allowed_pattern: (Optional) Pattern that this input value must match. Default: undefined
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_chars: (Optional) Maximum number of chars that this input value (string) must contain. Default: undefined
        :param min_chars: (Optional) Minimum number of chars that this input value (string) must contain. Default: undefined
        '''
        if __debug__:
            def stub(
                *,
                allowed_pattern: typing.Optional[builtins.str] = None,
                allowed_values: typing.Optional[typing.Sequence[builtins.str]] = None,
                default_value: typing.Optional[builtins.str] = None,
                description: typing.Optional[builtins.str] = None,
                max_chars: typing.Optional[jsii.Number] = None,
                min_chars: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument allowed_pattern", value=allowed_pattern, expected_type=type_hints["allowed_pattern"])
            check_type(argname="argument allowed_values", value=allowed_values, expected_type=type_hints["allowed_values"])
            check_type(argname="argument default_value", value=default_value, expected_type=type_hints["default_value"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument max_chars", value=max_chars, expected_type=type_hints["max_chars"])
            check_type(argname="argument min_chars", value=min_chars, expected_type=type_hints["min_chars"])
        self._values: typing.Dict[str, typing.Any] = {}
        if allowed_pattern is not None:
            self._values["allowed_pattern"] = allowed_pattern
        if allowed_values is not None:
            self._values["allowed_values"] = allowed_values
        if default_value is not None:
            self._values["default_value"] = default_value
        if description is not None:
            self._values["description"] = description
        if max_chars is not None:
            self._values["max_chars"] = max_chars
        if min_chars is not None:
            self._values["min_chars"] = min_chars

    @builtins.property
    def allowed_pattern(self) -> typing.Optional[builtins.str]:
        '''(Optional) Pattern that this input value must match.

        :default: undefined
        '''
        result = self._values.get("allowed_pattern")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def allowed_values(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(Optional) List of allowed values that this input may be.

        :default: undefined
        '''
        result = self._values.get("allowed_values")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def default_value(self) -> typing.Optional[builtins.str]:
        '''(Optional) Default value to use for this input if not specified when invoking the document.

        :default: undefined
        '''
        result = self._values.get("default_value")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) The description of the input.

        :default: name
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_chars(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Maximum number of chars that this input value (string) must contain.

        :default: undefined
        '''
        result = self._values.get("max_chars")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_chars(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Minimum number of chars that this input value (string) must contain.

        :default: undefined
        '''
        result = self._values.get("min_chars")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "StringInputProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.StringListInputProps",
    jsii_struct_bases=[],
    name_mapping={
        "allowed_values": "allowedValues",
        "default_value": "defaultValue",
        "description": "description",
        "max_items": "maxItems",
        "min_items": "minItems",
    },
)
class StringListInputProps:
    def __init__(
        self,
        *,
        allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
        default_value: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_items: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_items: (Optional) Maximum number of items that this input value (list) must contain. Default: undefined
        :param min_items: (Optional) Minimum number of items that this input value (list) must contain. Default: undefined
        '''
        if __debug__:
            def stub(
                *,
                allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
                default_value: typing.Any = None,
                description: typing.Optional[builtins.str] = None,
                max_items: typing.Optional[jsii.Number] = None,
                min_items: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument allowed_values", value=allowed_values, expected_type=type_hints["allowed_values"])
            check_type(argname="argument default_value", value=default_value, expected_type=type_hints["default_value"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument max_items", value=max_items, expected_type=type_hints["max_items"])
            check_type(argname="argument min_items", value=min_items, expected_type=type_hints["min_items"])
        self._values: typing.Dict[str, typing.Any] = {}
        if allowed_values is not None:
            self._values["allowed_values"] = allowed_values
        if default_value is not None:
            self._values["default_value"] = default_value
        if description is not None:
            self._values["description"] = description
        if max_items is not None:
            self._values["max_items"] = max_items
        if min_items is not None:
            self._values["min_items"] = min_items

    @builtins.property
    def allowed_values(self) -> typing.Optional[typing.List[typing.Any]]:
        '''(Optional) List of allowed values that this input may be.

        :default: undefined
        '''
        result = self._values.get("allowed_values")
        return typing.cast(typing.Optional[typing.List[typing.Any]], result)

    @builtins.property
    def default_value(self) -> typing.Any:
        '''(Optional) Default value to use for this input if not specified when invoking the document.

        :default: undefined
        '''
        result = self._values.get("default_value")
        return typing.cast(typing.Any, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) The description of the input.

        :default: name
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_items(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Maximum number of items that this input value (list) must contain.

        :default: undefined
        '''
        result = self._values.get("max_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_items(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Minimum number of items that this input value (list) must contain.

        :default: undefined
        '''
        result = self._values.get("min_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "StringListInputProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.StringMapInputProps",
    jsii_struct_bases=[],
    name_mapping={
        "allowed_values": "allowedValues",
        "default_value": "defaultValue",
        "description": "description",
        "max_items": "maxItems",
        "min_items": "minItems",
    },
)
class StringMapInputProps:
    def __init__(
        self,
        *,
        allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
        default_value: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_items: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_items: (Optional) Maximum number of items that this input value (list) must contain. Default: undefined
        :param min_items: (Optional) Minimum number of items that this input value (list) must contain. Default: undefined
        '''
        if __debug__:
            def stub(
                *,
                allowed_values: typing.Optional[typing.Sequence[typing.Any]] = None,
                default_value: typing.Any = None,
                description: typing.Optional[builtins.str] = None,
                max_items: typing.Optional[jsii.Number] = None,
                min_items: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument allowed_values", value=allowed_values, expected_type=type_hints["allowed_values"])
            check_type(argname="argument default_value", value=default_value, expected_type=type_hints["default_value"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument max_items", value=max_items, expected_type=type_hints["max_items"])
            check_type(argname="argument min_items", value=min_items, expected_type=type_hints["min_items"])
        self._values: typing.Dict[str, typing.Any] = {}
        if allowed_values is not None:
            self._values["allowed_values"] = allowed_values
        if default_value is not None:
            self._values["default_value"] = default_value
        if description is not None:
            self._values["description"] = description
        if max_items is not None:
            self._values["max_items"] = max_items
        if min_items is not None:
            self._values["min_items"] = min_items

    @builtins.property
    def allowed_values(self) -> typing.Optional[typing.List[typing.Any]]:
        '''(Optional) List of allowed values that this input may be.

        :default: undefined
        '''
        result = self._values.get("allowed_values")
        return typing.cast(typing.Optional[typing.List[typing.Any]], result)

    @builtins.property
    def default_value(self) -> typing.Any:
        '''(Optional) Default value to use for this input if not specified when invoking the document.

        :default: undefined
        '''
        result = self._values.get("default_value")
        return typing.cast(typing.Any, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) The description of the input.

        :default: name
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_items(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Maximum number of items that this input value (list) must contain.

        :default: undefined
        '''
        result = self._values.get("max_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_items(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Minimum number of items that this input value (list) must contain.

        :default: undefined
        '''
        result = self._values.get("min_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "StringMapInputProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class SynthUtils(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SynthUtils",
):
    '''Wraps SynthUtils from @aws-cdk/assert because that package is not exported via JSII.'''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="synthesize")
    @builtins.classmethod
    def synthesize(
        cls,
        stack: aws_cdk.Stack,
    ) -> typing.Optional[aws_cdk.cx_api.CloudAssembly]:
        '''Wraps @aws-cdk/assert SynthUtils.synthesize(stack). Synthesizes the stack provided.

        :param stack: -
        '''
        if __debug__:
            def stub(stack: aws_cdk.Stack) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
        return typing.cast(typing.Optional[aws_cdk.cx_api.CloudAssembly], jsii.sinvoke(cls, "synthesize", [stack]))


class UsernamePasswordAuthMethod(
    AuthMethod,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.UsernamePasswordAuthMethod",
):
    def __init__(
        self,
        auth_method: builtins.str,
        user_name: SecureVariable,
        password: SecureVariable,
    ) -> None:
        '''
        :param auth_method: -
        :param user_name: -
        :param password: -
        '''
        if __debug__:
            def stub(
                auth_method: builtins.str,
                user_name: SecureVariable,
                password: SecureVariable,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument auth_method", value=auth_method, expected_type=type_hints["auth_method"])
            check_type(argname="argument user_name", value=user_name, expected_type=type_hints["user_name"])
            check_type(argname="argument password", value=password, expected_type=type_hints["password"])
        jsii.create(self.__class__, self, [auth_method, user_name, password])

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="toEntry")
    def to_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toEntry", []))

    @builtins.property
    @jsii.member(jsii_name="authMethod")
    def auth_method(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "authMethod"))

    @builtins.property
    @jsii.member(jsii_name="password")
    def password(self) -> SecureVariable:
        return typing.cast(SecureVariable, jsii.get(self, "password"))

    @builtins.property
    @jsii.member(jsii_name="userName")
    def user_name(self) -> SecureVariable:
        return typing.cast(SecureVariable, jsii.get(self, "userName"))


class WaitForResourceSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.WaitForResourceSimulation",
):
    '''AutomationStep impl for aws:waitForAwsResourceProperty https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-waitForAwsResourceProperty.html.'''

    def __init__(
        self,
        step: "WaitForResourceStep",
        *,
        aws_invoker: IAwsInvoker,
        sleep_hook: ISleepHook,
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations). Default: - really perform sleep using SleeperImpl class.
        '''
        if __debug__:
            def stub(
                step: "WaitForResourceStep",
                *,
                aws_invoker: IAwsInvoker,
                sleep_hook: ISleepHook,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = WaitForResourceSimulationProps(
            aws_invoker=aws_invoker, sleep_hook=sleep_hook
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''As is the case in an SSM Automation execution, this will continue to sleep/execute until desired value is found.

        This function will throw if the timeoutSeconds is exceeded and the desired value is still not received from AWS.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.WaitForResourceSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker", "sleep_hook": "sleepHook"},
)
class WaitForResourceSimulationProps:
    def __init__(self, *, aws_invoker: IAwsInvoker, sleep_hook: ISleepHook) -> None:
        '''
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations). Default: - really perform sleep using SleeperImpl class.
        '''
        if __debug__:
            def stub(*, aws_invoker: IAwsInvoker, sleep_hook: ISleepHook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def aws_invoker(self) -> IAwsInvoker:
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast(IAwsInvoker, result)

    @builtins.property
    def sleep_hook(self) -> ISleepHook:
        '''(Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations).

        :default: - really perform sleep using SleeperImpl class.
        '''
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast(ISleepHook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "WaitForResourceSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IWebhook)
class WebhookImpl(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.WebhookImpl",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        integration_name: builtins.str,
        body: typing.Optional[builtins.str] = None,
    ) -> InvokeWebhookResult:
        '''Invoke the web hook.

        :param integration_name: The name of the Automation integration. For example, exampleIntegration. The integration you specify must already exist.
        :param body: (Optional) The payload you want to send when your webhook integration is invoked.
        '''
        _props = InvokeWebhookProps(integration_name=integration_name, body=body)

        return typing.cast(InvokeWebhookResult, jsii.invoke(self, "invoke", [_props]))


@jsii.implements(IExecuteAutomationHook)
class ApiExecuteAutomationHook(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ApiExecuteAutomationHook",
):
    '''ExecuteAutomation implementation using AWS API.'''

    def __init__(self, aws_invoker: IAwsInvoker, sleep_hook: ISleepHook) -> None:
        '''
        :param aws_invoker: -
        :param sleep_hook: -
        '''
        if __debug__:
            def stub(aws_invoker: IAwsInvoker, sleep_hook: ISleepHook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
        jsii.create(self.__class__, self, [aws_invoker, sleep_hook])

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        document_version: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[builtins.str] = None,
        max_errors: typing.Optional[builtins.str] = None,
        runtime_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        tags: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_locations: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_maps: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_parameter_name: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
    ) -> ExecuteAutomationOutputs:
        '''Simulate the aws:executeAutomation.

        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        '''
        props = ExecuteAutomationProps(
            document_name=document_name,
            document_version=document_version,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            runtime_parameters=runtime_parameters,
            tags=tags,
            target_locations=target_locations,
            target_maps=target_maps,
            target_parameter_name=target_parameter_name,
            targets=targets,
        )

        return typing.cast(ExecuteAutomationOutputs, jsii.invoke(self, "execute", [props]))

    @builtins.property
    @jsii.member(jsii_name="props")
    def props(self) -> ApiExecuteAutomationProps:
        return typing.cast(ApiExecuteAutomationProps, jsii.get(self, "props"))


@jsii.implements(IRunCommandHook)
class ApiRunCommandHook(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ApiRunCommandHook",
):
    '''RunCommand implementation using AWS API.'''

    def __init__(self, aws_invoker: IAwsInvoker, sleep_hook: ISleepHook) -> None:
        '''
        :param aws_invoker: -
        :param sleep_hook: -
        '''
        if __debug__:
            def stub(aws_invoker: IAwsInvoker, sleep_hook: ISleepHook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
            check_type(argname="argument sleep_hook", value=sleep_hook, expected_type=type_hints["sleep_hook"])
        jsii.create(self.__class__, self, [aws_invoker, sleep_hook])

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        targets: typing.Sequence[builtins.str],
        cloud_watch_output_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        document_hash: typing.Optional[builtins.str] = None,
        document_hash_type: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_errors: typing.Optional[jsii.Number] = None,
        notification_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        output_s3_bucket_name: typing.Optional[builtins.str] = None,
        output_s3_key_prefix: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        service_role_arn: typing.Optional[builtins.str] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
    ) -> RunCommandOutputs:
        '''Simulate the aws:runCommand.

        :param document_name: If the Command type document is owned by you or AWS, specify the name of the document. If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        :param targets: The instance IDs where you want the command to run. You can specify a maximum of 50 IDs. You can also use the pseudo parameter {{ RESOURCE_ID }} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_. Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        :param cloud_watch_output_config: (Optional) Configuration options for sending command output to Amazon CloudWatch Logs. For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        :param comment: (Optional) User-defined information about the command.
        :param document_hash: (Optional) The hash for the document.
        :param document_hash_type: (Optional) The type of the hash.
        :param max_concurrency: (Optional) The maximum concurrency.
        :param max_errors: (Optional) The maximum errors.
        :param notification_config: (Optional) The configurations for sending notifications.
        :param output_s3_bucket_name: (Optional) The name of the S3 bucket for command output responses.
        :param output_s3_key_prefix: (Optional) The prefix.
        :param parameters: (Optional) The required and optional parameters specified in the document.
        :param service_role_arn: (Optional) The ARN of the AWS Identity and Access Management (IAM) role.
        :param timeout_seconds: (Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance. If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        props = RunCommandProps(
            document_name=document_name,
            targets=targets,
            cloud_watch_output_config=cloud_watch_output_config,
            comment=comment,
            document_hash=document_hash,
            document_hash_type=document_hash_type,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            notification_config=notification_config,
            output_s3_bucket_name=output_s3_bucket_name,
            output_s3_key_prefix=output_s3_key_prefix,
            parameters=parameters,
            service_role_arn=service_role_arn,
            timeout_seconds=timeout_seconds,
        )

        return typing.cast(RunCommandOutputs, jsii.invoke(self, "execute", [props]))

    @builtins.property
    @jsii.member(jsii_name="awsInvoker")
    def aws_invoker(self) -> IAwsInvoker:
        return typing.cast(IAwsInvoker, jsii.get(self, "awsInvoker"))

    @builtins.property
    @jsii.member(jsii_name="props")
    def props(self) -> ApiRunCommandProps:
        return typing.cast(ApiRunCommandProps, jsii.get(self, "props"))

    @builtins.property
    @jsii.member(jsii_name="sleepHook")
    def sleep_hook(self) -> ISleepHook:
        return typing.cast(ISleepHook, jsii.get(self, "sleepHook"))


@jsii.implements(IApproveHook)
class ApproveImpl(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ApproveImpl",
):
    '''This IApproveHook implementation provides a real ask and waits for user input of Enter.

    This implementation does not work well on all exported JSII languages.
    Users can provide their own impl using the IAskHook interface.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="ask")
    def ask(self, approver: builtins.str) -> builtins.bool:
        '''Ask for approval.

        :param approver: -
        '''
        if __debug__:
            def stub(approver: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument approver", value=approver, expected_type=type_hints["approver"])
        return typing.cast(builtins.bool, jsii.invoke(self, "ask", [approver]))


class ApproveSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ApproveSimulation",
):
    '''AutomationStep implementation for aws:approve https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-approve.html.'''

    def __init__(self, step: "ApproveStep", *, approve_hook: IApproveHook) -> None:
        '''
        :param step: -
        :param approve_hook: (Optional) Approve hook to be called to pause the execution. To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class. Default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        '''
        if __debug__:
            def stub(step: "ApproveStep", *, approve_hook: IApproveHook) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = ApproveSimulationProps(approve_hook=approve_hook)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''May perform a real approval ask based on the params used during instance creation.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @jsii.member(jsii_name="provideDate")
    def _provide_date(self) -> datetime.datetime:
        '''Override to mock the date the reviewer approved.'''
        return typing.cast(datetime.datetime, jsii.invoke(self, "provideDate", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="approveHook")
    def approve_hook(self) -> IApproveHook:
        return typing.cast(IApproveHook, jsii.get(self, "approveHook"))


class ArnDoc(
    DocumentSource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ArnDoc",
):
    def __init__(self, arn: IStringVariable) -> None:
        '''
        :param arn: -
        '''
        if __debug__:
            def stub(arn: IStringVariable) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
        jsii.create(self.__class__, self, [arn])

    @jsii.member(jsii_name="formatRequest")
    def format_request(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatRequest", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property
    @jsii.member(jsii_name="arn")
    def arn(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "arn"))


class AssertAwsResourceSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AssertAwsResourceSimulation",
):
    '''AutomationStep implementation of aws:assertAwsResourceProperty.

    https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-assertAwsResourceProperty.html
    '''

    def __init__(
        self,
        step: "AssertAwsResourceStep",
        *,
        aws_invoker: IAwsInvoker,
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(
                step: "AssertAwsResourceStep",
                *,
                aws_invoker: IAwsInvoker,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument step", value=step, expected_type=type_hints["step"])
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''If the value found matches one of the desiredValues, then this function returns.

        Otherwise it throws.

        :param inputs: -

        :return: empty object if value matches desiredValues.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class AutomationDocument(
    SsmDocument,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationDocument",
):
    '''The AutomationDocuemnt used to both build the SSM Automation yaml/json and to use in simulation.

    The AutomationDocument will delegate execution responsibility to the AutomationSteps that it receives.
    The SsmDocument parent class contains methods to runSimulation() as well as print().
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        assume_role: typing.Optional[IStringVariable] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
        target_type: typing.Optional[builtins.str] = None,
        update_method: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param update_method: If the document resource you specify in your template already exists, this parameter determines whether a new version of the existing document is created, or the existing document is replaced. ``Replace`` is the default method. If you specify ``NewVersion`` for the ``UpdateMethod`` parameter, and the ``Name`` of the document does not match an existing resource, a new document is created. When you specify ``NewVersion`` , the default version of the document is changed to the newly created version.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                assume_role: typing.Optional[IStringVariable] = None,
                description: typing.Optional[builtins.str] = None,
                doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
                doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
                document_format: typing.Optional[DocumentFormat] = None,
                document_name: typing.Optional[builtins.str] = None,
                header: typing.Optional[builtins.str] = None,
                requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
                target_type: typing.Optional[builtins.str] = None,
                update_method: typing.Optional[builtins.str] = None,
                version_name: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AutomationDocumentProps(
            assume_role=assume_role,
            description=description,
            doc_inputs=doc_inputs,
            doc_outputs=doc_outputs,
            document_format=document_format,
            document_name=document_name,
            header=header,
            requires=requires,
            tags=tags,
            target_type=target_type,
            update_method=update_method,
            version_name=version_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addStep")
    def add_step(self, component: IAutomationComponent) -> None:
        '''
        :param component: -
        '''
        if __debug__:
            def stub(component: IAutomationComponent) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument component", value=component, expected_type=type_hints["component"])
        return typing.cast(None, jsii.invoke(self, "addStep", [component]))

    @jsii.member(jsii_name="buildSsmDocument")
    def _build_ssm_document(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Delegates building the SSM Document to be converted to a yaml/json to the subclass (Automation etc).

        :return: an object that can be used to build the json/yaml string representation of this document.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "buildSsmDocument", []))

    @jsii.member(jsii_name="collectedSteps")
    def collected_steps(self) -> typing.List["AutomationStep"]:
        return typing.cast(typing.List["AutomationStep"], jsii.invoke(self, "collectedSteps", []))

    @jsii.member(jsii_name="documentType")
    def document_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "documentType", []))

    @builtins.property
    @jsii.member(jsii_name="builder")
    def builder(self) -> AutomationDocumentBuilder:
        return typing.cast(AutomationDocumentBuilder, jsii.get(self, "builder"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationDocumentProps",
    jsii_struct_bases=[SsmDocumentProps],
    name_mapping={
        "assume_role": "assumeRole",
        "description": "description",
        "doc_inputs": "docInputs",
        "doc_outputs": "docOutputs",
        "document_format": "documentFormat",
        "document_name": "documentName",
        "header": "header",
        "requires": "requires",
        "tags": "tags",
        "target_type": "targetType",
        "update_method": "updateMethod",
        "version_name": "versionName",
    },
)
class AutomationDocumentProps(SsmDocumentProps):
    def __init__(
        self,
        *,
        assume_role: typing.Optional[IStringVariable] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
        target_type: typing.Optional[builtins.str] = None,
        update_method: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options for AutomationDocument.

        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param update_method: If the document resource you specify in your template already exists, this parameter determines whether a new version of the existing document is created, or the existing document is replaced. ``Replace`` is the default method. If you specify ``NewVersion`` for the ``UpdateMethod`` parameter, and the ``Name`` of the document does not match an existing resource, a new document is created. When you specify ``NewVersion`` , the default version of the document is changed to the newly created version.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        if __debug__:
            def stub(
                *,
                assume_role: typing.Optional[IStringVariable] = None,
                description: typing.Optional[builtins.str] = None,
                doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
                doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
                document_format: typing.Optional[DocumentFormat] = None,
                document_name: typing.Optional[builtins.str] = None,
                header: typing.Optional[builtins.str] = None,
                requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
                target_type: typing.Optional[builtins.str] = None,
                update_method: typing.Optional[builtins.str] = None,
                version_name: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument assume_role", value=assume_role, expected_type=type_hints["assume_role"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument doc_inputs", value=doc_inputs, expected_type=type_hints["doc_inputs"])
            check_type(argname="argument doc_outputs", value=doc_outputs, expected_type=type_hints["doc_outputs"])
            check_type(argname="argument document_format", value=document_format, expected_type=type_hints["document_format"])
            check_type(argname="argument document_name", value=document_name, expected_type=type_hints["document_name"])
            check_type(argname="argument header", value=header, expected_type=type_hints["header"])
            check_type(argname="argument requires", value=requires, expected_type=type_hints["requires"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument target_type", value=target_type, expected_type=type_hints["target_type"])
            check_type(argname="argument update_method", value=update_method, expected_type=type_hints["update_method"])
            check_type(argname="argument version_name", value=version_name, expected_type=type_hints["version_name"])
        self._values: typing.Dict[str, typing.Any] = {}
        if assume_role is not None:
            self._values["assume_role"] = assume_role
        if description is not None:
            self._values["description"] = description
        if doc_inputs is not None:
            self._values["doc_inputs"] = doc_inputs
        if doc_outputs is not None:
            self._values["doc_outputs"] = doc_outputs
        if document_format is not None:
            self._values["document_format"] = document_format
        if document_name is not None:
            self._values["document_name"] = document_name
        if header is not None:
            self._values["header"] = header
        if requires is not None:
            self._values["requires"] = requires
        if tags is not None:
            self._values["tags"] = tags
        if target_type is not None:
            self._values["target_type"] = target_type
        if update_method is not None:
            self._values["update_method"] = update_method
        if version_name is not None:
            self._values["version_name"] = version_name

    @builtins.property
    def assume_role(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Assume role to use for this document.

        If provided, this value MUST be included as one of the documentInput names.
        '''
        result = self._values.get("assume_role")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) Description of the document.

        Defaults to the document name.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def doc_inputs(self) -> typing.Optional[typing.List[Input]]:
        '''(Optional) Inputs required by the document.'''
        result = self._values.get("doc_inputs")
        return typing.cast(typing.Optional[typing.List[Input]], result)

    @builtins.property
    def doc_outputs(self) -> typing.Optional[typing.List[DocumentOutput]]:
        '''(Optional) Outputs to be emitted from the document.

        The outputs are placed in a StringSet called outputs (as is done in SSM).

        :default: []
        '''
        result = self._values.get("doc_outputs")
        return typing.cast(typing.Optional[typing.List[DocumentOutput]], result)

    @builtins.property
    def document_format(self) -> typing.Optional[DocumentFormat]:
        '''(Optional) Specifies whether this document should be written as YAML or JSON.

        :default: JSON
        '''
        result = self._values.get("document_format")
        return typing.cast(typing.Optional[DocumentFormat], result)

    @builtins.property
    def document_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the document.

        Will default to the id provided for the CDK node.
        '''
        result = self._values.get("document_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def header(self) -> typing.Optional[builtins.str]:
        '''(Optional) A Header/comment to include at the start of a YAML document.

        JSON documents do not support headers.
        '''
        result = self._values.get("header")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def requires(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]]:
        '''``AWS::SSM::Document.Requires``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-requires
        '''
        result = self._values.get("requires")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.CfnTag]]:
        '''``AWS::SSM::Document.Tags``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.CfnTag]], result)

    @builtins.property
    def target_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.TargetType``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-targettype
        '''
        result = self._values.get("target_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def update_method(self) -> typing.Optional[builtins.str]:
        '''If the document resource you specify in your template already exists, this parameter determines whether a new version of the existing document is created, or the existing document is replaced.

        ``Replace`` is the default method. If you specify ``NewVersion`` for the ``UpdateMethod`` parameter, and the ``Name`` of the document does not match an existing resource, a new document is created. When you specify ``NewVersion`` , the default version of the document is changed to the newly created version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-updatemethod
        '''
        result = self._values.get("update_method")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.VersionName``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-versionname
        '''
        result = self._values.get("version_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutomationDocumentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IAutomationComponent)
class AutomationStep(
    Step,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationStep",
):
    '''Parent class for AutomationSteps.

    These steps are using in an AutomationDocument.
    You can instantiate steps using the AutomationBuilder for convenience.
    You can use these steps to simulate their execution (mimics the SSM run) AND to build their yaml/json declaration.
    Control flow of the subsequent step is determined by the currently executing step.
    The flow of the execution therefore follows a chain-of-responsibility pattern.
    The inputs received into a step AND the outputs of previous steps are merged to form inputs of subsequent steps.
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AutomationStepProps(
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        if __debug__:
            def stub(doc: AutomationDocumentBuilder) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument doc", value=doc, expected_type=type_hints["doc"])
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="listUserOutputs")
    def list_user_outputs(self) -> typing.List[Output]:
        '''Lists the outputs defined by the user for this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listUserOutputs", []))

    @jsii.member(jsii_name="prepareSsmEntry")
    def _prepare_ssm_entry(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "prepareSsmEntry", [inputs]))

    @jsii.member(jsii_name="variables")
    def variables(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "variables", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DEFAULT_MAX_ATTEMPTS")
    def DEFAULT_MAX_ATTEMPTS(cls) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.sget(cls, "DEFAULT_MAX_ATTEMPTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DEFAULT_TIMEOUT")
    def DEFAULT_TIMEOUT(cls) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.sget(cls, "DEFAULT_TIMEOUT"))

    @builtins.property
    @jsii.member(jsii_name="isEnd")
    def is_end(self) -> builtins.bool:
        return typing.cast(builtins.bool, jsii.get(self, "isEnd"))

    @builtins.property
    @jsii.member(jsii_name="maxAttempts")
    def max_attempts(self) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.get(self, "maxAttempts"))

    @builtins.property
    @jsii.member(jsii_name="onCancel")
    def on_cancel(self) -> "OnCancel":
        return typing.cast("OnCancel", jsii.get(self, "onCancel"))

    @builtins.property
    @jsii.member(jsii_name="onFailure")
    def on_failure(self) -> OnFailure:
        return typing.cast(OnFailure, jsii.get(self, "onFailure"))

    @builtins.property
    @jsii.member(jsii_name="timeoutSeconds")
    def timeout_seconds(self) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.get(self, "timeoutSeconds"))

    @builtins.property
    @jsii.member(jsii_name="explicitNextStep")
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        return typing.cast(typing.Optional[StepRef], jsii.get(self, "explicitNextStep"))

    @builtins.property
    @jsii.member(jsii_name="allStepsInExecution")
    def all_steps_in_execution(self) -> typing.Optional[typing.List["AutomationStep"]]:
        return typing.cast(typing.Optional[typing.List["AutomationStep"]], jsii.get(self, "allStepsInExecution"))

    @all_steps_in_execution.setter
    def all_steps_in_execution(
        self,
        value: typing.Optional[typing.List["AutomationStep"]],
    ) -> None:
        if __debug__:
            def stub(value: typing.Optional[typing.List["AutomationStep"]]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allStepsInExecution", value)

    @builtins.property
    @jsii.member(jsii_name="nextStep")
    def next_step(self) -> typing.Optional["AutomationStep"]:
        return typing.cast(typing.Optional["AutomationStep"], jsii.get(self, "nextStep"))

    @next_step.setter
    def next_step(self, value: typing.Optional["AutomationStep"]) -> None:
        if __debug__:
            def stub(value: typing.Optional["AutomationStep"]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "nextStep", value)


class _AutomationStepProxy(
    AutomationStep,
    jsii.proxy_for(Step), # type: ignore[misc]
):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AutomationStep).__jsii_proxy_class__ = lambda : _AutomationStepProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationStepProps",
    jsii_struct_bases=[StepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
    },
)
class AutomationStepProps(StepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
        self._values: typing.Dict[str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutomationStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AwsApiStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AwsApiStep",
):
    '''AutomationStep implementation of aws:executeAwsApi.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        outputs: typing.Sequence[typing.Union[Output, typing.Dict[str, typing.Any]]],
        api_params: typing.Mapping[builtins.str, typing.Any],
        pascal_case_api: builtins.str,
        service: AwsService,
        java_script_api: typing.Optional[builtins.str] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param outputs: (Required) specify the outputs to extract from the JavaScript JSON response.
        :param api_params: (Required) API Params to submit with the request to the api. You may include variables which will be substitued for inputs during step execution as such {{ INPUT }}
        :param pascal_case_api: (Required) The AWS api represented in PascalCase. This value is used as-is in the SSM yaml/json. This is used as the default for javaScriptApi (see that param).
        :param service: (Required) The AWS service to be invoked.
        :param java_script_api: (Optional) The api as represented the AWS JavaScript API. This is usually lowerCamelCase. This is used in the simulation run to execute the AWS API against the JavaScript SDK. Default: - will use the camelCaseApi param and substitute the first character for lowercase by default.
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                outputs: typing.Sequence[typing.Union[Output, typing.Dict[str, typing.Any]]],
                api_params: typing.Mapping[builtins.str, typing.Any],
                pascal_case_api: builtins.str,
                service: AwsService,
                java_script_api: typing.Optional[builtins.str] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AwsApiStepProps(
            outputs=outputs,
            api_params=api_params,
            pascal_case_api=pascal_case_api,
            service=service,
            java_script_api=java_script_api,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Derives the inputs by parsing the apiParams to find matches for inputs in double circle braces ("{{ INPUT }}").

        :return: list of required inputs.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listUserOutputs")
    def list_user_outputs(self) -> typing.List[Output]:
        '''Lists the outputs defined by the user for this step.

        :return: Outputs as specified in params
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listUserOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="apiParams")
    def api_params(self) -> "DictFormat":
        return typing.cast("DictFormat", jsii.get(self, "apiParams"))

    @builtins.property
    @jsii.member(jsii_name="javaScriptApi")
    def java_script_api(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "javaScriptApi"))

    @builtins.property
    @jsii.member(jsii_name="outputs")
    def outputs(self) -> typing.List[Output]:
        return typing.cast(typing.List[Output], jsii.get(self, "outputs"))

    @builtins.property
    @jsii.member(jsii_name="pascalCaseApi")
    def pascal_case_api(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "pascalCaseApi"))

    @builtins.property
    @jsii.member(jsii_name="service")
    def service(self) -> AwsService:
        return typing.cast(AwsService, jsii.get(self, "service"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AwsInvocationProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "api_params": "apiParams",
        "pascal_case_api": "pascalCaseApi",
        "service": "service",
        "java_script_api": "javaScriptApi",
    },
)
class AwsInvocationProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        api_params: typing.Mapping[builtins.str, typing.Any],
        pascal_case_api: builtins.str,
        service: AwsService,
        java_script_api: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param api_params: (Required) API Params to submit with the request to the api. You may include variables which will be substitued for inputs during step execution as such {{ INPUT }}
        :param pascal_case_api: (Required) The AWS api represented in PascalCase. This value is used as-is in the SSM yaml/json. This is used as the default for javaScriptApi (see that param).
        :param service: (Required) The AWS service to be invoked.
        :param java_script_api: (Optional) The api as represented the AWS JavaScript API. This is usually lowerCamelCase. This is used in the simulation run to execute the AWS API against the JavaScript SDK. Default: - will use the camelCaseApi param and substitute the first character for lowercase by default.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                api_params: typing.Mapping[builtins.str, typing.Any],
                pascal_case_api: builtins.str,
                service: AwsService,
                java_script_api: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument api_params", value=api_params, expected_type=type_hints["api_params"])
            check_type(argname="argument pascal_case_api", value=pascal_case_api, expected_type=type_hints["pascal_case_api"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument java_script_api", value=java_script_api, expected_type=type_hints["java_script_api"])
        self._values: typing.Dict[str, typing.Any] = {
            "api_params": api_params,
            "pascal_case_api": pascal_case_api,
            "service": service,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if java_script_api is not None:
            self._values["java_script_api"] = java_script_api

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def api_params(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''(Required) API Params to submit with the request to the api.

        You may include variables which will be substitued for inputs during step execution as such {{ INPUT }}

        Example::

            { 'VolumeIds': ['{{ EbsDescribeInstance.VolumeId }}'] }
        '''
        result = self._values.get("api_params")
        assert result is not None, "Required property 'api_params' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    @builtins.property
    def pascal_case_api(self) -> builtins.str:
        '''(Required) The AWS api represented in PascalCase.

        This value is used as-is in the SSM yaml/json.
        This is used as the default for javaScriptApi (see that param).

        Example::

            DescribeInstances
        '''
        result = self._values.get("pascal_case_api")
        assert result is not None, "Required property 'pascal_case_api' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> AwsService:
        '''(Required) The AWS service to be invoked.

        Example::

            AwsService.S3
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(AwsService, result)

    @builtins.property
    def java_script_api(self) -> typing.Optional[builtins.str]:
        '''(Optional) The api as represented the AWS JavaScript API.

        This is usually lowerCamelCase.
        This is used in the simulation run to execute the AWS API against the JavaScript SDK.

        :default: - will use the camelCaseApi param and substitute the first character for lowercase by default.

        Example::

            describeInstances
        '''
        result = self._values.get("java_script_api")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsInvocationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BranchStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.BranchStep",
):
    '''AutomationStep implementation of aws:branch https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-branch.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        choices: typing.Sequence[Choice],
        default_step_name: typing.Optional[builtins.str] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param choices: (Required) list of choices. The first matched choice will be used to jump to the step specified in the choice.
        :param default_step_name: (Optional) default step in all of the choices evaluate to false. Default: undefined - the next step in the chain will be invoked. See AWS Documentation for branch below.
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                choices: typing.Sequence[Choice],
                default_step_name: typing.Optional[builtins.str] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BranchStepProps(
            choices=choices,
            default_step_name=default_step_name,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.

        :return: all of the inputsToTest from the choices provided to the constructor
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''There is no output from branch steps.

        :return: empty list
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="choices")
    def choices(self) -> typing.List[Choice]:
        return typing.cast(typing.List[Choice], jsii.get(self, "choices"))

    @builtins.property
    @jsii.member(jsii_name="defaultStepName")
    def default_step_name(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "defaultStepName"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.BranchStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "choices": "choices",
        "default_step_name": "defaultStepName",
    },
)
class BranchStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        choices: typing.Sequence[Choice],
        default_step_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param choices: (Required) list of choices. The first matched choice will be used to jump to the step specified in the choice.
        :param default_step_name: (Optional) default step in all of the choices evaluate to false. Default: undefined - the next step in the chain will be invoked. See AWS Documentation for branch below.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                choices: typing.Sequence[Choice],
                default_step_name: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument choices", value=choices, expected_type=type_hints["choices"])
            check_type(argname="argument default_step_name", value=default_step_name, expected_type=type_hints["default_step_name"])
        self._values: typing.Dict[str, typing.Any] = {
            "choices": choices,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if default_step_name is not None:
            self._values["default_step_name"] = default_step_name

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def choices(self) -> typing.List[Choice]:
        '''(Required) list of choices.

        The first matched choice will be used to jump to the step specified in the choice.
        '''
        result = self._values.get("choices")
        assert result is not None, "Required property 'choices' is missing"
        return typing.cast(typing.List[Choice], result)

    @builtins.property
    def default_step_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) default step in all of the choices evaluate to false.

        :default: undefined - the next step in the chain will be invoked. See AWS Documentation for branch below.
        '''
        result = self._values.get("default_step_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BranchStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ChangeInstanceStateSimulationProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "aws_invoker": "awsInvoker",
    },
)
class ChangeInstanceStateSimulationProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        aws_invoker: IAwsInvoker,
    ) -> None:
        '''Properties for ChangeInstanceStateStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                aws_invoker: IAwsInvoker,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument aws_invoker", value=aws_invoker, expected_type=type_hints["aws_invoker"])
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def aws_invoker(self) -> IAwsInvoker:
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast(IAwsInvoker, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ChangeInstanceStateSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ChangeInstanceStateStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ChangeInstanceStateStep",
):
    '''AutomationStep implemenation for aws:changeInstanceState https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-changestate.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        desired_state: "IDesiredStateVariable",
        instance_ids: IStringListVariable,
        additional_info: typing.Optional[IStringVariable] = None,
        check_state_only: typing.Optional["IBooleanVariable"] = None,
        force: typing.Optional["IBooleanVariable"] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param desired_state: The desired state. When set to running, this action waits for the Amazon EC2 state to be Running, the Instance Status to be OK, and the System Status to be OK before completing.
        :param instance_ids: The IDs of the instances.
        :param additional_info: (Optional) Reserved.
        :param check_state_only: (Optional) If false, sets the instance state to the desired state. If true, asserts the desired state using polling. Default: false
        :param force: (Optional) If set, forces the instances to stop. The instances don't have an opportunity to flush file system caches or file system metadata. If you use this option, you must perform file system check and repair procedures. This option isn't recommended for EC2 instances for Windows Server.
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                desired_state: "IDesiredStateVariable",
                instance_ids: IStringListVariable,
                additional_info: typing.Optional[IStringVariable] = None,
                check_state_only: typing.Optional["IBooleanVariable"] = None,
                force: typing.Optional["IBooleanVariable"] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ChangeInstanceStateStepProps(
            desired_state=desired_state,
            instance_ids=instance_ids,
            additional_info=additional_info,
            check_state_only=check_state_only,
            force=force,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''This step has no outputs.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="desiredState")
    def desired_state(self) -> "IDesiredStateVariable":
        return typing.cast("IDesiredStateVariable", jsii.get(self, "desiredState"))

    @builtins.property
    @jsii.member(jsii_name="instanceIds")
    def instance_ids(self) -> IStringListVariable:
        return typing.cast(IStringListVariable, jsii.get(self, "instanceIds"))

    @builtins.property
    @jsii.member(jsii_name="additionalInfo")
    def additional_info(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "additionalInfo"))

    @builtins.property
    @jsii.member(jsii_name="checkStateOnly")
    def check_state_only(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "checkStateOnly"))

    @builtins.property
    @jsii.member(jsii_name="force")
    def force(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "force"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ChangeInstanceStateStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "desired_state": "desiredState",
        "instance_ids": "instanceIds",
        "additional_info": "additionalInfo",
        "check_state_only": "checkStateOnly",
        "force": "force",
    },
)
class ChangeInstanceStateStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        desired_state: "IDesiredStateVariable",
        instance_ids: IStringListVariable,
        additional_info: typing.Optional[IStringVariable] = None,
        check_state_only: typing.Optional["IBooleanVariable"] = None,
        force: typing.Optional["IBooleanVariable"] = None,
    ) -> None:
        '''Properties for ChangeInstanceStateStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param desired_state: The desired state. When set to running, this action waits for the Amazon EC2 state to be Running, the Instance Status to be OK, and the System Status to be OK before completing.
        :param instance_ids: The IDs of the instances.
        :param additional_info: (Optional) Reserved.
        :param check_state_only: (Optional) If false, sets the instance state to the desired state. If true, asserts the desired state using polling. Default: false
        :param force: (Optional) If set, forces the instances to stop. The instances don't have an opportunity to flush file system caches or file system metadata. If you use this option, you must perform file system check and repair procedures. This option isn't recommended for EC2 instances for Windows Server.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                desired_state: "IDesiredStateVariable",
                instance_ids: IStringListVariable,
                additional_info: typing.Optional[IStringVariable] = None,
                check_state_only: typing.Optional["IBooleanVariable"] = None,
                force: typing.Optional["IBooleanVariable"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument desired_state", value=desired_state, expected_type=type_hints["desired_state"])
            check_type(argname="argument instance_ids", value=instance_ids, expected_type=type_hints["instance_ids"])
            check_type(argname="argument additional_info", value=additional_info, expected_type=type_hints["additional_info"])
            check_type(argname="argument check_state_only", value=check_state_only, expected_type=type_hints["check_state_only"])
            check_type(argname="argument force", value=force, expected_type=type_hints["force"])
        self._values: typing.Dict[str, typing.Any] = {
            "desired_state": desired_state,
            "instance_ids": instance_ids,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if additional_info is not None:
            self._values["additional_info"] = additional_info
        if check_state_only is not None:
            self._values["check_state_only"] = check_state_only
        if force is not None:
            self._values["force"] = force

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def desired_state(self) -> "IDesiredStateVariable":
        '''The desired state.

        When set to running, this action waits for the Amazon EC2 state to be Running, the Instance Status to be OK,
        and the System Status to be OK before completing.
        '''
        result = self._values.get("desired_state")
        assert result is not None, "Required property 'desired_state' is missing"
        return typing.cast("IDesiredStateVariable", result)

    @builtins.property
    def instance_ids(self) -> IStringListVariable:
        '''The IDs of the instances.'''
        result = self._values.get("instance_ids")
        assert result is not None, "Required property 'instance_ids' is missing"
        return typing.cast(IStringListVariable, result)

    @builtins.property
    def additional_info(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Reserved.'''
        result = self._values.get("additional_info")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def check_state_only(self) -> typing.Optional["IBooleanVariable"]:
        '''(Optional) If false, sets the instance state to the desired state.

        If true, asserts the desired state using polling.

        :default: false
        '''
        result = self._values.get("check_state_only")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    @builtins.property
    def force(self) -> typing.Optional["IBooleanVariable"]:
        '''(Optional) If set, forces the instances to stop.

        The instances don't have an opportunity to flush file system caches or file system metadata.
        If you use this option, you must perform file system check and repair procedures.
        This option isn't recommended for EC2 instances for Windows Server.
        '''
        result = self._values.get("force")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ChangeInstanceStateStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CommandDocument(
    SsmDocument,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandDocument",
):
    '''https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html The CommandDocument will delegate execution responsibility to the CammandSteps that it receives. The SsmDocument parent class contains methods to runSimulation() as well as print().'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        assume_role: typing.Optional[IStringVariable] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
        target_type: typing.Optional[builtins.str] = None,
        update_method: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param update_method: If the document resource you specify in your template already exists, this parameter determines whether a new version of the existing document is created, or the existing document is replaced. ``Replace`` is the default method. If you specify ``NewVersion`` for the ``UpdateMethod`` parameter, and the ``Name`` of the document does not match an existing resource, a new document is created. When you specify ``NewVersion`` , the default version of the document is changed to the newly created version.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                assume_role: typing.Optional[IStringVariable] = None,
                description: typing.Optional[builtins.str] = None,
                doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
                doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
                document_format: typing.Optional[DocumentFormat] = None,
                document_name: typing.Optional[builtins.str] = None,
                header: typing.Optional[builtins.str] = None,
                requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
                target_type: typing.Optional[builtins.str] = None,
                update_method: typing.Optional[builtins.str] = None,
                version_name: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CommandDocumentProps(
            assume_role=assume_role,
            description=description,
            doc_inputs=doc_inputs,
            doc_outputs=doc_outputs,
            document_format=document_format,
            document_name=document_name,
            header=header,
            requires=requires,
            tags=tags,
            target_type=target_type,
            update_method=update_method,
            version_name=version_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addStep")
    def add_step(self, component: ICommandComponent) -> None:
        '''
        :param component: -
        '''
        if __debug__:
            def stub(component: ICommandComponent) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument component", value=component, expected_type=type_hints["component"])
        return typing.cast(None, jsii.invoke(self, "addStep", [component]))

    @jsii.member(jsii_name="buildSsmDocument")
    def _build_ssm_document(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Delegates building the SSM Document to be converted to a yaml/json to the subclass (Automation etc).'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "buildSsmDocument", []))

    @jsii.member(jsii_name="collectedSteps")
    def collected_steps(self) -> typing.List["CommandStep"]:
        return typing.cast(typing.List["CommandStep"], jsii.invoke(self, "collectedSteps", []))

    @jsii.member(jsii_name="documentType")
    def document_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "documentType", []))

    @builtins.property
    @jsii.member(jsii_name="builder")
    def builder(self) -> CommandDocumentBuilder:
        return typing.cast(CommandDocumentBuilder, jsii.get(self, "builder"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CommandDocumentProps",
    jsii_struct_bases=[SsmDocumentProps],
    name_mapping={
        "assume_role": "assumeRole",
        "description": "description",
        "doc_inputs": "docInputs",
        "doc_outputs": "docOutputs",
        "document_format": "documentFormat",
        "document_name": "documentName",
        "header": "header",
        "requires": "requires",
        "tags": "tags",
        "target_type": "targetType",
        "update_method": "updateMethod",
        "version_name": "versionName",
    },
)
class CommandDocumentProps(SsmDocumentProps):
    def __init__(
        self,
        *,
        assume_role: typing.Optional[IStringVariable] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
        target_type: typing.Optional[builtins.str] = None,
        update_method: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param update_method: If the document resource you specify in your template already exists, this parameter determines whether a new version of the existing document is created, or the existing document is replaced. ``Replace`` is the default method. If you specify ``NewVersion`` for the ``UpdateMethod`` parameter, and the ``Name`` of the document does not match an existing resource, a new document is created. When you specify ``NewVersion`` , the default version of the document is changed to the newly created version.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        if __debug__:
            def stub(
                *,
                assume_role: typing.Optional[IStringVariable] = None,
                description: typing.Optional[builtins.str] = None,
                doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
                doc_outputs: typing.Optional[typing.Sequence[typing.Union[DocumentOutput, typing.Dict[str, typing.Any]]]] = None,
                document_format: typing.Optional[DocumentFormat] = None,
                document_name: typing.Optional[builtins.str] = None,
                header: typing.Optional[builtins.str] = None,
                requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, typing.Dict[str, typing.Any]], aws_cdk.IResolvable]]]] = None,
                tags: typing.Optional[typing.Sequence[typing.Union[aws_cdk.CfnTag, typing.Dict[str, typing.Any]]]] = None,
                target_type: typing.Optional[builtins.str] = None,
                update_method: typing.Optional[builtins.str] = None,
                version_name: typing.Optional[builtins.str] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument assume_role", value=assume_role, expected_type=type_hints["assume_role"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument doc_inputs", value=doc_inputs, expected_type=type_hints["doc_inputs"])
            check_type(argname="argument doc_outputs", value=doc_outputs, expected_type=type_hints["doc_outputs"])
            check_type(argname="argument document_format", value=document_format, expected_type=type_hints["document_format"])
            check_type(argname="argument document_name", value=document_name, expected_type=type_hints["document_name"])
            check_type(argname="argument header", value=header, expected_type=type_hints["header"])
            check_type(argname="argument requires", value=requires, expected_type=type_hints["requires"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument target_type", value=target_type, expected_type=type_hints["target_type"])
            check_type(argname="argument update_method", value=update_method, expected_type=type_hints["update_method"])
            check_type(argname="argument version_name", value=version_name, expected_type=type_hints["version_name"])
        self._values: typing.Dict[str, typing.Any] = {}
        if assume_role is not None:
            self._values["assume_role"] = assume_role
        if description is not None:
            self._values["description"] = description
        if doc_inputs is not None:
            self._values["doc_inputs"] = doc_inputs
        if doc_outputs is not None:
            self._values["doc_outputs"] = doc_outputs
        if document_format is not None:
            self._values["document_format"] = document_format
        if document_name is not None:
            self._values["document_name"] = document_name
        if header is not None:
            self._values["header"] = header
        if requires is not None:
            self._values["requires"] = requires
        if tags is not None:
            self._values["tags"] = tags
        if target_type is not None:
            self._values["target_type"] = target_type
        if update_method is not None:
            self._values["update_method"] = update_method
        if version_name is not None:
            self._values["version_name"] = version_name

    @builtins.property
    def assume_role(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Assume role to use for this document.

        If provided, this value MUST be included as one of the documentInput names.
        '''
        result = self._values.get("assume_role")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) Description of the document.

        Defaults to the document name.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def doc_inputs(self) -> typing.Optional[typing.List[Input]]:
        '''(Optional) Inputs required by the document.'''
        result = self._values.get("doc_inputs")
        return typing.cast(typing.Optional[typing.List[Input]], result)

    @builtins.property
    def doc_outputs(self) -> typing.Optional[typing.List[DocumentOutput]]:
        '''(Optional) Outputs to be emitted from the document.

        The outputs are placed in a StringSet called outputs (as is done in SSM).

        :default: []
        '''
        result = self._values.get("doc_outputs")
        return typing.cast(typing.Optional[typing.List[DocumentOutput]], result)

    @builtins.property
    def document_format(self) -> typing.Optional[DocumentFormat]:
        '''(Optional) Specifies whether this document should be written as YAML or JSON.

        :default: JSON
        '''
        result = self._values.get("document_format")
        return typing.cast(typing.Optional[DocumentFormat], result)

    @builtins.property
    def document_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the document.

        Will default to the id provided for the CDK node.
        '''
        result = self._values.get("document_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def header(self) -> typing.Optional[builtins.str]:
        '''(Optional) A Header/comment to include at the start of a YAML document.

        JSON documents do not support headers.
        '''
        result = self._values.get("header")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def requires(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]]:
        '''``AWS::SSM::Document.Requires``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-requires
        '''
        result = self._values.get("requires")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.CfnTag]]:
        '''``AWS::SSM::Document.Tags``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.CfnTag]], result)

    @builtins.property
    def target_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.TargetType``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-targettype
        '''
        result = self._values.get("target_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def update_method(self) -> typing.Optional[builtins.str]:
        '''If the document resource you specify in your template already exists, this parameter determines whether a new version of the existing document is created, or the existing document is replaced.

        ``Replace`` is the default method. If you specify ``NewVersion`` for the ``UpdateMethod`` parameter, and the ``Name`` of the document does not match an existing resource, a new document is created. When you specify ``NewVersion`` , the default version of the document is changed to the newly created version.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-updatemethod
        '''
        result = self._values.get("update_method")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.VersionName``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-versionname
        '''
        result = self._values.get("version_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommandDocumentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(ICommandComponent)
class CommandStep(
    Step,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandStep",
):
    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                exit_on_failure: typing.Optional[builtins.bool] = None,
                exit_on_success: typing.Optional[builtins.bool] = None,
                finally_step: typing.Optional[builtins.bool] = None,
                mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
                on_cancel: typing.Optional[Step] = None,
                precondition: typing.Optional[Precondition] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CommandStepProps(
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: CommandDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        if __debug__:
            def stub(doc: CommandDocumentBuilder) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument doc", value=doc, expected_type=type_hints["doc"])
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''RunCommand Steps do not have outputs.

        :return: []
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="prepareSsmEntry")
    def _prepare_ssm_entry(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "prepareSsmEntry", [inputs]))

    @jsii.member(jsii_name="variables")
    def variables(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "variables", []))

    @builtins.property
    @jsii.member(jsii_name="exitOnFailure")
    def exit_on_failure(self) -> builtins.bool:
        return typing.cast(builtins.bool, jsii.get(self, "exitOnFailure"))

    @builtins.property
    @jsii.member(jsii_name="exitOnSuccess")
    def exit_on_success(self) -> builtins.bool:
        return typing.cast(builtins.bool, jsii.get(self, "exitOnSuccess"))

    @builtins.property
    @jsii.member(jsii_name="finallyStep")
    def finally_step(self) -> builtins.bool:
        return typing.cast(builtins.bool, jsii.get(self, "finallyStep"))

    @builtins.property
    @jsii.member(jsii_name="markSuccessAndExitOnFailure")
    def mark_success_and_exit_on_failure(self) -> builtins.bool:
        return typing.cast(builtins.bool, jsii.get(self, "markSuccessAndExitOnFailure"))

    @builtins.property
    @jsii.member(jsii_name="platforms")
    @abc.abstractmethod
    def platforms(self) -> typing.List[Platform]:
        ...

    @builtins.property
    @jsii.member(jsii_name="precondition")
    def precondition(self) -> typing.Optional[Precondition]:
        return typing.cast(typing.Optional[Precondition], jsii.get(self, "precondition"))

    @builtins.property
    @jsii.member(jsii_name="allStepsInExecution")
    def all_steps_in_execution(self) -> typing.Optional[typing.List["CommandStep"]]:
        return typing.cast(typing.Optional[typing.List["CommandStep"]], jsii.get(self, "allStepsInExecution"))

    @all_steps_in_execution.setter
    def all_steps_in_execution(
        self,
        value: typing.Optional[typing.List["CommandStep"]],
    ) -> None:
        if __debug__:
            def stub(value: typing.Optional[typing.List["CommandStep"]]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allStepsInExecution", value)

    @builtins.property
    @jsii.member(jsii_name="nextStep")
    def next_step(self) -> typing.Optional["CommandStep"]:
        return typing.cast(typing.Optional["CommandStep"], jsii.get(self, "nextStep"))

    @next_step.setter
    def next_step(self, value: typing.Optional["CommandStep"]) -> None:
        if __debug__:
            def stub(value: typing.Optional["CommandStep"]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "nextStep", value)


class _CommandStepProxy(
    CommandStep,
    jsii.proxy_for(Step), # type: ignore[misc]
):
    @builtins.property
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CommandStep).__jsii_proxy_class__ = lambda : _CommandStepProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CommandStepProps",
    jsii_struct_bases=[StepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
    },
)
class CommandStepProps(StepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                exit_on_failure: typing.Optional[builtins.bool] = None,
                exit_on_success: typing.Optional[builtins.bool] = None,
                finally_step: typing.Optional[builtins.bool] = None,
                mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
                on_cancel: typing.Optional[Step] = None,
                precondition: typing.Optional[Precondition] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument exit_on_failure", value=exit_on_failure, expected_type=type_hints["exit_on_failure"])
            check_type(argname="argument exit_on_success", value=exit_on_success, expected_type=type_hints["exit_on_success"])
            check_type(argname="argument finally_step", value=finally_step, expected_type=type_hints["finally_step"])
            check_type(argname="argument mark_success_and_exit_on_failure", value=mark_success_and_exit_on_failure, expected_type=type_hints["mark_success_and_exit_on_failure"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument precondition", value=precondition, expected_type=type_hints["precondition"])
        self._values: typing.Dict[str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommandStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IAutomationComponent)
class CompositeAutomationStep(
    constructs.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.CompositeAutomationStep",
):
    def __init__(self, scope: constructs.Construct, id: builtins.str) -> None:
        '''
        :param scope: -
        :param id: -
        '''
        if __debug__:
            def stub(scope: constructs.Construct, id: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        jsii.create(self.__class__, self, [scope, id])

    @jsii.member(jsii_name="addToDocument")
    @abc.abstractmethod
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        ...


class _CompositeAutomationStepProxy(CompositeAutomationStep):
    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        if __debug__:
            def stub(doc: AutomationDocumentBuilder) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument doc", value=doc, expected_type=type_hints["doc"])
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CompositeAutomationStep).__jsii_proxy_class__ = lambda : _CompositeAutomationStepProxy


@jsii.implements(ICommandComponent)
class CompositeCommandStep(
    constructs.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.CompositeCommandStep",
):
    def __init__(self, scope: constructs.Construct, id: builtins.str) -> None:
        '''
        :param scope: -
        :param id: -
        '''
        if __debug__:
            def stub(scope: constructs.Construct, id: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        jsii.create(self.__class__, self, [scope, id])

    @jsii.member(jsii_name="addToDocument")
    @abc.abstractmethod
    def add_to_document(self, doc: CommandDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        ...


class _CompositeCommandStepProxy(CompositeCommandStep):
    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: CommandDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        if __debug__:
            def stub(doc: CommandDocumentBuilder) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument doc", value=doc, expected_type=type_hints["doc"])
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CompositeCommandStep).__jsii_proxy_class__ = lambda : _CompositeCommandStepProxy


class ConfigureDockerStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ConfigureDockerStep",
):
    '''AutomationStep implemenation for aws:UpdateAgent https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-configuredocker.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        action: "IActionVariable",
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param action: The type of action to perform. True correlates to "Install" false correlates to "Uninstall"
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                action: "IActionVariable",
                exit_on_failure: typing.Optional[builtins.bool] = None,
                exit_on_success: typing.Optional[builtins.bool] = None,
                finally_step: typing.Optional[builtins.bool] = None,
                mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
                on_cancel: typing.Optional[Step] = None,
                precondition: typing.Optional[Precondition] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ConfigureDockerStepProps(
            action=action,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command include agentName allowDowngrade source and targetVersion if version other than latest is desired.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="installAction")
    def install_action(self) -> "IActionVariable":
        return typing.cast("IActionVariable", jsii.get(self, "installAction"))

    @builtins.property
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ConfigureDockerStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "action": "action",
    },
)
class ConfigureDockerStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        action: "IActionVariable",
    ) -> None:
        '''Properties for ConfigureDocker step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param action: The type of action to perform. True correlates to "Install" false correlates to "Uninstall"
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                exit_on_failure: typing.Optional[builtins.bool] = None,
                exit_on_success: typing.Optional[builtins.bool] = None,
                finally_step: typing.Optional[builtins.bool] = None,
                mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
                on_cancel: typing.Optional[Step] = None,
                precondition: typing.Optional[Precondition] = None,
                action: "IActionVariable",
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument exit_on_failure", value=exit_on_failure, expected_type=type_hints["exit_on_failure"])
            check_type(argname="argument exit_on_success", value=exit_on_success, expected_type=type_hints["exit_on_success"])
            check_type(argname="argument finally_step", value=finally_step, expected_type=type_hints["finally_step"])
            check_type(argname="argument mark_success_and_exit_on_failure", value=mark_success_and_exit_on_failure, expected_type=type_hints["mark_success_and_exit_on_failure"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument precondition", value=precondition, expected_type=type_hints["precondition"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
        self._values: typing.Dict[str, typing.Any] = {
            "action": action,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def action(self) -> "IActionVariable":
        '''The type of action to perform.

        True correlates to "Install" false correlates to "Uninstall"
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast("IActionVariable", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ConfigureDockerStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ConfigurePackageStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ConfigurePackageStep",
):
    '''CommandStep implemenation for aws:configurePackage https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-configurepackage.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        package_action: "IActionVariable",
        package_name: "IPackageNameVariable",
        additional_arguments: typing.Optional[IStringMapVariable] = None,
        installation_type: typing.Optional["IInstallationTypeVariable"] = None,
        version: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param package_action: Install or uninstall a package.
        :param package_name: The name of the AWS package to install or uninstall.
        :param additional_arguments: The additional parameters to provide to your install, uninstall, or update scripts. Each parameter must be prefixed with SSM_. You can reference a Parameter Store parameter in your additional arguments by using the convention {{ ssm:parameter-name }}. To use the additional parameter in your install, uninstall, or update script, you must reference the parameter as an environment variable using the syntax appropriate for the operating system. For example, in PowerShell, you reference the SSM_arg argument as $Env:SSM_arg. There is no limit to the number of arguments you define, but the additional argument input has a 4096 character limit. This limit includes all of the keys and values you define.
        :param installation_type: The type of installation to perform. If you specify Uninstall and reinstall, the package is completely uninstalled, and then reinstalled. The application is unavailable until the reinstallation is complete. If you specify In-place update, only new or changed files are added to the existing installation according you instructions you provide in an update script. The application remains available throughout the update process. The In-place update option isn't supported for AWS-published packages. Uninstall and reinstall is the default value.
        :param version: A specific version of the package to install or uninstall. If installing, the system installs the latest published version, by default. If uninstalling, the system uninstalls the currently installed version, by default. If no installed version is found, the latest published version is downloaded, and the uninstall action is run.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                package_action: "IActionVariable",
                package_name: "IPackageNameVariable",
                additional_arguments: typing.Optional[IStringMapVariable] = None,
                installation_type: typing.Optional["IInstallationTypeVariable"] = None,
                version: typing.Optional[IStringVariable] = None,
                exit_on_failure: typing.Optional[builtins.bool] = None,
                exit_on_success: typing.Optional[builtins.bool] = None,
                finally_step: typing.Optional[builtins.bool] = None,
                mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
                on_cancel: typing.Optional[Step] = None,
                precondition: typing.Optional[Precondition] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ConfigurePackageStepProps(
            package_action=package_action,
            package_name=package_name,
            additional_arguments=additional_arguments,
            installation_type=installation_type,
            version=version,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command include ...'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="packageAction")
    def package_action(self) -> "IActionVariable":
        return typing.cast("IActionVariable", jsii.get(self, "packageAction"))

    @builtins.property
    @jsii.member(jsii_name="packageName")
    def package_name(self) -> "IPackageNameVariable":
        return typing.cast("IPackageNameVariable", jsii.get(self, "packageName"))

    @builtins.property
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property
    @jsii.member(jsii_name="additionalArguments")
    def additional_arguments(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "additionalArguments"))

    @builtins.property
    @jsii.member(jsii_name="installationType")
    def installation_type(self) -> typing.Optional["IInstallationTypeVariable"]:
        return typing.cast(typing.Optional["IInstallationTypeVariable"], jsii.get(self, "installationType"))

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "version"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ConfigurePackageStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "package_action": "packageAction",
        "package_name": "packageName",
        "additional_arguments": "additionalArguments",
        "installation_type": "installationType",
        "version": "version",
    },
)
class ConfigurePackageStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        package_action: "IActionVariable",
        package_name: "IPackageNameVariable",
        additional_arguments: typing.Optional[IStringMapVariable] = None,
        installation_type: typing.Optional["IInstallationTypeVariable"] = None,
        version: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties ConfigurePackage step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param package_action: Install or uninstall a package.
        :param package_name: The name of the AWS package to install or uninstall.
        :param additional_arguments: The additional parameters to provide to your install, uninstall, or update scripts. Each parameter must be prefixed with SSM_. You can reference a Parameter Store parameter in your additional arguments by using the convention {{ ssm:parameter-name }}. To use the additional parameter in your install, uninstall, or update script, you must reference the parameter as an environment variable using the syntax appropriate for the operating system. For example, in PowerShell, you reference the SSM_arg argument as $Env:SSM_arg. There is no limit to the number of arguments you define, but the additional argument input has a 4096 character limit. This limit includes all of the keys and values you define.
        :param installation_type: The type of installation to perform. If you specify Uninstall and reinstall, the package is completely uninstalled, and then reinstalled. The application is unavailable until the reinstallation is complete. If you specify In-place update, only new or changed files are added to the existing installation according you instructions you provide in an update script. The application remains available throughout the update process. The In-place update option isn't supported for AWS-published packages. Uninstall and reinstall is the default value.
        :param version: A specific version of the package to install or uninstall. If installing, the system installs the latest published version, by default. If uninstalling, the system uninstalls the currently installed version, by default. If no installed version is found, the latest published version is downloaded, and the uninstall action is run.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                exit_on_failure: typing.Optional[builtins.bool] = None,
                exit_on_success: typing.Optional[builtins.bool] = None,
                finally_step: typing.Optional[builtins.bool] = None,
                mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
                on_cancel: typing.Optional[Step] = None,
                precondition: typing.Optional[Precondition] = None,
                package_action: "IActionVariable",
                package_name: "IPackageNameVariable",
                additional_arguments: typing.Optional[IStringMapVariable] = None,
                installation_type: typing.Optional["IInstallationTypeVariable"] = None,
                version: typing.Optional[IStringVariable] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument exit_on_failure", value=exit_on_failure, expected_type=type_hints["exit_on_failure"])
            check_type(argname="argument exit_on_success", value=exit_on_success, expected_type=type_hints["exit_on_success"])
            check_type(argname="argument finally_step", value=finally_step, expected_type=type_hints["finally_step"])
            check_type(argname="argument mark_success_and_exit_on_failure", value=mark_success_and_exit_on_failure, expected_type=type_hints["mark_success_and_exit_on_failure"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument precondition", value=precondition, expected_type=type_hints["precondition"])
            check_type(argname="argument package_action", value=package_action, expected_type=type_hints["package_action"])
            check_type(argname="argument package_name", value=package_name, expected_type=type_hints["package_name"])
            check_type(argname="argument additional_arguments", value=additional_arguments, expected_type=type_hints["additional_arguments"])
            check_type(argname="argument installation_type", value=installation_type, expected_type=type_hints["installation_type"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[str, typing.Any] = {
            "package_action": package_action,
            "package_name": package_name,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if additional_arguments is not None:
            self._values["additional_arguments"] = additional_arguments
        if installation_type is not None:
            self._values["installation_type"] = installation_type
        if version is not None:
            self._values["version"] = version

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def package_action(self) -> "IActionVariable":
        '''Install or uninstall a package.'''
        result = self._values.get("package_action")
        assert result is not None, "Required property 'package_action' is missing"
        return typing.cast("IActionVariable", result)

    @builtins.property
    def package_name(self) -> "IPackageNameVariable":
        '''The name of the AWS package to install or uninstall.'''
        result = self._values.get("package_name")
        assert result is not None, "Required property 'package_name' is missing"
        return typing.cast("IPackageNameVariable", result)

    @builtins.property
    def additional_arguments(self) -> typing.Optional[IStringMapVariable]:
        '''The additional parameters to provide to your install, uninstall, or update scripts.

        Each parameter must be prefixed with SSM_.
        You can reference a Parameter Store parameter in your additional arguments by using the convention {{ ssm:parameter-name }}.
        To use the additional parameter in your install, uninstall, or update script,
        you must reference the parameter as an environment variable using the syntax appropriate for the operating system.
        For example, in PowerShell, you reference the SSM_arg argument as $Env:SSM_arg.
        There is no limit to the number of arguments you define, but the additional argument input has a 4096 character limit.
        This limit includes all of the keys and values you define.
        '''
        result = self._values.get("additional_arguments")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def installation_type(self) -> typing.Optional["IInstallationTypeVariable"]:
        '''The type of installation to perform.

        If you specify Uninstall and reinstall, the package is completely uninstalled, and then reinstalled.
        The application is unavailable until the reinstallation is complete.
        If you specify In-place update, only new or changed files are added to the existing installation according you instructions you provide in an update script.
        The application remains available throughout the update process.
        The In-place update option isn't supported for AWS-published packages. Uninstall and reinstall is the default value.
        '''
        result = self._values.get("installation_type")
        return typing.cast(typing.Optional["IInstallationTypeVariable"], result)

    @builtins.property
    def version(self) -> typing.Optional[IStringVariable]:
        '''A specific version of the package to install or uninstall.

        If installing, the system installs the latest published version, by default.
        If uninstalling, the system uninstalls the currently installed version, by default.
        If no installed version is found, the latest published version is downloaded, and the uninstall action is run.
        '''
        result = self._values.get("version")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ConfigurePackageStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Continue(
    OnFailure,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.Continue",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="stepToInvoke")
    def step_to_invoke(self, current_step: AutomationStep) -> builtins.str:
        '''
        :param current_step: -
        '''
        if __debug__:
            def stub(current_step: AutomationStep) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument current_step", value=current_step, expected_type=type_hints["current_step"])
        return typing.cast(builtins.str, jsii.invoke(self, "stepToInvoke", [current_step]))

    @jsii.member(jsii_name="toSsmValue")
    def to_ssm_value(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "toSsmValue", []))


class CopyImageStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CopyImageStep",
):
    '''AutomationStep implemenation for aws:copyImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-copyimage.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        image_name: IStringVariable,
        source_image_id: IStringVariable,
        source_region: IStringVariable,
        client_token: typing.Optional[IStringVariable] = None,
        encrypted: typing.Optional["IBooleanVariable"] = None,
        image_description: typing.Optional[IStringVariable] = None,
        kms_key_id: typing.Optional[IStringVariable] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param image_name: The name for the image.
        :param source_image_id: The AMI ID to copy from the source Region.
        :param source_region: The region where the source AMI exists.
        :param client_token: (Optional) A unique, case-sensitive identifier that you provide to ensure request idempotency.
        :param encrypted: (Optional) Encrypt the target AMI.
        :param image_description: (Optional) A description of the image.
        :param kms_key_id: (Optional) The full Amazon Resource Name (ARN) of the AWS KMS key to use when encrypting the snapshots of an image during a copy operation.
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                image_name: IStringVariable,
                source_image_id: IStringVariable,
                source_region: IStringVariable,
                client_token: typing.Optional[IStringVariable] = None,
                encrypted: typing.Optional["IBooleanVariable"] = None,
                image_description: typing.Optional[IStringVariable] = None,
                kms_key_id: typing.Optional[IStringVariable] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CopyImageStepProps(
            image_name=image_name,
            source_image_id=source_image_id,
            source_region=source_region,
            client_token=client_token,
            encrypted=encrypted,
            image_description=image_description,
            kms_key_id=kms_key_id,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="imageName")
    def image_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "imageName"))

    @builtins.property
    @jsii.member(jsii_name="sourceImageId")
    def source_image_id(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "sourceImageId"))

    @builtins.property
    @jsii.member(jsii_name="sourceRegion")
    def source_region(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "sourceRegion"))

    @builtins.property
    @jsii.member(jsii_name="clientToken")
    def client_token(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "clientToken"))

    @builtins.property
    @jsii.member(jsii_name="encrypted")
    def encrypted(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "encrypted"))

    @builtins.property
    @jsii.member(jsii_name="imageDescription")
    def image_description(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "imageDescription"))

    @builtins.property
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "kmsKeyId"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CopyImageStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "image_name": "imageName",
        "source_image_id": "sourceImageId",
        "source_region": "sourceRegion",
        "client_token": "clientToken",
        "encrypted": "encrypted",
        "image_description": "imageDescription",
        "kms_key_id": "kmsKeyId",
    },
)
class CopyImageStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        image_name: IStringVariable,
        source_image_id: IStringVariable,
        source_region: IStringVariable,
        client_token: typing.Optional[IStringVariable] = None,
        encrypted: typing.Optional["IBooleanVariable"] = None,
        image_description: typing.Optional[IStringVariable] = None,
        kms_key_id: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for CopyImageStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param image_name: The name for the image.
        :param source_image_id: The AMI ID to copy from the source Region.
        :param source_region: The region where the source AMI exists.
        :param client_token: (Optional) A unique, case-sensitive identifier that you provide to ensure request idempotency.
        :param encrypted: (Optional) Encrypt the target AMI.
        :param image_description: (Optional) A description of the image.
        :param kms_key_id: (Optional) The full Amazon Resource Name (ARN) of the AWS KMS key to use when encrypting the snapshots of an image during a copy operation.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                image_name: IStringVariable,
                source_image_id: IStringVariable,
                source_region: IStringVariable,
                client_token: typing.Optional[IStringVariable] = None,
                encrypted: typing.Optional["IBooleanVariable"] = None,
                image_description: typing.Optional[IStringVariable] = None,
                kms_key_id: typing.Optional[IStringVariable] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument image_name", value=image_name, expected_type=type_hints["image_name"])
            check_type(argname="argument source_image_id", value=source_image_id, expected_type=type_hints["source_image_id"])
            check_type(argname="argument source_region", value=source_region, expected_type=type_hints["source_region"])
            check_type(argname="argument client_token", value=client_token, expected_type=type_hints["client_token"])
            check_type(argname="argument encrypted", value=encrypted, expected_type=type_hints["encrypted"])
            check_type(argname="argument image_description", value=image_description, expected_type=type_hints["image_description"])
            check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
        self._values: typing.Dict[str, typing.Any] = {
            "image_name": image_name,
            "source_image_id": source_image_id,
            "source_region": source_region,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if client_token is not None:
            self._values["client_token"] = client_token
        if encrypted is not None:
            self._values["encrypted"] = encrypted
        if image_description is not None:
            self._values["image_description"] = image_description
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def image_name(self) -> IStringVariable:
        '''The name for the image.'''
        result = self._values.get("image_name")
        assert result is not None, "Required property 'image_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def source_image_id(self) -> IStringVariable:
        '''The AMI ID to copy from the source Region.'''
        result = self._values.get("source_image_id")
        assert result is not None, "Required property 'source_image_id' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def source_region(self) -> IStringVariable:
        '''The region where the source AMI exists.'''
        result = self._values.get("source_region")
        assert result is not None, "Required property 'source_region' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def client_token(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A unique, case-sensitive identifier that you provide to ensure request idempotency.'''
        result = self._values.get("client_token")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def encrypted(self) -> typing.Optional["IBooleanVariable"]:
        '''(Optional) Encrypt the target AMI.'''
        result = self._values.get("encrypted")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    @builtins.property
    def image_description(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A description of the image.'''
        result = self._values.get("image_description")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def kms_key_id(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The full Amazon Resource Name (ARN) of the AWS KMS key to use when encrypting the snapshots of an image during a copy operation.'''
        result = self._values.get("kms_key_id")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CopyImageStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CreateImageStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateImageStep",
):
    '''AutomationStep implemenation for aws:createImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-create.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        image_name: IStringVariable,
        instance_id: IStringVariable,
        block_device_mappings: typing.Optional[IStringMapVariable] = None,
        image_description: typing.Optional[IStringVariable] = None,
        no_reboot: typing.Optional["IBooleanVariable"] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param image_name: The name for the image.
        :param instance_id: The ID of the instance.
        :param block_device_mappings: (Optional) The block devices for the instance.
        :param image_description: (Optional) A description of the image.
        :param no_reboot: (Optional) By default, Amazon Elastic Compute Cloud (Amazon EC2) attempts to shut down and reboot the instance before creating the image. If the No Reboot option is set to true, Amazon EC2 doesn't shut down the instance before creating the image. When this option is used, file system integrity on the created image can't be guaranteed. If you don't want the instance to run after you create an AMI from it, first use the aws:changeInstanceState – Change or assert instance state action to stop the instance, and then use this aws:createImage action with the NoReboot option set to true.
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                image_name: IStringVariable,
                instance_id: IStringVariable,
                block_device_mappings: typing.Optional[IStringMapVariable] = None,
                image_description: typing.Optional[IStringVariable] = None,
                no_reboot: typing.Optional["IBooleanVariable"] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CreateImageStepProps(
            image_name=image_name,
            instance_id=instance_id,
            block_device_mappings=block_device_mappings,
            image_description=image_description,
            no_reboot=no_reboot,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="imageName")
    def image_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "imageName"))

    @builtins.property
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "instanceId"))

    @builtins.property
    @jsii.member(jsii_name="blockDeviceMappings")
    def block_device_mappings(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "blockDeviceMappings"))

    @builtins.property
    @jsii.member(jsii_name="imageDescription")
    def image_description(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "imageDescription"))

    @builtins.property
    @jsii.member(jsii_name="noReboot")
    def no_reboot(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "noReboot"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CreateImageStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "image_name": "imageName",
        "instance_id": "instanceId",
        "block_device_mappings": "blockDeviceMappings",
        "image_description": "imageDescription",
        "no_reboot": "noReboot",
    },
)
class CreateImageStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        image_name: IStringVariable,
        instance_id: IStringVariable,
        block_device_mappings: typing.Optional[IStringMapVariable] = None,
        image_description: typing.Optional[IStringVariable] = None,
        no_reboot: typing.Optional["IBooleanVariable"] = None,
    ) -> None:
        '''Properties for CreateImageStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param image_name: The name for the image.
        :param instance_id: The ID of the instance.
        :param block_device_mappings: (Optional) The block devices for the instance.
        :param image_description: (Optional) A description of the image.
        :param no_reboot: (Optional) By default, Amazon Elastic Compute Cloud (Amazon EC2) attempts to shut down and reboot the instance before creating the image. If the No Reboot option is set to true, Amazon EC2 doesn't shut down the instance before creating the image. When this option is used, file system integrity on the created image can't be guaranteed. If you don't want the instance to run after you create an AMI from it, first use the aws:changeInstanceState – Change or assert instance state action to stop the instance, and then use this aws:createImage action with the NoReboot option set to true.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                image_name: IStringVariable,
                instance_id: IStringVariable,
                block_device_mappings: typing.Optional[IStringMapVariable] = None,
                image_description: typing.Optional[IStringVariable] = None,
                no_reboot: typing.Optional["IBooleanVariable"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument image_name", value=image_name, expected_type=type_hints["image_name"])
            check_type(argname="argument instance_id", value=instance_id, expected_type=type_hints["instance_id"])
            check_type(argname="argument block_device_mappings", value=block_device_mappings, expected_type=type_hints["block_device_mappings"])
            check_type(argname="argument image_description", value=image_description, expected_type=type_hints["image_description"])
            check_type(argname="argument no_reboot", value=no_reboot, expected_type=type_hints["no_reboot"])
        self._values: typing.Dict[str, typing.Any] = {
            "image_name": image_name,
            "instance_id": instance_id,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if block_device_mappings is not None:
            self._values["block_device_mappings"] = block_device_mappings
        if image_description is not None:
            self._values["image_description"] = image_description
        if no_reboot is not None:
            self._values["no_reboot"] = no_reboot

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def image_name(self) -> IStringVariable:
        '''The name for the image.'''
        result = self._values.get("image_name")
        assert result is not None, "Required property 'image_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def instance_id(self) -> IStringVariable:
        '''The ID of the instance.'''
        result = self._values.get("instance_id")
        assert result is not None, "Required property 'instance_id' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def block_device_mappings(self) -> typing.Optional[IStringMapVariable]:
        '''(Optional) The block devices for the instance.'''
        result = self._values.get("block_device_mappings")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def image_description(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A description of the image.'''
        result = self._values.get("image_description")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def no_reboot(self) -> typing.Optional["IBooleanVariable"]:
        '''(Optional) By default, Amazon Elastic Compute Cloud (Amazon EC2) attempts to shut down and reboot the instance before creating the image.

        If the No Reboot option is set to true, Amazon EC2 doesn't shut down the instance before creating the image. When this option is used, file system integrity on the created image can't be guaranteed.
        If you don't want the instance to run after you create an AMI from it, first use the aws:changeInstanceState – Change or assert instance state action to stop the instance, and then use this aws:createImage action with the NoReboot option set to true.
        '''
        result = self._values.get("no_reboot")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateImageStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CreateStackStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateStackStep",
):
    '''AutomationStep implementation for aws:createStack https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createstack.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        stack_name: IStringVariable,
        template: typing.Union[BodyOrUrlProp, typing.Dict[str, typing.Any]],
        capabilities: typing.Optional[IStringListVariable] = None,
        client_request_token: typing.Optional[IStringVariable] = None,
        notification_ar_ns: typing.Optional[IStringListVariable] = None,
        on_stack_failure: typing.Optional["IOnFailureVariable"] = None,
        parameters: typing.Optional[IMapListVariable] = None,
        resource_types: typing.Optional[IStringListVariable] = None,
        role_arn: typing.Optional[IStringVariable] = None,
        stack_policy: typing.Optional[typing.Union[BodyOrUrlProp, typing.Dict[str, typing.Any]]] = None,
        tags: typing.Optional[IMapListVariable] = None,
        timeout_in_minutes: typing.Optional[INumberVariable] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param stack_name: The name that is associated with the stack. The name must be unique in the Region in which you're creating the stack.
        :param template: Template body or URL. For more information, see `Template Anatomy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html>`_.
        :param capabilities: (Optional) A list of values that you specify before CloudFormation can create certain stacks. Some stack templates include resources that can affect permissions in your AWS account. For example, creating new AWS Identity and Access Management (IAM) users can affect permissions in your account. For those stacks, you must explicitly acknowledge their capabilities by specifying this parameter. Valid values include CAPABILITY_IAM, CAPABILITY_NAMED_IAM, and CAPABILITY_AUTO_EXPAND.
        :param client_request_token: (Optional) A unique identifier for this CreateStack request. Specify this token if you set maxAttempts in this step to a value greater than 1. By specifying this token, CloudFormation knows that you aren't attempting to create a new stack with the same name.
        :param notification_ar_ns: (Optional) The Amazon Simple Notification Service (Amazon SNS) topic ARNs for publishing stack-related events.
        :param on_stack_failure: (Optional) Determines the action to take if stack creation failed. Default: - Rollback on failure
        :param parameters: (Optional) A list of Parameter structures that specify input parameters for the stack. For more information, see the `Parameter <https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html>`_ data type.
        :param resource_types: (Optional) The template resource types that you have permissions to work with for this create stack action. For example: AWS::EC2::Instance, AWS::EC2::*, or Custom::MyCustomInstance.
        :param role_arn: (Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack. CloudFormation uses the role's credentials to make calls on your behalf. CloudFormation always uses this role for all future operations on the stack. As long as users have permission to operate on the stack, CloudFormation uses this role even if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges. If you don't specify a value, CloudFormation uses the role that was previously associated with the stack. If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        :param stack_policy: (Optional) Stack policy body or URL. For more information, see `Prevent Updates to Stack Resources <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/protect-stack-resources.html>`_.
        :param tags: (Optional) Key-value pairs to associate with this stack. CloudFormation also propagates these tags to the resources created in the stack. You can specify a maximum number of 10 tags.
        :param timeout_in_minutes: (Optional) The amount of time that can pass before the stack status becomes CREATE_FAILED. If DisableRollback isn't set or is set to false, the stack will be rolled back.
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                stack_name: IStringVariable,
                template: typing.Union[BodyOrUrlProp, typing.Dict[str, typing.Any]],
                capabilities: typing.Optional[IStringListVariable] = None,
                client_request_token: typing.Optional[IStringVariable] = None,
                notification_ar_ns: typing.Optional[IStringListVariable] = None,
                on_stack_failure: typing.Optional["IOnFailureVariable"] = None,
                parameters: typing.Optional[IMapListVariable] = None,
                resource_types: typing.Optional[IStringListVariable] = None,
                role_arn: typing.Optional[IStringVariable] = None,
                stack_policy: typing.Optional[typing.Union[BodyOrUrlProp, typing.Dict[str, typing.Any]]] = None,
                tags: typing.Optional[IMapListVariable] = None,
                timeout_in_minutes: typing.Optional[INumberVariable] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CreateStackStepProps(
            stack_name=stack_name,
            template=template,
            capabilities=capabilities,
            client_request_token=client_request_token,
            notification_ar_ns=notification_ar_ns,
            on_stack_failure=on_stack_failure,
            parameters=parameters,
            resource_types=resource_types,
            role_arn=role_arn,
            stack_policy=stack_policy,
            tags=tags,
            timeout_in_minutes=timeout_in_minutes,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="stackName")
    def stack_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "stackName"))

    @builtins.property
    @jsii.member(jsii_name="capabilities")
    def capabilities(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "capabilities"))

    @builtins.property
    @jsii.member(jsii_name="clientRequestToken")
    def client_request_token(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "clientRequestToken"))

    @builtins.property
    @jsii.member(jsii_name="notificationARNs")
    def notification_ar_ns(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "notificationARNs"))

    @builtins.property
    @jsii.member(jsii_name="onStackFailure")
    def on_stack_failure(self) -> typing.Optional["IOnFailureVariable"]:
        return typing.cast(typing.Optional["IOnFailureVariable"], jsii.get(self, "onStackFailure"))

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "parameters"))

    @builtins.property
    @jsii.member(jsii_name="resourceTypes")
    def resource_types(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "resourceTypes"))

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "roleArn"))

    @builtins.property
    @jsii.member(jsii_name="stackPolicyBody")
    def stack_policy_body(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "stackPolicyBody"))

    @builtins.property
    @jsii.member(jsii_name="stackPolicyUrl")
    def stack_policy_url(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "stackPolicyUrl"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="templateBody")
    def template_body(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "templateBody"))

    @builtins.property
    @jsii.member(jsii_name="templateUrl")
    def template_url(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "templateUrl"))

    @builtins.property
    @jsii.member(jsii_name="timeoutInMinutes")
    def timeout_in_minutes(self) -> typing.Optional[INumberVariable]:
        return typing.cast(typing.Optional[INumberVariable], jsii.get(self, "timeoutInMinutes"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CreateStackStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "stack_name": "stackName",
        "template": "template",
        "capabilities": "capabilities",
        "client_request_token": "clientRequestToken",
        "notification_ar_ns": "notificationARNs",
        "on_stack_failure": "onStackFailure",
        "parameters": "parameters",
        "resource_types": "resourceTypes",
        "role_arn": "roleArn",
        "stack_policy": "stackPolicy",
        "tags": "tags",
        "timeout_in_minutes": "timeoutInMinutes",
    },
)
class CreateStackStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        stack_name: IStringVariable,
        template: typing.Union[BodyOrUrlProp, typing.Dict[str, typing.Any]],
        capabilities: typing.Optional[IStringListVariable] = None,
        client_request_token: typing.Optional[IStringVariable] = None,
        notification_ar_ns: typing.Optional[IStringListVariable] = None,
        on_stack_failure: typing.Optional["IOnFailureVariable"] = None,
        parameters: typing.Optional[IMapListVariable] = None,
        resource_types: typing.Optional[IStringListVariable] = None,
        role_arn: typing.Optional[IStringVariable] = None,
        stack_policy: typing.Optional[typing.Union[BodyOrUrlProp, typing.Dict[str, typing.Any]]] = None,
        tags: typing.Optional[IMapListVariable] = None,
        timeout_in_minutes: typing.Optional[INumberVariable] = None,
    ) -> None:
        '''Properties for CreateStackStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param stack_name: The name that is associated with the stack. The name must be unique in the Region in which you're creating the stack.
        :param template: Template body or URL. For more information, see `Template Anatomy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html>`_.
        :param capabilities: (Optional) A list of values that you specify before CloudFormation can create certain stacks. Some stack templates include resources that can affect permissions in your AWS account. For example, creating new AWS Identity and Access Management (IAM) users can affect permissions in your account. For those stacks, you must explicitly acknowledge their capabilities by specifying this parameter. Valid values include CAPABILITY_IAM, CAPABILITY_NAMED_IAM, and CAPABILITY_AUTO_EXPAND.
        :param client_request_token: (Optional) A unique identifier for this CreateStack request. Specify this token if you set maxAttempts in this step to a value greater than 1. By specifying this token, CloudFormation knows that you aren't attempting to create a new stack with the same name.
        :param notification_ar_ns: (Optional) The Amazon Simple Notification Service (Amazon SNS) topic ARNs for publishing stack-related events.
        :param on_stack_failure: (Optional) Determines the action to take if stack creation failed. Default: - Rollback on failure
        :param parameters: (Optional) A list of Parameter structures that specify input parameters for the stack. For more information, see the `Parameter <https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html>`_ data type.
        :param resource_types: (Optional) The template resource types that you have permissions to work with for this create stack action. For example: AWS::EC2::Instance, AWS::EC2::*, or Custom::MyCustomInstance.
        :param role_arn: (Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack. CloudFormation uses the role's credentials to make calls on your behalf. CloudFormation always uses this role for all future operations on the stack. As long as users have permission to operate on the stack, CloudFormation uses this role even if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges. If you don't specify a value, CloudFormation uses the role that was previously associated with the stack. If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        :param stack_policy: (Optional) Stack policy body or URL. For more information, see `Prevent Updates to Stack Resources <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/protect-stack-resources.html>`_.
        :param tags: (Optional) Key-value pairs to associate with this stack. CloudFormation also propagates these tags to the resources created in the stack. You can specify a maximum number of 10 tags.
        :param timeout_in_minutes: (Optional) The amount of time that can pass before the stack status becomes CREATE_FAILED. If DisableRollback isn't set or is set to false, the stack will be rolled back.
        '''
        if isinstance(template, dict):
            template = BodyOrUrlProp(**template)
        if isinstance(stack_policy, dict):
            stack_policy = BodyOrUrlProp(**stack_policy)
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                stack_name: IStringVariable,
                template: typing.Union[BodyOrUrlProp, typing.Dict[str, typing.Any]],
                capabilities: typing.Optional[IStringListVariable] = None,
                client_request_token: typing.Optional[IStringVariable] = None,
                notification_ar_ns: typing.Optional[IStringListVariable] = None,
                on_stack_failure: typing.Optional["IOnFailureVariable"] = None,
                parameters: typing.Optional[IMapListVariable] = None,
                resource_types: typing.Optional[IStringListVariable] = None,
                role_arn: typing.Optional[IStringVariable] = None,
                stack_policy: typing.Optional[typing.Union[BodyOrUrlProp, typing.Dict[str, typing.Any]]] = None,
                tags: typing.Optional[IMapListVariable] = None,
                timeout_in_minutes: typing.Optional[INumberVariable] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument stack_name", value=stack_name, expected_type=type_hints["stack_name"])
            check_type(argname="argument template", value=template, expected_type=type_hints["template"])
            check_type(argname="argument capabilities", value=capabilities, expected_type=type_hints["capabilities"])
            check_type(argname="argument client_request_token", value=client_request_token, expected_type=type_hints["client_request_token"])
            check_type(argname="argument notification_ar_ns", value=notification_ar_ns, expected_type=type_hints["notification_ar_ns"])
            check_type(argname="argument on_stack_failure", value=on_stack_failure, expected_type=type_hints["on_stack_failure"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument resource_types", value=resource_types, expected_type=type_hints["resource_types"])
            check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
            check_type(argname="argument stack_policy", value=stack_policy, expected_type=type_hints["stack_policy"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument timeout_in_minutes", value=timeout_in_minutes, expected_type=type_hints["timeout_in_minutes"])
        self._values: typing.Dict[str, typing.Any] = {
            "stack_name": stack_name,
            "template": template,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if capabilities is not None:
            self._values["capabilities"] = capabilities
        if client_request_token is not None:
            self._values["client_request_token"] = client_request_token
        if notification_ar_ns is not None:
            self._values["notification_ar_ns"] = notification_ar_ns
        if on_stack_failure is not None:
            self._values["on_stack_failure"] = on_stack_failure
        if parameters is not None:
            self._values["parameters"] = parameters
        if resource_types is not None:
            self._values["resource_types"] = resource_types
        if role_arn is not None:
            self._values["role_arn"] = role_arn
        if stack_policy is not None:
            self._values["stack_policy"] = stack_policy
        if tags is not None:
            self._values["tags"] = tags
        if timeout_in_minutes is not None:
            self._values["timeout_in_minutes"] = timeout_in_minutes

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def stack_name(self) -> IStringVariable:
        '''The name that is associated with the stack.

        The name must be unique in the Region in which you're creating the stack.
        '''
        result = self._values.get("stack_name")
        assert result is not None, "Required property 'stack_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def template(self) -> BodyOrUrlProp:
        '''Template body or URL.

        For more information, see `Template Anatomy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html>`_.
        '''
        result = self._values.get("template")
        assert result is not None, "Required property 'template' is missing"
        return typing.cast(BodyOrUrlProp, result)

    @builtins.property
    def capabilities(self) -> typing.Optional[IStringListVariable]:
        '''(Optional) A list of values that you specify before CloudFormation can create certain stacks.

        Some stack templates include resources that can affect permissions in your AWS account.
        For example, creating new AWS Identity and Access Management (IAM) users can affect permissions in your account.
        For those stacks, you must explicitly acknowledge their capabilities by specifying this parameter.

        Valid values include CAPABILITY_IAM, CAPABILITY_NAMED_IAM, and CAPABILITY_AUTO_EXPAND.
        '''
        result = self._values.get("capabilities")
        return typing.cast(typing.Optional[IStringListVariable], result)

    @builtins.property
    def client_request_token(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A unique identifier for this CreateStack request.

        Specify this token if you set maxAttempts in this step to a value greater than 1.
        By specifying this token, CloudFormation knows that you aren't attempting to create a new stack with the same name.
        '''
        result = self._values.get("client_request_token")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def notification_ar_ns(self) -> typing.Optional[IStringListVariable]:
        '''(Optional) The Amazon Simple Notification Service (Amazon SNS) topic ARNs for publishing stack-related events.'''
        result = self._values.get("notification_ar_ns")
        return typing.cast(typing.Optional[IStringListVariable], result)

    @builtins.property
    def on_stack_failure(self) -> typing.Optional["IOnFailureVariable"]:
        '''(Optional) Determines the action to take if stack creation failed.

        :default: - Rollback on failure
        '''
        result = self._values.get("on_stack_failure")
        return typing.cast(typing.Optional["IOnFailureVariable"], result)

    @builtins.property
    def parameters(self) -> typing.Optional[IMapListVariable]:
        '''(Optional) A list of Parameter structures that specify input parameters for the stack.

        For more information, see the `Parameter <https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html>`_ data type.
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def resource_types(self) -> typing.Optional[IStringListVariable]:
        '''(Optional) The template resource types that you have permissions to work with for this create stack action.

        For example: AWS::EC2::Instance, AWS::EC2::*, or Custom::MyCustomInstance.
        '''
        result = self._values.get("resource_types")
        return typing.cast(typing.Optional[IStringListVariable], result)

    @builtins.property
    def role_arn(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack.

        CloudFormation uses the role's credentials to make calls on your behalf.
        CloudFormation always uses this role for all future operations on the stack.
        As long as users have permission to operate on the stack, CloudFormation uses this role even
        if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges.

        If you don't specify a value, CloudFormation uses the role that was previously associated with the stack.
        If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        '''
        result = self._values.get("role_arn")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def stack_policy(self) -> typing.Optional[BodyOrUrlProp]:
        '''(Optional) Stack policy body or URL.

        For more information, see `Prevent Updates to Stack Resources <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/protect-stack-resources.html>`_.
        '''
        result = self._values.get("stack_policy")
        return typing.cast(typing.Optional[BodyOrUrlProp], result)

    @builtins.property
    def tags(self) -> typing.Optional[IMapListVariable]:
        '''(Optional) Key-value pairs to associate with this stack.

        CloudFormation also propagates these tags to the resources created in the stack. You can specify a maximum number of 10 tags.
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def timeout_in_minutes(self) -> typing.Optional[INumberVariable]:
        '''(Optional) The amount of time that can pass before the stack status becomes CREATE_FAILED.

        If DisableRollback isn't set or is set to false, the stack will be rolled back.
        '''
        result = self._values.get("timeout_in_minutes")
        return typing.cast(typing.Optional[INumberVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateStackStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CreateTagsStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateTagsStep",
):
    '''AutomationStep implemenation for aws:createTags https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createtag.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        resource_ids: IStringListVariable,
        tags: IMapListVariable,
        resource_type: typing.Optional["IResourceTypeVariable"] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param resource_ids: The IDs of the resource(s) to be tagged. If resource type isn't “EC2”, this field can contain only a single item.
        :param tags: The tags to associate with the resource(s).
        :param resource_type: (Optional) The type of resource(s) to be tagged. Valid values: EC2 | ManagedInstance | MaintenanceWindow | Parameter Default: EC2
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                resource_ids: IStringListVariable,
                tags: IMapListVariable,
                resource_type: typing.Optional["IResourceTypeVariable"] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CreateTagsStepProps(
            resource_ids=resource_ids,
            tags=tags,
            resource_type=resource_type,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''This step has no outputs.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="resourceIds")
    def resource_ids(self) -> IStringListVariable:
        return typing.cast(IStringListVariable, jsii.get(self, "resourceIds"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> IMapListVariable:
        return typing.cast(IMapListVariable, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="resourceType")
    def resource_type(self) -> typing.Optional["IResourceTypeVariable"]:
        return typing.cast(typing.Optional["IResourceTypeVariable"], jsii.get(self, "resourceType"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CreateTagsStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "resource_ids": "resourceIds",
        "tags": "tags",
        "resource_type": "resourceType",
    },
)
class CreateTagsStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        resource_ids: IStringListVariable,
        tags: IMapListVariable,
        resource_type: typing.Optional["IResourceTypeVariable"] = None,
    ) -> None:
        '''Properties for CreateTagStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param resource_ids: The IDs of the resource(s) to be tagged. If resource type isn't “EC2”, this field can contain only a single item.
        :param tags: The tags to associate with the resource(s).
        :param resource_type: (Optional) The type of resource(s) to be tagged. Valid values: EC2 | ManagedInstance | MaintenanceWindow | Parameter Default: EC2
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                resource_ids: IStringListVariable,
                tags: IMapListVariable,
                resource_type: typing.Optional["IResourceTypeVariable"] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument resource_ids", value=resource_ids, expected_type=type_hints["resource_ids"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument resource_type", value=resource_type, expected_type=type_hints["resource_type"])
        self._values: typing.Dict[str, typing.Any] = {
            "resource_ids": resource_ids,
            "tags": tags,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if resource_type is not None:
            self._values["resource_type"] = resource_type

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def resource_ids(self) -> IStringListVariable:
        '''The IDs of the resource(s) to be tagged.

        If resource type isn't “EC2”, this field can contain only a single item.
        '''
        result = self._values.get("resource_ids")
        assert result is not None, "Required property 'resource_ids' is missing"
        return typing.cast(IStringListVariable, result)

    @builtins.property
    def tags(self) -> IMapListVariable:
        '''The tags to associate with the resource(s).'''
        result = self._values.get("tags")
        assert result is not None, "Required property 'tags' is missing"
        return typing.cast(IMapListVariable, result)

    @builtins.property
    def resource_type(self) -> typing.Optional["IResourceTypeVariable"]:
        '''(Optional) The type of resource(s) to be tagged.

        Valid values: EC2 | ManagedInstance | MaintenanceWindow | Parameter

        :default: EC2
        '''
        result = self._values.get("resource_type")
        return typing.cast(typing.Optional["IResourceTypeVariable"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateTagsStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DeleteImageStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteImageStep",
):
    '''AutomationStep implementation for aws:deleteImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-delete.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        image_id: IStringVariable,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param image_id: The ID of the image to be deleted.
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                image_id: IStringVariable,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = DeleteImageStepProps(
            image_id=image_id,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''This step has no outputs.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="imageId")
    def image_id(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "imageId"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteImageStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "image_id": "imageId",
    },
)
class DeleteImageStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        image_id: IStringVariable,
    ) -> None:
        '''Properties for DeleteImageStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param image_id: The ID of the image to be deleted.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                image_id: IStringVariable,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument image_id", value=image_id, expected_type=type_hints["image_id"])
        self._values: typing.Dict[str, typing.Any] = {
            "image_id": image_id,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def image_id(self) -> IStringVariable:
        '''The ID of the image to be deleted.'''
        result = self._values.get("image_id")
        assert result is not None, "Required property 'image_id' is missing"
        return typing.cast(IStringVariable, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DeleteImageStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DeleteStackStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteStackStep",
):
    '''AutomationStep implemenation for aws:deleteStack https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-deletestack.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        stack_name_variable: IStringVariable,
        role_arn: typing.Optional[IStringVariable] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param stack_name_variable: (Optional) Variable that is fed into this step declaring which stack to delete. Default: - StackName is the default value.
        :param role_arn: (Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack. CloudFormation uses the role's credentials to make calls on your behalf. CloudFormation always uses this role for all future operations on the stack. As long as users have permission to operate on the stack, CloudFormation uses this role even if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges. If you don't specify a value, CloudFormation uses the role that was previously associated with the stack. If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                stack_name_variable: IStringVariable,
                role_arn: typing.Optional[IStringVariable] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = DeleteStackStepProps(
            stack_name_variable=stack_name_variable,
            role_arn=role_arn,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''The input required for this step is the stackNameVariable.

        :return: the stackNameVariable as declared in the constructor
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''No outputs emitted from Delete Stack step.

        :return: []
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="stackNameVariable")
    def stack_name_variable(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "stackNameVariable"))

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "roleArn"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteStackStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "stack_name_variable": "stackNameVariable",
        "role_arn": "roleArn",
    },
)
class DeleteStackStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        stack_name_variable: IStringVariable,
        role_arn: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for DeleteStackStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param stack_name_variable: (Optional) Variable that is fed into this step declaring which stack to delete. Default: - StackName is the default value.
        :param role_arn: (Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack. CloudFormation uses the role's credentials to make calls on your behalf. CloudFormation always uses this role for all future operations on the stack. As long as users have permission to operate on the stack, CloudFormation uses this role even if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges. If you don't specify a value, CloudFormation uses the role that was previously associated with the stack. If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                stack_name_variable: IStringVariable,
                role_arn: typing.Optional[IStringVariable] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument stack_name_variable", value=stack_name_variable, expected_type=type_hints["stack_name_variable"])
            check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
        self._values: typing.Dict[str, typing.Any] = {
            "stack_name_variable": stack_name_variable,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if role_arn is not None:
            self._values["role_arn"] = role_arn

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def stack_name_variable(self) -> IStringVariable:
        '''(Optional) Variable that is fed into this step declaring which stack to delete.

        :default: - StackName is the default value.
        '''
        result = self._values.get("stack_name_variable")
        assert result is not None, "Required property 'stack_name_variable' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def role_arn(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack.

        CloudFormation uses the role's credentials to make calls on your behalf.
        CloudFormation always uses this role for all future operations on the stack.
        As long as users have permission to operate on the stack, CloudFormation uses this role even
        if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges.

        If you don't specify a value, CloudFormation uses the role that was previously associated with the stack.
        If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        '''
        result = self._values.get("role_arn")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DeleteStackStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IStringVariable)
class DictFormat(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DictFormat",
):
    def __init__(self, format: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param format: -
        '''
        if __debug__:
            def stub(format: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument format", value=format, expected_type=type_hints["format"])
        jsii.create(self.__class__, self, [format])

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{ Variable }} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

    @jsii.member(jsii_name="resolveToDict")
    def resolve_to_dict(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "resolveToDict", [inputs]))

    @jsii.member(jsii_name="resolveToString")
    def resolve_to_string(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.str:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(builtins.str, jsii.invoke(self, "resolveToString", [inputs]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

    @builtins.property
    @jsii.member(jsii_name="format")
    def format(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "format"))


@jsii.implements(IEnvironment)
class DockerEnvironment(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DockerEnvironment",
):
    '''Provides a Docker client against which customers can execute their commands.

    This utility will not download docker images, rather will create containers from images provided.
    You can use this rather than running your run commands against a real EC2 machine.
    '''

    @jsii.member(jsii_name="fromContainer")
    @builtins.classmethod
    def from_container(cls, container_id: builtins.str) -> "DockerEnvironment":
        '''Use an existing container against which to run commands using the run function.

        :param container_id: -
        '''
        if __debug__:
            def stub(container_id: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument container_id", value=container_id, expected_type=type_hints["container_id"])
        return typing.cast("DockerEnvironment", jsii.sinvoke(cls, "fromContainer", [container_id]))

    @jsii.member(jsii_name="fromImage")
    @builtins.classmethod
    def from_image(cls, image: builtins.str) -> "DockerEnvironment":
        '''Create a container from the provided image.

        The container created will be used by this instance to run commands using the run function.

        :param image: -
        '''
        if __debug__:
            def stub(image: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
        return typing.cast("DockerEnvironment", jsii.sinvoke(cls, "fromImage", [image]))

    @jsii.member(jsii_name="removeContainer")
    def remove_container(self) -> None:
        '''Force removes the container associated with this instance.'''
        return typing.cast(None, jsii.invoke(self, "removeContainer", []))

    @jsii.member(jsii_name="run")
    def run(self, command: builtins.str) -> builtins.str:
        '''Runs commands against the docker specified during construction.

        This function runs synchronously.

        :param command: -

        :return: the response from the docker as a string.
        '''
        if __debug__:
            def stub(command: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
        return typing.cast(builtins.str, jsii.invoke(self, "run", [command]))

    @builtins.property
    @jsii.member(jsii_name="containerId")
    def container_id(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "containerId"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DocumentResult",
    jsii_struct_bases=[SimulationResult],
    name_mapping={
        "executed_steps": "executedSteps",
        "response_code": "responseCode",
        "outputs": "outputs",
        "stack_trace": "stackTrace",
        "document_outputs": "documentOutputs",
    },
)
class DocumentResult(SimulationResult):
    def __init__(
        self,
        *,
        executed_steps: typing.Sequence[builtins.str],
        response_code: ResponseCode,
        outputs: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        stack_trace: typing.Optional[builtins.str] = None,
        document_outputs: typing.Sequence[builtins.str],
    ) -> None:
        '''
        :param executed_steps: All the steps that were executed in this Simulation.
        :param response_code: 
        :param outputs: May be empty if responseCode is FAILED/CANCELLED. There are no outputs provided for Command steps or documents.
        :param stack_trace: undefined if responseCode is SUCCESS.
        :param document_outputs: 
        '''
        if __debug__:
            def stub(
                *,
                executed_steps: typing.Sequence[builtins.str],
                response_code: ResponseCode,
                outputs: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
                stack_trace: typing.Optional[builtins.str] = None,
                document_outputs: typing.Sequence[builtins.str],
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument executed_steps", value=executed_steps, expected_type=type_hints["executed_steps"])
            check_type(argname="argument response_code", value=response_code, expected_type=type_hints["response_code"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument stack_trace", value=stack_trace, expected_type=type_hints["stack_trace"])
            check_type(argname="argument document_outputs", value=document_outputs, expected_type=type_hints["document_outputs"])
        self._values: typing.Dict[str, typing.Any] = {
            "executed_steps": executed_steps,
            "response_code": response_code,
            "document_outputs": document_outputs,
        }
        if outputs is not None:
            self._values["outputs"] = outputs
        if stack_trace is not None:
            self._values["stack_trace"] = stack_trace

    @builtins.property
    def executed_steps(self) -> typing.List[builtins.str]:
        '''All the steps that were executed in this Simulation.'''
        result = self._values.get("executed_steps")
        assert result is not None, "Required property 'executed_steps' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def response_code(self) -> ResponseCode:
        result = self._values.get("response_code")
        assert result is not None, "Required property 'response_code' is missing"
        return typing.cast(ResponseCode, result)

    @builtins.property
    def outputs(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''May be empty if responseCode is FAILED/CANCELLED.

        There are no outputs provided for Command steps or documents.
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def stack_trace(self) -> typing.Optional[builtins.str]:
        '''undefined if responseCode is SUCCESS.'''
        result = self._values.get("stack_trace")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def document_outputs(self) -> typing.List[builtins.str]:
        result = self._values.get("document_outputs")
        assert result is not None, "Required property 'document_outputs' is missing"
        return typing.cast(typing.List[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DocumentResult(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DomainJoinStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DomainJoinStep",
):
    '''CommandStep implemenation for aws:softwareInventory https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-domainJoin.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        directory_id: IStringVariable,
        directory_name: IStringVariable,
        dns_ip_addresses: IStringListVariable,
        directory_ou: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param directory_id: The ID of the directory.
        :param directory_name: The name of the domain.
        :param dns_ip_addresses: The IP addresses of the DNS servers.
        :param directory_ou: (Optional) The organizational unit (OU).
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                directory_id: IStringVariable,
                directory_name: IStringVariable,
                dns_ip_addresses: IStringListVariable,
                directory_ou: typing.Optional[IStringVariable] = None,
                exit_on_failure: typing.Optional[builtins.bool] = None,
                exit_on_success: typing.Optional[builtins.bool] = None,
                finally_step: typing.Optional[builtins.bool] = None,
                mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
                on_cancel: typing.Optional[Step] = None,
                precondition: typing.Optional[Precondition] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = DomainJoinStepProps(
            directory_id=directory_id,
            directory_name=directory_name,
            dns_ip_addresses=dns_ip_addresses,
            directory_ou=directory_ou,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''All Inputs for this command are optional.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="directoryId")
    def directory_id(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "directoryId"))

    @builtins.property
    @jsii.member(jsii_name="directoryName")
    def directory_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "directoryName"))

    @builtins.property
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property
    @jsii.member(jsii_name="directoryOU")
    def directory_ou(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "directoryOU"))

    @builtins.property
    @jsii.member(jsii_name="dnsIpAddresses")
    def dns_ip_addresses(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "dnsIpAddresses"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DomainJoinStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "directory_id": "directoryId",
        "directory_name": "directoryName",
        "dns_ip_addresses": "dnsIpAddresses",
        "directory_ou": "directoryOU",
    },
)
class DomainJoinStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        directory_id: IStringVariable,
        directory_name: IStringVariable,
        dns_ip_addresses: IStringListVariable,
        directory_ou: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties of DomainJoin step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param directory_id: The ID of the directory.
        :param directory_name: The name of the domain.
        :param dns_ip_addresses: The IP addresses of the DNS servers.
        :param directory_ou: (Optional) The organizational unit (OU).
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                exit_on_failure: typing.Optional[builtins.bool] = None,
                exit_on_success: typing.Optional[builtins.bool] = None,
                finally_step: typing.Optional[builtins.bool] = None,
                mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
                on_cancel: typing.Optional[Step] = None,
                precondition: typing.Optional[Precondition] = None,
                directory_id: IStringVariable,
                directory_name: IStringVariable,
                dns_ip_addresses: IStringListVariable,
                directory_ou: typing.Optional[IStringVariable] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument exit_on_failure", value=exit_on_failure, expected_type=type_hints["exit_on_failure"])
            check_type(argname="argument exit_on_success", value=exit_on_success, expected_type=type_hints["exit_on_success"])
            check_type(argname="argument finally_step", value=finally_step, expected_type=type_hints["finally_step"])
            check_type(argname="argument mark_success_and_exit_on_failure", value=mark_success_and_exit_on_failure, expected_type=type_hints["mark_success_and_exit_on_failure"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument precondition", value=precondition, expected_type=type_hints["precondition"])
            check_type(argname="argument directory_id", value=directory_id, expected_type=type_hints["directory_id"])
            check_type(argname="argument directory_name", value=directory_name, expected_type=type_hints["directory_name"])
            check_type(argname="argument dns_ip_addresses", value=dns_ip_addresses, expected_type=type_hints["dns_ip_addresses"])
            check_type(argname="argument directory_ou", value=directory_ou, expected_type=type_hints["directory_ou"])
        self._values: typing.Dict[str, typing.Any] = {
            "directory_id": directory_id,
            "directory_name": directory_name,
            "dns_ip_addresses": dns_ip_addresses,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if directory_ou is not None:
            self._values["directory_ou"] = directory_ou

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def directory_id(self) -> IStringVariable:
        '''The ID of the directory.

        Example::

            "d-1234567890"
        '''
        result = self._values.get("directory_id")
        assert result is not None, "Required property 'directory_id' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def directory_name(self) -> IStringVariable:
        '''The name of the domain.

        Example::

            "example.com"
        '''
        result = self._values.get("directory_name")
        assert result is not None, "Required property 'directory_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def dns_ip_addresses(self) -> IStringListVariable:
        '''The IP addresses of the DNS servers.

        Example::

            ["198.51.100.1","198.51.100.2"]
        '''
        result = self._values.get("dns_ip_addresses")
        assert result is not None, "Required property 'dns_ip_addresses' is missing"
        return typing.cast(IStringListVariable, result)

    @builtins.property
    def directory_ou(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The organizational unit (OU).

        Example::

            "OU=test,DC=example,DC=com"
        '''
        result = self._values.get("directory_ou")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DomainJoinStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DownloadContentStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DownloadContentStep",
):
    '''AutomationStep implemenation for aws:downloadContent https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-downloadContent.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        downloadable_content: IDownloadableContent,
        destination_path: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param downloadable_content: The information required to retrieve the content from the required source. This is a dictionary whose format changes based on the sourceType See the aws documentation for more info https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-downloadContent
        :param destination_path: (Optional) An optional local path on the instance where you want to download the file. If you don't specify a path, the content is downloaded to a path relative to your command ID.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                downloadable_content: IDownloadableContent,
                destination_path: typing.Optional[IStringVariable] = None,
                exit_on_failure: typing.Optional[builtins.bool] = None,
                exit_on_success: typing.Optional[builtins.bool] = None,
                finally_step: typing.Optional[builtins.bool] = None,
                mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
                on_cancel: typing.Optional[Step] = None,
                precondition: typing.Optional[Precondition] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = DownloadContentStepProps(
            downloadable_content=downloadable_content,
            destination_path=destination_path,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command includes both the sourceType and sourceInfo variables and  destinationPath if provided.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="downloadableContent")
    def downloadable_content(self) -> IDownloadableContent:
        return typing.cast(IDownloadableContent, jsii.get(self, "downloadableContent"))

    @builtins.property
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property
    @jsii.member(jsii_name="destinationPath")
    def destination_path(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "destinationPath"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DownloadContentStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "downloadable_content": "downloadableContent",
        "destination_path": "destinationPath",
    },
)
class DownloadContentStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        downloadable_content: IDownloadableContent,
        destination_path: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties.json for ps Module step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param downloadable_content: The information required to retrieve the content from the required source. This is a dictionary whose format changes based on the sourceType See the aws documentation for more info https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-downloadContent
        :param destination_path: (Optional) An optional local path on the instance where you want to download the file. If you don't specify a path, the content is downloaded to a path relative to your command ID.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                exit_on_failure: typing.Optional[builtins.bool] = None,
                exit_on_success: typing.Optional[builtins.bool] = None,
                finally_step: typing.Optional[builtins.bool] = None,
                mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
                on_cancel: typing.Optional[Step] = None,
                precondition: typing.Optional[Precondition] = None,
                downloadable_content: IDownloadableContent,
                destination_path: typing.Optional[IStringVariable] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument exit_on_failure", value=exit_on_failure, expected_type=type_hints["exit_on_failure"])
            check_type(argname="argument exit_on_success", value=exit_on_success, expected_type=type_hints["exit_on_success"])
            check_type(argname="argument finally_step", value=finally_step, expected_type=type_hints["finally_step"])
            check_type(argname="argument mark_success_and_exit_on_failure", value=mark_success_and_exit_on_failure, expected_type=type_hints["mark_success_and_exit_on_failure"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument precondition", value=precondition, expected_type=type_hints["precondition"])
            check_type(argname="argument downloadable_content", value=downloadable_content, expected_type=type_hints["downloadable_content"])
            check_type(argname="argument destination_path", value=destination_path, expected_type=type_hints["destination_path"])
        self._values: typing.Dict[str, typing.Any] = {
            "downloadable_content": downloadable_content,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if destination_path is not None:
            self._values["destination_path"] = destination_path

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def downloadable_content(self) -> IDownloadableContent:
        '''The information required to retrieve the content from the required source.

        This is a dictionary whose format changes based on the sourceType
        See the aws documentation for more info
        https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-downloadContent
        '''
        result = self._values.get("downloadable_content")
        assert result is not None, "Required property 'downloadable_content' is missing"
        return typing.cast(IDownloadableContent, result)

    @builtins.property
    def destination_path(self) -> typing.Optional[IStringVariable]:
        '''(Optional) An optional local path on the instance where you want to download the file.

        If you don't specify a path, the content is downloaded to a path relative to your command ID.
        '''
        result = self._values.get("destination_path")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DownloadContentStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ExecuteAutomationStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteAutomationStep",
):
    '''AutomationStep implementation for aws:executeAutomation https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeAutomation.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        document_name: IStringVariable,
        document_version: typing.Optional[IStringVariable] = None,
        max_concurrency: typing.Optional[IStringVariable] = None,
        max_errors: typing.Optional[IStringVariable] = None,
        runtime_parameters: typing.Optional[IStringMapVariable] = None,
        tags: typing.Optional[IMapListVariable] = None,
        target_locations: typing.Optional[IMapListVariable] = None,
        target_maps: typing.Optional[IMapListVariable] = None,
        target_parameter_name: typing.Optional[IStringVariable] = None,
        targets: typing.Optional[IMapListVariable] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                document_name: IStringVariable,
                document_version: typing.Optional[IStringVariable] = None,
                max_concurrency: typing.Optional[IStringVariable] = None,
                max_errors: typing.Optional[IStringVariable] = None,
                runtime_parameters: typing.Optional[IStringMapVariable] = None,
                tags: typing.Optional[IMapListVariable] = None,
                target_locations: typing.Optional[IMapListVariable] = None,
                target_maps: typing.Optional[IMapListVariable] = None,
                target_parameter_name: typing.Optional[IStringVariable] = None,
                targets: typing.Optional[IMapListVariable] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ExecuteAutomationStepProps(
            document_name=document_name,
            document_version=document_version,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            runtime_parameters=runtime_parameters,
            tags=tags,
            target_locations=target_locations,
            target_maps=target_maps,
            target_parameter_name=target_parameter_name,
            targets=targets,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="documentName")
    def document_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "documentName"))

    @builtins.property
    @jsii.member(jsii_name="documentVersion")
    def document_version(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "documentVersion"))

    @builtins.property
    @jsii.member(jsii_name="maxConcurrency")
    def max_concurrency(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "maxConcurrency"))

    @builtins.property
    @jsii.member(jsii_name="maxErrors")
    def max_errors(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "maxErrors"))

    @builtins.property
    @jsii.member(jsii_name="runtimeParameters")
    def runtime_parameters(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "runtimeParameters"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="targetLocations")
    def target_locations(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "targetLocations"))

    @builtins.property
    @jsii.member(jsii_name="targetMaps")
    def target_maps(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "targetMaps"))

    @builtins.property
    @jsii.member(jsii_name="targetParameterName")
    def target_parameter_name(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "targetParameterName"))

    @builtins.property
    @jsii.member(jsii_name="targets")
    def targets(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "targets"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteAutomationStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "document_name": "documentName",
        "document_version": "documentVersion",
        "max_concurrency": "maxConcurrency",
        "max_errors": "maxErrors",
        "runtime_parameters": "runtimeParameters",
        "tags": "tags",
        "target_locations": "targetLocations",
        "target_maps": "targetMaps",
        "target_parameter_name": "targetParameterName",
        "targets": "targets",
    },
)
class ExecuteAutomationStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        document_name: IStringVariable,
        document_version: typing.Optional[IStringVariable] = None,
        max_concurrency: typing.Optional[IStringVariable] = None,
        max_errors: typing.Optional[IStringVariable] = None,
        runtime_parameters: typing.Optional[IStringMapVariable] = None,
        tags: typing.Optional[IMapListVariable] = None,
        target_locations: typing.Optional[IMapListVariable] = None,
        target_maps: typing.Optional[IMapListVariable] = None,
        target_parameter_name: typing.Optional[IStringVariable] = None,
        targets: typing.Optional[IMapListVariable] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                document_name: IStringVariable,
                document_version: typing.Optional[IStringVariable] = None,
                max_concurrency: typing.Optional[IStringVariable] = None,
                max_errors: typing.Optional[IStringVariable] = None,
                runtime_parameters: typing.Optional[IStringMapVariable] = None,
                tags: typing.Optional[IMapListVariable] = None,
                target_locations: typing.Optional[IMapListVariable] = None,
                target_maps: typing.Optional[IMapListVariable] = None,
                target_parameter_name: typing.Optional[IStringVariable] = None,
                targets: typing.Optional[IMapListVariable] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument document_name", value=document_name, expected_type=type_hints["document_name"])
            check_type(argname="argument document_version", value=document_version, expected_type=type_hints["document_version"])
            check_type(argname="argument max_concurrency", value=max_concurrency, expected_type=type_hints["max_concurrency"])
            check_type(argname="argument max_errors", value=max_errors, expected_type=type_hints["max_errors"])
            check_type(argname="argument runtime_parameters", value=runtime_parameters, expected_type=type_hints["runtime_parameters"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument target_locations", value=target_locations, expected_type=type_hints["target_locations"])
            check_type(argname="argument target_maps", value=target_maps, expected_type=type_hints["target_maps"])
            check_type(argname="argument target_parameter_name", value=target_parameter_name, expected_type=type_hints["target_parameter_name"])
            check_type(argname="argument targets", value=targets, expected_type=type_hints["targets"])
        self._values: typing.Dict[str, typing.Any] = {
            "document_name": document_name,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if document_version is not None:
            self._values["document_version"] = document_version
        if max_concurrency is not None:
            self._values["max_concurrency"] = max_concurrency
        if max_errors is not None:
            self._values["max_errors"] = max_errors
        if runtime_parameters is not None:
            self._values["runtime_parameters"] = runtime_parameters
        if tags is not None:
            self._values["tags"] = tags
        if target_locations is not None:
            self._values["target_locations"] = target_locations
        if target_maps is not None:
            self._values["target_maps"] = target_maps
        if target_parameter_name is not None:
            self._values["target_parameter_name"] = target_parameter_name
        if targets is not None:
            self._values["targets"] = targets

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def document_name(self) -> IStringVariable:
        result = self._values.get("document_name")
        assert result is not None, "Required property 'document_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def document_version(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("document_version")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def max_concurrency(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("max_concurrency")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def max_errors(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("max_errors")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def runtime_parameters(self) -> typing.Optional[IStringMapVariable]:
        result = self._values.get("runtime_parameters")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def tags(self) -> typing.Optional[IMapListVariable]:
        result = self._values.get("tags")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def target_locations(self) -> typing.Optional[IMapListVariable]:
        result = self._values.get("target_locations")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def target_maps(self) -> typing.Optional[IMapListVariable]:
        result = self._values.get("target_maps")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def target_parameter_name(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("target_parameter_name")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def targets(self) -> typing.Optional[IMapListVariable]:
        result = self._values.get("targets")
        return typing.cast(typing.Optional[IMapListVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExecuteAutomationStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ExecuteScriptStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteScriptStep",
):
    '''AutomationStep implementation for aws:executeScript https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html.'''

    def __init__(
        self,
        stage: constructs.Construct,
        id: builtins.str,
        *,
        code: ScriptCode,
        input_payload: typing.Mapping[builtins.str, IGenericVariable],
        language: ScriptLanguage,
        outputs: typing.Optional[typing.Sequence[typing.Union[Output, typing.Dict[str, typing.Any]]]] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param stage: -
        :param id: -
        :param code: Inline code to be executed. String will be used to produce function in yaml/json. Simulation will execute the function in this code using the language specified.
        :param input_payload: InputPayload that will be passed to the first parameter of the handler. This can be used to pass input data to the script. The key of this dict is the variable name that will be available to the code. The value is the Variable object.
        :param language: (Required) Language used to execute the script.
        :param outputs: (Optional) Outputs that the function is expected to return. Be sure to prefix the selector for these outputs with "$.Payload." for executeScript step outputs. Default: []
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                stage: constructs.Construct,
                id: builtins.str,
                *,
                code: ScriptCode,
                input_payload: typing.Mapping[builtins.str, IGenericVariable],
                language: ScriptLanguage,
                outputs: typing.Optional[typing.Sequence[typing.Union[Output, typing.Dict[str, typing.Any]]]] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ExecuteScriptStepProps(
            code=code,
            input_payload=input_payload,
            language=language,
            outputs=outputs,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [stage, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listUserOutputs")
    def list_user_outputs(self) -> typing.List[Output]:
        '''Lists the outputs defined by the user for this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listUserOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="code")
    def code(self) -> ScriptCode:
        return typing.cast(ScriptCode, jsii.get(self, "code"))

    @builtins.property
    @jsii.member(jsii_name="inputs")
    def inputs(self) -> typing.Mapping[builtins.str, IGenericVariable]:
        return typing.cast(typing.Mapping[builtins.str, IGenericVariable], jsii.get(self, "inputs"))

    @builtins.property
    @jsii.member(jsii_name="language")
    def language(self) -> ScriptLanguage:
        return typing.cast(ScriptLanguage, jsii.get(self, "language"))

    @builtins.property
    @jsii.member(jsii_name="outputs")
    def outputs(self) -> typing.List[Output]:
        return typing.cast(typing.List[Output], jsii.get(self, "outputs"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteScriptStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "code": "code",
        "input_payload": "inputPayload",
        "language": "language",
        "outputs": "outputs",
    },
)
class ExecuteScriptStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        code: ScriptCode,
        input_payload: typing.Mapping[builtins.str, IGenericVariable],
        language: ScriptLanguage,
        outputs: typing.Optional[typing.Sequence[typing.Union[Output, typing.Dict[str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for ExecuteScriptStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param code: Inline code to be executed. String will be used to produce function in yaml/json. Simulation will execute the function in this code using the language specified.
        :param input_payload: InputPayload that will be passed to the first parameter of the handler. This can be used to pass input data to the script. The key of this dict is the variable name that will be available to the code. The value is the Variable object.
        :param language: (Required) Language used to execute the script.
        :param outputs: (Optional) Outputs that the function is expected to return. Be sure to prefix the selector for these outputs with "$.Payload." for executeScript step outputs. Default: []
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                code: ScriptCode,
                input_payload: typing.Mapping[builtins.str, IGenericVariable],
                language: ScriptLanguage,
                outputs: typing.Optional[typing.Sequence[typing.Union[Output, typing.Dict[str, typing.Any]]]] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument input_payload", value=input_payload, expected_type=type_hints["input_payload"])
            check_type(argname="argument language", value=language, expected_type=type_hints["language"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
        self._values: typing.Dict[str, typing.Any] = {
            "code": code,
            "input_payload": input_payload,
            "language": language,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if outputs is not None:
            self._values["outputs"] = outputs

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def code(self) -> ScriptCode:
        '''Inline code to be executed.

        String will be used to produce function in yaml/json.
        Simulation will execute the function in this code using the language specified.
        '''
        result = self._values.get("code")
        assert result is not None, "Required property 'code' is missing"
        return typing.cast(ScriptCode, result)

    @builtins.property
    def input_payload(self) -> typing.Mapping[builtins.str, IGenericVariable]:
        '''InputPayload that will be passed to the first parameter of the handler.

        This can be used to pass input data to the script.
        The key of this dict is the variable name that will be available to the code.
        The value is the Variable object.
        '''
        result = self._values.get("input_payload")
        assert result is not None, "Required property 'input_payload' is missing"
        return typing.cast(typing.Mapping[builtins.str, IGenericVariable], result)

    @builtins.property
    def language(self) -> ScriptLanguage:
        '''(Required) Language used to execute the script.'''
        result = self._values.get("language")
        assert result is not None, "Required property 'language' is missing"
        return typing.cast(ScriptLanguage, result)

    @builtins.property
    def outputs(self) -> typing.Optional[typing.List[Output]]:
        '''(Optional) Outputs that the function is expected to return.

        Be sure to prefix the selector for these outputs with "$.Payload." for executeScript step outputs.

        :default: []
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Optional[typing.List[Output]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExecuteScriptStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ExecuteStateMachineStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteStateMachineStep",
):
    '''AutomationStep implementation of `aws:executeStateMachine <https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeStateMachine.html>`_.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        state_machine_arn: IStringVariable,
        execution_name: typing.Optional[IStringVariable] = None,
        input: typing.Optional[IStringVariable] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param state_machine_arn: The Amazon Resource Name (ARN) of the Step Functions state machine.
        :param execution_name: (Optional) The name of the execution.
        :param input: (Optional) A string that contains the JSON input data for the execution.
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        if __debug__:
            def stub(
                scope: constructs.Construct,
                id: builtins.str,
                *,
                state_machine_arn: IStringVariable,
                execution_name: typing.Optional[IStringVariable] = None,
                input: typing.Optional[IStringVariable] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ExecuteStateMachineStepProps(
            state_machine_arn=state_machine_arn,
            execution_name=execution_name,
            input=input,
            explicit_next_step=explicit_next_step,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''No outputs emitted from Delete Stack step.

        :return: []
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property
    @jsii.member(jsii_name="stateMachineArn")
    def state_machine_arn(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "stateMachineArn"))

    @builtins.property
    @jsii.member(jsii_name="executionName")
    def execution_name(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "executionName"))

    @builtins.property
    @jsii.member(jsii_name="input")
    def input(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "input"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteStateMachineStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "explicit_next_step": "explicitNextStep",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "state_machine_arn": "stateMachineArn",
        "execution_name": "executionName",
        "input": "input",
    },
)
class ExecuteStateMachineStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        explicit_next_step: typing.Optional[StepRef] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["OnCancel"] = None,
        on_failure: typing.Optional[OnFailure] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        state_machine_arn: IStringVariable,
        execution_name: typing.Optional[IStringVariable] = None,
        input: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Props for ExecuteStateMachine step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param explicit_next_step: (Optional) explicit step to go to after this step completes. https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp Default: will implicitly choose the next step in the sequence that the steps are added to the document.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Fallback action to take in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Fallback action to take in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param state_machine_arn: The Amazon Resource Name (ARN) of the Step Functions state machine.
        :param execution_name: (Optional) The name of the execution.
        :param input: (Optional) A string that contains the JSON input data for the execution.
        '''
        if __debug__:
            def stub(
                *,
                description: typing.Optional[builtins.str] = None,
                input_observer: typing.Optional[IObserver] = None,
                name: typing.Optional[builtins.str] = None,
                output_observer: typing.Optional[IObserver] = None,
                explicit_next_step: typing.Optional[StepRef] = None,
                is_end: typing.Optional[builtins.bool] = None,
                max_attempts: typing.Optional[jsii.Number] = None,
                on_cancel: typing.Optional["OnCancel"] = None,
                on_failure: typing.Optional[OnFailure] = None,
                timeout_seconds: typing.Optional[jsii.Number] = None,
                state_machine_arn: IStringVariable,
                execution_name: typing.Optional[IStringVariable] = None,
                input: typing.Optional[IStringVariable] = None,
            ) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument input_observer", value=input_observer, expected_type=type_hints["input_observer"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument output_observer", value=output_observer, expected_type=type_hints["output_observer"])
            check_type(argname="argument explicit_next_step", value=explicit_next_step, expected_type=type_hints["explicit_next_step"])
            check_type(argname="argument is_end", value=is_end, expected_type=type_hints["is_end"])
            check_type(argname="argument max_attempts", value=max_attempts, expected_type=type_hints["max_attempts"])
            check_type(argname="argument on_cancel", value=on_cancel, expected_type=type_hints["on_cancel"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument timeout_seconds", value=timeout_seconds, expected_type=type_hints["timeout_seconds"])
            check_type(argname="argument state_machine_arn", value=state_machine_arn, expected_type=type_hints["state_machine_arn"])
            check_type(argname="argument execution_name", value=execution_name, expected_type=type_hints["execution_name"])
            check_type(argname="argument input", value=input, expected_type=type_hints["input"])
        self._values: typing.Dict[str, typing.Any] = {
            "state_machine_arn": state_machine_arn,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if explicit_next_step is not None:
            self._values["explicit_next_step"] = explicit_next_step
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if execution_name is not None:
            self._values["execution_name"] = execution_name
        if input is not None:
            self._values["input"] = input

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def explicit_next_step(self) -> typing.Optional[StepRef]:
        '''(Optional) explicit step to go to after this step completes.

        https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#nextProp

        :default: will implicitly choose the next step in the sequence that the steps are added to the document.
        '''
        result = self._values.get("explicit_next_step")
        return typing.cast(typing.Optional[StepRef], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional["OnCancel"]:
        '''(Optional) Fallback action to take in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional["OnCancel"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[OnFailure]:
        '''(Optional) Fallback action to take in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[OnFailure], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def state_machine_arn(self) -> IStringVariable:
        '''The Amazon Resource Name (ARN) of the Step Functions state machine.'''
        result = self._values.get("state_machine_arn")
        assert result is not None, "Required property 'state_machine_arn' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def execution_name(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The name of the execution.'''
        result = self._values.get("execution_name")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def input(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A string that contains the JSON input data for the execution.'''
        result = self._values.get("input")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExecuteStateMachineStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class FileScriptCode(
    ScriptCode,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.FileScriptCode",
):
    def __init__(self, full_path: builtins.str) -> None:
        '''
        :param full_path: -
        '''
        if __debug__:
            def stub(full_path: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument full_path", value=full_path, expected_type=type_hints["full_path"])
        jsii.create(self.__class__, self, [full_path])

    @jsii.member(jsii_name="codeAsString")
    def code_as_string(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "codeAsString", []))

    @jsii.member(jsii_name="createOrGetFile")
    def create_or_get_file(self, _suffix: builtins.str) -> builtins.str:
        '''If there is a file for this code, return it.

        Otherwise, create a file with the specified suffix.

        :param _suffix: -
        '''
        if __debug__:
            def stub(_suffix: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument _suffix", value=_suffix, expected_type=type_hints["_suffix"])
        return typing.cast(builtins.str, jsii.invoke(self, "createOrGetFile", [_suffix]))

    @builtins.property
    @jsii.member(jsii_name="fullPath")
    def full_path(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "fullPath"))


@jsii.implements(IGenericVariable)
class GenericVariable(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.GenericVariable",
):
    '''Abstraction of SSM variables.

    Variables are printed as using this syntax: {{ myVariable }}
    To resolve a variable, you must supply the available inputs and the variable will resolve the value.
    '''

    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        if __debug__:
            def stub(reference: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument reference", value=reference, expected_type=type_hints["reference"])
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="assertType")
    @abc.abstractmethod
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        ...

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{ Variable }} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Returns a string representation of an object.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @builtins.property
    @jsii.member(jsii_name="reference")
    def reference(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "reference"))


class _GenericVariableProxy(GenericVariable):
    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        if __debug__:
            def stub(value: typing.Any) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, GenericVariable).__jsii_proxy_class__ = lambda : _GenericVariableProxy


@jsii.implements(IDownloadableContent)
class GitContent(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.GitContent",
):
    def __init__(
        self,
        *,
        repository: IStringVariable,
        branch: typing.Optional[builtins.str] = None,
        commit_id: typing.Optional[builtins.str] = None,
        password: typing.Optional[SecureVariable] = None,
        private_ssh_key: typing.Optional[SecureVariable] = None,
        skip_host_key_checking: typing.Optional["IBooleanVariable"] = None,
        user_name: typing.Optional[SecureVariable] = None,
    ) -> None:
        '''
        :param repository: The Git repository URL to the file or directory you want to download.
        :param branch: The default is master. branch parameter is required only if your SSM document is stored in a branch other than master. Supply either commitId or branch (or neither).
        :param commit_id: The default is head. To use the version of your SSM document in a commit other than the latest, specify the full commit ID. For example: "bbc1ddb94...b76d3bEXAMPLE". Supply either commitId or branch (or neither).
        :param password: The password to use when connecting to the repository you specify using HTTP.
        :param private_ssh_key: The SSH key to use when connecting to the repository you specify.
        :param skip_host_key_checking: Determines the value of the StrictHostKeyChecking option when connecting to the repository you specify. The default value is false.
        :param user_name: The username to use when connecting to the repository you specify using HTTP.
        '''
        props = GitContentProps(
            repository=repository,
            branch=branch,
            commit_id=commit_id,
            password=password,
            private_ssh_key=private_ssh_key,
            skip_host_key_checking=skip_host_key_checking,
            user_name=user_name,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property
    @jsii.member(jsii_name="repository")
    def repository(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "repository"))

    @builtins.property
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))

    @builtins.property
    @jsii.member(jsii_name="branch")
    def branch(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "branch"))

    @builtins.property
    @jsii.member(jsii_name="commitId")
    def commit_id(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "commitId"))

    @builtins.property
    @jsii.member(jsii_name="password")
    def password(self) -> typing.Optional[SecureVariable]:
        return typing.cast(typing.Optional[SecureVariable], jsii.get(self, "password"))

    @builtins.property
    @jsii.member(jsii_name="privateSshKey")
    def private_ssh_key(self) -> typing.Optional[SecureVariable]:
        return typing.cast(typing.Optional[SecureVariable], jsii.get(self, "privateSshKey"))

    @builtins.property
    @jsii.member(jsii_name="skipHostKeyChecking")
    def skip_host_key_checking(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "skipHostKeyChecking"))

    @builtins.property
    @jsii.member(jsii_name="userName")
    def user_name(self) -> typing.Optional[SecureVariable]:
        return typing.cast(typing.Optional[SecureVariable], jsii.get(self, "userName"))


@jsii.implements(IDownloadableContent)
class GitHubContent(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.GitHubContent",
):
    def __init__(
        self,
        *,
        owner: IStringVariable,
        path: IStringVariable,
        repository: IStringVariable,
        token_info: SecureVariable,
        branch: typing.Optional[builtins.str] = None,
        commit_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param owner: The repository owner.
        :param path: The path to the file or directory you want to download.
        :param repository: The name of the repository.
        :param token_info: The Systems Manager parameter (a SecureString parameter) where you store your GitHub access token information.
        :param branch: The default is master. branch parameter is required only if your SSM document is stored in a branch other than master. Supply either commitId or branch (or neither).
        :param commit_id: The default is head. To use the version of your SSM document in a commit other than the latest, specify the full commit ID. For example: "bbc1ddb94...b76d3bEXAMPLE". Supply either commitId or branch (or neither).
        '''
        props = GitHubContentProps(
            owner=owner,
            path=path,
            repository=repository,
            token_info=token_info,
            branch=branch,
            commit_id=commit_id,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property
    @jsii.member(jsii_name="owner")
    def owner(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "owner"))

    @builtins.property
    @jsii.member(jsii_name="path")
    def path(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "path"))

    @builtins.property
    @jsii.member(jsii_name="repository")
    def repository(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "repository"))

    @builtins.property
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))

    @builtins.property
    @jsii.member(jsii_name="tokenInfo")
    def token_info(self) -> SecureVariable:
        return typing.cast(SecureVariable, jsii.get(self, "tokenInfo"))

    @builtins.property
    @jsii.member(jsii_name="branch")
    def branch(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "branch"))

    @builtins.property
    @jsii.member(jsii_name="commitId")
    def commit_id(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "commitId"))


class HardCodedSecureVariable(
    SecureVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedSecureVariable",
):
    '''A hard-coded string variable.

    Used when not dependent on step inputs.
    '''

    def __init__(self, val: builtins.str) -> None:
        '''
        :param val: -
        '''
        if __debug__:
            def stub(val: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument val", value=val, expected_type=type_hints["val"])
        jsii.create(self.__class__, self, [val])

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{ Variable }} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, _inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param _inputs: -
        '''
        if __debug__:
            def stub(_inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument _inputs", value=_inputs, expected_type=type_hints["_inputs"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [_inputs]))

    @builtins.property
    @jsii.member(jsii_name="val")
    def val(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "val"))


@jsii.implements(IGenericVariable)
class HardCodedValueBase(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedValueBase",
):
    def __init__(self, val: typing.Any) -> None:
        '''
        :param val: -
        '''
        if __debug__:
            def stub(val: typing.Any) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument val", value=val, expected_type=type_hints["val"])
        jsii.create(self.__class__, self, [val])

    @jsii.member(jsii_name="assertType")
    @abc.abstractmethod
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        ...

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{ Variable }} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, _inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param _inputs: -
        '''
        if __debug__:
            def stub(_inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument _inputs", value=_inputs, expected_type=type_hints["_inputs"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [_inputs]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Returns a string representation of an object.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @builtins.property
    @jsii.member(jsii_name="val")
    def val(self) -> typing.Any:
        return typing.cast(typing.Any, jsii.get(self, "val"))


class _HardCodedValueBaseProxy(HardCodedValueBase):
    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        if __debug__:
            def stub(value: typing.Any) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, HardCodedValueBase).__jsii_proxy_class__ = lambda : _HardCodedValueBaseProxy


@jsii.implements(IDownloadableContent)
class HttpContent(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HttpContent",
):
    def __init__(
        self,
        *,
        url: IStringVariable,
        allow_insecure_download: typing.Optional["IBooleanVariable"] = None,
        auth_method: typing.Optional[AuthMethod] = None,
    ) -> None:
        '''
        :param url: The URL to the file or directory you want to download.
        :param allow_insecure_download: Determines whether a download can be performed over a connection that isn't encrypted with Secure Socket Layer (SSL) or Transport Layer Security (TLS). The default value is false. We don't recommend performing downloads without encryption. If you choose to do so, you assume all associated risks. Security is a shared responsibility between AWS and you. This is described as the shared responsibility model. To learn more, see the shared responsibility model.
        :param auth_method: Determines whether a username and password are used for authentication when connecting to the url you specify. If you specify Basic or Digest, you must provide values for the username and password parameters. To use the Digest method, SSM Agent version 3.0.1181.0 or later must be installed on your instance. The Digest method supports MD5 and SHA256 encryption.
        '''
        props = HttpContentProps(
            url=url,
            allow_insecure_download=allow_insecure_download,
            auth_method=auth_method,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))

    @builtins.property
    @jsii.member(jsii_name="url")
    def url(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "url"))

    @builtins.property
    @jsii.member(jsii_name="allowInsecureDownload")
    def allow_insecure_download(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "allowInsecureDownload"))

    @builtins.property
    @jsii.member(jsii_name="authMethod")
    def auth_method(self) -> typing.Optional[AuthMethod]:
        return typing.cast(typing.Optional[AuthMethod], jsii.get(self, "authMethod"))


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IActionVariable")
class IActionVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IActionVariableProxy(
    jsii.proxy_for(IStringVariable), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IActionVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IActionVariable).__jsii_proxy_class__ = lambda : _IActionVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IBooleanVariable")
class IBooleanVariable(IGenericVariable, typing_extensions.Protocol):
    '''A boolean variable.'''

    @jsii.member(jsii_name="resolveToBoolean")
    def resolve_to_boolean(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.bool:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _IBooleanVariableProxy(
    jsii.proxy_for(IGenericVariable), # type: ignore[misc]
):
    '''A boolean variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IBooleanVariable"

    @jsii.member(jsii_name="resolveToBoolean")
    def resolve_to_boolean(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.bool:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        if __debug__:
            def stub(inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
        return typing.cast(builtins.bool, jsii.invoke(self, "resolveToBoolean", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IBooleanVariable).__jsii_proxy_class__ = lambda : _IBooleanVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IDesiredStateVariable")
class IDesiredStateVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IDesiredStateVariableProxy(
    jsii.proxy_for(IStringVariable), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IDesiredStateVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IDesiredStateVariable).__jsii_proxy_class__ = lambda : _IDesiredStateVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IDocumentHashTypeVariable")
class IDocumentHashTypeVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IDocumentHashTypeVariableProxy(
    jsii.proxy_for(IStringVariable), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IDocumentHashTypeVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IDocumentHashTypeVariable).__jsii_proxy_class__ = lambda : _IDocumentHashTypeVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IInstallUninstallRepairVariable")
class IInstallUninstallRepairVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IInstallUninstallRepairVariableProxy(
    jsii.proxy_for(IStringVariable), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IInstallUninstallRepairVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IInstallUninstallRepairVariable).__jsii_proxy_class__ = lambda : _IInstallUninstallRepairVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IInstallationTypeVariable")
class IInstallationTypeVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IInstallationTypeVariableProxy(
    jsii.proxy_for(IStringVariable), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IInstallationTypeVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IInstallationTypeVariable).__jsii_proxy_class__ = lambda : _IInstallationTypeVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IOnFailureVariable")
class IOnFailureVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IOnFailureVariableProxy(
    jsii.proxy_for(IStringVariable), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IOnFailureVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IOnFailureVariable).__jsii_proxy_class__ = lambda : _IOnFailureVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IPackageNameVariable")
class IPackageNameVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IPackageNameVariableProxy(
    jsii.proxy_for(IStringVariable), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IPackageNameVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IPackageNameVariable).__jsii_proxy_class__ = lambda : _IPackageNameVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IResourceTypeVariable")
class IResourceTypeVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IResourceTypeVariableProxy(
    jsii.proxy_for(IStringVariable), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IResourceTypeVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IResourceTypeVariable).__jsii_proxy_class__ = lambda : _IResourceTypeVariableProxy


class InlineScriptCode(
    ScriptCode,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.InlineScriptCode",
):
    def __init__(self, inline_code: builtins.str) -> None:
        '''
        :param inline_code: -
        '''
        if __debug__:
            def stub(inline_code: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument inline_code", value=inline_code, expected_type=type_hints["inline_code"])
        jsii.create(self.__class__, self, [inline_code])

    @jsii.member(jsii_name="codeAsString")
    def code_as_string(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "codeAsString", []))

    @jsii.member(jsii_name="createOrGetFile")
    def create_or_get_file(self, suffix: builtins.str) -> builtins.str:
        '''If there is a file for this code, return it.

        Otherwise, create a file with the specified suffix.

        :param suffix: -
        '''
        if __debug__:
            def stub(suffix: builtins.str) -> None:
                ...
            type_hints = typing.get_type_hints(stub)
            check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
        return typing.cast(builtins.str, jsii.invoke(self, "createOrGetFile", [suffix]))

    @builtins.property
    @jsii.member(jsii_name="inlineCode")
    def inline_code(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "inlineCode"))


class InvokeLambdaFunctionStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeLambdaFunctionStep",
):
    '''AutomationStep implemenation for aws:invokeLambdaFunction https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-lamb.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
    