"""Test libconeangle."""

import numpy as np
from numpy.testing import assert_allclose
from numpy.typing import NDArray
import pytest

from libconeangle import cone_angle


def test_PdCO():
    """Test PdCO with cone tangent to one atom."""
    coordinates: NDArray[np.float64] = np.array(
        [[0.0, 0.0, -0.52], [0.0, 0.0, 1.76], [0.0, 0.0, 2.86]]
    )
    radii: NDArray[np.float64] = np.array([2.1, 1.7, 1.52])
    c_angle, axis, tangent_atoms = cone_angle(coordinates, radii, 0)

    assert_allclose(c_angle, 96.423, atol=0.001)
    assert_allclose(axis, np.array([0.0, 0.0, 1.0]), atol=0.0001)
    assert set(tangent_atoms) == set([1])


def test_PdCO_close():
    """Test PdCO with one atom within vdW distance of Pd atom."""
    coordinates: NDArray[np.float64] = np.array(
        [[0.0, 0.0, 0.1], [0.0, 0.0, 1.76], [0.0, 0.0, 2.86]]
    )  # Pd atom too close
    radii: NDArray[np.float64] = np.array([2.1, 1.7, 1.52])
    with pytest.raises(ValueError):
        cone_angle(coordinates, radii, 0)


def test_PdCO_bounds():
    """Test PdCO with metal index out of bounds."""
    coordinates: NDArray[np.float64] = np.array(
        [[0.0, 0.0, 0.1], [0.0, 0.0, 1.76], [0.0, 0.0, 2.86]]
    )  # Pd atom too close
    radii: NDArray[np.float64] = np.array([2.1, 1.7, 1.52])
    with pytest.raises(ValueError):
        cone_angle(coordinates, radii, -1)


def test_PdCO_mismatch():
    """Test PdCO with metal index out of bounds."""
    coordinates: NDArray[np.float64] = np.array(
        [[0.0, 0.0, 0.1], [0.0, 0.0, 1.76], [0.0, 0.0, 2.86]]
    )  # Pd atom too close
    radii: NDArray[np.float64] = np.array([2.1, 1.7])
    with pytest.raises(ValueError):
        cone_angle(coordinates, radii, 0)


def test_Pdbpy():
    """Test Pd(bpy) with cone tangent to two atoms."""
    coordinates: NDArray[np.float64] = np.array(
        [
            [-1.899000e-03, -1.846004e00, -2.007000e-03],
            [-1.343609e00, -8.359200e-02, 4.197960e-01],
            [1.345137e00, -8.591400e-02, -4.193200e-01],
            [-7.399130e-01, 1.062540e00, 8.798000e-03],
            [-1.473982e00, 2.183787e00, -3.961990e-01],
            [-2.865881e00, 2.130461e00, -3.962120e-01],
            [-3.487476e00, 9.490540e-01, 1.374000e-02],
            [-2.690562e00, -1.192190e-01, 4.151720e-01],
            [-9.542920e-01, 3.073408e00, -7.377720e-01],
            [-3.451265e00, 2.985788e00, -7.207940e-01],
            [-4.569174e00, 8.579540e-01, 3.831000e-02],
            [-3.135734e00, -1.043925e00, 7.700380e-01],
            [2.691916e00, -1.229550e-01, -4.128390e-01],
            [3.489163e00, 9.451910e-01, -1.169700e-02],
            [2.868283e00, 2.127761e00, 3.960000e-01],
            [1.476591e00, 2.182376e00, 3.941950e-01],
            [7.424670e-01, 1.061320e00, -1.040300e-02],
            [3.136172e00, -1.048666e00, -7.660710e-01],
            [4.570699e00, 8.531320e-01, -3.487800e-02],
            [3.453586e00, 2.983188e00, 7.202120e-01],
            [9.563590e-01, 3.072406e00, 7.338950e-01],
        ]
    )
    radii: NDArray[np.float64] = np.array(
        [
            2.1,
            1.55,
            1.55,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.1,
            1.1,
            1.1,
            1.1,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.1,
            1.1,
            1.1,
            1.1,
        ]
    )
    c_angle, axis, tangent_atoms = cone_angle(coordinates, radii, 0)

    assert_allclose(c_angle, 190.799, atol=0.001)
    assert_allclose(axis, np.array([0.00200079, 0.99998474, 0.00514848]), atol=0.0001)
    assert set(tangent_atoms) == set([11, 17])


def test_PdPMe3():
    """Test PdPMe3 with cone tangent to three atoms."""
    coordinates: NDArray[np.float64] = np.array(
        [
            [-7.433970e-01, -9.920000e-04, -1.518770e00],
            [1.005896e00, 2.102000e-03, -1.636180e-01],
            [2.689042e00, -1.142000e-03, -9.464240e-01],
            [2.797104e00, -8.888730e-01, -1.576886e00],
            [3.486761e00, 3.481000e-03, -1.921520e-01],
            [2.795646e00, 8.800330e-01, -1.586258e00],
            [1.164898e00, 1.431506e00, 1.010324e00],
            [1.222204e00, 2.365419e00, 4.431760e-01],
            [2.057780e00, 1.338752e00, 1.642300e00],
            [2.773700e-01, 1.479634e00, 1.648431e00],
            [1.165320e00, -1.429939e00, 1.006886e00],
            [1.222850e00, -2.362601e00, 4.376840e-01],
            [2.776990e-01, -1.479572e00, 1.644733e00],
            [2.058112e00, -1.338384e00, 1.639153e00],
        ]
    )
    radii: NDArray[np.float64] = np.array(
        [2.1, 1.8, 1.7, 1.1, 1.1, 1.1, 1.7, 1.1, 1.1, 1.1, 1.7, 1.1, 1.1, 1.1]
    )
    c_angle, axis, tangent_atoms = cone_angle(coordinates, radii, 0)

    assert_allclose(c_angle, 117.110, atol=0.001)
    assert_allclose(
        axis, np.array([7.89524317e-01, 5.22544038e-05, 6.13719276e-01]), atol=0.0001
    )
    assert set(tangent_atoms) == set([5, 9, 12])


def test_no_cone():
    """Test AdBrettPhos with CO ligands still left so no cone is detected."""
    coordinates: NDArray[np.float64] = np.array(
        [
            [-0.150731, -0.19074103, -1.92057203],
            [3.37788306, 0.05130647, 3.61481396],
            [2.54925444, 0.10907458, 2.4732705],
            [1.16592121, 0.19499813, 2.7295883],
            [0.67750873, 0.32027115, 4.03553064],
            [-0.66459397, 0.33472346, 4.29526449],
            [-1.5896386, 0.22951037, 3.26335973],
            [-2.97600312, 0.16907618, 3.47545234],
            [-3.37273754, 0.14349526, 4.82745361],
            [-1.12082686, 0.16072819, 1.92806203],
            [0.25612043, 0.13107103, 1.67287536],
            [1.02958224, 0.02476611, 0.01490941],
            [1.82059756, 1.7253701, -0.29829352],
            [0.70905762, 2.73064994, 0.07168378],
            [1.17203462, 4.18917778, -0.1847218],
            [2.40741965, 4.4864593, 0.70070731],
            [3.54375578, 3.49571492, 0.34372851],
            [3.93048362, 3.66565495, -1.14642971],
            [2.69574266, 3.37243964, -2.03329984],
            [1.55037404, 4.35567902, -1.67841579],
            [2.20574022, 1.92384277, -1.78094054],
            [3.05713071, 2.04198933, 0.57964817],
            [2.15728409, -1.50864448, -0.07241542],
            [1.99383629, -2.27757743, -1.40472633],
            [2.86192227, -3.56211624, -1.41764689],
            [4.35051981, -3.16593263, -1.25560966],
            [4.53490985, -2.40802958, 0.08272307],
            [4.11213979, -3.32596808, 1.25833128],
            [2.6248743, -3.72959249, 1.09122431],
            [2.43708516, -4.48368875, -0.24816519],
            [1.7486867, -2.45094363, 1.08450343],
            [3.64796946, -1.13454067, 0.07974954],
            [-2.15873843, 0.02557711, 0.87494921],
            [-2.83375913, 1.15424363, 0.36371904],
            [-3.79170515, 0.96990112, -0.61710415],
            [-4.13561567, -0.28988183, -1.07699306],
            [-3.55863471, -1.3919975, -0.47177744],
            [-2.58828911, -1.26278612, 0.50348897],
            [-2.1100334, -2.48589077, 1.26782191],
            [-3.1332671, -2.87618502, 2.33469829],
            [-1.78019071, -3.675817, 0.36281612],
            [-5.05101947, -0.44670698, -2.27345372],
            [-4.23171871, -0.79345761, -3.5188832],
            [-6.16277239, -1.46827493, -2.02460123],
            [-2.6389261, 2.53465503, 0.97093586],
            [-2.6731693, 3.66659474, -0.06385225],
            [-3.67816057, 2.79319683, 2.06269304],
            [3.09870866, -0.79978635, 4.23676883],
            [3.30075044, 0.98089069, 4.17942435],
            [4.41250643, -0.07494945, 3.29490855],
            [1.36612294, 0.39439745, 4.85692446],
            [-1.00413428, 0.41590767, 5.31160633],
            [-4.4574898, 0.04228008, 4.87101366],
            [-3.08700057, 1.07437316, 5.31820892],
            [-2.91961721, -0.70934337, 5.33392628],
            [0.45320601, 2.60280398, 1.12970462],
            [-0.19085692, 2.5009366, -0.50473306],
            [0.35492371, 4.88682367, 0.07034344],
            [2.14421904, 4.3943906, 1.76081388],
            [2.74277135, 5.51883252, 0.54920794],
            [4.42373568, 3.69813931, 0.97994855],
            [4.75063997, 2.98584776, -1.40467617],
            [4.29620677, 4.68241516, -1.32997566],
            [2.96839395, 3.48856684, -3.09739297],
            [0.67652035, 4.15885389, -2.31075326],
            [1.85887259, 5.38767303, -1.88008983],
            [2.96534664, 1.19348819, -2.07803376],
            [1.33404653, 1.74391149, -2.4132579],
            [2.80000551, 1.90433112, 1.63257294],
            [3.86967954, 1.34944608, 0.35361523],
            [2.30473569, -1.63346618, -2.23048312],
            [0.943587, -2.53527988, -1.56287883],
            [2.72089862, -4.09188083, -2.37664043],
            [4.66676824, -2.53530469, -2.09457136],
            [4.98553074, -4.058925, -1.27717134],
            [5.59425957, -2.11901399, 0.20154492],
            [4.25663636, -2.80691628, 2.21277006],
            [4.74609151, -4.2194651, 1.28838594],
            [2.32101659, -4.38137367, 1.9295423],
            [1.39110083, -4.78849902, -0.36828869],
            [3.03412685, -5.40277855, -0.25327918],
            [1.84793775, -1.92904579, 2.04037241],
            [0.69355273, -2.71517113, 0.97742744],
            [3.94364514, -0.47412877, -0.74156635],
            [3.79104303, -0.58608794, 1.01295901],
            [-4.2742406, 1.82643368, -1.05108578],
            [-3.84581055, -2.37458531, -0.79712727],
            [-1.18368241, -2.19257929, 1.78763211],
            [-3.3536679, -2.03497956, 2.99220055],
            [-2.77928716, -3.70502614, 2.94884454],
            [-4.07120364, -3.18222374, 1.87169859],
            [-1.32736176, -4.49395165, 0.92444772],
            [-1.0952821, -3.39356505, -0.43550308],
            [-2.6789532, -4.06780733, -0.11110905],
            [-5.52924589, 0.53389222, -2.44222512],
            [-4.86089039, -0.88510392, -4.40472287],
            [-3.70765326, -1.73971975, -3.3807132],
            [-3.47439401, -0.03299876, -3.71557572],
            [-6.72153597, -1.23264259, -1.11797902],
            [-5.75277225, -2.47091635, -1.90793774],
            [-6.86768348, -1.50183613, -2.85638946],
            [-1.64495071, 2.53371097, 1.44508932],
            [-3.68359058, 3.8164286, -0.4422523],
            [-2.03981694, 3.4530384, -0.92276893],
            [-2.34994421, 4.61433395, 0.36862794],
            [-4.68247165, 2.80296196, 1.63957451],
            [-3.51501765, 3.75298046, 2.55428919],
            [-3.65502807, 2.01491302, 2.8234341],
            [-1.09122209, 1.25105272, -2.2802411],
            [0.94507581, -0.26624979, -3.32603572],
            [-1.11868769, -1.6330022, -2.21693533],
            [-1.73355125, -2.57286054, -2.41694099],
            [-1.68224675, 2.19788714, -2.51393017],
            [2.00766997, -0.2091706, -3.73779609],
        ]
    )
    radii: NDArray[np.float64] = np.array(
        [
            1.97,
            1.7,
            1.52,
            1.7,
            1.7,
            1.7,
            1.7,
            1.52,
            1.7,
            1.7,
            1.7,
            1.8,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.7,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.1,
            1.7,
            1.7,
            1.7,
            1.52,
            1.52,
            1.52,
        ]
    )
    with pytest.raises(ValueError):
        cone_angle(coordinates, radii, 0)
