"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.startExecution = exports.cfnResponseFailed = exports.cfnResponseSuccess = exports.CREATE_FAILED_PHYSICAL_ID_MARKER = void 0;
const aws_sdk_1 = require("aws-sdk"); // eslint-disable-line import/no-extraneous-dependencies
const http_1 = require("./http");
exports.CREATE_FAILED_PHYSICAL_ID_MARKER = 'AWSCDK::StateMachineProvider::CREATE_FAILED';
async function cfnResponseSuccess(event) {
    console.log('Event: %j', event);
    await http_1.respond('SUCCESS', {
        ...event.Input,
        PhysicalResourceId: event.Output?.PhysicalResourceId ?? event.Input.PhysicalResourceId ?? event.Input.RequestId,
        Data: event.Output?.Data ?? {},
        NoEcho: event.Output?.NoEcho,
    });
}
exports.cfnResponseSuccess = cfnResponseSuccess;
async function cfnResponseFailed(event) {
    console.log('Event: %j', event);
    const parsedCause = JSON.parse(event.Cause);
    const executionResult = {
        ...parsedCause,
        Input: JSON.parse(parsedCause.Input),
    };
    console.log('Execution result: %j', executionResult);
    let physicalResourceId = executionResult.Output?.PhysicalResourceId ?? executionResult.Input.PhysicalResourceId;
    if (!physicalResourceId) {
        // special case: if CREATE fails, which usually implies, we usually don't
        // have a physical resource id. in this case, the subsequent DELETE
        // operation does not have any meaning, and will likely fail as well. to
        // address this, we use a marker so the provider framework can simply
        // ignore the subsequent DELETE.
        if (executionResult.Input.RequestType === 'Create') {
            console.log('CREATE failed, responding with a marker physical resource id so that the subsequent DELETE will be ignored');
            physicalResourceId = exports.CREATE_FAILED_PHYSICAL_ID_MARKER;
        }
        else {
            console.log(`ERROR: Malformed event. "PhysicalResourceId" is required: ${JSON.stringify(event)}`);
        }
    }
    await http_1.respond('FAILED', {
        ...executionResult.Input,
        Reason: `${event.Error}: ${event.Cause}`,
        PhysicalResourceId: physicalResourceId,
    });
}
exports.cfnResponseFailed = cfnResponseFailed;
async function startExecution(event) {
    try {
        console.log('Event: %j', event);
        if (!process.env.STATE_MACHINE_ARN) {
            throw new Error('Missing STATE_MACHINE_ARN.');
        }
        // ignore DELETE event when the physical resource ID is the marker that
        // indicates that this DELETE is a subsequent DELETE to a failed CREATE
        // operation.
        if (event.RequestType === 'Delete' && event.PhysicalResourceId === exports.CREATE_FAILED_PHYSICAL_ID_MARKER) {
            console.log('ignoring DELETE event caused by a failed CREATE event');
            await http_1.respond('SUCCESS', event);
            return;
        }
        const stepFunctions = new aws_sdk_1.StepFunctions();
        await stepFunctions.startExecution({
            stateMachineArn: process.env.STATE_MACHINE_ARN,
            input: JSON.stringify(event),
        }).promise();
    }
    catch (err) {
        console.log(err);
        await http_1.respond('FAILED', {
            ...event,
            Reason: err.message,
        });
    }
}
exports.startExecution = startExecution;
//# sourceMappingURL=data:application/json;base64,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