"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComponentBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const imagebuilder = require("@aws-cdk/aws-imagebuilder");
const yaml = require("yaml");
const Validator = require("yaml-validator");
const component_version_bump_1 = require("./component_version_bump");
const index_1 = require("./index");
const utils = require("./utils");
/**
 * @stability stable
 */
class ComponentBuilder {
    /**
     * @stability stable
     */
    constructor(scope, componentDefs, pipeline_name, platform) {
        /**
         * @stability stable
         */
        this.cacheDir = 'component_cache';
        this.componentDeps = componentDefs;
        this.scope = scope;
        this.pipelineName = pipeline_name;
        this.platform = platform;
        if (!fs.existsSync(this.cacheDir)) {
            fs.mkdirSync(this.cacheDir);
        }
    }
    /**
     * @stability stable
     */
    createComponentDependenciesMap() {
        const components = this.componentDeps.map(c => this.createComponent(c)).filter(c => c);
        return components.reduce((a, el) => ({
            ...a,
            [el.name.substr(this.pipelineName.length + 3)]: el,
        }), {}); // remove the prefix
    }
    /**
     * @stability stable
     */
    createComponent(componentConfig) {
        let componentYaml;
        switch (componentConfig.type) {
            case 'local':
                componentYaml = this.assembleComponent(componentConfig, 'local_components');
                break;
            case 'git':
                componentYaml = this.assembleComponent(componentConfig, this.cacheDir);
                break;
            case 'aws_arn':
                return new index_1.ArnComponentRef(componentConfig.arn, `${this.pipelineName} - ${componentConfig.name}`);
            default:
                componentYaml = '';
        }
        if (componentConfig && componentYaml) {
            return new imagebuilder.CfnComponent(this.scope, `${utils.pascalize(componentConfig.name)}Component`, componentYaml);
        }
        return undefined;
    }
    /**
     * @stability stable
     */
    assembleComponent(componentConfig, basedir) {
        let componentPath = componentConfig.type === 'local' ? componentConfig.name : `${this.pipelineName}@${componentConfig.name}`;
        if (componentConfig.path) {
            componentPath = `${componentPath}/${componentConfig.path}`;
        }
        if (this.checkComponentYaml(basedir, componentPath)) {
            // Get metadata YAML
            const metadata = fs.readFileSync(path.join(basedir, componentPath, 'metadata.yaml'), 'utf-8');
            const metayaml = yaml.parse(metadata);
            // Get base component.yaml
            const baseComponent = fs.readFileSync(path.join(basedir, componentPath, 'component.yaml'), 'utf-8');
            const baseYaml = yaml.parse(baseComponent);
            let assembledYaml = baseYaml;
            if (metayaml.distribution_configs) {
                const distributionconfigcomp = metayaml.distribution_configs[this.platform];
                // if there is a platform specific component yaml specified, load it and add steps to the assembled component
                if (distributionconfigcomp) {
                    const componentExtension = fs.readFileSync(path.join(basedir, componentPath, distributionconfigcomp), 'utf-8');
                    const componentExtYaml = yaml.parse(componentExtension);
                    const phase_extensions = componentExtYaml.phase_extensions;
                    // Add phase extensions to base component
                    this.addPhaseExtensions(phase_extensions, assembledYaml);
                }
            }
            this.addDependencyConstants(componentConfig, assembledYaml);
            const name = `${this.pipelineName} - ${componentConfig.name}`;
            const componentVersionBump = new component_version_bump_1.ComponentVersionBump(this.scope, `${utils.pascalize(`${this.pipelineName} ${componentConfig.name}`)}VersionBump`, name);
            return {
                name: name,
                platform: metayaml.platform,
                version: componentVersionBump.nextVersion,
                data: yaml.stringify(assembledYaml),
                tags: {},
            };
        }
    }
    addDependencyConstants(componentConfig, assembledYaml) {
        if ('constants' in componentConfig) {
            let constantsDict = componentConfig.constants;
            let compConstants = Object.keys(constantsDict).map((k) => {
                let constantObj = {};
                constantObj[k] = {
                    type: 'string',
                    value: constantsDict[k],
                };
                return constantObj;
            });
            if ('constants' in assembledYaml) {
                // Override the keys from the component itself with the ones from the component dependencies.
                let compConstantsKeys = Object.keys(constantsDict);
                assembledYaml.constants = assembledYaml.constants.filter(function (c) {
                    // See if their is an intersection between the keys from component dependency and
                    // the ones defined in the original component.yaml file and remove those from component.yaml file.
                    return Object.keys(c).filter(d => compConstantsKeys.includes(d)).length == 0;
                });
                assembledYaml.constants.splice(assembledYaml.constants.length, 0, ...compConstants);
            }
            else {
                assembledYaml.constants = compConstants;
            }
        }
    }
    addPhaseExtensions(phase_extensions, assembledYaml) {
        for (let phase_extension of phase_extensions) {
            // For prebuild phases, copy the steps before the build step
            if (phase_extension.name === 'prebuild') {
                const buildphase = assembledYaml.phases.find((el) => el.name === 'build');
                if (buildphase) {
                    // Insert the prebuild before the build steps.
                    buildphase.steps.splice(0, 0, ...phase_extension.steps);
                }
                else {
                    // In case the original componen yaml doesn't include a build phase,
                    // copy the phase from the extension directly in the assembled YAML
                    const copied_phase = { ...phase_extension };
                    copied_phase.name = 'build';
                    assembledYaml.phases.push(copied_phase);
                }
            }
            else { // Else just copy the build phase.
                const extphase = assembledYaml.phases.find((el) => el.name === phase_extension.name);
                if (extphase) {
                    extphase.steps.push(...phase_extension.steps);
                }
            }
        }
    }
    checkComponentYaml(baseDir, componentPath) {
        const metadatafile = path.join(baseDir, componentPath, 'metadata.yaml');
        const componentfile = path.join(baseDir, componentPath, 'component.yaml');
        const metadataExists = fs.existsSync(metadatafile);
        const componentYamlExists = fs.existsSync(componentfile);
        const yamlValidator = new Validator({});
        yamlValidator.validate([
            metadatafile,
            componentfile,
        ]);
        return metadataExists && componentYamlExists && yamlValidator.report() == 0;
    }
}
exports.ComponentBuilder = ComponentBuilder;
_a = JSII_RTTI_SYMBOL_1;
ComponentBuilder[_a] = { fqn: "halloumi-ami-pipelines.ComponentBuilder", version: "0.0.13" };
class PhaseExtension {
}
class Step {
}
class Inputs {
}
//# sourceMappingURL=data:application/json;base64,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