"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiPipelineLib = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const imagebuilder = require("@aws-cdk/aws-imagebuilder");
const kms = require("@aws-cdk/aws-kms");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const component_builder_1 = require("./component_builder");
const image_recipe_version_bump_1 = require("./image_recipe_version_bump");
const slack_notification_1 = require("./slack_notification");
const source_action_builder_1 = require("./source_action_builder");
const ssmupdate = require("./ssm_update");
const utils = require("./utils");
/**
 * Construct for creating a Codepipeline, EC2 Image builder pipeline from 1 pipeline configuration.
 *
 * @stability stable
 */
class AmiPipelineLib extends cdk.Construct {
    /**
     * Constructor.
     *
     * @stability stable
     */
    constructor(scope, id, pipelineConfig, componentDepsConfig, slackConfig) {
        super(scope, id);
        this.id = id;
        this.pipelineConfig = pipelineConfig;
        this.pipelineConfig.name = this.id;
        this.componentDepsConfig = componentDepsConfig;
        this.slackConfig = slackConfig;
        this.componentBuilder = new component_builder_1.ComponentBuilder(this, componentDepsConfig, this.id, this.pipelineConfig.parent_image);
        this.sourceActionBuilder = new source_action_builder_1.SourceActionBuilder(this, pipelineConfig.sources, this.id);
        this.createImagebuilderPipeline();
        this.createCodepipelineProject();
        this.createScheduledTask();
        if (this.topic) {
            new ssmupdate.SsmUpdateConstruct(this, 'SSMUpdate', this.topic, this.pipelineConfig);
            new slack_notification_1.SlackNotification(this, 'SlackNotification', this.topic, slackConfig, `${this.id}Recipe`);
        }
    }
    /**
     * @stability stable
     */
    createScheduledTask() {
        if (this.codepipeline && this.pipelineConfig.schedule) {
            const pipelineTarget = new event_targets.CodePipeline(this.codepipeline);
            new events.Rule(this, 'ScheduleRule', {
                schedule: events.Schedule.expression(this.pipelineConfig.schedule),
                targets: [pipelineTarget],
            });
        }
    }
    /**
     * @stability stable
     */
    createImagebuilderPipeline() {
        let compMap = this.componentBuilder.createComponentDependenciesMap();
        let image_id;
        let parentImage;
        if ('image_id' in this.pipelineConfig && this.pipelineConfig.image_id) {
            image_id = this.pipelineConfig.image_id;
        }
        else {
            switch (this.pipelineConfig.parent_image) {
                case 'AmazonLinux2':
                    parentImage = ec2.MachineImage.latestAmazonLinux({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        edition: ec2.AmazonLinuxEdition.STANDARD,
                        virtualization: ec2.AmazonLinuxVirt.HVM,
                        storage: ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
                        cpuType: ec2.AmazonLinuxCpuType.X86_64,
                    });
                    break;
                default:
                    parentImage = ec2.MachineImage.lookup(this.getLookupCriteria(this.pipelineConfig.parent_image));
                    break;
            }
        }
        // Use a custom block device mapping if encryption is required.
        let blockDeviceMappings = this.createBlockDeviceMapping();
        this.recipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
            name: `${this.id}Recipe`,
            parentImage: image_id ? image_id : parentImage ? parentImage.getImage(this).imageId : null,
            version: this.getNextRecipeVersion(`${this.id}Recipe`),
            components: this.pipelineConfig.recipe.components.map((c) => ({
                componentArn: compMap[c.name].ref,
            })),
            blockDeviceMappings: blockDeviceMappings,
        });
        const builderRole = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
            ],
            inlinePolicies: this.getInlinePolicies(this.pipelineConfig),
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [builderRole.roleName],
        });
        this.topic = new sns.Topic(this, 'AmiPipelineTopic');
        this.infrastructure = new imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: `${this.id} - EC2 IB Infrastructure`,
            instanceProfileName: instanceProfile.ref,
            snsTopicArn: this.topic.topicArn,
            instanceTypes: this.pipelineConfig.instance_type ? [this.pipelineConfig.instance_type] : undefined,
            terminateInstanceOnFailure: !('terminate_on_failure' in this.pipelineConfig) || this.pipelineConfig.terminate_on_failure,
            subnetId: this.pipelineConfig.subnet_id,
            securityGroupIds: this.pipelineConfig.security_group_ids ? this.pipelineConfig.security_group_ids.split(',') : undefined,
        });
        this.distributionConfig = this.pipelineConfig.shared_with ? new imagebuilder.CfnDistributionConfiguration(this, 'DistributionConfig', {
            name: `${this.id} - Distribution`,
            distributions: this.pipelineConfig.shared_with.map((s) => ({
                region: s.region,
                amiDistributionConfiguration: {
                    Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                    Description: `${this.id} Shared image`,
                    AmiTags: {
                        Name: this.id,
                    },
                    TargetAccountIds: s.account_ids,
                },
            })),
        }) : undefined;
        this.imagePipeline = new imagebuilder.CfnImagePipeline(this, 'AmiPipeline', {
            name: `${this.id} - AMI Pipeline`,
            imageRecipeArn: this.recipe.ref,
            infrastructureConfigurationArn: this.infrastructure.ref,
            distributionConfigurationArn: this.distributionConfig ? this.distributionConfig.ref : undefined,
        });
    }
    getInlinePolicies(pipelineConfig) {
        let result = {};
        const statements = pipelineConfig.policy_statements;
        if (statements) {
            for (let statement in statements) {
                result[statement] = new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            sid: statement,
                            actions: statements[statement].actions,
                            effect: iam.Effect.ALLOW,
                            resources: statements[statement].resources,
                        }),
                    ],
                });
            }
        }
        return result;
    }
    createBlockDeviceMapping() {
        const key = this.pipelineConfig.encrypted ? new kms.Key(this, 'PipelineEncryption', {
            description: `Image encryption for ${this.pipelineConfig.name}`,
            alias: `${utils.pascalize(this.pipelineConfig.name)}_key`,
        }) : null;
        let blockDeviceMappings = key ? [
            {
                deviceName: '/dev/xvda',
                ebs: {
                    encrypted: true,
                    volumeSize: 8,
                    volumeType: 'gp2',
                    kmsKeyId: key.keyArn,
                },
            },
        ] : undefined;
        return blockDeviceMappings;
    }
    /**
     * @stability stable
     */
    getNextRecipeVersion(recipeName) {
        const recipeBumpCustomResource = new image_recipe_version_bump_1.ImageRecipeVersionBump(this, 'ImageRecipeVersionBump', recipeName);
        return recipeBumpCustomResource.nextVersion;
    }
    /**
     * @stability stable
     */
    createCodepipelineProject() {
        var _b;
        if (this.imagePipeline) {
            // Create CodePipeline project
            let imagebuilderBuild = new codebuild.PipelineProject(this, 'AmiPipelineBuild', {
                buildSpec: codebuild.BuildSpec.fromObject({
                    version: '0.2',
                    phases: {
                        install: {
                            commands: [
                                'apt-get update',
                                'apt-get install awscli -y',
                            ],
                        },
                        pre_build: {
                            commands: [
                                'curl -X POST -H \'Content-type: application/json\' --data "{\\"text\\":\\"$PIPELINE_NAME build has started.\\"}" $SLACK_WEBHOOK_URL',
                            ],
                        },
                        build: {
                            commands: 'aws imagebuilder start-image-pipeline-execution --image-pipeline-arn $IMAGE_PIPELINE_ARN',
                        },
                    },
                }),
                environment: {
                    buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                    environmentVariables: {
                        IMAGE_PIPELINE_ARN: { value: this.imagePipeline.ref },
                        SLACK_WEBHOOK_URL: { value: this.slackConfig.slackWebhookUrl },
                        SLACK_CHANNEL: { value: this.slackConfig.channel },
                        SLACK_USERNAME: { value: this.slackConfig.username },
                        PIPELINE_NAME: { value: this.imagePipeline.name },
                    },
                },
            });
            imagebuilderBuild.addToRolePolicy(iam.PolicyStatement.fromJson({
                Sid: 'imagebuilderstart',
                Effect: 'Allow',
                Action: ['imagebuilder:StartImagePipelineExecution'],
                Resource: '*',
            }));
            const imageBuilderOutput = new codepipeline.Artifact('CdkBuildOutput');
            let sourceActions = this.sourceActionBuilder.createPipelineSources();
            const buildAction = new codepipeline_actions.CodeBuildAction({
                actionName: 'StartImageBuilder',
                project: imagebuilderBuild,
                input: (_b = sourceActions.find(a => a && a.sourceOutput)) === null || _b === void 0 ? void 0 : _b.sourceOutput,
                extraInputs: sourceActions.length > 1 ? sourceActions.slice(1).map(a => a.sourceOutput) : [],
                outputs: [imageBuilderOutput],
            });
            // Complete Pipeline Project
            this.codepipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                restartExecutionOnUpdate: true,
                stages: [
                    {
                        stageName: 'Source',
                        actions: sourceActions.map(c => c.action).filter((c) => c != null),
                    },
                    {
                        stageName: 'Build',
                        actions: [buildAction],
                    },
                ],
            });
        }
    }
    /**
     * @stability stable
     */
    getLookupCriteria(parentImage) {
        switch (parentImage) {
            case 'Ubuntu1804':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'Ubuntu2004':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'CentOS7':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['cvugziknvmxgqna9noibqnnsy'],
                    },
                };
            case 'CentOS8':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['47k9ia2igxpcce2bzo8u3kj03'],
                    },
                };
            default:
                return {
                    name: 'amzn-linux',
                };
        }
    }
}
exports.AmiPipelineLib = AmiPipelineLib;
_a = JSII_RTTI_SYMBOL_1;
AmiPipelineLib[_a] = { fqn: "halloumi-ami-pipelines.AmiPipelineLib", version: "0.0.13" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1pX3BpcGVsaW5lX2xpYi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9hbWlfcGlwZWxpbmVfbGliLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsb0RBQW9EO0FBQ3BELDBEQUEwRDtBQUMxRCwwRUFBMEU7QUFDMUUsd0NBQXdDO0FBQ3hDLDhDQUE4QztBQUM5Qyw2REFBNkQ7QUFDN0Qsd0NBQXdDO0FBQ3hDLDBEQUEwRDtBQUMxRCx3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLHFDQUFxQztBQUdyQywyREFBdUQ7QUFDdkQsMkVBQXFFO0FBQ3JFLDZEQUE2RTtBQUM3RSxtRUFBOEQ7QUFDOUQsMENBQTBDO0FBQzFDLGlDQUFpQzs7Ozs7O0FBS2pDLE1BQWEsY0FBZSxTQUFRLEdBQUcsQ0FBQyxTQUFTOzs7Ozs7SUFxQi9DLFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsY0FBbUIsRUFBRSxtQkFBMEIsRUFBRSxXQUErQjtRQUM1SCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxFQUFFLEdBQUcsRUFBRSxDQUFDO1FBQ2IsSUFBSSxDQUFDLGNBQWMsR0FBRyxjQUFjLENBQUM7UUFDckMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLEVBQUUsQ0FBQztRQUNuQyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsbUJBQW1CLENBQUM7UUFDL0MsSUFBSSxDQUFDLFdBQVcsR0FBRyxXQUFXLENBQUM7UUFFL0IsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksb0NBQWdCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFLElBQUksQ0FBQyxFQUFFLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUNuSCxJQUFJLENBQUMsbUJBQW1CLEdBQUcsSUFBSSwyQ0FBbUIsQ0FBQyxJQUFJLEVBQUUsY0FBYyxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFMUYsSUFBSSxDQUFDLDBCQUEwQixFQUFFLENBQUM7UUFDbEMsSUFBSSxDQUFDLHlCQUF5QixFQUFFLENBQUM7UUFDakMsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7UUFFM0IsSUFBSSxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ2QsSUFBSSxTQUFTLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRSxJQUFJLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUNyRixJQUFJLHNDQUFpQixDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRSxJQUFJLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRSxHQUFHLElBQUksQ0FBQyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1NBRS9GO0lBQ0gsQ0FBQzs7OztJQUVELG1CQUFtQjtRQUNqQixJQUFJLElBQUksQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUU7WUFDckQsTUFBTSxjQUFjLEdBQUcsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUN6RSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtnQkFDcEMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDO2dCQUNsRSxPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7YUFDMUIsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDOzs7O0lBRUQsMEJBQTBCO1FBQ3hCLElBQUksT0FBTyxHQUF1QixJQUFJLENBQUMsZ0JBQWdCLENBQUMsOEJBQThCLEVBQUUsQ0FBQztRQUV6RixJQUFJLFFBQVEsQ0FBQztRQUNiLElBQUksV0FBVyxDQUFDO1FBQ2hCLElBQUksVUFBVSxJQUFJLElBQUksQ0FBQyxjQUFjLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUU7WUFDckUsUUFBUSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDO1NBQ3pDO2FBQU07WUFDTCxRQUFRLElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxFQUFFO2dCQUN4QyxLQUFLLGNBQWM7b0JBQ2pCLFdBQVcsR0FBRyxHQUFHLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDO3dCQUMvQyxVQUFVLEVBQUUsR0FBRyxDQUFDLHFCQUFxQixDQUFDLGNBQWM7d0JBQ3BELE9BQU8sRUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsUUFBUTt3QkFDeEMsY0FBYyxFQUFFLEdBQUcsQ0FBQyxlQUFlLENBQUMsR0FBRzt3QkFDdkMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxlQUFlO3dCQUMvQyxPQUFPLEVBQUUsR0FBRyxDQUFDLGtCQUFrQixDQUFDLE1BQU07cUJBQ3ZDLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUNSO29CQUNFLFdBQVcsR0FBRyxHQUFHLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDO29CQUNoRyxNQUFNO2FBQ1Q7U0FDRjtRQUVELCtEQUErRDtRQUMvRCxJQUFJLG1CQUFtQixHQUFHLElBQUksQ0FBQyx3QkFBd0IsRUFBRSxDQUFDO1FBRTFELElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxZQUFZLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDakUsSUFBSSxFQUFFLEdBQUcsSUFBSSxDQUFDLEVBQUUsUUFBUTtZQUN4QixXQUFXLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUEsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUEsQ0FBQyxDQUFDLElBQUk7WUFDeEYsT0FBTyxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxHQUFHLElBQUksQ0FBQyxFQUFFLFFBQVEsQ0FBQztZQUN0RCxVQUFVLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztnQkFDakUsWUFBWSxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRzthQUNsQyxDQUFDLENBQUM7WUFDSCxtQkFBbUIsRUFBRSxtQkFBbUI7U0FDekMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDN0MsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDO1lBQ3hELGVBQWUsRUFBRTtnQkFDZixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDhCQUE4QixDQUFDO2dCQUMxRSxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLHFEQUFxRCxDQUFDO2dCQUNqRyxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLG1DQUFtQyxDQUFDO2FBQ2hGO1lBQ0QsY0FBYyxFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDO1NBQzVELENBQUMsQ0FBQztRQUVILE1BQU0sZUFBZSxHQUFHLElBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUMxRSxLQUFLLEVBQUUsQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDO1NBQzlCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO1FBQ3JELElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxZQUFZLENBQUMsOEJBQThCLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO1lBQzVGLElBQUksRUFBRSxHQUFHLElBQUksQ0FBQyxFQUFFLDBCQUEwQjtZQUMxQyxtQkFBbUIsRUFBRSxlQUFlLENBQUMsR0FBRztZQUN4QyxXQUFXLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRO1lBQ2hDLGFBQWEsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2xHLDBCQUEwQixFQUFFLENBQUMsQ0FBQyxzQkFBc0IsSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxvQkFBb0I7WUFDeEgsUUFBUSxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsU0FBUztZQUN2QyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztTQUN6SCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLElBQUksWUFBWSxDQUFDLDRCQUE0QixDQUFDLElBQUksRUFBRSxvQkFBb0IsRUFBRTtZQUNwSSxJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUMsRUFBRSxpQkFBaUI7WUFDakMsYUFBYSxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUssRUFBRSxFQUFFLENBQUMsQ0FDNUQ7Z0JBQ0UsTUFBTSxFQUFFLENBQUMsQ0FBQyxNQUFNO2dCQUNoQiw0QkFBNEIsRUFBRTtvQkFDNUIsSUFBSSxFQUFFLEdBQUcsSUFBSSxDQUFDLEVBQUUsaUNBQWlDO29CQUNqRCxXQUFXLEVBQUUsR0FBRyxJQUFJLENBQUMsRUFBRSxlQUFlO29CQUN0QyxPQUFPLEVBQUU7d0JBQ1AsSUFBSSxFQUFFLElBQUksQ0FBQyxFQUFFO3FCQUNkO29CQUNELGdCQUFnQixFQUFFLENBQUMsQ0FBQyxXQUFXO2lCQUNoQzthQUNGLENBQUMsQ0FBQztTQUNOLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBRWYsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQzFFLElBQUksRUFBRSxHQUFHLElBQUksQ0FBQyxFQUFFLGlCQUFpQjtZQUNqQyxjQUFjLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHO1lBQy9CLDhCQUE4QixFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRztZQUN2RCw0QkFBNEIsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDaEcsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGlCQUFpQixDQUFDLGNBQW1CO1FBQzNDLElBQUksTUFBTSxHQUFTLEVBQUUsQ0FBQztRQUN0QixNQUFNLFVBQVUsR0FBRyxjQUFjLENBQUMsaUJBQWlCLENBQUM7UUFDcEQsSUFBSSxVQUFVLEVBQUU7WUFDZCxLQUFLLElBQUksU0FBUyxJQUFJLFVBQVUsRUFBRTtnQkFDaEMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxHQUFHLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQztvQkFDekMsVUFBVSxFQUFFO3dCQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzs0QkFDdEIsR0FBRyxFQUFFLFNBQVM7NEJBQ2QsT0FBTyxFQUFFLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPOzRCQUN0QyxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLOzRCQUN4QixTQUFTLEVBQUUsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFNBQVM7eUJBQzNDLENBQUM7cUJBQ0g7aUJBQ0YsQ0FBQyxDQUFDO2FBQ0o7U0FDRjtRQUVELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFFTyx3QkFBd0I7UUFDOUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEVBQUU7WUFDbEYsV0FBVyxFQUFFLHdCQUF3QixJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRTtZQUMvRCxLQUFLLEVBQUUsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLE1BQU07U0FDMUQsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7UUFDVixJQUFJLG1CQUFtQixHQUFHLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDOUI7Z0JBQ0UsVUFBVSxFQUFFLFdBQVc7Z0JBQ3ZCLEdBQUcsRUFBRTtvQkFDSCxTQUFTLEVBQUUsSUFBSTtvQkFDZixVQUFVLEVBQUUsQ0FBQztvQkFDYixVQUFVLEVBQUUsS0FBSztvQkFDakIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxNQUFNO2lCQUNyQjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ2QsT0FBTyxtQkFBbUIsQ0FBQztJQUM3QixDQUFDOzs7O0lBRUQsb0JBQW9CLENBQUMsVUFBa0I7UUFDckMsTUFBTSx3QkFBd0IsR0FBRyxJQUFJLGtEQUFzQixDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUN4RyxPQUFPLHdCQUF3QixDQUFDLFdBQVcsQ0FBQztJQUM5QyxDQUFDOzs7O0lBRUQseUJBQXlCOztRQUN2QixJQUFJLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdEIsOEJBQThCO1lBQzlCLElBQUksaUJBQWlCLEdBQUcsSUFBSSxTQUFTLENBQUMsZUFBZSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtnQkFDOUUsU0FBUyxFQUFFLFNBQVMsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDO29CQUN4QyxPQUFPLEVBQUUsS0FBSztvQkFDZCxNQUFNLEVBQUU7d0JBQ04sT0FBTyxFQUFFOzRCQUNQLFFBQVEsRUFBRTtnQ0FDUixnQkFBZ0I7Z0NBQ2hCLDJCQUEyQjs2QkFDNUI7eUJBQ0Y7d0JBQ0QsU0FBUyxFQUFFOzRCQUNULFFBQVEsRUFBRTtnQ0FDUixxSUFBcUk7NkJBQ3RJO3lCQUNGO3dCQUNELEtBQUssRUFBRTs0QkFDTCxRQUFRLEVBQUUsMEZBQTBGO3lCQUNyRztxQkFDRjtpQkFDRixDQUFDO2dCQUNGLFdBQVcsRUFBRTtvQkFDWCxVQUFVLEVBQUUsU0FBUyxDQUFDLGVBQWUsQ0FBQyxZQUFZO29CQUNsRCxvQkFBb0IsRUFBRTt3QkFDcEIsa0JBQWtCLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUU7d0JBQ3JELGlCQUFpQixFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsZUFBZSxFQUFFO3dCQUM5RCxhQUFhLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLEVBQUU7d0JBQ2xELGNBQWMsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsRUFBRTt3QkFDcEQsYUFBYSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFO3FCQUNsRDtpQkFDRjthQUNGLENBQUMsQ0FBQztZQUVILGlCQUFpQixDQUFDLGVBQWUsQ0FDL0IsR0FBRyxDQUFDLGVBQWUsQ0FBQyxRQUFRLENBQUM7Z0JBQzNCLEdBQUcsRUFBRSxtQkFBbUI7Z0JBQ3hCLE1BQU0sRUFBRSxPQUFPO2dCQUNmLE1BQU0sRUFBRSxDQUFDLDBDQUEwQyxDQUFDO2dCQUNwRCxRQUFRLEVBQUUsR0FBRzthQUNkLENBQUMsQ0FBQyxDQUFDO1lBRU4sTUFBTSxrQkFBa0IsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztZQUV2RSxJQUFJLGFBQWEsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMscUJBQXFCLEVBQUUsQ0FBQztZQUNyRSxNQUFNLFdBQVcsR0FBRyxJQUFJLG9CQUFvQixDQUFDLGVBQWUsQ0FBQztnQkFDM0QsVUFBVSxFQUFFLG1CQUFtQjtnQkFDL0IsT0FBTyxFQUFFLGlCQUFpQjtnQkFDMUIsS0FBSyxFQUFFLE1BQUEsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsWUFBWSxDQUFDLDBDQUFFLFlBQWE7Z0JBQ2xFLFdBQVcsRUFBRSxhQUFhLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQzVGLE9BQU8sRUFBRSxDQUFDLGtCQUFrQixDQUFDO2FBQzlCLENBQUMsQ0FBQztZQUVILDRCQUE0QjtZQUM1QixJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO2dCQUM5RCx3QkFBd0IsRUFBRSxJQUFJO2dCQUM5QixNQUFNLEVBQUU7b0JBQ047d0JBQ0UsU0FBUyxFQUFFLFFBQVE7d0JBQ25CLE9BQU8sRUFBRSxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBNkIsRUFBRSxDQUFDLENBQUMsSUFBSSxJQUFJLENBQUM7cUJBQzlGO29CQUNEO3dCQUNFLFNBQVMsRUFBRSxPQUFPO3dCQUNsQixPQUFPLEVBQUUsQ0FBQyxXQUFXLENBQUM7cUJBQ3ZCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDOzs7O0lBRUQsaUJBQWlCLENBQUMsV0FBZ0I7UUFDaEMsUUFBUSxXQUFXLEVBQUU7WUFDbkIsS0FBSyxZQUFZO2dCQUNmLE9BQU87b0JBQ0wsSUFBSSxFQUFFLGtEQUFrRDtvQkFDeEQsTUFBTSxFQUFFLENBQUMsY0FBYyxDQUFDO2lCQUN6QixDQUFDO1lBQ0osS0FBSyxZQUFZO2dCQUNmLE9BQU87b0JBQ0wsSUFBSSxFQUFFLGlEQUFpRDtvQkFDdkQsTUFBTSxFQUFFLENBQUMsY0FBYyxDQUFDO2lCQUN6QixDQUFDO1lBQ0osS0FBSyxTQUFTO2dCQUNaLE9BQU87b0JBQ0wsSUFBSSxFQUFFLEdBQUc7b0JBQ1QsTUFBTSxFQUFFLENBQUMsaUJBQWlCLENBQUM7b0JBQzNCLE9BQU8sRUFBRTt3QkFDUCxjQUFjLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQztxQkFDOUM7aUJBQ0YsQ0FBQztZQUNKLEtBQUssU0FBUztnQkFDWixPQUFPO29CQUNMLElBQUksRUFBRSxHQUFHO29CQUNULE1BQU0sRUFBRSxDQUFDLGlCQUFpQixDQUFDO29CQUMzQixPQUFPLEVBQUU7d0JBQ1AsY0FBYyxFQUFFLENBQUMsMkJBQTJCLENBQUM7cUJBQzlDO2lCQUNGLENBQUM7WUFDSjtnQkFDRSxPQUFPO29CQUNMLElBQUksRUFBRSxZQUFZO2lCQUNuQixDQUFDO1NBQ0w7SUFDSCxDQUFDOztBQWpTSCx3Q0FtU0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjb2RlYnVpbGQgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVidWlsZCc7XG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZSc7XG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmVfYWN0aW9ucyBmcm9tICdAYXdzLWNkay9hd3MtY29kZXBpcGVsaW5lLWFjdGlvbnMnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMnO1xuaW1wb3J0ICogYXMgZXZlbnRfdGFyZ2V0cyBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzLXRhcmdldHMnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgaW1hZ2VidWlsZGVyIGZyb20gJ0Bhd3MtY2RrL2F3cy1pbWFnZWJ1aWxkZXInO1xuaW1wb3J0ICogYXMga21zIGZyb20gJ0Bhd3MtY2RrL2F3cy1rbXMnO1xuaW1wb3J0ICogYXMgc25zIGZyb20gJ0Bhd3MtY2RrL2F3cy1zbnMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5pbXBvcnQgeyBTdHJpbmdDb21wb25lbnRNYXAgfSBmcm9tICcuJztcbmltcG9ydCB7IENvbXBvbmVudEJ1aWxkZXIgfSBmcm9tICcuL2NvbXBvbmVudF9idWlsZGVyJztcbmltcG9ydCB7IEltYWdlUmVjaXBlVmVyc2lvbkJ1bXAgfSBmcm9tICcuL2ltYWdlX3JlY2lwZV92ZXJzaW9uX2J1bXAnO1xuaW1wb3J0IHsgU2xhY2tDb25maWd1cmF0aW9uLCBTbGFja05vdGlmaWNhdGlvbiB9IGZyb20gJy4vc2xhY2tfbm90aWZpY2F0aW9uJztcbmltcG9ydCB7IFNvdXJjZUFjdGlvbkJ1aWxkZXIgfSBmcm9tICcuL3NvdXJjZV9hY3Rpb25fYnVpbGRlcic7XG5pbXBvcnQgKiBhcyBzc211cGRhdGUgZnJvbSAnLi9zc21fdXBkYXRlJztcbmltcG9ydCAqIGFzIHV0aWxzIGZyb20gJy4vdXRpbHMnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEFtaVBpcGVsaW5lTGliIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIHBpcGVsaW5lQ29uZmlnOiBhbnlcbiAgY29tcG9uZW50RGVwc0NvbmZpZzogYW55W11cbiAgaWQ6IHN0cmluZ1xuICBzb3VyY2VBY3Rpb25CdWlsZGVyOiBTb3VyY2VBY3Rpb25CdWlsZGVyO1xuICBjb21wb25lbnRCdWlsZGVyOiBDb21wb25lbnRCdWlsZGVyO1xuICBpbWFnZVBpcGVsaW5lOiBpbWFnZWJ1aWxkZXIuQ2ZuSW1hZ2VQaXBlbGluZSB8IHVuZGVmaW5lZDtcbiAgdG9waWM6IHNucy5Ub3BpYyB8IHVuZGVmaW5lZDtcbiAgY29kZXBpcGVsaW5lOiBjb2RlcGlwZWxpbmUuUGlwZWxpbmUgfCB1bmRlZmluZWQ7XG4gIHNsYWNrQ29uZmlnOiBTbGFja0NvbmZpZ3VyYXRpb247XG4gIHJlY2lwZTogaW1hZ2VidWlsZGVyLkNmbkltYWdlUmVjaXBlIHwgdW5kZWZpbmVkO1xuICBpbmZyYXN0cnVjdHVyZTogaW1hZ2VidWlsZGVyLkNmbkluZnJhc3RydWN0dXJlQ29uZmlndXJhdGlvbiB8IHVuZGVmaW5lZDtcbiAgZGlzdHJpYnV0aW9uQ29uZmlnOiBpbWFnZWJ1aWxkZXIuQ2ZuRGlzdHJpYnV0aW9uQ29uZmlndXJhdGlvbiB8IHVuZGVmaW5lZDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwaXBlbGluZUNvbmZpZzogYW55LCBjb21wb25lbnREZXBzQ29uZmlnOiBhbnlbXSwgc2xhY2tDb25maWc6IFNsYWNrQ29uZmlndXJhdGlvbikge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmlkID0gaWQ7XG4gICAgdGhpcy5waXBlbGluZUNvbmZpZyA9IHBpcGVsaW5lQ29uZmlnO1xuICAgIHRoaXMucGlwZWxpbmVDb25maWcubmFtZSA9IHRoaXMuaWQ7XG4gICAgdGhpcy5jb21wb25lbnREZXBzQ29uZmlnID0gY29tcG9uZW50RGVwc0NvbmZpZztcbiAgICB0aGlzLnNsYWNrQ29uZmlnID0gc2xhY2tDb25maWc7XG5cbiAgICB0aGlzLmNvbXBvbmVudEJ1aWxkZXIgPSBuZXcgQ29tcG9uZW50QnVpbGRlcih0aGlzLCBjb21wb25lbnREZXBzQ29uZmlnLCB0aGlzLmlkLCB0aGlzLnBpcGVsaW5lQ29uZmlnLnBhcmVudF9pbWFnZSk7XG4gICAgdGhpcy5zb3VyY2VBY3Rpb25CdWlsZGVyID0gbmV3IFNvdXJjZUFjdGlvbkJ1aWxkZXIodGhpcywgcGlwZWxpbmVDb25maWcuc291cmNlcywgdGhpcy5pZCk7XG5cbiAgICB0aGlzLmNyZWF0ZUltYWdlYnVpbGRlclBpcGVsaW5lKCk7XG4gICAgdGhpcy5jcmVhdGVDb2RlcGlwZWxpbmVQcm9qZWN0KCk7XG4gICAgdGhpcy5jcmVhdGVTY2hlZHVsZWRUYXNrKCk7XG5cbiAgICBpZiAodGhpcy50b3BpYykge1xuICAgICAgbmV3IHNzbXVwZGF0ZS5Tc21VcGRhdGVDb25zdHJ1Y3QodGhpcywgJ1NTTVVwZGF0ZScsIHRoaXMudG9waWMsIHRoaXMucGlwZWxpbmVDb25maWcpO1xuICAgICAgbmV3IFNsYWNrTm90aWZpY2F0aW9uKHRoaXMsICdTbGFja05vdGlmaWNhdGlvbicsIHRoaXMudG9waWMsIHNsYWNrQ29uZmlnLCBgJHt0aGlzLmlkfVJlY2lwZWApO1xuXG4gICAgfVxuICB9XG5cbiAgY3JlYXRlU2NoZWR1bGVkVGFzaygpIHtcbiAgICBpZiAodGhpcy5jb2RlcGlwZWxpbmUgJiYgdGhpcy5waXBlbGluZUNvbmZpZy5zY2hlZHVsZSkge1xuICAgICAgY29uc3QgcGlwZWxpbmVUYXJnZXQgPSBuZXcgZXZlbnRfdGFyZ2V0cy5Db2RlUGlwZWxpbmUodGhpcy5jb2RlcGlwZWxpbmUpO1xuICAgICAgbmV3IGV2ZW50cy5SdWxlKHRoaXMsICdTY2hlZHVsZVJ1bGUnLCB7XG4gICAgICAgIHNjaGVkdWxlOiBldmVudHMuU2NoZWR1bGUuZXhwcmVzc2lvbih0aGlzLnBpcGVsaW5lQ29uZmlnLnNjaGVkdWxlKSxcbiAgICAgICAgdGFyZ2V0czogW3BpcGVsaW5lVGFyZ2V0XSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIGNyZWF0ZUltYWdlYnVpbGRlclBpcGVsaW5lKCkge1xuICAgIGxldCBjb21wTWFwOiBTdHJpbmdDb21wb25lbnRNYXAgPSB0aGlzLmNvbXBvbmVudEJ1aWxkZXIuY3JlYXRlQ29tcG9uZW50RGVwZW5kZW5jaWVzTWFwKCk7XG5cbiAgICBsZXQgaW1hZ2VfaWQ7XG4gICAgbGV0IHBhcmVudEltYWdlO1xuICAgIGlmICgnaW1hZ2VfaWQnIGluIHRoaXMucGlwZWxpbmVDb25maWcgJiYgdGhpcy5waXBlbGluZUNvbmZpZy5pbWFnZV9pZCkge1xuICAgICAgaW1hZ2VfaWQgPSB0aGlzLnBpcGVsaW5lQ29uZmlnLmltYWdlX2lkO1xuICAgIH0gZWxzZSB7XG4gICAgICBzd2l0Y2ggKHRoaXMucGlwZWxpbmVDb25maWcucGFyZW50X2ltYWdlKSB7XG4gICAgICAgIGNhc2UgJ0FtYXpvbkxpbnV4Mic6XG4gICAgICAgICAgcGFyZW50SW1hZ2UgPSBlYzIuTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHtcbiAgICAgICAgICAgIGdlbmVyYXRpb246IGVjMi5BbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIsXG4gICAgICAgICAgICBlZGl0aW9uOiBlYzIuQW1hem9uTGludXhFZGl0aW9uLlNUQU5EQVJELFxuICAgICAgICAgICAgdmlydHVhbGl6YXRpb246IGVjMi5BbWF6b25MaW51eFZpcnQuSFZNLFxuICAgICAgICAgICAgc3RvcmFnZTogZWMyLkFtYXpvbkxpbnV4U3RvcmFnZS5HRU5FUkFMX1BVUlBPU0UsXG4gICAgICAgICAgICBjcHVUeXBlOiBlYzIuQW1hem9uTGludXhDcHVUeXBlLlg4Nl82NCxcbiAgICAgICAgICB9KTtcbiAgICAgICAgICBicmVhaztcbiAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICBwYXJlbnRJbWFnZSA9IGVjMi5NYWNoaW5lSW1hZ2UubG9va3VwKHRoaXMuZ2V0TG9va3VwQ3JpdGVyaWEodGhpcy5waXBlbGluZUNvbmZpZy5wYXJlbnRfaW1hZ2UpKTtcbiAgICAgICAgICBicmVhaztcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBVc2UgYSBjdXN0b20gYmxvY2sgZGV2aWNlIG1hcHBpbmcgaWYgZW5jcnlwdGlvbiBpcyByZXF1aXJlZC5cbiAgICBsZXQgYmxvY2tEZXZpY2VNYXBwaW5ncyA9IHRoaXMuY3JlYXRlQmxvY2tEZXZpY2VNYXBwaW5nKCk7XG5cbiAgICB0aGlzLnJlY2lwZSA9IG5ldyBpbWFnZWJ1aWxkZXIuQ2ZuSW1hZ2VSZWNpcGUodGhpcywgJ0ltYWdlUmVjaXBlJywge1xuICAgICAgbmFtZTogYCR7dGhpcy5pZH1SZWNpcGVgLFxuICAgICAgcGFyZW50SW1hZ2U6IGltYWdlX2lkID8gaW1hZ2VfaWQ6IHBhcmVudEltYWdlID8gcGFyZW50SW1hZ2UuZ2V0SW1hZ2UodGhpcykuaW1hZ2VJZDogbnVsbCxcbiAgICAgIHZlcnNpb246IHRoaXMuZ2V0TmV4dFJlY2lwZVZlcnNpb24oYCR7dGhpcy5pZH1SZWNpcGVgKSxcbiAgICAgIGNvbXBvbmVudHM6IHRoaXMucGlwZWxpbmVDb25maWcucmVjaXBlLmNvbXBvbmVudHMubWFwKChjOiBhbnkpID0+ICh7XG4gICAgICAgIGNvbXBvbmVudEFybjogY29tcE1hcFtjLm5hbWVdLnJlZixcbiAgICAgIH0pKSxcbiAgICAgIGJsb2NrRGV2aWNlTWFwcGluZ3M6IGJsb2NrRGV2aWNlTWFwcGluZ3MsXG4gICAgfSk7XG5cbiAgICBjb25zdCBidWlsZGVyUm9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdlYzIuYW1hem9uYXdzLmNvbScpLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbXG4gICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQW1hem9uU1NNTWFuYWdlZEluc3RhbmNlQ29yZScpLFxuICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0VDMkluc3RhbmNlUHJvZmlsZUZvckltYWdlQnVpbGRlckVDUkNvbnRhaW5lckJ1aWxkcycpLFxuICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0VDMkluc3RhbmNlUHJvZmlsZUZvckltYWdlQnVpbGRlcicpLFxuICAgICAgXSxcbiAgICAgIGlubGluZVBvbGljaWVzOiB0aGlzLmdldElubGluZVBvbGljaWVzKHRoaXMucGlwZWxpbmVDb25maWcpLFxuICAgIH0pO1xuXG4gICAgY29uc3QgaW5zdGFuY2VQcm9maWxlID0gbmV3IGlhbS5DZm5JbnN0YW5jZVByb2ZpbGUodGhpcywgJ0luc3RhbmNlUHJvZmlsZScsIHtcbiAgICAgIHJvbGVzOiBbYnVpbGRlclJvbGUucm9sZU5hbWVdLFxuICAgIH0pO1xuXG4gICAgdGhpcy50b3BpYyA9IG5ldyBzbnMuVG9waWModGhpcywgJ0FtaVBpcGVsaW5lVG9waWMnKTtcbiAgICB0aGlzLmluZnJhc3RydWN0dXJlID0gbmV3IGltYWdlYnVpbGRlci5DZm5JbmZyYXN0cnVjdHVyZUNvbmZpZ3VyYXRpb24odGhpcywgJ0luZnJhc3RydWN0dXJlJywge1xuICAgICAgbmFtZTogYCR7dGhpcy5pZH0gLSBFQzIgSUIgSW5mcmFzdHJ1Y3R1cmVgLFxuICAgICAgaW5zdGFuY2VQcm9maWxlTmFtZTogaW5zdGFuY2VQcm9maWxlLnJlZixcbiAgICAgIHNuc1RvcGljQXJuOiB0aGlzLnRvcGljLnRvcGljQXJuLFxuICAgICAgaW5zdGFuY2VUeXBlczogdGhpcy5waXBlbGluZUNvbmZpZy5pbnN0YW5jZV90eXBlID8gW3RoaXMucGlwZWxpbmVDb25maWcuaW5zdGFuY2VfdHlwZV0gOiB1bmRlZmluZWQsXG4gICAgICB0ZXJtaW5hdGVJbnN0YW5jZU9uRmFpbHVyZTogISgndGVybWluYXRlX29uX2ZhaWx1cmUnIGluIHRoaXMucGlwZWxpbmVDb25maWcpIHx8IHRoaXMucGlwZWxpbmVDb25maWcudGVybWluYXRlX29uX2ZhaWx1cmUsXG4gICAgICBzdWJuZXRJZDogdGhpcy5waXBlbGluZUNvbmZpZy5zdWJuZXRfaWQsXG4gICAgICBzZWN1cml0eUdyb3VwSWRzOiB0aGlzLnBpcGVsaW5lQ29uZmlnLnNlY3VyaXR5X2dyb3VwX2lkcyA/IHRoaXMucGlwZWxpbmVDb25maWcuc2VjdXJpdHlfZ3JvdXBfaWRzLnNwbGl0KCcsJykgOiB1bmRlZmluZWQsXG4gICAgfSk7XG5cbiAgICB0aGlzLmRpc3RyaWJ1dGlvbkNvbmZpZyA9IHRoaXMucGlwZWxpbmVDb25maWcuc2hhcmVkX3dpdGggPyBuZXcgaW1hZ2VidWlsZGVyLkNmbkRpc3RyaWJ1dGlvbkNvbmZpZ3VyYXRpb24odGhpcywgJ0Rpc3RyaWJ1dGlvbkNvbmZpZycsIHtcbiAgICAgIG5hbWU6IGAke3RoaXMuaWR9IC0gRGlzdHJpYnV0aW9uYCxcbiAgICAgIGRpc3RyaWJ1dGlvbnM6IHRoaXMucGlwZWxpbmVDb25maWcuc2hhcmVkX3dpdGgubWFwKChzOmFueSkgPT4gKFxuICAgICAgICB7XG4gICAgICAgICAgcmVnaW9uOiBzLnJlZ2lvbixcbiAgICAgICAgICBhbWlEaXN0cmlidXRpb25Db25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICBOYW1lOiBgJHt0aGlzLmlkfSAtIHt7IGltYWdlYnVpbGRlcjpidWlsZERhdGUgfX1gLFxuICAgICAgICAgICAgRGVzY3JpcHRpb246IGAke3RoaXMuaWR9IFNoYXJlZCBpbWFnZWAsXG4gICAgICAgICAgICBBbWlUYWdzOiB7XG4gICAgICAgICAgICAgIE5hbWU6IHRoaXMuaWQsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgVGFyZ2V0QWNjb3VudElkczogcy5hY2NvdW50X2lkcyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KSksXG4gICAgfSkgOiB1bmRlZmluZWQ7XG5cbiAgICB0aGlzLmltYWdlUGlwZWxpbmUgPSBuZXcgaW1hZ2VidWlsZGVyLkNmbkltYWdlUGlwZWxpbmUodGhpcywgJ0FtaVBpcGVsaW5lJywge1xuICAgICAgbmFtZTogYCR7dGhpcy5pZH0gLSBBTUkgUGlwZWxpbmVgLFxuICAgICAgaW1hZ2VSZWNpcGVBcm46IHRoaXMucmVjaXBlLnJlZixcbiAgICAgIGluZnJhc3RydWN0dXJlQ29uZmlndXJhdGlvbkFybjogdGhpcy5pbmZyYXN0cnVjdHVyZS5yZWYsXG4gICAgICBkaXN0cmlidXRpb25Db25maWd1cmF0aW9uQXJuOiB0aGlzLmRpc3RyaWJ1dGlvbkNvbmZpZyA/IHRoaXMuZGlzdHJpYnV0aW9uQ29uZmlnLnJlZiA6IHVuZGVmaW5lZCxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgZ2V0SW5saW5lUG9saWNpZXMocGlwZWxpbmVDb25maWc6IGFueSk6IHsgW25hbWU6IHN0cmluZ106IGlhbS5Qb2xpY3lEb2N1bWVudCB9IHwgdW5kZWZpbmVkIHtcbiAgICBsZXQgcmVzdWx0IDogYW55ID0ge307XG4gICAgY29uc3Qgc3RhdGVtZW50cyA9IHBpcGVsaW5lQ29uZmlnLnBvbGljeV9zdGF0ZW1lbnRzO1xuICAgIGlmIChzdGF0ZW1lbnRzKSB7XG4gICAgICBmb3IgKGxldCBzdGF0ZW1lbnQgaW4gc3RhdGVtZW50cykge1xuICAgICAgICByZXN1bHRbc3RhdGVtZW50XSA9IG5ldyBpYW0uUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgc2lkOiBzdGF0ZW1lbnQsXG4gICAgICAgICAgICAgIGFjdGlvbnM6IHN0YXRlbWVudHNbc3RhdGVtZW50XS5hY3Rpb25zLFxuICAgICAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgICAgIHJlc291cmNlczogc3RhdGVtZW50c1tzdGF0ZW1lbnRdLnJlc291cmNlcyxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiByZXN1bHQ7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZUJsb2NrRGV2aWNlTWFwcGluZygpIHtcbiAgICBjb25zdCBrZXkgPSB0aGlzLnBpcGVsaW5lQ29uZmlnLmVuY3J5cHRlZCA/IG5ldyBrbXMuS2V5KHRoaXMsICdQaXBlbGluZUVuY3J5cHRpb24nLCB7XG4gICAgICBkZXNjcmlwdGlvbjogYEltYWdlIGVuY3J5cHRpb24gZm9yICR7dGhpcy5waXBlbGluZUNvbmZpZy5uYW1lfWAsXG4gICAgICBhbGlhczogYCR7dXRpbHMucGFzY2FsaXplKHRoaXMucGlwZWxpbmVDb25maWcubmFtZSl9X2tleWAsXG4gICAgfSkgOiBudWxsO1xuICAgIGxldCBibG9ja0RldmljZU1hcHBpbmdzID0ga2V5ID8gW1xuICAgICAge1xuICAgICAgICBkZXZpY2VOYW1lOiAnL2Rldi94dmRhJyxcbiAgICAgICAgZWJzOiB7XG4gICAgICAgICAgZW5jcnlwdGVkOiB0cnVlLFxuICAgICAgICAgIHZvbHVtZVNpemU6IDgsXG4gICAgICAgICAgdm9sdW1lVHlwZTogJ2dwMicsXG4gICAgICAgICAga21zS2V5SWQ6IGtleS5rZXlBcm4sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIF0gOiB1bmRlZmluZWQ7XG4gICAgcmV0dXJuIGJsb2NrRGV2aWNlTWFwcGluZ3M7XG4gIH1cblxuICBnZXROZXh0UmVjaXBlVmVyc2lvbihyZWNpcGVOYW1lOiBzdHJpbmcpIHtcbiAgICBjb25zdCByZWNpcGVCdW1wQ3VzdG9tUmVzb3VyY2UgPSBuZXcgSW1hZ2VSZWNpcGVWZXJzaW9uQnVtcCh0aGlzLCAnSW1hZ2VSZWNpcGVWZXJzaW9uQnVtcCcsIHJlY2lwZU5hbWUpO1xuICAgIHJldHVybiByZWNpcGVCdW1wQ3VzdG9tUmVzb3VyY2UubmV4dFZlcnNpb247XG4gIH1cblxuICBjcmVhdGVDb2RlcGlwZWxpbmVQcm9qZWN0KCkge1xuICAgIGlmICh0aGlzLmltYWdlUGlwZWxpbmUpIHtcbiAgICAgIC8vIENyZWF0ZSBDb2RlUGlwZWxpbmUgcHJvamVjdFxuICAgICAgbGV0IGltYWdlYnVpbGRlckJ1aWxkID0gbmV3IGNvZGVidWlsZC5QaXBlbGluZVByb2plY3QodGhpcywgJ0FtaVBpcGVsaW5lQnVpbGQnLCB7XG4gICAgICAgIGJ1aWxkU3BlYzogY29kZWJ1aWxkLkJ1aWxkU3BlYy5mcm9tT2JqZWN0KHtcbiAgICAgICAgICB2ZXJzaW9uOiAnMC4yJyxcbiAgICAgICAgICBwaGFzZXM6IHtcbiAgICAgICAgICAgIGluc3RhbGw6IHtcbiAgICAgICAgICAgICAgY29tbWFuZHM6IFtcbiAgICAgICAgICAgICAgICAnYXB0LWdldCB1cGRhdGUnLFxuICAgICAgICAgICAgICAgICdhcHQtZ2V0IGluc3RhbGwgYXdzY2xpIC15JyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBwcmVfYnVpbGQ6IHtcbiAgICAgICAgICAgICAgY29tbWFuZHM6IFtcbiAgICAgICAgICAgICAgICAnY3VybCAtWCBQT1NUIC1IIFxcJ0NvbnRlbnQtdHlwZTogYXBwbGljYXRpb24vanNvblxcJyAtLWRhdGEgXCJ7XFxcXFwidGV4dFxcXFxcIjpcXFxcXCIkUElQRUxJTkVfTkFNRSBidWlsZCBoYXMgc3RhcnRlZC5cXFxcXCJ9XCIgJFNMQUNLX1dFQkhPT0tfVVJMJyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBidWlsZDoge1xuICAgICAgICAgICAgICBjb21tYW5kczogJ2F3cyBpbWFnZWJ1aWxkZXIgc3RhcnQtaW1hZ2UtcGlwZWxpbmUtZXhlY3V0aW9uIC0taW1hZ2UtcGlwZWxpbmUtYXJuICRJTUFHRV9QSVBFTElORV9BUk4nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KSxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBidWlsZEltYWdlOiBjb2RlYnVpbGQuTGludXhCdWlsZEltYWdlLlNUQU5EQVJEXzRfMCxcbiAgICAgICAgICBlbnZpcm9ubWVudFZhcmlhYmxlczoge1xuICAgICAgICAgICAgSU1BR0VfUElQRUxJTkVfQVJOOiB7IHZhbHVlOiB0aGlzLmltYWdlUGlwZWxpbmUucmVmIH0sXG4gICAgICAgICAgICBTTEFDS19XRUJIT09LX1VSTDogeyB2YWx1ZTogdGhpcy5zbGFja0NvbmZpZy5zbGFja1dlYmhvb2tVcmwgfSxcbiAgICAgICAgICAgIFNMQUNLX0NIQU5ORUw6IHsgdmFsdWU6IHRoaXMuc2xhY2tDb25maWcuY2hhbm5lbCB9LFxuICAgICAgICAgICAgU0xBQ0tfVVNFUk5BTUU6IHsgdmFsdWU6IHRoaXMuc2xhY2tDb25maWcudXNlcm5hbWUgfSxcbiAgICAgICAgICAgIFBJUEVMSU5FX05BTUU6IHsgdmFsdWU6IHRoaXMuaW1hZ2VQaXBlbGluZS5uYW1lIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuXG4gICAgICBpbWFnZWJ1aWxkZXJCdWlsZC5hZGRUb1JvbGVQb2xpY3koXG4gICAgICAgIGlhbS5Qb2xpY3lTdGF0ZW1lbnQuZnJvbUpzb24oe1xuICAgICAgICAgIFNpZDogJ2ltYWdlYnVpbGRlcnN0YXJ0JyxcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgQWN0aW9uOiBbJ2ltYWdlYnVpbGRlcjpTdGFydEltYWdlUGlwZWxpbmVFeGVjdXRpb24nXSxcbiAgICAgICAgICBSZXNvdXJjZTogJyonLFxuICAgICAgICB9KSk7XG5cbiAgICAgIGNvbnN0IGltYWdlQnVpbGRlck91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoJ0Nka0J1aWxkT3V0cHV0Jyk7XG5cbiAgICAgIGxldCBzb3VyY2VBY3Rpb25zID0gdGhpcy5zb3VyY2VBY3Rpb25CdWlsZGVyLmNyZWF0ZVBpcGVsaW5lU291cmNlcygpO1xuICAgICAgY29uc3QgYnVpbGRBY3Rpb24gPSBuZXcgY29kZXBpcGVsaW5lX2FjdGlvbnMuQ29kZUJ1aWxkQWN0aW9uKHtcbiAgICAgICAgYWN0aW9uTmFtZTogJ1N0YXJ0SW1hZ2VCdWlsZGVyJyxcbiAgICAgICAgcHJvamVjdDogaW1hZ2VidWlsZGVyQnVpbGQsXG4gICAgICAgIGlucHV0OiBzb3VyY2VBY3Rpb25zLmZpbmQoYSA9PiBhICYmIGEuc291cmNlT3V0cHV0KT8uc291cmNlT3V0cHV0ISxcbiAgICAgICAgZXh0cmFJbnB1dHM6IHNvdXJjZUFjdGlvbnMubGVuZ3RoID4gMSA/IHNvdXJjZUFjdGlvbnMuc2xpY2UoMSkubWFwKGEgPT4gYS5zb3VyY2VPdXRwdXQpIDogW10sXG4gICAgICAgIG91dHB1dHM6IFtpbWFnZUJ1aWxkZXJPdXRwdXRdLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIENvbXBsZXRlIFBpcGVsaW5lIFByb2plY3RcbiAgICAgIHRoaXMuY29kZXBpcGVsaW5lID0gbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZSh0aGlzLCAnUGlwZWxpbmUnLCB7XG4gICAgICAgIHJlc3RhcnRFeGVjdXRpb25PblVwZGF0ZTogdHJ1ZSxcbiAgICAgICAgc3RhZ2VzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgc3RhZ2VOYW1lOiAnU291cmNlJyxcbiAgICAgICAgICAgIGFjdGlvbnM6IHNvdXJjZUFjdGlvbnMubWFwKGMgPT4gYy5hY3Rpb24pLmZpbHRlcigoYyk6IGMgaXMgY29kZXBpcGVsaW5lLklBY3Rpb24gPT4gYyAhPSBudWxsKSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIHN0YWdlTmFtZTogJ0J1aWxkJyxcbiAgICAgICAgICAgIGFjdGlvbnM6IFtidWlsZEFjdGlvbl0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIGdldExvb2t1cENyaXRlcmlhKHBhcmVudEltYWdlOiBhbnkpOiBlYzIuTG9va3VwTWFjaGluZUltYWdlUHJvcHMge1xuICAgIHN3aXRjaCAocGFyZW50SW1hZ2UpIHtcbiAgICAgIGNhc2UgJ1VidW50dTE4MDQnOlxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIG5hbWU6ICd1YnVudHUvaW1hZ2VzL2h2bS1zc2QvdWJ1bnR1LWJpb25pYy0xOC4wNC1hbWQ2NConLFxuICAgICAgICAgIG93bmVyczogWycwOTk3MjAxMDk0NzcnXSxcbiAgICAgICAgfTtcbiAgICAgIGNhc2UgJ1VidW50dTIwMDQnOlxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIG5hbWU6ICd1YnVudHUvaW1hZ2VzL2h2bS1zc2QvdWJ1bnR1LWZvY2FsLTIwLjA0LWFtZDY0KicsXG4gICAgICAgICAgb3duZXJzOiBbJzA5OTcyMDEwOTQ3NyddLFxuICAgICAgICB9O1xuICAgICAgY2FzZSAnQ2VudE9TNyc6XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgbmFtZTogJyonLFxuICAgICAgICAgIG93bmVyczogWydhd3MtbWFya2V0cGxhY2UnXSxcbiAgICAgICAgICBmaWx0ZXJzOiB7XG4gICAgICAgICAgICAncHJvZHVjdC1jb2RlJzogWydjdnVnemlrbnZteGdxbmE5bm9pYnFubnN5J10sXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICAgIGNhc2UgJ0NlbnRPUzgnOlxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIG5hbWU6ICcqJyxcbiAgICAgICAgICBvd25lcnM6IFsnYXdzLW1hcmtldHBsYWNlJ10sXG4gICAgICAgICAgZmlsdGVyczoge1xuICAgICAgICAgICAgJ3Byb2R1Y3QtY29kZSc6IFsnNDdrOWlhMmlneHBjY2UyYnpvOHUza2owMyddLFxuICAgICAgICAgIH0sXG4gICAgICAgIH07XG4gICAgICBkZWZhdWx0OlxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIG5hbWU6ICdhbXpuLWxpbnV4JyxcbiAgICAgICAgfTtcbiAgICB9XG4gIH1cblxufSJdfQ==