import * as iam from '@aws-cdk/aws-iam';
import { IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Connections, IConnectable } from './connections';
import { ISecurityGroup } from './security-group';
import { IVpc, SubnetSelection } from './vpc';
/**
 * A VPC endpoint.
 *
 * @stability stable
 */
export interface IVpcEndpoint extends IResource {
    /**
     * The VPC endpoint identifier.
     *
     * @stability stable
     * @attribute true
     */
    readonly vpcEndpointId: string;
}
/**
 * @stability stable
 */
export declare abstract class VpcEndpoint extends Resource implements IVpcEndpoint {
    /**
     * The VPC endpoint identifier.
     *
     * @stability stable
     */
    abstract readonly vpcEndpointId: string;
    /**
     * @stability stable
     */
    protected policyDocument?: iam.PolicyDocument;
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add.
     * @stability stable
     */
    addToPolicy(statement: iam.PolicyStatement): void;
}
/**
 * A gateway VPC endpoint.
 *
 * @stability stable
 */
export interface IGatewayVpcEndpoint extends IVpcEndpoint {
}
/**
 * The type of VPC endpoint.
 *
 * @stability stable
 */
export declare enum VpcEndpointType {
    /**
     * Interface.
     *
     * An interface endpoint is an elastic network interface with a private IP
     * address that serves as an entry point for traffic destined to a supported
     * service.
     *
     * @stability stable
     */
    INTERFACE = "Interface",
    /**
     * Gateway.
     *
     * A gateway endpoint is a gateway that is a target for a specified route in
     * your route table, used for traffic destined to a supported AWS service.
     *
     * @stability stable
     */
    GATEWAY = "Gateway"
}
/**
 * A service for a gateway VPC endpoint.
 *
 * @stability stable
 */
export interface IGatewayVpcEndpointService {
    /**
     * The name of the service.
     *
     * @stability stable
     */
    readonly name: string;
}
/**
 * An AWS service for a gateway VPC endpoint.
 *
 * @stability stable
 */
export declare class GatewayVpcEndpointAwsService implements IGatewayVpcEndpointService {
    /**
     * @stability stable
     */
    static readonly DYNAMODB: GatewayVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly S3: GatewayVpcEndpointAwsService;
    /**
     * The name of the service.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * @stability stable
     */
    constructor(name: string, prefix?: string);
}
/**
 * Options to add a gateway endpoint to a VPC.
 *
 * @stability stable
 */
export interface GatewayVpcEndpointOptions {
    /**
     * The service to use for this gateway VPC endpoint.
     *
     * @stability stable
     */
    readonly service: IGatewayVpcEndpointService;
    /**
     * Where to add endpoint routing.
     *
     * By default, this endpoint will be routable from all subnets in the VPC.
     * Specify a list of subnet selection objects here to be more specific.
     *
     * @default - All subnets in the VPC
     * @stability stable
     * @example
     *
     * vpc.addGatewayEndpoint('DynamoDbEndpoint', {
     *   service: ec2.GatewayVpcEndpointAwsService.DYNAMODB,
     *   // Add only to ISOLATED subnets
     *   subnets: [
     *     { subnetType: ec2.SubnetType.ISOLATED }
     *   ]
     * });
     */
    readonly subnets?: SubnetSelection[];
}
/**
 * Construction properties for a GatewayVpcEndpoint.
 *
 * @stability stable
 */
export interface GatewayVpcEndpointProps extends GatewayVpcEndpointOptions {
    /**
     * The VPC network in which the gateway endpoint will be used.
     *
     * @stability stable
     */
    readonly vpc: IVpc;
}
/**
 * A gateway VPC endpoint.
 *
 * @stability stable
 * @resource AWS::EC2::VPCEndpoint
 */
export declare class GatewayVpcEndpoint extends VpcEndpoint implements IGatewayVpcEndpoint {
    /**
     * @stability stable
     */
    static fromGatewayVpcEndpointId(scope: Construct, id: string, gatewayVpcEndpointId: string): IGatewayVpcEndpoint;
    /**
     * The gateway VPC endpoint identifier.
     *
     * @stability stable
     */
    readonly vpcEndpointId: string;
    /**
     * The date and time the gateway VPC endpoint was created.
     *
     * @stability stable
     * @attribute true
     */
    readonly vpcEndpointCreationTimestamp: string;
    /**
     * @stability stable
     * @attribute true
     */
    readonly vpcEndpointNetworkInterfaceIds: string[];
    /**
     * @stability stable
     * @attribute true
     */
    readonly vpcEndpointDnsEntries: string[];
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: GatewayVpcEndpointProps);
}
/**
 * A service for an interface VPC endpoint.
 *
 * @stability stable
 */
export interface IInterfaceVpcEndpointService {
    /**
     * The name of the service.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * The port of the service.
     *
     * @stability stable
     */
    readonly port: number;
    /**
     * Whether Private DNS is supported by default.
     *
     * @stability stable
     */
    readonly privateDnsDefault?: boolean;
}
/**
 * A custom-hosted service for an interface VPC endpoint.
 *
 * @stability stable
 */
export declare class InterfaceVpcEndpointService implements IInterfaceVpcEndpointService {
    /**
     * The name of the service.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * The port of the service.
     *
     * @stability stable
     */
    readonly port: number;
    /**
     * Whether Private DNS is supported by default.
     *
     * @stability stable
     */
    readonly privateDnsDefault?: boolean;
    /**
     * @stability stable
     */
    constructor(name: string, port?: number);
}
/**
 * An AWS service for an interface VPC endpoint.
 *
 * @stability stable
 */
export declare class InterfaceVpcEndpointAwsService implements IInterfaceVpcEndpointService {
    /**
     * @stability stable
     */
    static readonly SAGEMAKER_NOTEBOOK: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly ATHENA: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CLOUDFORMATION: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CLOUDTRAIL: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CODEBUILD: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CODEBUILD_FIPS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CODECOMMIT: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CODECOMMIT_FIPS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CODEPIPELINE: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CONFIG: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly EC2: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly EC2_MESSAGES: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly ECR: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly ECR_DOCKER: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly ECS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly ECS_AGENT: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly ECS_TELEMETRY: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly ELASTIC_FILESYSTEM: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly ELASTIC_FILESYSTEM_FIPS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly ELASTIC_INFERENCE_RUNTIME: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly ELASTIC_LOAD_BALANCING: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CLOUDWATCH_EVENTS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly APIGATEWAY: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CODECOMMIT_GIT: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CODECOMMIT_GIT_FIPS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly GLUE: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly KINESIS_STREAMS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly KINESIS_FIREHOSE: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly KMS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CLOUDWATCH_LOGS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly CLOUDWATCH: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly SAGEMAKER_API: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly SAGEMAKER_RUNTIME: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly SAGEMAKER_RUNTIME_FIPS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly SECRETS_MANAGER: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly SERVICE_CATALOG: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly SNS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly SQS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly SSM: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly SSM_MESSAGES: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly STS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly TRANSFER: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly STORAGE_GATEWAY: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly REKOGNITION: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly REKOGNITION_FIPS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly STEP_FUNCTIONS: InterfaceVpcEndpointAwsService;
    /**
     * @stability stable
     */
    static readonly LAMBDA: InterfaceVpcEndpointAwsService;
    /**
     * The name of the service.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * The port of the service.
     *
     * @stability stable
     */
    readonly port: number;
    /**
     * Whether Private DNS is supported by default.
     *
     * @stability stable
     */
    readonly privateDnsDefault?: boolean;
    /**
     * @stability stable
     */
    constructor(name: string, prefix?: string, port?: number);
}
/**
 * Options to add an interface endpoint to a VPC.
 *
 * @stability stable
 */
export interface InterfaceVpcEndpointOptions {
    /**
     * The service to use for this interface VPC endpoint.
     *
     * @stability stable
     */
    readonly service: IInterfaceVpcEndpointService;
    /**
     * Whether to associate a private hosted zone with the specified VPC.
     *
     * This
     * allows you to make requests to the service using its default DNS hostname.
     *
     * @default set by the instance of IInterfaceVpcEndpointService, or true if
     * not defined by the instance of IInterfaceVpcEndpointService
     * @stability stable
     */
    readonly privateDnsEnabled?: boolean;
    /**
     * The subnets in which to create an endpoint network interface.
     *
     * At most one
     * per availability zone.
     *
     * @default - private subnets
     * @stability stable
     */
    readonly subnets?: SubnetSelection;
    /**
     * The security groups to associate with this interface VPC endpoint.
     *
     * @default - a new security group is created
     * @stability stable
     */
    readonly securityGroups?: ISecurityGroup[];
    /**
     * Whether to automatically allow VPC traffic to the endpoint.
     *
     * If enabled, all traffic to the endpoint from within the VPC will be
     * automatically allowed. This is done based on the VPC's CIDR range.
     *
     * @default true
     * @stability stable
     */
    readonly open?: boolean;
    /**
     * Limit to only those availability zones where the endpoint service can be created.
     *
     * Setting this to 'true' requires a lookup to be performed at synthesis time. Account
     * and region must be set on the containing stack for this to work.
     *
     * @default false
     * @stability stable
     */
    readonly lookupSupportedAzs?: boolean;
}
/**
 * Construction properties for an InterfaceVpcEndpoint.
 *
 * @stability stable
 */
export interface InterfaceVpcEndpointProps extends InterfaceVpcEndpointOptions {
    /**
     * The VPC network in which the interface endpoint will be used.
     *
     * @stability stable
     */
    readonly vpc: IVpc;
}
/**
 * An interface VPC endpoint.
 *
 * @stability stable
 */
export interface IInterfaceVpcEndpoint extends IVpcEndpoint, IConnectable {
}
/**
 * A interface VPC endpoint.
 *
 * @stability stable
 * @resource AWS::EC2::VPCEndpoint
 */
export declare class InterfaceVpcEndpoint extends VpcEndpoint implements IInterfaceVpcEndpoint {
    /**
     * Imports an existing interface VPC endpoint.
     *
     * @stability stable
     */
    static fromInterfaceVpcEndpointAttributes(scope: Construct, id: string, attrs: InterfaceVpcEndpointAttributes): IInterfaceVpcEndpoint;
    /**
     * The interface VPC endpoint identifier.
     *
     * @stability stable
     */
    readonly vpcEndpointId: string;
    /**
     * The date and time the interface VPC endpoint was created.
     *
     * @stability stable
     * @attribute true
     */
    readonly vpcEndpointCreationTimestamp: string;
    /**
     * The DNS entries for the interface VPC endpoint.
     *
     * @stability stable
     * @attribute true
     */
    readonly vpcEndpointDnsEntries: string[];
    /**
     * One or more network interfaces for the interface VPC endpoint.
     *
     * @stability stable
     * @attribute true
     */
    readonly vpcEndpointNetworkInterfaceIds: string[];
    /**
     * (deprecated) The identifier of the first security group associated with this interface VPC endpoint.
     *
     * @deprecated use the `connections` object
     */
    readonly securityGroupId: string;
    /**
     * Access to network connections.
     *
     * @stability stable
     */
    readonly connections: Connections;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: InterfaceVpcEndpointProps);
    /**
     * Determine which subnets to place the endpoint in. This is in its own function
     * because there's a lot of code.
     */
    private endpointSubnets;
    private availableAvailabilityZones;
}
/**
 * Construction properties for an ImportedInterfaceVpcEndpoint.
 *
 * @stability stable
 */
export interface InterfaceVpcEndpointAttributes {
    /**
     * The interface VPC endpoint identifier.
     *
     * @stability stable
     */
    readonly vpcEndpointId: string;
    /**
     * (deprecated) The identifier of the security group associated with the interface VPC endpoint.
     *
     * @deprecated use `securityGroups` instead
     */
    readonly securityGroupId?: string;
    /**
     * The security groups associated with the interface VPC endpoint.
     *
     * @stability stable
     */
    readonly securityGroups?: ISecurityGroup[];
    /**
     * The port of the service of the interface VPC endpoint.
     *
     * @stability stable
     */
    readonly port: number;
}
