"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * A CloudFormation-init configuration.
 *
 * @stability stable
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * Build a new config from a set of Init Elements.
     *
     * @stability stable
     */
    static fromElements(...elements) {
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * Use an existing InitConfig object as the default and only config.
     *
     * @stability stable
     */
    static fromConfig(config) {
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * Build a CloudFormationInit from config sets.
     *
     * @stability stable
     */
    static fromConfigSets(props) {
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * Add a config with the given name to this CloudFormationInit object.
     *
     * @stability stable
     */
    addConfig(configName, config) {
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * Add a config set with the given name to this CloudFormationInit object.
     *
     * The new configset will reference the given configs in the given order.
     *
     * @stability stable
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * Attach the CloudFormation Init config to the given resource.
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *    metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *    next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *    `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *    referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     *
     * @stability stable
     */
    attach(attachedResource, attachOptions) {
        var _a, _b, _c;
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        const CFN_INIT_METADATA_KEY = 'AWS::CloudFormation::Init';
        if (attachedResource.getMetadata(CFN_INIT_METADATA_KEY) !== undefined) {
            throw new Error(`Cannot bind CfnInit: resource '${attachedResource.node.path}' already has '${CFN_INIT_METADATA_KEY}' attached`);
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata(CFN_INIT_METADATA_KEY, bindResult.configData);
        // Need to resolve the various tokens from assets in the config,
        // as well as include any asset hashes provided so the fingerprint is accurate.
        const resolvedConfig = attachedResource.stack.resolve(bindResult.configData);
        const fingerprintInput = { config: resolvedConfig, assetHash: bindResult.assetHash };
        const fingerprint = contentHash(JSON.stringify(fingerprintInput)).substr(0, 16);
        attachOptions.instanceRole.addToPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        const resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        const configSets = ((_a = attachOptions.configSets) !== null && _a !== void 0 ? _a : ['default']).join(',');
        const printLog = (_b = attachOptions.printLog) !== null && _b !== void 0 ? _b : true;
        if ((_c = attachOptions.embedFingerprint) !== null && _c !== void 0 ? _c : true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : [],
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['  cat /var/log/cfn-init.log >&2'] : [],
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(Object.values(configNameToBindResult).map(c => c.assetHash)),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
/**
 * A collection of configuration elements.
 *
 * @stability stable
 */
class InitConfig {
    /**
     * @stability stable
     */
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * Whether this configset has elements or not.
     *
     * @stability stable
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * Add one or more elements to the config.
     *
     * @stability stable
     */
    add(...elements) {
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const allConfig = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig];
        const authentication = allConfig.map(c => c === null || c === void 0 ? void 0 : c.authentication).reduce(deepMerge, undefined);
        const assetHash = combineAssetHashesOrUndefined(allConfig.map(c => c === null || c === void 0 ? void 0 : c.assetHash));
        return {
            config: {
                packages: packageConfig === null || packageConfig === void 0 ? void 0 : packageConfig.config,
                groups: groupsConfig === null || groupsConfig === void 0 ? void 0 : groupsConfig.config,
                users: usersConfig === null || usersConfig === void 0 ? void 0 : usersConfig.config,
                sources: sourcesConfig === null || sourcesConfig === void 0 ? void 0 : sourcesConfig.config,
                files: filesConfig === null || filesConfig === void 0 ? void 0 : filesConfig.config,
                commands: commandsConfig === null || commandsConfig === void 0 ? void 0 : commandsConfig.config,
                services: servicesConfig === null || servicesConfig === void 0 ? void 0 : servicesConfig.config,
            },
            authentication,
            assetHash,
        };
    }
    bindForType(elementType, renderOptions) {
        var _a;
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: (_a = bindResults.map(r => r.config).reduce(deepMerge, undefined)) !== null && _a !== void 0 ? _a : {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(bindResults.map(r => r.assetHash)),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    var _a, _b;
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...(_a = target[key]) !== null && _a !== void 0 ? _a : [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge((_b = target[key]) !== null && _b !== void 0 ? _b : {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
// Combines all input asset hashes into one, or if no hashes are present, returns undefined.
function combineAssetHashesOrUndefined(hashes) {
    const hashArray = hashes.filter((x) => x !== undefined);
    return hashArray.length > 0 ? hashArray.join('') : undefined;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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