"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
/**
 * A stack that will make a Lambda that will launch a lambda to glue
 * together all the DynamoDB tables into a global table
 */
class GlobalTableCoordinator extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        const lambdaFunction = new lambda.SingletonFunction(this, 'SingletonLambda', {
            code: lambda.Code.fromAsset(path.resolve(__dirname, '../', 'lambda-packages', 'aws-global-table-coordinator', 'lib')),
            description: 'Lambda to make DynamoDB a global table',
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            timeout: cdk.Duration.minutes(5),
            uuid: 'D38B65A6-6B54-4FB6-9BAD-9CD40A6DAC12',
        });
        grantCreateGlobalTableLambda(lambdaFunction.role);
        new cfn.CustomResource(this, 'CfnCustomResource', {
            provider: cfn.CustomResourceProvider.lambda(lambdaFunction),
            properties: {
                regions: props.regions,
                resourceType: 'Custom::DynamoGlobalTableCoordinator',
                tableName: props.tableName,
            },
            removalPolicy: props.removalPolicy,
        });
    }
}
exports.GlobalTableCoordinator = GlobalTableCoordinator;
/**
 * Permits an IAM Principal to create a global dynamodb table.
 * @param principal The principal (no-op if undefined)
 */
function grantCreateGlobalTableLambda(principal) {
    if (principal) {
        principal.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'iam:CreateServiceLinkedRole',
                'application-autoscaling:DeleteScalingPolicy',
                'application-autoscaling:DeregisterScalableTarget',
                'dynamodb:CreateGlobalTable', 'dynamodb:DescribeLimits',
                'dynamodb:DeleteTable', 'dynamodb:DescribeGlobalTable',
                'dynamodb:UpdateGlobalTable',
            ],
        }));
    }
}
//# sourceMappingURL=data:application/json;base64,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