"""
This module defines the core data structures for representing reasoning processes,
including the `ReasoningNode` and `SearchResult` classes.
`ReasoningNode` represents a single step in a chain of thought, storing information
about the strategy used, the prompt, the response, the chain of thought (CoT) data,
parent and child nodes, and metadata. It also provides methods for traversing the
reasoning tree and extracting the complete CoT.
`SearchResult` encapsulates the outcome of a reasoning search, including the final
reasoning node, the question, the ground truth answer, all terminal nodes, the success
status, the final answer, and any relevant metadata.
"""

from collections import deque
from typing import Any, Optional
from uuid import uuid4 as uuid

from cot_forge.reasoning.strategies import Strategy, StrategyRegistry
from cot_forge.utils.parsing import extract_final_answer_from_cot

# TODO: Change get_full_cot to give a better printout including strategy names


class ReasoningNode:
  """
  A node in a reasoning graph/tree/chain representing a step in the reasoning process.
  Attributes:
      id (UUID): UUID for identifying node in serialization and deserialization
      strategy (Strategy | None): The strategy applied at this node.
      prompt (str): The prompt used to generate the response.
      response (str): The response generated by the prompt.
      cot (list[dict[str, Any]] | None): Chain of thought steps for this node. 
          Each step is a dictionary.
      parent (Optional['ReasoningNode']): The parent node in the reasoning chain.
      children (list[ReasoningNode]): The child nodes of this node.
      is_final (bool): Indicates if this node is the final node in its chain.
      success (bool): Indicates if the reasoning at this node was successful.
      pruned (bool): Indicates if this node was pruned during the search.
      metadata (dict[str, Any]): Additional information about this node.
  Methods:
      add_child(child: 'ReasoningNode'): Adds a child node to this node.
      get_full_node_chain(): Returns the complete chain of nodes from the root to this node.
      get_full_cot(): Returns the complete chain of thought from the root to this node.
  """

  def __init__(self,
               strategy: Strategy,
               prompt: str,
               response: str,
               cot: list[dict[str, Any]],
               id: str = None,
               parent: Optional['ReasoningNode'] = None,
               is_final: bool = False,
               success: bool = False,
               pruned: bool = False,
               metadata: dict[str, Any] = None
               ):
    self.id = id if id else str(uuid())
    self.strategy = strategy
    self.prompt = prompt
    self.response = response
    self.cot: dict[str, Any] = cot
    self.parent = parent
    self.children: list[ReasoningNode] = []
    self.is_final = is_final
    self.success = success
    self.pruned = pruned
    self.metadata = {} if metadata is None else metadata

  def add_child(self, child: 'ReasoningNode'):
    self.children.append(child)

  def get_full_node_chain(self) -> list['ReasoningNode']:
    """Get the complete chain from the root to this node."""
    chain = []
    current_node = self
    while current_node:
      chain.append(current_node)
      current_node = current_node.parent
    return list(reversed(chain))

  def get_full_cot(self) -> list[dict[str, Any]]:
    """Get the complete chain of thought (CoT) from the root to this node."""
    nodes = self.get_full_node_chain()  # Already in root-to-current order
    result = []
    for node in nodes:
      if node.cot:
        result.extend(node.cot)
    return result

  def get_root(self) -> 'ReasoningNode':
    """Get the root node of the reasoning chain."""
    current_node = self
    while current_node.parent:
      current_node = current_node.parent
    return current_node

  def __repr__(self):
    # Count CoT steps instead of showing them
    cot_count = len(self.cot) if isinstance(self.cot, list) else "N/A"

    # Count children instead of showing them
    child_count = len(self.children)

    return (
        f"ReasoningNode(strategy={self.strategy.name}, "
        f"has_parent={self.parent is not None}, "
        f"children={child_count}, is_final={self.is_final}, "
        f"success={self.success}, cot_steps={cot_count})"
    )

  def __str__(self):
    return (
        f"ReasoningNode(strategy="
        f"{self.strategy.name if hasattr(self.strategy, 'name') else self.strategy}, "
        f"success={self.success}, final={self.is_final}, "
        f"cot_steps={len(self.cot) if isinstance(self.cot, list) else 0})"
    )

  def to_dict(self) -> dict[str, Any]:
    """
    Converts the ReasoningNode to a dictionary representation, excluding parent and children
    to avoid circular references.

    Returns:
        dict: Dictionary containing all node properties
    """
    return {
        "id": str(self.id),
        "strategy": self.strategy.name,
        "prompt": self.prompt,
        "response": self.response,
        "cot": self.cot,
        "is_final": self.is_final,
        "success": self.success,
        "metadata": self.metadata
    }


class SearchResult:
  """
  Represents the outcome of a search process for a reasoning task.
  Attributes:
      question (str): The original question posed to the search algorithm.
      ground_truth_answer (str): The known correct answer to the question.
      root_node (ReasoningNode): The root node of the reasoning graph.
      terminal_nodes (list[ReasoningNode] | None): A list of all terminal nodes reached in search.
      success (bool): Indicates whether the search was successful in finding a valid answer.
      metadata (dict[str, Any]): Additional information or statistics about the search process.
  """

  def __init__(
      self,
      question: str,
      ground_truth_answer: str,
      terminal_nodes: list[ReasoningNode] | None = None,
      success: bool = False,
      metadata: dict[str, Any] = None
  ):
    self.question = question
    self.ground_truth_answer = ground_truth_answer
    self.terminal_nodes = terminal_nodes if terminal_nodes else []
    self.success = success
    self.metadata = metadata if metadata else {}
    # Assumption is that all terminal nodes share the same root node
    self.root_node = self.terminal_nodes[0].get_root(
    ) if self.terminal_nodes else None

  def get_successful_terminal_nodes(self) -> list[ReasoningNode]:
    """Returns a list of successful terminal nodes."""
    return [node for node in self.terminal_nodes if node.success]

  def get_successful_final_answers(self) -> list[str]:
    """Returns a list of final answers from successful terminal nodes."""
    return [extract_final_answer_from_cot(node.cot) for node in self.get_successful_terminal_nodes()]

  def get_all_final_answers(self) -> list[str]:
    """Returns a list of all final answers from terminal nodes."""
    return [extract_final_answer_from_cot(node.cot) for node in self.terminal_nodes]

  def __repr__(self):
    question_preview = f"{self.question[:30]}..." if len(
        self.question) > 30 else self.question
    ground_truth_preview = (
        f"{self.ground_truth_answer[:30]}..."
        if len(self.ground_truth_answer) > 30
        else self.ground_truth_answer
    )
    return (
        f"SearchResult(question={question_preview}, "
        f"ground_truth_answer={ground_truth_preview}, "
        f"num_terminal_nodes={len(self.terminal_nodes)}, "
        f"success={self.success}, "
        f"successful_nodes={len(self.get_successful_terminal_nodes())}, "
        f"metadata={self.metadata})"
    )

  def __str__(self):
    successful_answers = self.get_successful_final_answers()
    question_preview = f"{self.question[:30]}..." if len(
        self.question) > 30 else self.question
    return (
        f"SearchResult(success={self.success}, "
        f"question={question_preview}, "
        f"num_terminal_nodes={len(self.terminal_nodes)}, "
        f"num_successful_nodes={len(self.get_successful_terminal_nodes())}, "
        f"successful_answers={successful_answers})"
    )

  def serialize(self):
    """
    Serializes the SearchResult into a dictionary representation.

    This method converts the entire reasoning graph structure into a serializable format
    by creating an adjacency list of parent-child relationships and a map of node IDs to
    node data. The graph is traversed starting from terminal nodes back to root nodes.

    The resulting dictionary contains:
    - adjacency_list: List of (parent_id, child_id) tuples representing graph edges
    - node_map: Dictionary mapping node IDs to serialized node data
    - question: The original question
    - ground_truth_answer: The known correct answer
    - success: Whether the search was successful
    - metadata: Additional search information

    Returns:
        dict[str, Any]: A serializable dictionary representation of the SearchResult
    """
    adjacency_list = []  # format: (parent_id, child_id)
    traversed_relationships = set()
    node_map = {}  # format: node_map[id] = ReasoningNode

    # Breadth-first traversal of the reasoning graph
    queue = deque([self.root_node])
    while queue:
      current_node = queue.popleft()
      if current_node:
        # Add to the node map
        node_map[current_node.id] = current_node

        # Add to the adjacency_list if not already traversed
        current_relationship = (
            str(current_node.parent.id) if current_node.parent else None, str(
                current_node.id)
        )
        if current_relationship not in traversed_relationships:
          adjacency_list.append(current_relationship)
          traversed_relationships.add(current_relationship)
        # Add children to the queue
        for child in current_node.children:
          queue.append(child)

    serialized_dict = {
        "adjacency_list": adjacency_list,
        "node_map": {k: v.to_dict() for k, v in node_map.items()},
        "question": self.question,
        "ground_truth_answer": self.ground_truth_answer,
        "success": self.success,
        "metadata": self.metadata
    }

    return serialized_dict

  @classmethod
  def deserialize(
      cls,
      serialized_dict: dict[str, Any],
      strategy_registry: StrategyRegistry
  ) -> type['SearchResult']:
    """
    Reconstructs a SearchResult object from its serialized dictionary representation.

    This method rebuilds the complete reasoning graph structure by:
    1. Validating the serialized dictionary contains required keys
    2. Reconstructing all ReasoningNode objects
    3. Using provided strategy_registry to reconstruct the nodes' Strategy
    4. Reestablishing parent-child relationships between nodes
    5. Identifying terminal nodes (nodes with no children)

    Args:
        serialized_dict (dict[str, Any]): The serialized SearchResult dictionary 
            (see SearchResult.serializer)
        strategy_registry (StrategyRegistry): Registry to convert strategy names to Strategy objs

    Returns:
        SearchResult: A fully reconstructed SearchResult with the complete reasoning graph

    Raises:
        ValueError: If the serialized dictionary is missing required keys
    """
    if "node_map" not in serialized_dict or "adjacency_list" not in serialized_dict:
      raise ValueError(
          "SearchResult: deserialized_dict requires keys 'node_map' and 'adjacency_list'.")

    if "question" not in serialized_dict or not "ground_truth_answer":
      raise ValueError(
          "SearchResult.deserialize() requires keys 'question' and 'ground_truth_answer'")

    question = serialized_dict['question']
    ground_truth_answer = serialized_dict['ground_truth_answer']
    node_map = serialized_dict['node_map']
    adjacency_list = serialized_dict['adjacency_list']
    success = serialized_dict.get("success", False)
    metadata = serialized_dict.get("metadata", {})

    # Construct all of the reasoning nodes
    for k, v in node_map.items():
      v['strategy'] = strategy_registry.get_strategy(v['strategy'])
      node_map[k] = ReasoningNode(**v)

    # Assume all nodes are terminal nodes and reduce set as we prove otherwise
    terminal_node_ids = {k for k in node_map}

    # Loop through adjacency list to add child/parents relationships
    for p, c in adjacency_list:
      terminal_node_ids.discard(p)
      if not p:
        node_map[c].parent = None
        continue
      node_map[p].children.append(node_map[c])
      node_map[c].parent = node_map[p]

    terminal_nodes = [node_map[k] for k in terminal_node_ids]

    return cls(
        question=question,
        ground_truth_answer=ground_truth_answer,
        terminal_nodes=terminal_nodes,
        success=success,
        metadata=metadata
    )
