"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InputType = exports.TaskInput = void 0;
const fields_1 = require("./fields");
/**
 * Type union for task classes that accept multiple types of payload.
 *
 * @stability stable
 */
class TaskInput {
    /**
     *
     * @param type type of task input
     * @param value payload for the corresponding input type.
     * It can be a JSON-encoded object, context, data, etc.
     */
    constructor(type, value) {
        this.type = type;
        this.value = value;
    }
    /**
     * Use a literal string as task input.
     *
     * This might be a JSON-encoded object, or just a text.
     *
     * @stability stable
     */
    static fromText(text) {
        return new TaskInput(InputType.TEXT, text);
    }
    /**
     * Use an object as task input.
     *
     * This object may contain Data and Context fields
     * as object values, if desired.
     *
     * @stability stable
     */
    static fromObject(obj) {
        return new TaskInput(InputType.OBJECT, obj);
    }
    /**
     * Use a part of the execution data or task context as task input.
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution or the current task context
     * as complete payload to a task.
     *
     * @stability stable
     */
    static fromJsonPathAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
    /**
     * Use a part of the execution data as task input.
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution as complete payload
     * to a task.
     *
     * @stability stable
     */
    static fromDataAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
    /**
     * Use a part of the task context as task input.
     *
     * Use this when you want to use a subobject or string from
     * the current task context as complete payload
     * to a task.
     *
     * @stability stable
     */
    static fromContextAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
}
exports.TaskInput = TaskInput;
/**
 * The type of task input.
 *
 * @stability stable
 */
var InputType;
(function (InputType) {
    InputType[InputType["TEXT"] = 0] = "TEXT";
    InputType[InputType["OBJECT"] = 1] = "OBJECT";
})(InputType = exports.InputType || (exports.InputType = {}));
//# sourceMappingURL=data:application/json;base64,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