#!/usr/bin/env python
"""
Script for generating various plots of package usage data read from logfiles
generated by Afterburner's TCP server-based package monitoring solution.

Currently, the following plots are generated:

* package usage by version
* package usage by user (normal and, optionally, log-scale variants)
* app usage by name
* app usage by user (normal and, optionally, log-scale variants)
* python usage by version
* scitools usage by version
* sub-daily invocation frequency (in 15-minute bins by default)
* daily invocation frequency
* monthly invocation frequency

If the --html option is enabled then an HTML page (named 'usage_plots.html') is
created containing all of the requested plots in a basic tabular layout.

To view this script's command-line usage, enter

% python plot_pkg_usage_stats.py --help

TODO:

* Implement a -y/--yearly option to generate annual sub-total graphs
* Add option to define bin size to use for sub-daily frequency plot
"""
from __future__ import (absolute_import, print_function)

import os
import re
import sys
import logging
import argparse
import subprocess
from datetime import date, datetime

import jinja2
import matplotlib
matplotlib.use('Agg')
import matplotlib.pyplot as plt
import pandas as pd
import numpy as np

try:
    import afterburner
except ImportError:
    # Try to import afterburner from MetO-specific location
    sys.path.append('/data/users/afterburner/software/turbofan/latest/lib/python')
    import afterburner

from afterburner.utils import is_true
from afterburner.utils.fileutils import expand_path

# Regular expression for parsing the main components of a log record.
RECORD_REGEX = re.compile(r'(\[\w+\])\s*(.+)\s*\|(.*)')

# Regular expression for parsing the app name from a log message.
APP_REGEX = re.compile(r'^Invoked the (\w+) app$')

# Column names for use with pandas dataframes.
DF_COL_NAMES = ['user', 'host', 'pkg', 'pkg_vn', 'py', 'sci', 'msg', 'app']

# Default plot size (in inches).
PLOT_SIZE = (8, 6)


def main():
    """Main function"""
    args = parse_args()
    dframe = load_usage_data(args)
    create_plots(dframe, args)
    if args.html:
        create_html_page(args)


def parse_args():
    """Parse command-line arguments."""

    parser = argparse.ArgumentParser(
        usage="%(prog)s [options]",
        description="Plot Python package & app usage statistics")

    parser.add_argument('-n', '--dryrun', action='store_true',
        help='Enable dry-run mode')
    parser.add_argument('-v', '--verbose', action='store_true',
        help='Enable verbose mode')
    parser.add_argument('-D', '--debug', action='store_true',
        help='Enable debug mode')

    # Developer note: most of the following options for enabling/disabling
    # plots are intentionally configured to accept y/n (or equivalent) values
    # rather than T or F values since this provides users with greater flexibility
    # for selecting which plots to generate or skip.

    parser.add_argument('-s', '--subdaily', default='n',
        help='Create sub-daily usage plots: y or n (default)')
    parser.add_argument('-d', '--daily', default='n',
        help='Create daily usage plots: y or n (default)')
    parser.add_argument('-m', '--monthly', default='n',
        help='Create monthly usage plots: y or n (default)')
    # parser.add_argument('-y', '--yearly', default='n',
    #     help='Create yearly usage plots: y or n (default)')
    parser.add_argument('-t', '--totals', default='y',
        help='Create plots of totals for full time period: y (default) or n')
    parser.add_argument('-S', '--subtotals', default='n',
        help='Create plots of sub-daily, daily & monthly sub-totals: y or n (default)')
    parser.add_argument('-a', '--all', action='store_true',
        help='Create all supported plots')
    parser.add_argument('--log-linear', action='store_true',
        help='Create selected per-user plots on a log-linear scale')
    parser.add_argument('--html', action='store_true',
        help='Create an html page of the plots')

    parser.add_argument('--start-date',
        help='Start date in YYYY-MM-DD format (default: date of earliest logfile)')
    parser.add_argument('--end-date',
        help='End date in YYYY-MM-DD format (default: today)')

    parser.add_argument("--log-dir",
        help="Pathname of directory in which to find log files "
             "(default: ~afterburner/logs/afterburner")
    parser.add_argument("--img-dir",
        help="Pathname of directory in which to create plot images (default: .)")
    parser.add_argument("--img-fmt", default='png',
        help="Format of output images (default: png)")

    parser.add_argument('--pkg-name', default='afterburner',
        help='Python package name being monitored (default: afterburner)')
    parser.add_argument('--pkg-title',
        help='Python package title (default: package name converted to title case)')

    args = parser.parse_args()

    # Set package name and title.
    if not args.pkg_name:
        args.pkg_name = 'afterburner'
    if not args.pkg_title:
        args.pkg_title = args.pkg_name.title()

    # If dry-run is enabled then we also need to enable verbose mode.
    if args.dryrun:
        args.verbose = True

    # Configure simple logging.
    if args.debug:
        level = logging.DEBUG
    elif args.verbose:
        level = logging.INFO
    else:
        level = logging.WARNING
    logging.basicConfig(level=level, format='%(levelname)s: %(message)s')

    # Expand log directory path.
    if args.log_dir:
        args.log_dir = expand_path(args.log_dir)
    else:
        args.log_dir = os.path.expanduser('~afterburner/logs/afterburner')

    # Expand image directory path.
    if args.img_dir:
        args.img_dir = expand_path(args.img_dir)
    else:
        args.img_dir = os.getcwd()

    # Create image output directory if it doesn't exist.
    if not os.path.isdir(args.img_dir):
        os.makedirs(args.img_dir)

    # Determine start date if not specfied by user.
    if not args.start_date:
        start = get_oldest_file_date(args.log_dir)
        if start:
            args.start_date = start
            logging.info("Start date not specified. Using %s", args.start_date)
        else:
            raise RuntimeError("Unable to determine date of earliest logfile.\n"
                "Please specify a date using the --start-date option.")

    # Determine end date if not specfied by user.
    if not args.end_date:
        args.end_date = str(date.today())
        logging.info("End date not specified. Using %s", args.end_date)

    args.totals = is_true(args.totals)
    args.subtotals = is_true(args.subtotals)
    args.subdaily = is_true(args.subdaily)
    args.daily = is_true(args.daily)
    args.monthly = is_true(args.monthly)

    if args.all:
        args.totals = True
        args.subtotals = True

    if args.subtotals:
        args.subdaily = True
        args.daily = True
        args.monthly = True

    args.date_range = "Time period covered: {0.start_date} to {0.end_date}".format(args)

    # Attach an empty list object which will be used to store the full paths to
    # created image files.
    args.img_paths = []

    return args


def load_usage_data(args):
    """
    Load usage data from daily log files. A sample log entry appears as follows:

    [afterburner_invoc] utc=2019-11-25T07:00:25, user=yamazaki@expspicesrv054,
    ... pkg=afterburner-1.3.1b1, py=2.7.15, sci=default_legacy/2019_02_27 |
    ... Invoked the ModelMonitor2 app
    """

    logging.info("Loading package usage data...")

    data = []
    times = []

    for dt in pd.date_range(args.start_date, args.end_date):
        year = dt.strftime('%Y')
        month = dt.strftime('%b').lower()
        logfile = dt.strftime('%Y-%m-%d.log')
        logfile_rel = os.path.join(year, month, logfile)
        logpath = os.path.join(args.log_dir, logfile_rel)

        if os.path.isfile(logpath):
            logging.info("Processing file <logdir>/%s...", logfile_rel)
        else:
            logging.warning("Logfile does not exist: %s", logpath)
            continue

        if args.dryrun:
            continue

        with open(logpath) as fh:
            nrecs = 0
            for record in fh.readlines():
                utc, rundict = parse_log_record(record.strip())
                if utc and rundict:
                    data.append(rundict)
                    times.append(utc)
                    nrecs += 1
            logging.info("\tRead %d records", nrecs)

    # Create a pandas data frame from the raw log records
    if args.dryrun:
        dframe = pd.DataFrame(np.array([range(len(DF_COL_NAMES))]), columns=DF_COL_NAMES)
    else:
        dframe = pd.DataFrame(data, index=times, columns=DF_COL_NAMES)
        logging.info("Read a total of %d records", len(dframe))
        logging.debug("First record: %r", data[0])
        logging.debug("Last record: %r", data[-1])

    return dframe


def parse_log_record(record):
    """Parse a log record."""

    utc = None
    rundict = {}

    mtch = RECORD_REGEX.match(record)
    if mtch:
        #logname = mtch.group(1).strip('[]')   # not used at present
        runinfo = mtch.group(2).strip()
        message = mtch.group(3).strip()
        utc, rundict = parse_runtime_info(runinfo)
        if rundict:
            #rundict['logname'] = logname
            rundict['msg'] = message
            # parse app name if mentioned in the message
            app_mtch = APP_REGEX.match(message)
            rundict['app'] = app_mtch.group(1) if app_mtch else None

    return utc, rundict


def parse_runtime_info(text):
    """Parse runtime information from a log record."""

    utc = None
    rundict = {}

    if text:
        rundict = {k:v for k, v in [x.strip().split('=') for x in text.split(',')]}
        utc = rundict.pop('utc')
        utc = datetime.strptime(utc, '%Y-%m-%dT%H:%M:%S')
        userhost = rundict.pop('user')
        user, host = userhost.split('@')
        rundict['user'] = user
        rundict['host'] = host
        pkg_name, pkg_vn = rundict['pkg'].split('-')
        rundict['pkg'] = pkg_name
        rundict['pkg_vn'] = pkg_vn

    return utc, rundict


def create_plots(dframe, args):
    """Create plot images."""

    if args.totals:
        plot_pkg_usage_by_version(dframe, args)
        plot_pkg_usage_by_user(dframe, args)
        if args.log_linear:
            plot_log_pkg_usage_by_user(dframe, args)
        plot_app_usage_by_name(dframe, args)
        plot_app_usage_by_user(dframe, args)
        if args.log_linear:
            plot_log_app_usage_by_user(dframe, args)
        plot_scitools_usage_by_version(dframe, args)
        plot_python_usage_by_version(dframe, args)

    if args.subdaily:
        plot_subdaily_invocation_freq(dframe, args)

    if args.daily:
        plot_daily_invocation_freq(dframe, args)

    if args.monthly:
        plot_monthly_invocation_freq(dframe, args)


def create_html_page(args):
    """Generate an HTML page containing the plots."""

    logging.info("Generating HTML page...")
    html_path = os.path.join(args.img_dir, 'usage_plots.html')

    try:
        jenv = jinja2.Environment(loader=jinja2.BaseLoader())
        template = jenv.from_string(HTML_TEMPLATE)

        # Convert the list of image filenames into a Nx2 array.
        first_in_row = True
        table = []
        for pth in args.img_paths:
            img_file = os.path.basename(pth)
            if first_in_row:
                row = [img_file]
            else:
                row.append(img_file)
                table.append(row)
            first_in_row = not first_in_row

        if not first_in_row:
            row.append(None)
            table.append(row)

        vn, rev = afterburner.__version__, afterburner.__version_info__[4]
        utcnow = datetime.utcnow().replace(microsecond=0)
        timestamp = utcnow.isoformat().replace('T', ' ')
        html_text = template.render(table=table, image_dir=args.img_dir,
            pkg_title=args.pkg_title, version=vn, rev=rev, timestamp=timestamp)

        with open(html_path, 'w') as fh:
            fh.write(html_text)
        logging.info("Wrote HTML file: " + html_path)

    except IOError:
        logging.error("Unable to create HTML output file: " + html_path)
        raise

    except Exception:
        logging.error("Problem trying to generate HTML page.")
        raise


def plot_bar_chart(dframe, img_path, orient='h', size=None, stacked=False, log=False,
        title=None, subtitle=None, xlabel=None, ylabel=None, xticks=None, yticks=None):
    """Plot a horizontal bar chart of the data held in a dataframe/series."""

    fig, axes = plt.subplots()
    fig.set_size_inches(size or PLOT_SIZE)

    utcnow = datetime.utcnow().replace(microsecond=0)
    time_str = "Chart generated at: " + utcnow.isoformat().replace('T', ' ')

    if orient == 'h':
        # create horizontal bar chart
        axes = dframe.plot.barh(stacked=stacked, log=log)
        axes.text(0.98, 0.05, time_str, verticalalignment='center',
            horizontalalignment='right', transform=axes.transAxes,
            color='darkgray', fontsize=10)
    else:
        # create vertical bar chart
        axes = dframe.plot.bar(stacked=stacked, log=log)
        axes.text(0.98, 0.95, time_str, verticalalignment='center',
            horizontalalignment='right', transform=axes.transAxes,
            color='darkgray', fontsize=10)

    if stacked:
        # move legend to RHS of the chart
        plt.legend(loc='center left', bbox_to_anchor=(1, 0.5))

    if title:
        plt.suptitle(title)
    if subtitle:
        plt.title(subtitle)

    if xticks:
        axes.set_xticklabels(xticks)
    if xlabel:
        axes.set_xlabel(xlabel)

    if yticks:
        axes.set_yticklabels(yticks)
    if ylabel:
        axes.set_ylabel(ylabel)

    plt.savefig(img_path, bbox_inches='tight')
    plt.close()


def plot_pkg_usage_by_version(dframe, args):
    """Plot package usage by version."""

    logging.info("Generating bar chart of package usage by version...")
    if args.dryrun: return

    df = dframe[['pkg', 'pkg_vn', 'app']]
    df = df[pd.isna(df['app'])]
    logging.debug("number of non-app records: %d", len(df))
    grouped = df.groupby('pkg_vn').size().sort_values()
    img_path = os.path.join(args.img_dir, 'pkg_usage_by_vn.' + args.img_fmt)
    plot_bar_chart(grouped, img_path, title=args.pkg_title+" Package Usage by Version",
        subtitle=args.date_range, xlabel='Number of Invocations', ylabel='Version')

    args.img_paths.append(img_path)


def plot_pkg_usage_by_user(dframe, args):
    """Plot package usage by user."""

    logging.info("Generating bar chart of package usage by user...")
    if args.dryrun: return

    df = dframe[['user', 'pkg', 'app']]
    df = df[pd.isna(df['app'])]
    logging.debug("number of non-app records: %d", len(df))
    grouped = df.groupby('user').size().sort_values()
    img_path = os.path.join(args.img_dir, 'pkg_usage_by_user.' + args.img_fmt)
    plot_bar_chart(grouped, img_path, title=args.pkg_title+" Package Usage by User",
        subtitle=args.date_range, xlabel='Number of Invocations', ylabel='Username')

    args.img_paths.append(img_path)


def plot_log_pkg_usage_by_user(dframe, args):
    """Plot package usage by user (on log scale)."""

    logging.info("Generating bar chart of package usage by user (on log scale)...")
    if args.dryrun: return

    df = dframe[['user', 'pkg', 'app']]
    df = df[pd.isna(df['app'])]
    logging.debug("number of non-app records: %d", len(df))
    grouped = df.groupby('user').size().sort_values()
    img_path = os.path.join(args.img_dir, 'log_pkg_usage_by_user.' + args.img_fmt)
    plot_bar_chart(grouped, img_path, title=args.pkg_title+" Package Usage by User (log scale)",
        subtitle=args.date_range, xlabel='Number of Invocations', ylabel='Username',
        log=True)

    args.img_paths.append(img_path)


def plot_app_usage_by_name(dframe, args):
    """Plot app usage by name."""

    logging.info("Generating bar chart of app usage by name...")
    if args.dryrun: return

    grouped = dframe.groupby('app').size().sort_values()
    img_path = os.path.join(args.img_dir, 'app_usage_by_name.' + args.img_fmt)
    plot_bar_chart(grouped, img_path, title=args.pkg_title+" App Usage by Name",
        subtitle=args.date_range, xlabel='Number of Invocations', ylabel='App Name')

    args.img_paths.append(img_path)


def plot_app_usage_by_user(dframe, args):
    """Plot app usage by user."""

    logging.info("Generating bar chart of app usage by user...")
    if args.dryrun: return

    grouped = dframe.groupby(['app', 'user']).size().sort_values().unstack()
    img_path = os.path.join(args.img_dir, 'app_usage_by_user.' + args.img_fmt)
    plot_bar_chart(grouped, img_path, title=args.pkg_title+" App Usage by User",
        subtitle=args.date_range, xlabel='Cumulative Number of Invocations',
        ylabel='App Name', stacked=True)

    args.img_paths.append(img_path)


def plot_log_app_usage_by_user(dframe, args):
    """Plot app usage by user (on log scale)."""

    logging.info("Generating bar chart of app usage by user (on log scale)...")
    if args.dryrun: return

    grouped = dframe.groupby(['app', 'user']).size().sort_values().unstack()
    img_path = os.path.join(args.img_dir, 'log_app_usage_by_user.' + args.img_fmt)
    plot_bar_chart(grouped, img_path, title=args.pkg_title+" App Usage by User (log scale)",
        subtitle=args.date_range, xlabel='Cumulative Number of Invocations',
        ylabel='App Name', stacked=True, log=True)

    args.img_paths.append(img_path)


def plot_scitools_usage_by_version(dframe, args):
    """Plot scitools usage by version."""

    logging.info("Generating bar chart of SciTools/SSS usage by version...")
    if args.dryrun: return

    grouped = dframe.groupby('sci').size().sort_values()
    img_path = os.path.join(args.img_dir, 'sci_usage_by_vn.' + args.img_fmt)
    yticks = []

    for name in grouped.index:
        if name == 'n/a':
            yticks.append('unspecified')
        else:
            yticks.append(name.replace('/', '\n'))

    plot_bar_chart(grouped, img_path, title="SciTools/SSS Usage by Version",
        subtitle=args.date_range, xlabel='Number of Invocations', ylabel='SSS Version',
        yticks=yticks)

    args.img_paths.append(img_path)


def plot_python_usage_by_version(dframe, args):
    """Plot python usage by version."""

    logging.info("Generating bar chart of Python usage by version...")
    if args.dryrun: return

    grouped = dframe.groupby('py').size().sort_values()
    img_path = os.path.join(args.img_dir, 'python_usage_by_vn.' + args.img_fmt)
    plot_bar_chart(grouped, img_path, title="Python Usage by Version",
        subtitle=args.date_range, xlabel='Number of Invocations', ylabel='Python Version')

    args.img_paths.append(img_path)


def plot_subdaily_invocation_freq(dframe, args, time_int=15):
    """Plot subdaily invocation frequency in N-minute bins."""

    logging.info("Generating bar chart of subdaily invocation frequency...")
    if args.dryrun: return

    nints = 1440 // time_int
    min_of_day = dframe.index.hour * 60 + dframe.index.minute
    min_of_day = np.round(min_of_day.to_numpy()/time_int)
    time_of_day = ['{:02}:{:02}'.format(x//60, x%60) for x in range(0, 1440, time_int)]
    df = pd.DataFrame(min_of_day, index=dframe.index, columns=['minuteofday'])
    freq = df.groupby('minuteofday').size().reindex(range(nints), fill_value=0)
    freq.index = time_of_day
    logging.debug("subdaily frequency:\n%s", freq.head())
    img_path = os.path.join(args.img_dir, 'subdaily_invoc_freq.' + args.img_fmt)

    if time_int < 60:
        dt = (60 // time_int) or 1
        xticks = [''] * len(time_of_day)
        xticks[::dt] = time_of_day[::dt]
    else:
        xticks = time_of_day

    plot_bar_chart(freq, img_path, orient='v',
        title="Sub-Daily {0} Package Invocation Frequency ({1} mins)".format(
            args.pkg_title, time_int),
        subtitle=args.date_range, xlabel='Time of Day', ylabel='Frequency',
        xticks=xticks)

    args.img_paths.append(img_path)


def plot_daily_invocation_freq(dframe, args):
    """Plot daily package invocation frequency"""

    logging.info("Generating bar chart of daily invocation frequency...")
    if args.dryrun: return

    df = dframe[['pkg', 'app']]
    df = df[pd.isna(df['app'])]
    freq = df.groupby(pd.Grouper(freq='D')).size()
    freq = pd.Series(freq.values, index=freq.index.strftime('%Y-%m-%d'))
    logging.debug("daily frequency:\n%s", freq.head())
    img_path = os.path.join(args.img_dir, 'daily_invoc_freq.' + args.img_fmt)
    title = "Daily {0} Package Invocation Frequency".format(args.pkg_title)
    plot_bar_chart(freq, img_path, orient='v', title=title, subtitle=args.date_range,
        xlabel='Day', ylabel='Frequency')

    args.img_paths.append(img_path)


def plot_monthly_invocation_freq(dframe, args):
    """Plot monthly package invocation frequency"""

    logging.info("Generating bar chart of monthly invocation frequency...")
    if args.dryrun: return

    df = dframe[['pkg', 'app']]
    df = df[pd.isna(df['app'])]
    freq = df.groupby(pd.Grouper(freq='MS')).size()
    freq = pd.Series(freq.values, index=freq.index.strftime('%Y-%m'))
    logging.debug("monthly frequency:\n%s", freq.head())
    img_path = os.path.join(args.img_dir, 'monthly_invoc_freq.' + args.img_fmt)
    title = "Monthly {0} Package Invocation Frequency".format(args.pkg_title)
    plot_bar_chart(freq, img_path, orient='v', title=title,
        subtitle=args.date_range, xlabel='Month', ylabel='Frequency')

    args.img_paths.append(img_path)


def get_oldest_file_date(log_dir):
    """
    Determine the date of the oldest logfile present in the directory hierarchy
    rooted at log_dir.

    This can be achieved using, for example, the find command as shown below.
    The resulting output line would then need to be parsed to get the date.

    find <logdir> -type f -name '????-??-??.log' -printf '%T+\t%p\n' | sort | head -n 1

    This should return a single text string containing "<datetime> <filename>".
    An empty string is returned if a date could not be determined.
    """

    cmd = ("find {} -type f -name '????-??-??.log' -printf '%T+\t%p\n' | sort | "
           "head -n 1".format(log_dir))

    try:
        cmd_out = subprocess.check_output(cmd, shell=True)
        cmd_out = cmd_out.decode('utf8')  # for Python 3 compatibility
        fdate, _fname = cmd_out.split()
        return fdate[:10]
    except:
        return ''


HTML_TEMPLATE = """
<!DOCTYPE html>
<html>
<head>
<style>
BODY {background-color: #f5efdf; font-family: Verdana, Arial, Helvetica, sans-serif; margin: 0px}

A:link {color: black; text-decoration: none}
A:visited {color: black; text-decoration: none}
A:active {color: black; text-decoration: underline}
A:hover {color: black; text-decoration: underline}

#titlebar {
    color: #555;
    background-color: #f5e1ab; 
	top: 3px; 
	padding:6px;
	position: relative;
	text-align: center;
	font-weight: bold;
	font-size: 120%;
}

#image-grid {
    padding-left: 20px;
	padding-right: 20px;
	padding-top: 30px;
	padding-bottom:15px
}
</style>
<title>{{pkg_title}} Package Usage Graphs</title>
</head>

<body class="Body">
<div id="titlebar"><b>{{pkg_title}} Package Usage Graphs</b></div>

<div id="image-grid">
<table width="100%">
{% for row in table %}
    <tr><td width="50%"><a href="{{row[0]}}">
        <img src="{{row[0]}}" style="display:block" width="100%">
        </a></td>
    {% if row[1] == None %}
        <td>&nbsp;</td></tr>
    {% else %}
        <td width="50%"><a href="{{row[1]}}">
        <img src="{{row[1]}}" style="display:block" width="100%">
        </a></td></tr>
    {% endif %}
{% endfor %}
</table>
</div>

<div align="center">
<p>Page generated by the plot_pkg_usage_stats.py script<br>
Timestamp: {{timestamp}} | Afterburner version: v{{version}}-r{{rev}}</p>
</div>

</body>
</html>
"""


if __name__ == '__main__':
    main()
