# (C) British Crown Copyright 2016-2021, Met Office
#
# See the LICENSE.TXT file included with the Afterburner
# software distribution for full license details.
"""
This module contains classes that encapsulate the logic for generating the names
of files produced by various climate models or their associated post-processing
systems.

**Index of Classes in this Module**

.. autosummary::
   :nosignatures:

   FilenameProvider
   UmFilenameProvider
   NemoFilenameProvider
   CiceFilenameProvider
   TemplateDrivenFilenameProvider
   TemplateStringFormatter
"""
from __future__ import (absolute_import, division, print_function)
from six.moves import (filter, input, map, range, zip)
from six import add_metaclass

import os
import abc
import string
import logging
import cf_units
from copy import copy
from pkg_resources import parse_version

import afterburner.utils.dateutils as dateutils
import afterburner.contrib.umfilelist as umfilelist
from afterburner.metavar import (MetaVariable, UmMetaVariable, NemoMetaVariable,
    CiceMetaVariable)

__all__ = (
    'POSTPROC_VN1_NEMO_TEMPLATE', 'POSTPROC_VN1_CICE_TEMPLATE',
    'POSTPROC_VN2_NEMO_TEMPLATE', 'POSTPROC_VN2_CICE_TEMPLATE',
    'FilenameProvider', 'UmFilenameProvider',
    'NemoFilenameProvider', 'CiceFilenameProvider',
    'TemplateDrivenFilenameProvider', 'TemplateStringFormatter')

# The following constants define filename templates for some commonly encountered
# filenaming schemes. In most instances they represent a canonical name template,
# but one which can be copied and amended to work in slightly different contexts.
# For example, some post-processing tools use {suite_id} instead of {runid}.

#: Generic template for NEMO netcdf filenames generated by the postproc vn1.0 script.
POSTPROC_VN1_NEMO_TEMPLATE = '{runid}o_{meaning_period}_{start_date}_{end_date}_{grid_id}.nc'

#: Generic template for CICE netcdf filenames generated by the postproc vn1.0 script.
POSTPROC_VN1_CICE_TEMPLATE = '{runid}i.{meaning_period}.{end_date}.nc'

#: Generic template for NEMO netcdf filenames generated by the postproc vn2.0 script.
POSTPROC_VN2_NEMO_TEMPLATE = 'nemo_{runid}o_{meaning_period}_{start_date}-{end_date}_{grid_id}.nc'

#: Generic template for CICE netcdf filenames generated by the postproc vn2.0 script.
POSTPROC_VN2_CICE_TEMPLATE = 'cice_{runid}i_{meaning_period}_{start_date}-{end_date}.nc'


@add_metaclass(abc.ABCMeta)
class FilenameProvider(object):
    """
    Abstract base class which defines the interface that subclasses should
    implement in order to provide the names of files generated by a particular
    climate model or associated post-processing system.
    """

    def __init__(self, version=None, ext='', **kwargs):
        """
        Constructor Arguments:

        :param str version: The version number of the climate model, or related
            post-processing system, responsible for generating model output files.
            Intended to act as a hint to facilitate version-specific filename
            construction but NOT CURRENTLY USED.
        :param str ext: Default extension, including the '.' separator, to append
            to generated filenames, e.g. '.nc'.
        """
        self.version = version
        self.ext = ext
        self.logger = logging.getLogger(__name__)

    @staticmethod
    def from_metavar(var):
        """
        Create a FilenameProvider instance suitable for use with the specified
        meta-variable.

        :param object var: An instance of one of the concrete subclasses of
            :class:`afterburner.metavar.MetaVariable`.
        """
        if isinstance(var, UmMetaVariable):
            return UmFilenameProvider(version=var.model_vn)
        elif isinstance(var, NemoMetaVariable):
            return NemoFilenameProvider(version=var.model_vn)
        elif isinstance(var, CiceMetaVariable):
            return CiceFilenameProvider(version=var.model_vn)
        else:
            raise TypeError("Unrecognised meta-variable object type: %s" % type(var))

    @abc.abstractmethod
    def iter_filenames(self, var_or_varlist, **kwargs):
        """
        Iterate over the set of model filenames corresponding to the specified
        variable or variables. Client code should not assume that the set of
        paths generated for a given variable are in any particular order (though
        typically they will be in ASCII sort order). The :meth:`get_filenames`
        method may be used to return a sorted list of filenames.

        :param var_or_varlist: An instance of one of the concrete subclasses of
            :class:`afterburner.metavar.MetaVariable`, or a list of such objects.
        :returns: A sequence of filenames corresponding to the specified model
            variables.
        """
        raise NotImplementedError()

    @abc.abstractmethod
    def get_filenames(self, var_or_varlist, sort=False, **kwargs):
        """
        Return a list of model filenames corresponding to the specified model
        variable or variables.

        :param var_or_varlist: An instance of one of the concrete subclasses of
            :class:`afterburner.metavar.MetaVariable`, or a list of such objects.
        :param bool sort: Specifies whether or not to ASCII-sort the returned list of
            filenames. For very large lists this might be an expensive operation.

        :returns: A list of model filenames corresponding to the specified model
            variables.
        """
        raise NotImplementedError()


class UmFilenameProvider(FilenameProvider):
    """
    Provides the names of files generated by the Met Office Unified Model.

    Refer to UM Documentation Paper 007 for a description of UM file-naming
    conventions.

    .. note:: At present this class is only able to provide usable UM filenames
       when the start date associated with target variables is aligned with the
       standard UM reference date of December 1st (e.g. 1859-12-01). Although a
       different reference date can be defined (see :class:`afterburner.metavar.UmMetaVariable`),
       the generated filenames might not correspond to those held, for example, in MASS.
    """

    def __init__(self, ext='.pp', **kwargs):
        """
        Constructor Arguments:

        :param str ext: Default extension, including the '.' separator, to append
            to generated filenames.
        """
        super(UmFilenameProvider, self).__init__(ext=ext, **kwargs)

    def iter_filenames(self, var_or_varlist, **kwargs):
        """
        Iterate over the set of UM model filenames corresponding to the specified
        variable or variables. If a given meta-variable has no associated time
        range then the corresponding time element of the generated filename(s)
        is replaced with a '*' character, e.g. 'expida.pm*.pp'.

        :param var_or_varlist: An instance of UmMetaVariable, or a list of these.
        :returns: A sequence of filenames corresponding to the requested model
            variables.
        """
        varlist = _as_list(var_or_varlist)

        for var in varlist:
            if var.time_range:
                startdate, enddate = self._get_start_end_dates(var)
                cmrdate = var.cmrdate
            else:
                # Use proxy start and end dates in order to generate sample
                # filename(s) suitable for subsequent manipulation. The start
                # and end dates must span at least 1 meaning period.
                cmrdate = var.cmrdate or '185912010000'
                syear = int(cmrdate[:4])
                ndays = umfilelist.stream_interval.get(var.stream_id[1:], 360)
                nyears = (ndays // 360) or 1
                startdate = cmrdate
                enddate = '{0:04d}{1}'.format(syear+nyears, cmrdate[4:])

            # Build the list of mandatory arguments to pass through to the
            # umfilelist.main function.
            args = []
            args.append('--prefix=%s' % var.run_id)
            args.append('--suffix=%s' % self.ext)
            args.append('--stream=%s' % var.stream_id)
            args.append('--startdate=%s' % startdate)
            args.append('--enddate=%s' % enddate)
            args.append('--calendar=%s' % (var.calendar or '360_day'))
            args.append('--%s_absolute_time' % (var.time_format or 'standard'))

            # Extract any umfilelist options defined on the current meta-variable.
            if cmrdate:
                args.append('--cmrdate=%s' % cmrdate)
            if var.reinit:
                args.append('--reinit=%s' % var.reinit)
            if var.newmode:
                args.append('--newmode')
            if var.zeropad_dates is False:   # treat None as an indeterminate setting
                args.append('--no_zeropad_dates')

            try:
                # Get an iterator over the expected filenames for the current variable.
                it = umfilelist.main_as_iterator(args)

                # If no time range was defined then we only need a version of the
                # first filename with the time information replaced with '*'.
                if not var.time_range:
                    fname = next(it)
                    fname = self._replace_time_component(var, fname)
                    if var.realization_id:
                        fname = self._fix_ensemble_member_filename(var, fname)
                    yield fname
                else:
                    for fname in it:
                        # If the meta-variable belongs to an ensemble then insert the
                        # realization name into the filename.
                        if var.realization_id:
                            fname = self._fix_ensemble_member_filename(var, fname)
                        yield fname

            except GeneratorExit:
                self.logger.debug('Caught GeneratorExit exception.')
                return

            except:
                msg = "Error generating filenames via the umfilelist\n" \
                      "module with the following arguments:\n" + ' '.join(args)
                self.logger.error(msg)
                raise

    def get_filenames(self, var_or_varlist, sort=False, **kwargs):
        """
        Return a list of UM model filenames corresponding to the specified
        variable or variables. If a given meta-variable has no associated time
        range then the corresponding time element of the generated filename(s)
        is replaced with a '*' character, e.g. 'expida.pm*.pp'.

        This method currently uses the umfilelist.py module from the AR5 project
        to generate UM filenames. The original source code for this module is at:
        https://code.metoffice.gov.uk/trac/cdds/browser/main/trunk/hadsdk/hadsdk/umfilelist.py.
        The local copy of this module can be found at :mod:`afterburner.contrib.umfilelist`.

        :param var_or_varlist: An instance of UmMetaVariable, or a list of these.
        :param bool sort: Specifies whether or not to ASCII-sort the returned list of
            filenames. For very large lists this might be an expensive operation.
        :returns: A list of UM filenames corresponding to the requested variables.
        """
        varlist = _as_list(var_or_varlist)
        filenames = set()

        for var in varlist:
            for fname in self.iter_filenames(var, **kwargs):
                filenames.add(fname)

        if sort:
            return sorted(filenames)
        else:
            return list(filenames)

    def _get_start_end_dates(self, var):
        """
        Get start and end date-times in YYYYMMDDhhmm format, as required by the
        umfilelist module. Local var.time_range date-times are in ISO-like
        YYYY-MM-DDThh:mm:ss format.
        """
        if not (var.start_time and var.end_time):
            msg = "Time range attributes not set on meta-variable:\n{0}".format(var)
            raise ValueError(msg)
        fmt = "{0.year:04d}{0.month:02d}{0.day:02d}{0.hour:02d}{0.minute:02d}"
        startdate = fmt.format(var.start_time)
        enddate = fmt.format(var.end_time)
        return startdate[:12], enddate[:12]

    def _fix_ensemble_member_filename(self, var, filename):
        """
        If meta-variable ``var`` references an ensemble member model run then
        insert the ensemble member name into the correct place within ``filename``.
        At present, in the case of UM PP files held in MASS, the ensemble member
        name appears after the 5-letter run id separated by a hyphen character,
        e.g. 'ad035-r1i2p3a.pa2005jan.pp'
        """
        if var.run_id not in filename:
            return filename
        else:
            tokens = filename.partition(var.run_id)
            return ''.join(tokens[:2]) + '-' + var.realization_id + tokens[2]

    def _replace_time_component(self, var, filename, repl='*'):
        """
        Replace the time component of ``filename`` with the text in ``repl``.
        Assumes that ``filename`` is in the format '<runid><stream><time><ext>',
        where <stream> includes a separator character (one of '.@-').
        """
        stem, ext = os.path.splitext(filename)
        l = len(var.run_id) + len(var.stream_id)
        return stem[:l+1] + repl + ext


class NemoFilenameProvider(FilenameProvider):
    """
    Provides the names of files generated by the NEMO model. In practice the
    filenames are as generated by local post-processing scripts rather than
    the actual NEMO model code. NEMO itself does not prescribe any particular
    file naming convention.

    Annual mean filenames follow these patterns::

        <expid>o_1y_YYYYMMDD_YYYYMMDD_grid_[TUVW].nc
        <expid>o_1y_YYYYMMDD_YYYYMMDD_diaptr.nc
        <expid>o_1y_YYYYMMDD_YYYYMMDD_scalar.nc

    Seasonal mean filenames follow these patterns::

        <expid>o_1s_YYYYMMDD_YYYYMMDD_grid_[TUVW].nc
        <expid>o_1s_YYYYMMDD_YYYYMMDD_diaptr.nc
        <expid>o_1s_YYYYMMDD_YYYYMMDD_scalar.nc

    Monthly mean filenames follow these patterns::

        <expid>o_1m_YYYYMMDD_YYYYMMDD_grid_[TUVW].nc
        <expid>o_1m_YYYYMMDD_YYYYMMDD_diaptr.nc
        <expid>o_1m_YYYYMMDD_YYYYMMDD_scalar.nc

    Note that NEMO filenames adopt the human convention for labelling start and
    end dates, i.e. 19700401_19700430 for April 1970. This differs from the CF
    convention, which would use 19700401_19700501 for the same monthly period.

    This class now provides support for generating filenames compliant with the
    postproc vn2.0 post-processing software. Such filenames follow the pattern::

        nemo_<expid>o_<frequency>_<start-date>-<end-date>_<grid>.nc

    Filenames adhering to this convention can be obtained by setting an attribute
    named ``postproc_vn``, and with a value of ``'2.0'``, on the meta-variable
    objects passed to the :meth:`get_filenames` method.
    """

    #: Mappings from NEMO stream names to corresponding filename tokens.
    STREAM_MAPPINGS = {'ony': '1y', 'ons': '1s', 'onm': '1m'}

    def __init__(self, ext='.nc', **kwargs):
        """
        Constructor Arguments:

        :param str ext: Default extension, including the '.' separator, to append
            to generated filenames.
        """
        super(NemoFilenameProvider, self).__init__(ext=ext, **kwargs)

    def iter_filenames(self, var_or_varlist, **kwargs):
        """
        Iterate over the set of NEMO model filenames corresponding to the specified
        variable or variables. If a given meta-variable has no associated time
        range then the corresponding time element of the generated filename(s)
        is replaced with a '*' character, e.g. 'expido_1y_*_grid_T.nc'.

        :param var_or_varlist: An instance of NemoMetaVariable, or a list of these.

        Extra Keyword Arguments (`**kwargs`):

        :param bool padding: If set to true (the default) then the generated
            filenames will include any extra files needed to cover time periods
            not aligned with month, season or year boundaries. If set to false
            then only files corresponding to whole months (or seasons or years)
            contained within the user-specified time range (as attached to the
            input variables) will be returned.

        :returns: A sequence of filenames corresponding to the requested NEMO
            model variables.
        """
        varlist = _as_list(var_or_varlist)
        padding = kwargs.get('padding', True)

        for var in varlist:
            if var.stream_id not in self.STREAM_MAPPINGS:
                raise ValueError("Unrecognised stream identifier: " + var.stream_id)
            meanp = self.STREAM_MAPPINGS.get(var.stream_id)

            if _uses_postproc2(var):
                grid_name = self._get_grid_name(var.grid_type, sep='-')
                if var.time_range:
                    it = dateutils.iter_date_chunks(var.start_time, var.end_time,
                        var.stream_id, calendar=var.calendar, padding=padding)
                    for dtr in it:
                        dt_str = dtr.as_name_token(dates_only=True, compact=True)
                        startdate, enddate = dt_str.split('_')
                        fn = "nemo_*{0}o_{1}_{2}-{3}_{4}{5}".format(var.run_id, meanp,
                            startdate[:8], enddate[:8], grid_name, self.ext)
                        yield fn
                else:
                    # No time range specified so replace time elements with '*'
                    fn = "nemo_*{0}o_{1}_*-*_{2}{3}".format(var.run_id, meanp,
                        grid_name, self.ext)
                    yield fn

            else:
                grid_name = self._get_grid_name(var.grid_type)
                if var.time_range:
                    it = dateutils.iter_date_chunks(var.start_time, var.end_time,
                        var.stream_id, calendar=var.calendar, padding=padding,
                        end_offset=-24)
                    for dtr in it:
                        dt_str = dtr.as_name_token(dates_only=True, compact=True)
                        startdate, enddate = dt_str.split('_')
                        fn = "{0}o_{1}_{2}_{3}_{4}{5}".format(var.run_id, meanp,
                            startdate[:8], enddate[:8], grid_name, self.ext)
                        yield fn
                else:
                    # No time range specified so replace time elements with '*'
                    fn = "{0}o_{1}_*_{2}{3}".format(var.run_id, meanp, grid_name,
                        self.ext)
                    yield fn

    def get_filenames(self, var_or_varlist, sort=False, **kwargs):
        """
        Return a list of NEMO model filenames corresponding to the specified
        variable or variables. If a given meta-variable has no associated time
        range then the corresponding time element of the generated filename(s)
        is replaced with a '*' character, e.g. 'expido_1y_*_grid_T.nc'.

        :param var_or_varlist: An instance of NemoMetaVariable, or a list of these.
        :param bool sort: Specifies whether or not to ASCII-sort the returned list of
            filenames. For very large lists this might be an expensive operation.

        Extra Keyword Arguments (`**kwargs`):

        :param bool padding: If set to true (the default) then the returned
            filelist will include any extra files needed to cover time periods
            not aligned with month, season or year boundaries. If set to false
            then only files corresponding to whole months (or seasons or years)
            contained within the user-specified time range (as attached to the
            input variables) will be returned.

        :returns: A list of NEMO filenames corresponding to the requested variables.
        """
        varlist = _as_list(var_or_varlist)
        filenames = set()

        for var in varlist:
            for fname in self.iter_filenames(var, **kwargs):
                filenames.add(fname)

        if sort:
            return sorted(filenames)
        else:
            return list(filenames)

    def _get_grid_name(self, grid_type, sep='_'):
        """
        Return the token used in a NEMO filename to represent the specified
        grid type.
        """
        if grid_type in 'TUVW':
            grid_name = 'grid' + sep + grid_type
        else:
            grid_name = grid_type
        return grid_name


class CiceFilenameProvider(FilenameProvider):
    """
    Provides the names of files generated by the CICE sea-ice model. In practice
    the filenames are as generated by local post-processing scripts rather than
    the actual CICE model code, which itself does not prescribe any particular
    file naming convention.

    Annual mean filenames follow this pattern::

        <expid>i.1y.YYYY-MM.nc

    Seasonal mean filenames follow this pattern::

        <expid>i.1s.YYYY-MM.nc

    Monthly mean filenames follow this pattern::

        <expid>i.1m.YYYY-MM.nc

    This class now provides support for generating filenames compliant with the
    postproc vn2.0 post-processing software. Such filenames follow the pattern::

        cice_<expid>i_<frequency>_<start-date>-<end-date>.nc

    Filenames adhering to this convention can be obtained by setting an attribute
    named ``postproc_vn``, and with a value of ``'2.0'``, on the meta-variable
    objects passed to the :meth:`get_filenames` method.
    """

    #: Mappings from CICE stream names to corresponding filename tokens.
    STREAM_MAPPINGS = {'iny': '1y', 'ins': '1s', 'inm': '1m'}

    def __init__(self, ext='.nc', **kwargs):
        """
        Constructor Arguments:

        :param str ext: Default extension, including the '.' separator, to append
            to generated filenames.
        """
        super(CiceFilenameProvider, self).__init__(ext=ext, **kwargs)

    def iter_filenames(self, var_or_varlist, **kwargs):
        """
        Iterate over the set of CICE model filenames corresponding to the specified
        variable or variable. If a given meta-variable has no associated time
        range then the corresponding time element of the generated filename(s)
        is replaced with a '*' character, e.g. 'expidi.1y.*.nc'.

        :param var_or_varlist: An instance of CiceMetaVariable, or a list of these.

        Extra Keyword Arguments (`**kwargs`):

        :param bool padding: If set to true (the default) then the generated
            filenames will include any extra files needed to cover time periods
            not aligned with month, season or year boundaries. If set to false
            then only files corresponding to whole months (or seasons or years)
            contained within the user-specified time range (as attached to the
            input variables) will be returned.

        :returns: A sequence of filenames corresponding to the requested CICE
            model variables.
        """
        varlist = _as_list(var_or_varlist)
        padding = kwargs.get('padding', True)

        for var in varlist:
            if var.stream_id not in self.STREAM_MAPPINGS:
                raise ValueError("Unrecognised stream identifier: " + var.stream_id)
            meanp = self.STREAM_MAPPINGS.get(var.stream_id)

            if _uses_postproc2(var):
                if var.time_range:
                    it = dateutils.iter_date_chunks(var.start_time, var.end_time,
                        var.stream_id, calendar=var.calendar, padding=padding)
                    for dtr in it:
                        dt_str = dtr.as_name_token(dates_only=True, compact=True)
                        startdate, enddate = dt_str.split('_')
                        fn = "cice_*{0}i_{1}_{2}-{3}{4}".format(var.run_id, meanp,
                            startdate[:8], enddate[:8], self.ext)
                        yield fn
                else:
                    # No time range specified so replace time elements with '*'
                    fn = "cice_*{0}i_{1}_*-*{2}".format(var.run_id, meanp, self.ext)
                    yield fn

            else:
                if var.time_range:
                    it = dateutils.iter_date_chunks(var.start_time, var.end_time,
                        var.stream_id, calendar=var.calendar, padding=padding,
                        end_offset=-24)
                    for dtr in it:
                        enddate = dtr.end[:7]   # 'YYYY-MM' portion of end date
                        fn = "{0}i.{1}.{2}{3}".format(var.run_id, meanp, enddate, self.ext)
                        yield fn
                else:
                    # No time range specified so replace time elements with '*'
                    fn = "{0}i.{1}.*{2}".format(var.run_id, meanp, self.ext)
                    yield fn

    def get_filenames(self, var_or_varlist, sort=False, **kwargs):
        """
        Return a list of CICE model filenames corresponding to the specified
        variable or variables. If a given meta-variable has no associated time
        range then the corresponding time element of the generated filename(s)
        is replaced with a '*' character, e.g. 'expidi.1y.*.nc'.

        :param var_or_varlist: An instance of CiceMetaVariable, or a list of these.
        :param bool sort: Specifies whether or not to ASCII sort the returned list
            of filenames. For very large lists this might be an expensive operation.

        Extra Keyword Arguments (`**kwargs`):

        :param bool padding: If set to true (the default) then the returned
            filelist will include any extra files needed to cover time periods
            not aligned with month, season or year boundaries. If set to false
            then only files corresponding to whole months (or seasons or years)
            contained within the user-specified time range (as attached to the
            input variables) will be returned.

        :returns: A list of CICE filenames corresponding to the requested variables.
        """
        varlist = _as_list(var_or_varlist)
        filenames = set()

        for var in varlist:
            for fname in self.iter_filenames(var, **kwargs):
                filenames.add(fname)

        if sort:
            return sorted(filenames)
        else:
            return list(filenames)


class TemplateDrivenFilenameProvider(FilenameProvider):
    """
    Provides functionality for generating filenames corresponding to one or more
    :mod:`meta-variable objects <afterburner.metavar>`. Filenames are constructed
    according to a template string compatible with Python's
    `string.format() <https://docs.python.org/2/library/string.html#format-string-syntax>`_
    method. An example template string for a NEMO model filename might appear thus::

        template = '{runid}o_{meaning_period}_{start_date}-{end_date}_{grid_id}.nc'

    Each meta-variable object passed to the :meth:`get_filenames` method must
    possess instance attributes matching the fields (or tokens) delimited by
    brace characters in the template string. An exception is raised if this
    condition does not hold.

    If a meta-variable object has a ``time_range`` attribute that defines a tuple
    of date strings (start_date, end_date), then the time range is expanded to
    yield the consecutive start and end dates for each meaning period covered by
    the time range. These date pairs can be referenced within the template string
    using fields named ``$startdate`` and ``$enddate``, respectively.

    Thus, the aforementioned template string could be recast as follows if it
    was desirable to obtain multiple filenames for consecutive meaning periods::

        template = '{runid}o_{meaning_period}_{$startdate}-{$enddate}_{grid_id}.nc'

    In the case where a meta-variable object does not define a ``time_range``
    attribute (or it is set to None) then these two special fields are assigned
    the value '*'. This can be useful when one wishes to process files for any
    time period.

    .. note:: The $ prefix is used merely to disambiguate 'special' fields from
       regular fields. Despite their appearance, the former are not interpreted
       as shell or environment variables.

    A custom string formatter class (see :class:`TemplateStringFormatter`) is
    used to parse the template string. This class adds the following conversion
    specifiers to those provided by the standard Python string formatter:

    * ``!l`` - convert the field value to lower case
    * ``!u`` - convert the field value to upper case
    * ``!t`` - convert the field value to title case

    To obtain a *lowercase* version of the model_name attribute, for instance,
    one would include the field ``{model_name!l}`` in the template string.
    """

    _special_date_fields = ['$startdate', '$enddate']

    def __init__(self, template, ext='', **kwargs):
        """
        Constructor Arguments:

        :param str template: The template string used to generate filenames.
            The template should conform to the syntax accepted by Python's
            string.format() method.
        :param str ext: Default extension to append to generated filenames.
            This can be left blank if the template string includes a hard-coded
            filename extension.
        """
        super(TemplateDrivenFilenameProvider, self).__init__(ext=ext, **kwargs)
        self.formatter = TemplateStringFormatter()
        self.template = template

    def iter_filenames(self, var_or_varlist, **kwargs):
        """
        Iterate over the set of filenames corresponding to the specified variable
        or variables. Each filename is constructed by replacing tokens in the
        template string with corresponding attribute values from the specified
        meta-variable(s).

        :param var_or_varlist: Typically an object (or list of objects) whose
            type is a subclass of :class:`afterburner.metavar.MetaVariable`. In
            practice, objects of pretty much any type can be passed in so long
            as they contain the attributes referenced by the template string.

        Refer to the :meth:`get_filenames` method for a description of the extra
        keyword arguments recognised by the current method.
        """
        varlist = _as_list(var_or_varlist)

        chunk_dates = sum([1 for x in self._special_date_fields if x in self.template])
        date_fmt = kwargs.get('date_fmt', '%Y%m%d')
        date_chunker = kwargs.get('date_chunker', dateutils.iter_date_chunks)
        kwds = {k:v for k, v in kwargs.items() if k in ['padding', 'ref_date',
            'end_offset']}

        for var in varlist:
            tokens = vars(var).copy()
            if isinstance(var, MetaVariable) and var.time_range and chunk_dates:
                it = date_chunker(var.start_time, var.end_time, var.stream_id,
                    calendar=var.calendar, **kwds)
                for dtr in it:
                    tokens['$startdate'] = dtr.start_ncdt.strftime(date_fmt)
                    tokens['$enddate'] = dtr.end_ncdt.strftime(date_fmt)
                    fn = self.formatter.format(self.template, **tokens)
                    if self.ext: fn += self.ext
                    yield fn
            else:
                tokens['$startdate'] = '*'
                tokens['$enddate'] = '*'
                fn = self.formatter.format(self.template, **tokens)
                if self.ext: fn += self.ext
                yield fn

    def get_filenames(self, var_or_varlist, sort=False, **kwargs):
        """
        Return a list of filenames corresponding to the specified variable or
        variables. Each filename is constructed by replacing tokens in the
        template string with corresponding attribute values from the specified
        meta-variable(s).

        :param var_or_varlist: Typically an object (or list of objects) whose
            type is a subclass of :class:`afterburner.metavar.MetaVariable`. In
            practice, objects of pretty much any type can be passed in so long
            as they contain the attributes referenced by the template string.
        :param bool sort: Specifies whether or not to ASCII sort the returned list
            of filenames. For very large lists this might be an expensive operation.

        Extra Keyword Arguments (`**kwargs`):

        :param func date_chunker: The generator function to call if a meta-variable
            object has a ``time_range`` attribute whose associated time period
            needs splitting into consecutive date chunks sized according to the
            meta-variable's standard meaning period. The function signature
            should mirror that defined for the :func:`afterburner.utils.dateutils.iter_date_chunks`
            function. That function acts as the default callback if one is not
            specified by the calling program.
        :param bool padding: If specified, this argument is passed through to
            the nominated (or default) date_chunker function.
        :param str ref_date: If specified, this argument is passed through to
            the nominated (or default) date_chunker function.
        :param int end_offset: If specified, this argument is passed through to
            the nominated (or default) date_chunker function.
        :param str date_fmt: The strftime-compatible format to use in constructing
            date-time strings to insert into filenames. The default format is
            '%Y%m%d'.

        :returns: A list of filenames corresponding to the requested variables.
        :raises KeyError: Raised if the template string contains a token which
            does not have a matching attribute on the input meta-variable(s).
        """
        varlist = _as_list(var_or_varlist)
        filenames = set()

        for var in varlist:
            for fname in self.iter_filenames(var, **kwargs):
                filenames.add(fname)

        if sort:
            return sorted(filenames)
        else:
            return list(filenames)


class TemplateStringFormatter(string.Formatter):
    """
    Custom string formatter class which adds support for lowercase (!l),
    uppercase (!u) and titlecase (!t) conversion specifiers to the base class.
    """
    def __init__(self):
        super(TemplateStringFormatter, self).__init__()

    def convert_field(self, value, conversion):
        if conversion is None:
            return value
        elif conversion == 's':
            return str(value)
        elif conversion == 'r':
            return repr(value)
        elif conversion == 'l':
            return value.lower()
        elif conversion == 'u':
            return value.upper()
        elif conversion == 't':
            return value.title()
        raise ValueError("Unknown conversion specifier {0!s}".format(conversion))


def _iter_date_chunks(var, padding=True, date_fmt='%Y%m%d', end_offset=0):
    """
    Iterate over the start and end dates of all time chunks in the time period
    specified in the meta-variable definition. The length of the time chunks
    will depend upon the meaning period, e.g. 3 months for seasonal means.
    Also, the returned dates will reflect the calendar associated with the
    variable.

    .. deprecated:: 1.2.0
       Use the :func:`afterburner.utils.dateutils.iter_date_chunks` function instead.

    :param xxxMetaVariable var: A meta-variable definition containing time range,
        calendar and meaning period information.
    :param bool padding: Indicates whether or not to pad out the time chunks to
        cover the full time range. This is enabled by default since it is the
        desired behaviour in most scenarios.
    :param str date_fmt: The strfime-compatible format to use in constructing
        the returned date strings.
    :param int end_offset: The offset, in whole hours, to apply to returned end
        dates. For example, an offset of -24 might be specified in order to
        generate date strings for use in NEMO filenames prior to postproc-2.0.
        By default no offset is applied.
    :returns: A list of (start, end) tuples where start and end are date-time
        strings formatted according to the value of the ``date_fmt`` argument.
    """

    # Get start and end dates as iris PDT objects and as numeric coords.
    tunits = cf_units.Unit('hours since {:04d}-01-01 0:0:0'.format(var.start_time.year-1),
        calendar=var.calendar)
    end_coord = tunits.date2num(dateutils.pdt_to_nc_datetime(var.end_time))

    # Assign the date of the first time chunk.
    if padding:
        epdt = dateutils.round_date_to_stream_boundary(var.start_time, var.stream_id)
    else:
        epdt = dateutils.round_date_to_stream_boundary(var.start_time, var.stream_id, end=True)
    ecoord = tunits.date2num(dateutils.pdt_to_nc_datetime(epdt))

    # Loop over successive time chunks until we go past the end date.
    # Use a cycle counter to preclude any chance of an infinite loop.
    niter = 0
    while ecoord < end_coord and niter < 100000:
        niter += 1
        spdt = epdt
        epdt = copy(spdt)

        # TODO: generalise this code block to cater for wider range of meaning periods?
        if var.stream_id in ['apy', 'iny', 'ony']:
            epdt.year += 1
        elif var.stream_id in ['aps', 'ins', 'ons']:
            epdt.month += 3
        elif var.stream_id in ['apm', 'inm', 'onm']:
            epdt.month += 1

        if epdt.month > 12:
            # roll over to next year
            epdt.year += 1
            epdt.month %= 12

        scoord = tunits.date2num(dateutils.pdt_to_nc_datetime(spdt))
        ecoord = tunits.date2num(dateutils.pdt_to_nc_datetime(epdt))
        if ecoord > end_coord and not padding:
            # break out if the current time chunk extends beyond the end
            # date and padding has been disabled.
            break

        sdt = tunits.num2date(scoord)
        edt = tunits.num2date(ecoord + end_offset)   # apply offset to end date
        yield (sdt.strftime(date_fmt), edt.strftime(date_fmt))


def _uses_postproc2(var):
    """Returns true if the specified meta-variable uses postproc 2.0 or later."""
    if var.postproc_vn:
        return parse_version(var.postproc_vn) >= parse_version('2.0')
    else:
        return False


def _as_list(x):
    """Convert x to [x] unless x is already a list, tuple or set."""
    if isinstance(x, (list, tuple, set)):
        return x
    else:
        return [x]
