# (C) British Crown Copyright 2016-2020, Met Office
#
# See the LICENSE.TXT file included with the Afterburner
# software distribution for full license details.
"""
Provides an implementation of the Afterburner ModelMonitor2 application, which
enables scientists to monitor selected model diagnostics as they are being
generated by a climate simulation.

In its present form the ModelMonitor2 application has a number of limitations,
as follows:

* As standard, only works with annual-mean or monthly-mean diagnostics produced
  by the UM, NEMO or CICE climate models (other data streams can be specified
  though your mileage may vary).
* Retrieving NEMO or CICE model data from MASS can be unreliable because MASS
  does not currently maintain temporal metadata for netCDF data collections.
  It is recommended that such model data be restored to disk as a separate task.
* Only plots 2D surface fields, or a specified level of a 3D (Z-Y-X) field.
* All input diagnostics are assumed to have the same calendar type which, by
  default, is '360_day'.
* If source model data is to be retrieved from the MASS data archive, then the
  runtime environment must have access to the MOOSE command-line interface.

Refer to the :class:`ModelMonitor2` class for further information.
"""
from __future__ import (absolute_import, division, print_function)
from six.moves import (filter, input, map, range, zip)
from six import text_type

import os
import sys
import logging
import datetime
import cf_units

try:
    import cftime as cft
except ImportError:
    import netcdftime as cft

import numpy as np
import numpy.ma as ma

import jinja2
import matplotlib
matplotlib.use('Agg')   # directs output to file rather than screen
import matplotlib.pyplot as plt
import matplotlib.ticker as mtick
import matplotlib.colors as mplc

import iris
import iris.analysis
import iris.analysis.cartography
import iris.coords
import iris.exceptions
import iris.util
import iris.quickplot as qplt
try:
    from iris.util import equalise_attributes
except ImportError:
    from iris.experimental.equalise_cubes import equalise_attributes

import afterburner
from afterburner.io import moose2 as moose
from afterburner.apps import AbstractApp
from afterburner.config import ConfigProvider
from afterburner.processors.diags import (ToaRadiationBalance, NetHeatFluxIntoOcean,
    NaoIndex)
from afterburner.processors.diags.derived import SimpleDerivedDiagnostic
from afterburner.utils import (NamespacePlus, get_class_object_from_class_path, is_true,
    INTERVAL_OPEN, INTERVAL_LEFTOPEN, INTERVAL_LEFTCLOSED, INTERVAL_CLOSED)
from afterburner.utils.cubeutils import (is_scalar_coord, is_mean_of_all_time_steps,
    set_history_attribute, are_time_axes_equal, compare_cubes)
from afterburner.utils.dateutils import (pdt_from_date_string, pdt_to_date_string,
    pdt_to_nc_datetime, pdt_compare, DateTimeRange)
from afterburner.utils.fileutils import expand_path
from afterburner.io.datastores import MassDataStore
from afterburner.io.datacaches import VAR_SPLIT_SCHEME, DataCache
from afterburner.metavar import *
from afterburner.modelmeta import (mass_collection_from_stream, is_msi_stash_code,
    decode_dates_from_nemo_filename, decode_dates_from_cice_filename)
from afterburner.exceptions import (DataProcessingError, DataStoreError,
    MooseUnavailableError, AppConfigError)

try:
    from functools import lru_cache
except ImportError:
    from afterburner.utils import lru_cache

# MOOSE read-only flags.
MOOSE_READ = moose.MOOSE_GET | moose.MOOSE_SELECT

# IDs of namelists used in the application configuration file.
MODEL_NAMELIST = 'models'
DIAGS_NAMELIST = 'diags'

# Define the properties of supported custom diagnostics.
# Those below relate to the original prototype app. Where practicable, new
# custom diagnostics should use one of the new derived diagnostic classes.
CUSTOM_DIAGNOSTICS = {
    'toa_radiation_balance': {
        'stashcodes': ['m01s01i207', 'm01s01i208', 'm01s03i332'],
        'class': ToaRadiationBalance
    },
    'net_heat_flux_into_ocean': {
        #'stashcodes': ['m01s01i203', 'm01s02i207', 'm01s03i228', 'm01s00i507', 'm01s03i232'],
        'stashcodes': ['m01s01i203', 'm01s02i201', 'm01s03i228', 'm01s03i234'],
        'class': NetHeatFluxIntoOcean
    },
}

# HTML template filename.
HTML_TEMPLATE = 'cmm2_app_template.html'

# Names of recognised vertical coordinates.
LEVEL_COORD_NAMES = ['level', 'model_level', 'model_level_number',
    'soil_model_level_number', 'pseudo_level', 'depth']

# Graph sorting options.
SORT_GRAPHS_BY_DIAG_NAME = 'diag_and_region_name'
SORT_GRAPHS_BY_ORDER_KEY = 'graph_order_key'

# Default STASH code for land-area fraction data.
LAF_STASH_CODE = 'm01s03i395'

# Dictionary of supported statistical methods. The 'func' key defines the
# function to call in order to calculate the statistic for a dataset. The
# 'label' key defines the label to use to annotate plots.
STAT_METHODS = {
    'min': {'func': '_calc_min', 'label': 'Minimum'},
    'max': {'func': '_calc_max', 'label': 'Maximum'},
    'mean': {'func': '_calc_mean', 'label': 'Mean'},
    'awmean': {'func': '_calc_mean', 'label': 'Area-Weighted Mean'},
    'sum': {'func': '_calc_sum', 'label': 'Sum'},
    'awsum': {'func': '_calc_sum', 'label': 'Area-weighted Sum'},
    'nao_index': {'func': '_calc_nao_index', 'label': 'NAO Index'},
}

# Dictionary of mappings from model calendar type to hourly-mean LBTIM value.
CAL_LBTIM_MAP = {
    cf_units.CALENDAR_STANDARD: 121, cf_units.CALENDAR_GREGORIAN: 121,
    cf_units.CALENDAR_PROLEPTIC_GREGORIAN: 121,
    cf_units.CALENDAR_360_DAY: 122, cf_units.CALENDAR_365_DAY: 124,
}

# Default symbology for climate model time-series graphs.
DEFAULT_SYMBOLOGY = {
    'transparency': 1.0,
    'line_style': 'solid',
    'line_colour': 'black',
    'line_width': None,
    'marker_style': None,
    'marker_size': None,
    'marker_edge_colour': 'auto',
    'marker_face_colour': 'auto',
}

# Create a logger object.
_logger = logging.getLogger(__name__)


class ModelMonitor2(AbstractApp):
    """
    Provides an implementation of the Climate Model Monitor v2 (CMM2) application,
    which enables scientists to monitor selected model diagnostics as they are
    being generated by a climate simulation.

    In normal operation the application retrieves UM annual-mean or monthly-mean
    data from the MASS archive, extracts the requested diagnostics, calculates
    the area-weighted global or regional mean of the data, plots the resultant
    mean values as a time-series graph, and finally saves the mean data to disk
    in netCDF format. Other statistic measures, such as the area-weighted sum,
    may be generated instead of, or in addition to, the spatial-mean.

    The target diagnostics must be either surface fields (e.g. 1.5m air temperature),
    or 3D (Z-Y-X) fields from which a single, user-specified level is extracted.
    It is possible, however, to monitor multiple vertical levels from the same
    diagnostic.

    The CMM2 app now includes a limited capability to monitor NEMO and CICE model
    diagnostics serialized in netCDF format. Owing to current limitations relating
    to the type and extent of metadata held in MASS in connection with netCDF data
    collections, it is recommended that requisite netCDF data files are placed
    on disk using a separate process (or even manually) rather than through use
    of the CMM2 app (in practical terms this means disabling the ``sync_with_mass``
    configuration option).
    """

    def __init__(self, arglist=None, **kwargs):
        """
        :param list arglist: List of raw options and/or arguments passed from the
            calling environment. Typically these will be unprocessed command-line
            arguments, e.g. ``['-f', 'foo', '--foe=fum', 'infile']``.
        """
        super(ModelMonitor2, self).__init__(version='2.0.0b7', **kwargs)

        # Parse command-line arguments and read app configuration information
        # from a Rose configuration file.
        self._parse_args(arglist, desc="Climate Model Monitor v2: generates "
            "time-series graphs of climate model diagnostics.")
        self._parse_app_config()
        self._set_message_level()
        self.logger.info("Initialising ModelMonitor2 app...")

        # Extract general options from app config object.
        self.app_options = self._get_app_options()

        # Before proceeding, check access to MASS.
        self._check_mass_access()

        # Extract model definitions from app config object.
        self.model_defns = self._get_model_definitions()

        # Extract diag definitions from app config object.
        self.diag_defns = self._get_diag_definitions()

        # Initialise dictionary to store memory-cached source diagnostic data.
        self.model_data_memcache = {}

        # Initialise model data cache.
        self._init_model_data_cache()

        # Initialise set object to record new files uploaded into cache.
        self.new_files_in_cache = set()

        # Plot size (w, h) in inches.
        self.plot_size = (9.5, 6.1)

    @property
    def cli_spec(self):
        """Defines the command-line interface specification for the application."""
        return [
            {'names': ['-c', '--config-file'], 'required': True,
                'help': 'Pathname of app configuration file'},
        ]

    def run(self):
        """Run the application."""
        self.logger.info("Running ModelMonitor2 app...")

        if not self._check_moose_status():
            return

        self._create_output_dirs()

        if self.app_options['clear_netcdf_cache']:
            self._clear_nc_cache()

        self._refresh_cached_diag_data()

        self._generate_plot_images()

        self._generate_html_page()

        self._teardown()

        self.logger.info("ModelMonitor2 app execution completed.")

    @property
    def model_names(self):
        """A list of names of all defined models, sorted by plot order."""
        names = [defn.name for defn in sorted(self.model_defns.values(),
            key=lambda sk: sk.plot_order)]
        return names

    @property
    def enabled_model_names(self):
        """A list of names of enabled models, sorted by plot order."""
        names = [defn.name for defn in sorted(self.model_defns.values(),
            key=lambda sk: sk.plot_order) if defn.enabled]
        return names

    @property
    def diag_names(self):
        """A sorted list of the names of all defined diagnostics."""
        return [self.diag_defns[k].name for k in self._sorted_diag_keys()]

    @property
    def enabled_diag_names(self):
        """A sorted list of the names of enabled diagnostics."""
        return [self.diag_defns[k].name for k in self._sorted_diag_keys(True)]

    @property
    def enabled_diag_keys(self):
        """A sorted list of (var_name, region_name) keys for enabled diagnostics."""
        return self._sorted_diag_keys(True)

    def _sorted_diag_keys(self, enabled_only=False):
        """
        Return a list of diagnostic keys sorted according to the value of the
        'sort_graphs_by' configuration option. If the enabled_only argument is
        set to true then only keys for enabled diagnostics are returned.
        """
        if self.app_options['sort_graphs_by'] == SORT_GRAPHS_BY_ORDER_KEY:
            keys = [k for k, defn in sorted(self.diag_defns.items(),
                    key=lambda kv: kv[1].graph_order)]

        else:    # SORT_GRAPHS_BY_DIAG_NAME
            keys = sorted(self.diag_defns)

        if enabled_only:
            keys = [k for k in keys if self.diag_defns[k].enabled]

        return keys

    def _get_app_options(self):
        """Read general app config options and store them in a dictionary."""

        section = 'general'
        app_opts = {}

        # directory locations
        output_dir = self.app_config.get_property(section, 'output_dir')
        output_dir = expand_path(output_dir)
        cache_dir = self.app_config.get_property(section, 'cache_dir')
        app_opts['cache_dir'] = expand_path(cache_dir)

        self.output_dir = output_dir
        self.nc_output_dir = os.path.join(output_dir, 'nc')
        self.img_output_dir = os.path.join(output_dir, 'images')
        self.html_output_dir = output_dir

        # cache management options
        app_opts['cache_type'] = self.app_config.get_property(section,
            'cache_type', default=VAR_SPLIT_SCHEME)
        app_opts['cache_readonly'] = self.app_config.get_bool_property(
            section, 'cache_readonly', default=False)
        app_opts['sync_with_mass'] = self.app_config.get_bool_property(
            section, 'sync_with_mass', default=True)
        app_opts['clear_model_cache'] = self.app_config.get_bool_property(
            section, 'clear_model_cache', default=False)
        app_opts['clear_netcdf_cache'] = self.app_config.get_bool_property(
            section, 'clear_netcdf_cache', default=False)

        # disable sync_with_mass option if read-only cache access selected
        if app_opts['cache_readonly']:
            app_opts['sync_with_mass'] = False

        # disable clear_model_cache if sync_with_mass option is disabled
        if not app_opts['sync_with_mass']:
            app_opts['clear_model_cache'] = False

        # default stream identifier, calendar, and reinit interval.
        app_opts['stream'] = self.app_config.get_property(section,
            'stream', default='apy')
        app_opts['calendar'] = self.app_config.get_property(section,
            'calendar', default='360_day')
        app_opts['reinit'] = self.app_config.get_int_property(section,
            'reinit', default=0)
        app_opts['zeropad_dates'] = self.app_config.get_bool_property(section,
            'zeropad_dates', default=True)

        # set default values for lbtim and lbproc
        app_opts['lbtim'] = CAL_LBTIM_MAP.get(app_opts['calendar'], 122)
        app_opts['lbproc'] = 128

        # set default list of levels to use in MOOSE time queries
        surface_lblevs = self.app_config.get_property(section, 'surface_lblevs',
            default='0,1,7777,8888,9999')
        if surface_lblevs == 'none':
            app_opts['surface_lblevs'] = None
        else:
            app_opts['surface_lblevs'] = [int(x) for x in surface_lblevs.split(',')]

        # treatment of region extent coordinates, e.g. closed, left-closed
        app_opts['treat_region_coords_as'] = self.app_config.get_property(
            section, 'treat_region_coords_as', default=INTERVAL_LEFTCLOSED)

        # graph annotation and sorting options
        for attname in ['include_stash', 'include_date', 'include_region']:
            app_opts[attname] = self.app_config.get_bool_property(section, attname)

        app_opts['legend_vis'] = self.app_config.get_property(section,
            'legend_vis', default='all')
        app_opts['legend_style'] = self.app_config.get_property(section,
            'legend_style', default='name')
        app_opts['sort_graphs_by'] = self.app_config.get_property(section,
            'sort_graphs_by', default=SORT_GRAPHS_BY_DIAG_NAME)

        # parallel processing options
        app_opts['num_processors'] = self.app_config.get_int_property(section,
            'num_processors', default=1)

        # equalise_attributes and ignoreable_attributes options
        app_opts['equalise_attributes'] = self.app_config.get_bool_property(
            section, 'equalise_attributes', default=False)
        attnames = self.app_config.get_property(section, 'ignoreable_attributes',
            default='um_version')
        if attnames:
            attnames = [s.strip() for s in attnames.split(',')]
        else:
            attnames = []
        app_opts['ignoreable_attributes'] = attnames

        return app_opts

    def _get_model_definitions(self):
        """
        Read all of the target model definitions from the MODEL_NAMELIST portion
        of the configuration file. Model definitions are stored as simple
        NamespacePlus objects.
        """
        default_reinit = self.app_options['reinit']
        default_zeropad_dates = self.app_options['zeropad_dates']

        model_defns = {}
        for mdict in self.app_config.iter_nl(MODEL_NAMELIST):
            mdict['enabled'] = is_true(mdict.get('enabled', 't'))
            if not mdict.get('um_vn'): mdict['um_vn'] = '0'

            mdefn = NamespacePlus(**mdict)
            mdefn.plot_order = int(mdefn.plot_order)

            # The 'name' config option for a model can take one of four forms:
            #
            # 1. an old-style 5-character UM jobid, e.g. 'anqjm'
            # 2. a plain Rose suite name, e.g. 'u-xy123'
            # 3. a Rose suite name + '/' + ensemble member , e.g. 'u-xy123/r1i2p3'
            # 4. a Rose suite name + '_' + jobid, e.g. 'u-xy123_atmos'
            #
            # For historic reasons, options 3 & 4 are hacks. A better solution
            # would be to have each of the above properties separately definable
            # within the app config file.

            mdefn.ens_id = ''
            if '/' in mdefn.name:
                mdefn.suite_id, mdefn.ens_id = mdefn.name.split('/')
            elif '_' in mdefn.name:
                mdefn.suite_id, mdefn.job_id = mdefn.name.split('_')
            else:
                mdefn.suite_id = mdefn.name
            mdefn.data_class = 'ens' if mdefn.ens_id else 'crum'

            # Set run_id using job_id, if that's defined, else parse it from
            # the suite_id value.
            mdefn.run_id = mdefn.job_id or mdefn.suite_id.split('-')[-1]

            mdefn.reinit = int(mdefn.reinit or default_reinit)
            mdefn.zeropad_dates = is_true(mdefn.zeropad_dates or default_zeropad_dates)
            mdefn.time_offset = int(mdefn.time_offset or 0)
            if mdefn.laf_file: mdefn.laf_file = expand_path(mdefn.laf_file)
            mdefn.laf_threshold = float(mdefn.laf_threshold or 0.5)

            _set_model_symbology(mdefn)

            model_defns[mdefn.name] = mdefn

        self.logger.debug("Read in %d model definitions.", len(model_defns))

        # For ensemble setups, check that all model definitions belong to
        # ensemble-based suites.
        msg = ''
        ems = [mdl.ens_id for mdl in model_defns.values() if mdl.enabled]
        if self._is_ensemble_data_cache():
            if not all(ems):
                msg = ("Ensemble-style model data cache has been selected but\n"
                    "1 or more configured models are NOT part of an ensemble.")
        elif any(ems):
            msg = ("1 or more configured models are part of an ensemble but\n"
                "an ensemble-style model data cache has NOT been selected.")
        if msg:
            self.logger.error(msg)
            raise AppConfigError(msg)

        return model_defns

    def _get_diag_definitions(self):
        """
        Read all of the target diagnostic definitions from the DIAGS_NAMELIST
        portion of the configuration file. Diagnostic definitions are stored as
        simple NamespacePlus objects. A potential enhancement would be to use
        Afterburner's MetaVariable classes.
        """
        default_stream = self.app_options['stream']
        default_calendar = self.app_options['calendar']
        default_lbtim = self.app_options['lbtim']
        default_lbproc = self.app_options['lbproc']
        default_lblevs = self.app_options['surface_lblevs']

        diag_defns = {}
        for ddict in self.app_config.iter_nl(DIAGS_NAMELIST):
            ddict['enabled'] = is_true(ddict.get('enabled', 't'))
            ddefn = NamespacePlus(**ddict)

            if not ddefn.stream: ddefn.stream = default_stream
            if not ddefn.calendar: ddefn.calendar = default_calendar
            if not ddefn.region_name: ddefn.region_name = 'Global'
            if not ddefn.statistic: ddefn.statistic = 'awmean'

            # Store the name of the model (UM, NEMO, etc) with which this
            # diagnostic is associated.
            ddefn.model_name = _model_name_from_stream(ddefn.stream)

            # Check that either the stashcode or the var_name option is defined.
            if not bool(ddefn.stashcode) ^ bool(ddefn.var_name):
                msg = ("Invalid definition for diagnostic '{0}'.\n"
                    "Either the 'stashcode' option or the 'var_name' option "
                    "must be defined (but not both).".format(ddefn._index))
                self.logger.error(msg)
                raise AppConfigError(msg)

            # Check that the stashcode option, if defined, is valid.
            if ddefn.stashcode and not is_msi_stash_code(ddefn.stashcode):
                msg = ("Invalid definition for diagnostic '{0}'.\n"
                    "The 'stashcode' option is not in MSI format.".format(
                    ddefn._index))
                self.logger.error(msg)
                raise AppConfigError(msg)

            # If the var_name option defines a STASH code then copy it to the
            # stashcode option. This means that users can standardise on using
            # just the var_name option in the app config file.
            if ddefn.var_name:
                if is_msi_stash_code(ddefn.var_name):
                    ddefn.stashcode = ddefn.var_name
            else:
                ddefn.var_name = ddefn.stashcode

            # Determine if this is a custom diagnostic, i.e. non STASH code and
            # either (a) is one of the predefined custom diagnostics, or (b) has
            # a formula defined.
            ddefn.is_custom = False
            if not is_msi_stash_code(ddefn.var_name):
                if ddefn.var_name in CUSTOM_DIAGNOSTICS or ddefn.formula:
                    ddefn.is_custom = True

            # If auxiliary variables are defined (e.g. for netCDF data sources)
            # then convert from string value to a list of var names.
            if ddefn.aux_var_names:
                ddefn.aux_var_names = ddefn.aux_var_names.split(',')

            # Determine if a land-area xor sea-area correction is to be applied.
            ddefn.apply_laf_corr = is_true(ddefn.apply_laf_corr)
            ddefn.apply_saf_corr = is_true(ddefn.apply_saf_corr)
            if ddefn.apply_laf_corr:
                ddefn.landsea = 'land'
                ddefn.apply_saf_corr = False   # ensure mutual exclusivity
            elif ddefn.apply_saf_corr:
                ddefn.landsea = 'sea'
            else:
                ddefn.landsea = 'land+sea'

            # Define a unique multi-part key for this diagnostic.
            level = 'lev%s' % (ddefn.level or '0')
            key = (ddefn.var_name, level, ddefn.landsea, ddefn.region_name,
                ddefn.statistic)
            if key in diag_defns:
                msg = "Diagnostic definition with key %s is not unique.\n" % str(key)
                msg += "Please check app config file for duplicate definitions."
                self.logger.error(msg)
                raise AppConfigError(msg)

            if not ddefn.name: ddefn.name = '/'.join(key)

            # Coerce lbproc/lbtim values, if defined, from strings to integers.
            for att, val in [('lbproc', default_lbproc), ('lbtim', default_lbtim)]:
                x = getattr(ddefn, att, None)
                setattr(ddefn, att, int(x) if x else val)

            # Coerce surface_lblevs value, if defined, to an integer list.
            if ddefn.surface_lblevs == 'none':
                ddefn.surface_lblevs = None
            else:
                try:
                    ddefn.surface_lblevs = [int(x) for x in ddefn.surface_lblevs.split(',')]
                except:
                    ddefn.surface_lblevs = default_lblevs

            ddefn.graph_order = int(ddefn.graph_order or 0)

            ddefn.key = key
            diag_defns[key] = ddefn

        self.logger.debug("Read in %d diagnostic definitions.", len(diag_defns))

        return diag_defns

    def _get_model_line_style(self, model_defn):
        """
        Returns a dictionary of the line style properties (colour, style, marker)
        for the specified model. See also the :meth:`_set_model_symbology` method.
        """

        line_spec = {
            'alpha': model_defn.transparency,
            'color': model_defn.line_colour,
            'linestyle': model_defn.line_style,
            'linewidth': model_defn.line_width,
            'marker': model_defn.marker_style,
            'markersize': model_defn.marker_size,
            'markeredgecolor': model_defn.marker_edge_colour,
            'markerfacecolor': model_defn.marker_face_colour,
        }

        return line_spec

    def _get_legend_text(self, model_defn, diag_defn=None):
        """
        Generates the text string to appear in the legend of a plot for the
        specified model and, if defined, diagnostic.
        """
        legend_style = self.app_options['legend_style'].lower()
        label = model_defn.label or '?undefined?'
        stream = diag_defn.stream if diag_defn else ''

        if legend_style in ('name+description', 'name+label'):
            if stream:
                legend_text = "{0}/{1} {2}".format(model_defn.name, stream,
                    label)
            else:
                legend_text = "{0} {1}".format(model_defn.name, label)
        elif legend_style in ('description', 'label'):
            legend_text = label
        else:
            legend_text = model_defn.name
            if stream: legend_text += '/' + stream

        return legend_text

    def _check_mass_access(self):
        """If synchronisation with MASS is required then check its availability."""
        if self.app_options['sync_with_mass'] and not moose.has_moose_support():
            msg = ("The runtime environment does not appear to support\n"
                "the MOOSE command interface to the MASS data archive.")
            self.logger.error(msg)
            raise MooseUnavailableError(msg)

    def _check_moose_status(self):
        """If synchronisation with MASS is required then check MOOSE is available."""
        if (self.app_options['sync_with_mass'] and not
            moose.check_moose_commands_enabled(MOOSE_READ)):
            self.logger.warning("The MASS archive is not currently available for "
                "read operations.")
            status = False
        else:
            status = True
        return status

    def _is_ensemble_data_cache(self):
        """Return true if an ensemble-type data cache has been configured."""
        cache_type = self.app_options.get('cache_type', '').lower()
        return cache_type.startswith('ensemble')

    def _init_model_data_cache(self):
        """Initialise a data cache object based on the VarSplit scheme."""
        try:
            cache_type = self.app_options['cache_type']
            cache_dir = self.app_options['cache_dir']
            cache_readonly = self.app_options['cache_readonly']
            data_store = MassDataStore()
            self.logger.info("Initialising %s data cache...", cache_type)
            self.data_cache = DataCache.create_cache(cache_type, data_store,
                cache_dir, read_only=cache_readonly)
        except Exception as exc:
            self.logger.error("Problem trying to initialise data cache rooted "
                "at directory:\n%s.", cache_dir)
            self.logger.error(str(exc))
            raise

    def _create_output_dirs(self):
        """Create any required output directories."""

        for opdir in [self.output_dir, self.nc_output_dir, self.img_output_dir]:
            if not os.path.exists(opdir):
                try:
                    os.makedirs(opdir)
                    self.logger.info("Created output directory: %s", opdir)
                except OSError:
                    self.logger.error("Problem trying to create output directory: %s",
                        opdir)
                    raise

            # Directory exists so do a write permission check.
            elif not os.access(opdir, os.W_OK):
                msg = "You do not have write permission to directory: %s" % opdir
                self.logger.error(msg)
                raise OSError(msg)

    def _clear_nc_cache(self):
        """
        Clears the netCDF data cache of any spatial-mean data files. Currently
        this entails a brute force removal of all files with a .nc extension.
        """
        self.logger.info("Clearing netCDF cache files from %s...", self.nc_output_dir)
        ndel = nerr = 0
        for dpath, _dnames, fnames in os.walk(self.nc_output_dir):
            for fn in fnames:
                if fn.endswith('.nc'):
                    try:
                        os.remove(os.path.join(dpath, fn))
                        ndel += 1
                    except OSError:
                        nerr += 1
        if ndel:
            self.logger.info("Deleted %s .nc files.", ndel)
        if nerr:
            self.logger.info("Unable to delete %s .nc files.", nerr)

    def _refresh_cached_diag_data(self):
        """Refresh the cached netCDF files with the latest meaned model data."""

        for model_name in self.enabled_model_names:
            model_defn = self.model_defns[model_name]
            self.model_data_memcache = {}
            self.logger.info("\nRefreshing data caches for model %s...", model_name)

            for diag_key in self.enabled_diag_keys:
                diag_defn = self.diag_defns[diag_key]

                # Load existing spatial-mean data, if any, from netCDF cache.
                diag_data, diag_dtrange = self._load_cached_diag_data(model_defn,
                    diag_defn)

                # Check to see if this is a custom diagnostic.
                if diag_defn.is_custom:
                    diag_list = _get_custom_diag_defns(diag_defn)
                else:
                    diag_list = [diag_defn]

                # If required, synchronise model data files from MASS.
                model_data = iris.cube.CubeList()
                if self.app_options['sync_with_mass']:
                    for diag in diag_list:
                        cubes = self._update_model_data_cache(model_defn,
                            diag, diag_dtrange)
                        if cubes:
                            model_data.extend(cubes)

                # Otherwise load model data from files in disk cache.
                else:
                    for diag in diag_list:
                        cubes = self._load_latest_model_data(model_defn,
                            diag, diag_dtrange)
                        if cubes:
                            model_data.extend(cubes)

                if not model_data: continue

                try:
                    self.logger.debug("Model data cubes:\n%s", str(model_data))

                    # For a custom diagnostic, replace the input model data with
                    # the computed target diagnostic data.
                    if diag_defn.is_custom:
                        model_data = self._extract_custom_data_subset(model_data,
                            model_defn, diag_list)
                        model_data = self._compute_custom_diagnostic(model_data,
                            diag_defn)
                        # result cube should be first element of returned cubelist
                        model_data = model_data[0]

                    # Extract required model data subset into a single cube.
                    else:
                        model_data = self._extract_model_data_subset(model_data,
                            model_defn, diag_defn)

                    # Apply land (or sea) area fraction correction if requested.
                    if diag_defn.apply_laf_corr or diag_defn.apply_saf_corr:
                        self._apply_laf_correction(model_data, model_defn, diag_defn)

                    # Compute the statistical measure for the model data.
                    smean_data = self._compute_statistic(model_data,
                        model_defn, diag_defn)

                    # Set the var_name attribute equal to that in the diagnostic
                    # definition (either a STASH code or a variable name).
                    smean_data.var_name = diag_defn.var_name

                    # Concatenate with existing statistical data, if present.
                    if diag_data:
                        smean_data = self._concat_mean_data(diag_data, smean_data)

                    # Update netCDF data files.
                    self._update_diag_data_cache(smean_data, model_defn, diag_defn)

                except Exception as exc:
                    # Catch and report errors trying to compute the statistic.
                    # This will prevent the app exiting and ensure that the user
                    # should at least see some useful diagnostic output.
                    self.logger.error("Problem trying to generate %s for "
                        "diagnostic %s/%s", diag_defn.statistic, model_name,
                        diag_defn.name)
                    self.logger.error(str(exc))

    def _load_cached_diag_data(self, model_defn, diag_defn):
        """Load spatial-mean model data from cached netCDF files."""

        ncpath = _make_netcdf_filename(model_defn, diag_defn, dirpath=self.nc_output_dir)

        diag_data = dt_range = None
        if os.path.exists(ncpath):
            self.logger.info("\nLoading cached netCDF data for %s/%s/%s...",
                model_defn.name, diag_defn.stream, diag_defn.name)
            diag_data = _read_cached_diag_data(ncpath)
            if diag_data:
                dt_range = DateTimeRange.from_cube(diag_data, use_bounds=True)
                self.logger.debug("Cube summary:\n%s", diag_data.summary(shorten=True))
                self.logger.debug("Time extent of data in netCDF cache: %s to %s",
                    dt_range.start, dt_range.end)
            else:
                self.logger.warning("Unable to load data from file: %s", ncpath)
        else:
            self.logger.info("No cached netCDF data found.")

        return diag_data, dt_range

    def _update_model_data_cache(self, model_defn, diag_defn, diag_dtrange):
        """Update the on-disk model data cache by retrieving files from MASS."""

        self.logger.info("Updating cached model data for %s/%s/%s...",
            model_defn.name, diag_defn.stream, diag_defn.var_name)

        # Check to see if data for this model/diagnostic combination has already
        # been updated.
        key = (model_defn.name, diag_defn.var_name)
        memcache = self.model_data_memcache.setdefault(key, {})
        visited = memcache.setdefault('visited', False)

        # Get time extent of data currently held in MASS for the current
        # diagnostic.
        if visited:
            mass_dtrange = memcache['mass_dtrange']
        else:
            try:
                if model_defn.start_date and model_defn.end_date:
                    # If both model start and end date have been specified then
                    # use these to constrain MASS data extent. This is to work
                    # around the issue querying very long runs in MASS.
                    mass_dtrange = DateTimeRange(model_defn.start_date, model_defn.end_date)
                else:
                    mass_dtrange = _get_mass_time_extent(model_defn, diag_defn)
                    self.logger.debug("Time extent (T1->T1) of data in MASS: %s to %s",
                        mass_dtrange.start, mass_dtrange.end)
                memcache['mass_dtrange'] = mass_dtrange
            except DataStoreError:
                self.logger.warning("Problem querying time extent of diagnostic "
                    "%s/%s in MASS.", model_defn.name, diag_defn.var_name)
                return None

        # Determine time extent (based on T1 field) of data to fetch from MASS.
        t1_dtrange = _get_mass_fetch_extent(model_defn, diag_defn, diag_dtrange,
            mass_dtrange)

        # Fetch any required files into the model data cache.
        if t1_dtrange:
            cubes = None

            # Create a meta-variable object from model + diag definitions.
            metavar = _create_metavariable(model_defn, diag_defn, time_range=t1_dtrange)

            try:
                # Get a list of actual data files on disk before the refresh.
                files_before = set(self.data_cache.get_filepaths([metavar]))

                # Fetch any new files. Note that the time_range argument passed
                # to the fetch_files() method is needed by StreamSplit-style
                # data caches.
                self.logger.info("Fetching model data files for period %s to %s...",
                    t1_dtrange.start, t1_dtrange.end)
                self.data_cache.fetch_files([metavar], time_range=metavar.time_range)

                # Get a list of data files on disk after the refresh.
                files_after = set(self.data_cache.get_filepaths([metavar]))
                new_files = files_after - files_before
                if new_files: self.new_files_in_cache.update(new_files)
                self.logger.debug("Number of new files retrieved: %d", len(new_files))

                # Load data from any new files.
                futures = afterburner.compare_iris_version('2', 'lt') and \
                    {'cell_datetime_objects': True} or {}
                with iris.FUTURE.context(**futures):
                    cubes = self.data_cache.load_data([metavar], minimal_data=True)
                    _promote_scalar_time_coord(cubes)
                    cubes = self._normalise_cube_attributes(cubes, do_concat=True)

                self.logger.debug("Cubes loaded from MASS:\n%s", str(cubes))

            except DataStoreError:
                self.logger.warning("Problem fetching data files for %s/%s/%s.",
                    model_defn.name, diag_defn.stream, diag_defn.var_name)

        else:
            cubes = None
            self.logger.info("Model data cache is up to date.")

        # Mark the current model/diagnostic combination as visited so that we
        # don't make repeated retrievals of the same files.
        memcache['visited'] = True

        return cubes

    def _load_latest_model_data(self, model_defn, diag_defn, diag_dtrange):
        """Load latest model data from the data cache."""
        self.logger.info("Loading latest model data for %s/%s/%s...",
            model_defn.name, diag_defn.stream, diag_defn.var_name)

        # Get time extent of new model data, if any, within the data cache.
        extent = _get_cached_data_extent(model_defn, diag_defn, diag_dtrange)
        if extent is False:
            self.logger.debug("Data in netCDF cache covers required date range.")
            return None
        elif isinstance(extent, DateTimeRange):
            time_range = (extent.start, extent.end)
            time_cons = None
        else:
            time_cons = extent   # can also be None
            time_range = None

        # Create a meta-variable object from model + diag definitions.
        metavar = _create_metavariable(model_defn, diag_defn, time_range)

        # Load model data into a cubelist.
        futures = afterburner.compare_iris_version('2', 'lt') and \
            {'cell_datetime_objects': True} or {}
        with iris.FUTURE.context(**futures):
            cubes = self.data_cache.load_data([metavar], constraints=time_cons,
                minimal_data=True)
            _promote_scalar_time_coord(cubes)
            cubes = self._normalise_cube_attributes(cubes, do_concat=True)

        self.logger.debug("Cubes loaded from model data cache:\n%s", str(cubes))

        return cubes

    def _normalise_cube_attributes(self, cubes, do_concat=False):
        """
        Normalise selected cube attributes in order to avoid later cube merge or
        concatenation errors. The cubes argument should normally relate to one
        diagnostic/variable, though this is not essential.
        """
        if self.app_options.get('equalise_attributes') and cubes:
            equalise_attributes(cubes)
            atts_changed = True
        elif self.app_options.get('ignoreable_attributes'):
            for cube in cubes:
                for att_name in self.app_options['ignoreable_attributes']:
                    if att_name in cube.attributes:
                        del cube.attributes[att_name]
            atts_changed = True
        else:
            atts_changed = False

        if atts_changed and do_concat and len(cubes) > 1:
            cubes = cubes.concatenate()

        return cubes

    def _compute_custom_diagnostic(self, model_data, diag_defn):
        """
        Compute the custom diagnostic defined by diag_defn using the supplied
        model source data.
        """
        self.logger.info("Computing custom diagnostic %s...", diag_defn.var_name)

        # Check that all source cubes have equal time axes.
        if not are_time_axes_equal(model_data):
            msg = "Source cubes for custom diagnostic '%s' have unequal time axes." \
                % diag_defn.var_name
            self.logger.error(msg)
            raise DataProcessingError(msg)

        metadata = {}
        for att_name in ['standard_name', 'long_name', 'var_name', 'units']:
            value = getattr(diag_defn, att_name, None)
            if value: metadata[att_name] = value

        # Instantiate the diagnostic processor class, then run it.
        if diag_defn.formula:
            diag_proc = diag_defn.cls(diag_defn.formula, result_metadata=metadata)
        else:
            diag_proc = diag_defn.cls(result_metadata=metadata)

        cubes = diag_proc.run(model_data)
        if cubes:
            self.logger.debug("Cube summary:\n%s", cubes[0].summary(shorten=True))

        return cubes

    def _extract_custom_data_subset(self, cubes, model_defn, diag_defns):
        """
        Extract the required subset of model data needed as input to the custom
        diagnostic generation task. The returned cubelist should be a stripped
        down version of the input cubelist, i.e. with any diurnal mean fields
        and unneeded coordinates removed, and subsetted to a particular vertical
        level, if one has been specified as part of the diagnostic definition.
        """
        cube_subset = iris.cube.CubeList()

        for diag_defn in diag_defns:
            diag_cubes = cubes.extract(iris.AttributeConstraint(STASH=diag_defn.stashcode))
            cube = self._extract_model_data_subset(diag_cubes, model_defn, diag_defn)
            cube_subset.append(cube)

        return cube_subset

    def _extract_model_data_subset(self, cubes, model_defn, diag_defn):
        """Extract the required subset of model data for the specified diagnostic"""
        self.logger.info("Extracting model data subset for %s/%s...",
            model_defn.name, diag_defn.name)

        # Extract cubes which correspond to data averaged over all time steps.
        if len(cubes) > 1:
            mean_constraint = iris.Constraint(cube_func=is_mean_of_all_time_steps)
            tmp_cubes = cubes.extract(mean_constraint)
            if tmp_cubes and len(tmp_cubes) == 1:
                model_data = tmp_cubes[0]
            else:
                raise DataProcessingError("Error trying to extract mean data "
                    "(over all time steps) from the following cubes:\n" +
                    str(cubes))
        else:
            model_data = cubes[0]

        # If a vertical level coordinate was defined for the current diagnostic
        # then extract it.
        if diag_defn.level:
            self.logger.info("Extracting vertical level %s...", diag_defn.level)
            model_data = _extract_vertical_level(model_data, diag_defn)
        self.logger.debug("Cube summary:\n%s", model_data.summary(shorten=True))

        # Remove unwanted coordinates.
        removed = _remove_redundant_coords(model_data)
        if removed:
            self.logger.debug("Removed the following coordinates:\n%s", removed)

        return model_data

    def _apply_laf_correction(self, model_data, model_defn, diag_defn):
        """Apply a land area correction in situ to the specified cube."""
        self.logger.info("Applying land area correction for %s/%s...",
            model_defn.name, diag_defn.name)

        #laf_cube = _read_laf_data(expand_path(model_defn.laf_file))
        laf_cube = self._load_laf_data(model_defn, diag_defn)
        if not laf_cube:
            raise DataProcessingError("Unable to retrieve land area fraction data "
                "from file: " + model_defn.laf_file)

        if laf_cube.shape != model_data.shape[-2:]:
            raise DataProcessingError("Shape of land area fraction data {0}\n"
                "does not match that of diagnostic {1}.".format(laf_cube.shape,
                model_data.shape[-2:]))

        if diag_defn.apply_laf_corr:
            frac_data = laf_cube.data
            if not ma.is_masked(frac_data):
                # mask where frac == 0 or frac < threshold
                frac_data = ma.masked_where(
                    (frac_data == 0) | (frac_data < model_defn.laf_threshold),
                    frac_data)
        else:
            # convert land-area fraction to sea-area fraction
            self.logger.debug("Converting LAF array to SAF array...")
            frac_data = _convert_laf_to_saf(laf_cube.data, model_defn.laf_threshold)

        model_data.data = model_data.data * frac_data

    def _load_laf_data(self, model_defn, diag_defn):
        """
        Load land area fraction data, either from a user-specified file (such as
        an ancillary file), or else from a suitable PP file retrieved from MASS.
        """

        # Check that one or other of the laf_file or laf_stashcode options is
        # defined.
        if not (model_defn.laf_file or model_defn.laf_stashcode):
            raise AppConfigError("No land area fraction file or STASH code "
                "defined for model {0}.".format(model_defn.name))

        # If laf_file option has not yet been assigned then try to fetch a
        # suitable PP file from MASS, as dictated by the laf_stashcode option.
        if not model_defn.laf_file:
            laf_metavar = UmMetaVariable(model_defn.um_vn, model_defn.suite_id,
                realization_id=model_defn.ens_id, stream_id=diag_defn.stream,
                stash_code=model_defn.laf_stashcode)

            filepaths = self.data_cache.get_filepaths([laf_metavar])
            if filepaths:
                # At least one LAF pp file is present in the model data cache,
                # so use the first one from the list.
                model_defn.laf_file = filepaths[0]

            else:
                # No LAF pp file(s) present in the model data cache, so try to
                # fetch one from MASS.
                dest_dir = os.path.join(self.nc_output_dir, 'laf')
                if not os.path.exists(dest_dir): os.makedirs(dest_dir)
                model_defn.laf_file = _fetch_laf_file_from_mass(dest_dir,
                    model_defn, diag_defn)

        try:
            self.logger.info("Loading LAF data from file %s...", model_defn.laf_file)
            laf_cube = _read_cached_laf_data(model_defn.laf_file,
                model_defn.laf_stashcode)
            self.logger.debug("LAF cube:\n%s", laf_cube.summary(shorten=True))
        except iris.exceptions.IrisError:
            msg = ("Error encountered trying to load LAF data from file "
                   "{0}...".format(model_defn.laf_file))
            self.logger.error(msg)
            raise

        return laf_cube

    def _compute_statistic(self, model_data, model_defn, diag_defn):
        """Compute the required statistic for the specified diagnostic"""
        self.logger.info("Calculating %s for %s/%s...", diag_defn.statistic,
            model_defn.name, diag_defn.name)

        # Ensure that lat/long coordinates have bounds.
        # NB: this can only be done for 1-dimensional coordinates.
        latcrd, loncrd = _get_lat_long_coords(model_data)
        for coord in [latcrd, loncrd]:
            if not coord.has_bounds():
                if coord.ndim == 1:
                    coord.guess_bounds()
                else:
                    self.logger.warning("Unable to guess bounds for multi-dimensional "
                        "%s coordinate.", coord.name())

        if diag_defn.region_name.lower() == 'global':
            region = None
            cube = model_data
        else:
            region = list(map(float, diag_defn.region_extent.split(',')))
            if len(region) != 4:
                raise AppConfigError("Region definition must comprise four "
                    " coordinates: found %d" % len(region))
            minlon, minlat, maxlon, maxlat = region[:]
            mininc, maxinc = _interval_type_as_iris_boolean_flags(
                self.app_options['treat_region_coords_as'])
            if latcrd.standard_name == 'grid_latitude':
                cube = model_data.intersection(
                    grid_latitude=(minlat, maxlat, mininc, maxinc),
                    grid_longitude=(minlon, maxlon, mininc, maxinc),
                    ignore_bounds=True)
            else:
                cube = model_data.intersection(
                    latitude=(minlat, maxlat, mininc, maxinc),
                    longitude=(minlon, maxlon, mininc, maxinc),
                    ignore_bounds=True)

        try:
            func_name = STAT_METHODS[diag_defn.statistic]['func']
            func = globals()[func_name]
            self.logger.debug("Calling statistical function '%s'...", func_name)
            stat_cube = func(cube, diag_defn)
            if hasattr(stat_cube, 'cell_measures'):
                for cm in stat_cube.cell_measures():
                    stat_cube.remove_cell_measure(cm)
            self.logger.debug("Cube summary:\n%s", stat_cube.summary(shorten=True))
        except:
            msg = "Error encountered trying to calculate {0} for {1}/{2}.".format(
                diag_defn.statistic, model_defn.name, diag_defn.name)
            self.logger.error(msg)
            raise

        return stat_cube

    def _concat_mean_data(self, old_mean_data, new_mean_data):
        """Concatenate the existing (on-disk) and new spatial-mean data cubes."""
        self.logger.info("Concatenating computed statistics values...")

        # Take care of potential diffs in cube.*_name attributes. Assume that
        # those in the cube of newly loaded data take precedence.
        for att_name in ['standard_name', 'long_name', 'var_name']:
            setattr(old_mean_data, att_name, getattr(new_mean_data, att_name))

        mean_cubes = iris.cube.CubeList([old_mean_data, new_mean_data])
        contiguous = _make_data_contiguous(mean_cubes)

        # Adjust cube metadata so that they are amenable to concatenation.
        prepared = _prep_data_for_concatenation(contiguous)

        try:
            concatenated = prepared.concatenate_cube()
        except iris.exceptions.ConcatenateError as exc:
            msg = "Problem trying to concatenate the following cubes:\n"
            for cube in prepared:
                msg += cube.summary(shorten=True) + '\n'
            self.logger.error(msg)
            self.logger.error(str(exc))
            if self.logger.level == logging.DEBUG:
                compare_cubes(prepared[0], prepared[1], sys.stderr)
            raise

        self.logger.debug("Cube summary:\n%s", concatenated.summary(shorten=True))

        return concatenated

    def _update_diag_data_cache(self, mean_data, model_defn, diag_defn):
        """Update the on-disk, netCDF-based data cache."""
        self.logger.info("Updating netCDF data cache for %s/%s...",
            model_defn.name, diag_defn.name)

        mean_data.attributes['region'] = diag_defn.region_name
        set_history_attribute(mean_data, "Statistical data generated by the "
            "Climate Model Monitor v2 application.")

        ncpath = _make_netcdf_filename(model_defn, diag_defn, dirpath=self.nc_output_dir)
        dirpath = os.path.dirname(ncpath)
        if not os.path.exists(dirpath): os.makedirs(dirpath)
        tmppath = os.path.splitext(ncpath)[0] + '.tmp'

        try:
            futures = afterburner.compare_iris_version('2', 'lt') and \
                {'netcdf_no_unlimited': True} or {}
            with iris.FUTURE.context(**futures):
                iris.save(mean_data, tmppath, saver=iris.fileformats.netcdf.save)
            if os.path.exists(ncpath):
                # not strictly necessary as the rename command should do this
                os.remove(ncpath)
            os.rename(tmppath, ncpath)
            self.logger.info("Statistical data saved to netCDF file: %s", ncpath)
        except (OSError, iris.exceptions.IrisError) as exc:
            self.logger.warning("Unable to save Statistical data to file: %s", ncpath)
            self.logger.warning(str(exc))
        finally:
            if os.path.exists(tmppath): os.remove(tmppath)

    def _generate_plot_images(self):
        """Generate time-series plots for each model-diagnostic combination."""
        self.logger.info("\nGenerating time-series plots...")
        legvis = self.app_options['legend_vis'].lower()

        for ndiag, key in enumerate(self.enabled_diag_keys):
            diag_defn = self.diag_defns[key]
            self.logger.info("Generating plot image for diagnostic %s...",
                diag_defn.name)

            fig, axes = plt.subplots()
            fig.set_size_inches(self.plot_size)

            # See if we need to add a legend to the current plot.
            legend = bool(legvis == 'all' or (legvis == 'first' and ndiag == 0))

            dt_range = self._plot_data_series(axes, diag_defn, add_legend=legend)
            self._plot_annotation(axes, diag_defn, dt_range)

            # Save the plot image to file.
            try:
                imgpath = _make_image_filename(diag_defn, dirpath=self.img_output_dir)
                dirpath = os.path.dirname(imgpath)
                if not os.path.exists(dirpath): os.makedirs(dirpath)
                plt.savefig(imgpath)
            except IOError as exc:
                self.logger.warning("Unable to save plot for diagnostic %s "
                    "to file:\n%s", diag_defn.name, imgpath)
                self.logger.warning(str(exc))

            plt.close(fig)

        # Create the plot legend in a separate image file.
        if legvis == 'extra':
            self._plot_legend()

    def _plot_data_series(self, axes, diag_defn, add_legend=False):
        """Plot the data series for a diagnostic for all enabled models."""
        plotted_something = False
        diag_label = diag_defn.var_name
        stat_label = STAT_METHODS[diag_defn.statistic]['label']
        title_text = ''
        dt_range = None

        for model_name in self.enabled_model_names:
            model_defn = self.model_defns[model_name]
            stat_file = _make_netcdf_filename(model_defn, diag_defn, dirpath=self.nc_output_dir)
            if os.path.exists(stat_file):
                smean_data = _read_cached_diag_data(stat_file)
                if not smean_data: continue
            else:
                self.logger.warning("NetCDF file not found: %s", stat_file)
                continue

            # Update overall date-time range.
            dtr = DateTimeRange.from_cube(smean_data)
            if dtr:
                if dt_range:
                    if pdt_compare(dtr.start_pdt, 'lt', dt_range.start_pdt):
                        dt_range.start = dtr.start
                    if pdt_compare(dtr.end_pdt, 'lt', dt_range.end_pdt):
                        dt_range.end = dtr.end
                else:
                    dt_range = dtr

            try:
                linestyle_args = self._get_model_line_style(model_defn)
                label = self._get_legend_text(model_defn, diag_defn=diag_defn)
                tcoord = smean_data.coord('time')
                if model_defn.time_offset:
                    tunits = _offset_time_origin(tcoord.units, model_defn.time_offset)
                    tcoord = iris.coords.DimCoord(points=tcoord.points,
                        standard_name='time', units=tunits)
                    #tcoord.guess_bounds()
                qplt.plot(tcoord, smean_data, label=label, **linestyle_args)
                plotted_something = True
                diag_label = smean_data.name()
            except ValueError as exc:
                self.logger.warning("Unable to generate plot for model %s, "
                    "diagnostic %s.", model_name, diag_defn.name)
                self.logger.warning(str(exc))
                self.logger.debug("Final cube:\n%s", smean_data.summary())

        # Update time axis label.
        axes.set_xlabel('Model Time')

        if plotted_something:
            plt.grid(True)
            if add_legend:
                plt.legend(loc=1, fontsize=10, framealpha=0.5, fancybox=True)
            title_text = diag_label.replace('_', ' ').title()
            title_text = '{0} of {1}'.format(stat_label, title_text)
            if self.app_options['include_stash']:
                title_text += ' (' + diag_defn.var_name + ')'
            plt.title(title_text)
        else:
            title_text = '{0} of {1}'.format(stat_label, diag_label)
            plt.title(title_text)
            axes.text(0.5, 0.5, 'No Data Available', verticalalignment='center',
                horizontalalignment='center', transform=axes.transAxes,
                fontsize=14)

        # Return the overall time range spanned by the current diagnostic.
        return dt_range

    def _plot_annotation(self, axes, diag_defn, dt_range=None):
        """Add annotation to the time-series plot for the specified diagnostic."""
        if self.app_options['include_date']:
            time_str = 'Graph generated at: ' + _utc_now_string().replace('T', ' ')
            axes.text(0.98, 0.05, time_str, verticalalignment='center',
                horizontalalignment='right', transform=axes.transAxes,
                color='darkgray', fontsize=10)

        # Add a label for the geographical region and, if set, vertical level.
        if diag_defn.region_name.lower() == 'global':
            region_text = 'Region: Globe / Full Extent'
        else:
            region_text = 'Region: ' + diag_defn.region_name.title()

        if self.app_options['include_region'] and diag_defn.region_extent:
            coords = diag_defn.region_extent.split(',')
            region_text += " (Lat:{1},{3}; Long:{0},{2})".format(*coords)

        if diag_defn.level:
            region_text += '\nLevel: ' + diag_defn.level

        if diag_defn.apply_laf_corr:
            region_text += '\nLand-area correction: Yes'
        elif diag_defn.apply_saf_corr:
            region_text += '\nSea-area correction: Yes'

        axes.text(0.02, 0.95, region_text, horizontalalignment='left',
            verticalalignment='center', transform=axes.transAxes, fontsize=10,
            color='gray')

        # set the y-axis (diagnostic data value) limits
        if diag_defn.ymin and diag_defn.ymax:
            ymin = float(diag_defn.ymin)
            ymax = float(diag_defn.ymax)
            axes.set_ylim(ymin, ymax)
            # if the limits are large or small then use scientific notation
            if ymax < 0.01 or ymax >= 10000:
                axes.yaxis.set_major_formatter(mtick.FormatStrFormatter('%.2e'))

        # set the x-axis (time coordinate) limits
        if diag_defn.xmin or diag_defn.xmax:
            xmin = xmax = None

            if diag_defn.xmin:
                xmin = int(diag_defn.xmin)
            elif dt_range:
                xmin = dt_range.start_pdt.year

            if diag_defn.xmax:
                xmax = int(diag_defn.xmax)
            elif dt_range:
                xmax = dt_range.end_pdt.year

            if not None in (xmin, xmax):
                try:
                    from nc_time_axis import CalendarDateTime
                    start_date = cft.datetime(xmin, 1, 1)
                    end_date = cft.datetime(xmax, 1, 1)
                    axes.set_xlim(CalendarDateTime(start_date, diag_defn.calendar),
                        CalendarDateTime(end_date, diag_defn.calendar))
                except (ImportError, ValueError, AttributeError) as exc:
                    self.logger.debug(str(exc))
                    start_date = datetime.datetime(xmin, 1, 1)
                    end_date = datetime.datetime(xmax, 1, 1)
                    axes.set_xlim(start_date, end_date)

    def _plot_legend(self):
        """Create the plot legend as a separate figure."""
        self.logger.info("Generating plot legend as separate figure...")

        fig, ax = plt.subplots()
        fig.set_size_inches(self.plot_size)
        plt.title('Key To Plotted Climate Models')

        try:
            ax.set_facecolor('none')
        except AttributeError:
            ax.set_axis_bgcolor('none')   # deprecated at MPL v2.0

        ax.get_xaxis().set_visible(0)
        ax.get_yaxis().set_visible(0)

        for model_name in self.enabled_model_names:
            model_defn = self.model_defns[model_name]
            linestyle_args = self._get_model_line_style(model_defn)
            label = self._get_legend_text(model_defn)
            plt.plot([0], [0], label=label, **linestyle_args)

        plt.legend(loc=2, ncol=2, fontsize=10, fancybox=True)

        try:
            imgfile = 'cmm_plot_legend.png'
            imgpath = os.path.join(self.img_output_dir, imgfile)
            plt.savefig(imgpath)
        except IOError as exc:
            self.logger.warning("Unable to save plot legend image to file:\n%s", imgpath)
            self.logger.warning(str(exc))

        plt.close(fig)

    def _generate_html_page(self):
        """Generate an HTML page containing the plots."""
        self.logger.info("\nGenerating HTML page...")

        # Load the jinja-based HTML template for the CMM app.
        ab_config = ConfigProvider()
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(ab_config.template_dir))
        template = env.get_template(HTML_TEMPLATE)

        # Create the HTML output file.
        try:
            htmlpath = os.path.join(self.html_output_dir, 'cmm.html')
            fh = open(htmlpath, 'w')
        except IOError:
            self.logger.error("Unable to create HTML output file: " + htmlpath)
            raise

        try:
            # Build a list of image file paths
            imgdir = os.path.basename(self.img_output_dir)
            image_paths = []
            for key in self.enabled_diag_keys:
                diag_defn = self.diag_defns[key]
                imgfile = _make_image_filename(diag_defn)
                image_paths.append(os.path.join(imgdir, imgfile))

            # Append separate legend image file if it was created.
            if self.app_options['legend_vis'].lower() == 'extra':
                image_paths.append(os.path.join(imgdir, 'cmm_plot_legend.png'))

            # Convert the list of filenames into a Nx2 array.
            first_in_row = True
            table = []
            for item in image_paths:
                if first_in_row:
                    row = [item]
                else:
                    row.append(item)
                    table.append(row)
                first_in_row = not first_in_row

            if not first_in_row:
                row.append(None)
                table.append(row)

            # process the template and write to file
            app_name = 'Climate Model Monitor v' + self.version
            vn, rev = afterburner.__version__, afterburner.__version_info__[4]
            timestamp = _utc_now_string().replace('T', ' ')
            fh.write(template.render(table=table, version=vn, rev=rev,
                timestamp=timestamp, app_name=app_name))
            self.logger.info("Wrote HTML file: " + htmlpath)

        except Exception as exc:
            self.logger.error("Problem trying to generate HTML page.")
            self.logger.error(str(exc))
            raise

        finally:
            fh.close()

    def _teardown(self):
        """Perform teardown operations."""

        if self.app_options.get('clear_model_cache') and self.new_files_in_cache:
            nfiles = len(self.new_files_in_cache)
            self.logger.info("Deleting %d files from model data cache...", nfiles)
            for fpath in self.new_files_in_cache:
                try:
                    os.remove(fpath)
                except OSError:
                    self.logger.warning("Unable to delete file: " + fpath)


def _model_name_from_stream(stream_id):
    """Return the name of the parent climate model associated with stream_id."""

    # NB: The decision logic here is very simplistic. It should be amended
    # if and when a more robust method presents itself.
    if stream_id.startswith('a'):
        model_name = afterburner.MODEL_UM
    elif stream_id.startswith('o'):
        model_name = afterburner.MODEL_NEMO
    elif stream_id.startswith('i'):
        model_name = afterburner.MODEL_CICE
    else:
        model_name = ''

    return model_name


def _create_metavariable(model_defn, diag_defn, time_range=None):
    """
    Create a MetaVariable object from the given model and diagnostic definitions.
    """
    if diag_defn.model_name == afterburner.MODEL_UM:
        model_vn = model_defn.um_vn or '0'
        metavar = UmMetaVariable(model_vn, model_defn.suite_id,
            run_id=model_defn.run_id,
            realization_id=model_defn.ens_id, stream_id=diag_defn.stream,
            stash_code=diag_defn.stashcode, lbproc=diag_defn.lbproc,
            lbtim=diag_defn.lbtim, time_range=time_range,
            calendar=diag_defn.calendar, reinit=model_defn.reinit,
            zeropad_dates=model_defn.zeropad_dates,
            postproc_vn=model_defn.postproc_vn)

    elif diag_defn.model_name == afterburner.MODEL_NEMO:
        model_vn = model_defn.nemo_vn or '0'
        metavar = NemoMetaVariable(model_vn, model_defn.suite_id,
            run_id=model_defn.run_id,
            realization_id=model_defn.ens_id, stream_id=diag_defn.stream,
            var_name=diag_defn.var_name, aux_var_names=diag_defn.aux_var_names,
            time_range=time_range, calendar=diag_defn.calendar,
            grid_type=diag_defn.grid_type or 'T',
            postproc_vn=model_defn.postproc_vn)

    elif diag_defn.model_name == afterburner.MODEL_CICE:
        model_vn = model_defn.cice_vn or '0'
        metavar = CiceMetaVariable(model_vn, model_defn.suite_id,
            run_id=model_defn.run_id,
            realization_id=model_defn.ens_id, stream_id=diag_defn.stream,
            var_name=diag_defn.var_name, aux_var_names=diag_defn.aux_var_names,
            time_range=time_range, calendar=diag_defn.calendar,
            grid_type=diag_defn.grid_type or 'T',
            postproc_vn=model_defn.postproc_vn)

    else:
        raise AppConfigError("Unsupported climate model name: {0}.".format(
            diag_defn.model_name))

    return metavar


def _make_netcdf_filename(model_defn, diag_defn, sep='_', ext='.nc', dirpath=''):
    """
    Returns the name of a NetCDF file based upon runid, stream, variable name,
    level, and region name, e.g. '<runid>_<stream>_<stash>_<region>.nc'. Extra
    tokens are included in the case of ensemble runs and level-specific fields.
    """
    region = diag_defn.region_name or 'global'
    region = ''.join(x for x in region.lower() if x.isalnum())

    runid = model_defn.suite_id.split('-')[-1]
    if model_defn.ens_id:
        runid += '_' + model_defn.ens_id
    elif model_defn.job_id:
        runid += '_' + model_defn.job_id

    level = landsea = None

    if diag_defn.level:
        level = 'lev' + diag_defn.level

    if diag_defn.landsea != 'land+sea':
        landsea = diag_defn.landsea

    fn = sep.join(filter(lambda x: x is not None, [runid, diag_defn.stream,
        diag_defn.var_name, level, landsea, region])) + ext

    # Prepend path with a directory named after the statistic, e.g. 'awmean'
    if diag_defn.statistic:
        fn = os.path.join(diag_defn.statistic, fn)

    # Prepend with dirpath, if supplied.
    if dirpath:
        fn = os.path.join(dirpath, fn)

    return fn


def _make_image_filename(diag_defn, sep='_', ext='.png', dirpath=''):
    """
    Returns the name of an image file based upon stream, variable name,
    level, and region name, i.e. '<stream>_<stash>_<region>.png'.
    """
    region = diag_defn.region_name or 'global'
    region = ''.join(x for x in region.lower() if x.isalnum())

    level = landsea = None

    if diag_defn.level:
        level = 'lev' + diag_defn.level

    if diag_defn.landsea != 'land+sea':
        landsea = diag_defn.landsea

    fn = sep.join(filter(lambda x: x is not None, [diag_defn.stream,
        diag_defn.var_name, level, landsea, region])) + ext

    # Prepend path with a directory named after the statistic, e.g. 'awmean'
    if diag_defn.statistic:
        fn = os.path.join(diag_defn.statistic, fn)

    # Prepend with dirpath, if supplied.
    if dirpath:
        fn = os.path.join(dirpath, fn)

    return fn


# NB: disabled LRU caching feature in order to pick up expected file updates.
def _read_cached_diag_data(stat_file):
    """
    Caching function which reads a statistical field from the specified netCDF
    file, which is assumed to yield a single cube on loading. If the field has
    previously been read and cached then it is returned, thus avoiding the
    overhead of reloading the data. Returns None if an error was encountered.
    """
    try:
        futures = afterburner.compare_iris_version('2', 'lt') and \
            {'netcdf_promote': True} or {}
        with iris.FUTURE.context(**futures):
            stat_data = iris.load_cube(stat_file)
    except (IOError, iris.exceptions.IrisError) as exc:
        _logger.error(str(exc))
        stat_data = None

    return stat_data


@lru_cache()
def _read_cached_laf_data(laf_file, laf_stash_code):
    """
    Caching function which reads land area fraction data from the specified file,
    which could be in UM fieldsfile, pp or netcdf format. If the LAF field has
    previously been read and cached then it is returned, thus avoiding the
    overhead of reloading the data. Returns None if an error was encountered.
    """
    try:
        # If the passed-in file contains a land area fraction field then the
        # following code should load 1 or 2 cubes. In the latter case they'll
        # typically refer to the same field.
        cons = iris.AttributeConstraint(STASH=laf_stash_code or LAF_STASH_CODE)
        futures = afterburner.compare_iris_version('2', 'lt') and \
            {'netcdf_promote': True} or {}
        with iris.FUTURE.context(**futures):
            laf_cubes = iris.load(laf_file, ['land_area_fraction', cons])
        if laf_cubes:
            laf_data = laf_cubes[0]
            laf_data.units = '1'
            laf_data.var_name = 'land_area_fraction'
        else:
            laf_data = None
    except (IOError, iris.exceptions.IrisError) as exc:
        _logger.error(str(exc))
        laf_data = None

    return laf_data


def _fetch_laf_file_from_mass(dest_dir, model_defn, diag_defn):
    """
    Fetch a file of land area fraction data from MASS for the model run and LAF
    STASH code combination specified via the model_defn & diag_defn arguments.
    Deposit the resulting file, converted to netCDF format, in dest_dir.
    """

    # If a LAF file already exists, return its pathname.
    laf_ncfile = "{0}_{1}.nc".format(model_defn.run_id, model_defn.laf_stashcode)
    laf_ncfile = os.path.join(dest_dir, laf_ncfile)
    if os.path.isfile(laf_ncfile): return laf_ncfile

    # Check to see if there are any files in MASS for the specified model and
    # STASH code.
    try:
        collection = mass_collection_from_stream(afterburner.MODEL_UM,
            diag_defn.stream)
        filelist = moose.metadata_list_struct(model_defn.suite_id,
            collection, stashcodes=[model_defn.laf_stashcode], sort='T1')
        moose_uri = os.path.dirname(filelist[0])
        ppfile = os.path.basename(filelist[0])
        laf_ppfile = os.path.join(dest_dir, ppfile)
    except DataStoreError:
        msg = ("No PP files containing LAF STASH code {0} could be "
               "found in MASS.".format(model_defn.laf_stashcode))
        _logger.error(msg)
        raise DataProcessingError(msg)

    try:
        _logger.info("Fetching LAF file %s from %s...", ppfile, moose_uri)
        moose.retrieve_files(dest_dir, moose_uri, files=[ppfile],
            stashcodes=[model_defn.laf_stashcode], fill_gaps=True)
        futures = afterburner.compare_iris_version('2', 'lt') and \
            {'netcdf_promote': True, 'netcdf_no_unlimited': True} or {}
        with iris.FUTURE.context(**futures):
            cube = iris.load_cube(laf_ppfile)
        cube.standard_name = 'land_area_fraction'
        cube.var_name = 'land_area_fraction'
        cube.units = '1'
        _logger.info("Saving LAF data to file %s...", laf_ncfile)
        with iris.FUTURE.context(**futures):
            iris.save(cube, laf_ncfile)
        os.remove(laf_ppfile)

    except DataStoreError:
        msg = ("Unable to fetch LAF data for suite/run {0}, STASH code {1} "
            "from MASS.".format(model_defn.suite_id, model_defn.laf_stashcode))
        _logger.error(msg)
        raise DataProcessingError(msg)

    except iris.exceptions.IrisError:
        msg = "Problem trying to load LAF data from file " + laf_ppfile
        _logger.error(msg)
        raise

    return laf_ncfile


def _convert_laf_to_saf(laf_data, threshold=None):
    """
    Convert a land-area fraction data array to a sea-area fraction data array.
    If the LAF data array is masked then we assume that sea grid points are
    represented by masked values, and that the returned array should therefore
    have land-only grid points masked out.

    Ordinarily the type of the returned array (masked or unmasked) is the same
    as the input array. However, if the threshold argument is used to define a
    land-sea threshold then the returned array will always be masked.
    """

    if ma.is_masked(laf_data):
        saf_data = 1 - ma.filled(laf_data, 0)     # convert to sea-area fraction
        saf_data = ma.masked_equal(saf_data, 0)   # mask out land-only points
    else:
        saf_data = 1 - laf_data                   # convert to sea-area fraction

    if threshold is not None:
        # mask out *land* points where frac <= 1-threshold
        saf_data = ma.masked_where((saf_data == 0) | (saf_data < 1-threshold),
            saf_data)

    return saf_data


def _calc_cube_area_weights(cube):
    """Calculate area weights for a cube."""

    # First check to see if the cube possesses a cell measure named 'area'.
    area_cms = None
    if hasattr(cube, 'cell_measures'):
        area_cms = [cm for cm in cube.cell_measures() if cm.measure == 'area']

    if area_cms:
        area = area_cms[0]
        weights = np.tile(area.data, (cube.shape[0],) + (1,)*area.ndim)
        #weights = np.broadcast_to(area.data, cube.shape)   # requires numpy 1.10+
        #weights = iris.util.broadcast_to_shape(area.data, (cube.shape[0],)+area.shape, range(1,area.ndim+1))
        _logger.debug("Shape of weights array: %r", weights.shape)

    # If not then try the usual method of generating weights from the latitude
    # and longitude axes (which must be full dimension coords, not aux coords).
    else:
        weights = iris.analysis.cartography.area_weights(cube)

    return weights


def _calc_mean(cube, diag_defn):
    """
    Calculates the area-weighted spatial mean of the passed-in cube, which
    is assumed to be dimensioned (time, lat, long).
    """
    # Calculate area weights if required.
    if diag_defn.statistic == 'awmean':
        weights = _calc_cube_area_weights(cube)
    else:
        weights = None

    # Calculate the mean of the input data.
    mean_cube = cube.collapsed(_get_lat_long_coords(cube), iris.analysis.MEAN,
        weights=weights)

    return mean_cube


def _calc_sum(cube, diag_defn):
    """
    Calculates the sum of the data payload of the passed-in cube, which
    is assumed to be dimensioned (time, lat, long).
    """

    # Calculate area weights if required.
    if diag_defn.statistic == 'awsum':
        weights = _calc_cube_area_weights(cube)
    else:
        weights = None

    # Calculate the sum of the input data.
    sum_cube = cube.collapsed(_get_lat_long_coords(cube), iris.analysis.SUM,
        weights=weights)

    # If present, remove the 'm-2' part from the units definition.
    if weights is not None and sum_cube.units.origin:
        parts = sum_cube.units.origin.split()
        if 'm-2' in parts and len(parts) > 1:
            parts.remove('m-2')
            sum_cube.units = ' '.join(parts)

    return sum_cube


def _calc_min(cube, diag_defn):
    """
    Calculates the minimum of the data payload of the passed-in cube, which
    is assumed to be dimensioned (time, lat, long).
    """
    # Calculate the minimum of the input data.
    min_cube = cube.collapsed(_get_lat_long_coords(cube), iris.analysis.MIN)

    return min_cube


def _calc_max(cube, diag_defn):
    """
    Calculates the maximum of the data payload of the passed-in cube, which
    is assumed to be dimensioned (time, lat, long).
    """
    # Calculate the maximum of the input data.
    max_cube = cube.collapsed(_get_lat_long_coords(cube), iris.analysis.MAX)

    return max_cube


def _calc_nao_index(cube, diag_defn):
    """
    Calculates the NAO Index of the data payload of the passed-in cube, which
    is assumed to be dimensioned (time, lat, long).
    """
    try:
        kwargs = dict(mslp_stashcode=diag_defn.stashcode)
        if diag_defn.interp_method:
            kwargs['interp_method'] = diag_defn.interp_method
        for stn in ['azores_station', 'iceland_station']:
            coords = getattr(diag_defn, stn, None)
            if coords: kwargs[stn] = [float(x) for x in coords.split(',')]
        proc = NaoIndex(**kwargs)
        result = proc.run(iris.cube.CubeList([cube]))
        nao_cube = result[0]
    except DataProcessingError:
        _logger.error("Error encountered running NAO Index processor object.")
        raise

    return nao_cube


def _extract_vertical_level(cube, diag_defn):
    """
    Extract a single vertical level from ``cube``. The level to extract is
    specified by the ``diag_defn.level`` attribute.
    """
    try:
        level = int(diag_defn.level)
    except ValueError:
        level = float(diag_defn.level)

    level_coord = None
    try:
        # First search for a dimension coordinate marked as the Z axis.
        level_coord = cube.coord(axis='Z', dim_coords=True)
    except iris.exceptions.CoordinateNotFoundError:
        # Otherwise search for a coordinate with a recognised level name.
        for coord in cube.coords():
            if coord.name() in LEVEL_COORD_NAMES:
                level_coord = coord
                break

    if not level_coord:
        msg = "Unable to determine vertical coordinate for cube:\n{0}.".format(
            cube.summary(shorten=True))
        raise iris.exceptions.CoordinateNotFoundError(msg)

    level_constraint = iris.Constraint(**{level_coord.name(): level})
    return cube.extract(level_constraint)


def _remove_redundant_coords(cube):
    """
    Remove non-essential coordinates from ``cube``. This should prevent a number
    of subsequent cube concatenation issues, plus reduce output file size. Note,
    however, that scalar coordinates, such as level number in the case of a
    vertical subset, are also removed.
    """
    removed_coord_names = []
    dim_coords_to_keep = ['time', 'latitude', 'longitude', 'grid_latitude', 'grid_longitude']

    # First remove any auxiliary coordinates.
    for coord in cube.coords(dim_coords=False):
        if coord.standard_name not in dim_coords_to_keep:
            removed_coord_names.append(coord.name())
            cube.remove_coord(coord)

    # Then remove any dimension coordinates.
    for coord in cube.coords(dim_coords=True):
        if coord.standard_name not in dim_coords_to_keep:
            removed_coord_names.append(coord.name())
            cube.remove_coord(coord)

    return removed_coord_names


def _make_data_contiguous(cubelist):
    """
    Checks that all of the cubes in a cubelist are time contiguous. If one isn't
    then that cube is split into two at the break and the original cube is
    replaced in the cubelist by the two new ones. This function is then called
    recursively to check the new cubelist.
    """
    all_contiguous = True
    new_list = iris.cube.CubeList()

    for cube in cubelist:
        time_coord = cube.coord('time')
        if not time_coord.has_bounds():
            time_coord.guess_bounds()  # bounds will be contiguous by definition
            new_list.append(cube)
        elif time_coord.is_contiguous():
            new_list.append(cube)
        else:
            all_contiguous = False
            bounds = time_coord.bounds
            bi = 1
            bmax = len(bounds)
            while bi < bmax and iris.util.approx_equal(bounds[bi,0], bounds[bi-1,1]):
                bi += 1
            if bi < bmax:
                bval = bounds[bi-1,1]
                earlier_constraint = iris.Constraint(time=lambda c, x=bval: c <= x)
                later_constraint = iris.Constraint(time=lambda c, x=bval: c > x)
                new_list.append(cube.extract(earlier_constraint))
                new_list.append(cube.extract(later_constraint))

    if not all_contiguous:
        new_list = _make_data_contiguous(new_list)

    return new_list


def _prep_data_for_concatenation(cubelist):
    """
    Equalises the metadata in a cubelist to prepare the cubes for concatenation.
    It loops through the cubes in the cubelist and makes sure that variable and
    coordinate names are set correctly, that the cube is a vector rather than
    scalar along the time coordinate, and that any coordinates that won't
    concatenate have been removed.
    """
    # It's necessary to make a new list and add the cubes to this because if
    # the time axis is scalar then a new cube is created, which won't
    # then appear in the list
    new_list = iris.cube.CubeList()
    equalise_attributes(cubelist)

    # Find the first non-null geographic coordinate system, if any.
    cslist = list(filter(None, [x.coord_system('GeogCS') for x in cubelist]))
    coord_system0 = cslist[0] if cslist else None

    for cube in cubelist:
        # read the data into memory or else the concatenation may not work
        _touch_data = cube.data

        # If the time axis is scalar then promote it to a dimension coordinate.
        if is_scalar_coord(cube, 'time'):
            cube = iris.util.new_axis(cube, 'time')

        # Make sure that the coordinate names are set correctly. Cubes loaded
        # from NetCDF will have coordinate names in unicode and so make sure
        # all are.
        for coord in cube.coords():
            try:
                coord.standard_name = text_type(coord.standard_name or '')
            except ValueError:
                coord.standard_name = text_type('')
            coord.long_name = text_type(coord.long_name or '')
            if coord.var_name:
                coord.var_name = text_type(coord.var_name)
            else:
                coord.var_name = coord.standard_name

        # Make sure that the circular flag is consistent on longitude.
        # FIXME: does this really need to be set to false?
        latcrd, loncrd = _get_lat_long_coords(cube)
        loncrd.circular = False

        # Make sure that coord systems are consistent.
        latcrd.coord_system = coord_system0
        loncrd.coord_system = coord_system0

        # Remove cell measures as they may prevent cube concatenation.
        if hasattr(cube, 'cell_measures'):
            for cm in cube.cell_measures():
                cube.remove_cell_measure(cm)

        # add the cube to the new cube list
        new_list.append(cube)

    return new_list


def _get_lat_long_coords(cube):
    """
    Return the latitude and longitude DimCoords associated with the specified
    cube. Normally these will be the coordinates named 'latitude' and 'longitude'.
    In the case of rotated pole data, however, the coordinates will be named
    'grid_latitude' and 'grid_longitude'.
    """
    try:
        # check for 'latitude' and 'longitude' dimension coordinates
        latcrd = cube.coord('latitude', dim_coords=True)
        loncrd = cube.coord('longitude', dim_coords=True)
    except iris.exceptions.CoordinateNotFoundError:
        try:
            # check for 'grid_latitude' and 'grid_longitude' dimension coordinates
            latcrd = cube.coord('grid_latitude', dim_coords=True)
            loncrd = cube.coord('grid_longitude', dim_coords=True)
        except iris.exceptions.CoordinateNotFoundError:
            # check for 'latitude' and 'longitude' auxiliary coordinates
            latcrd = cube.coord('latitude', dim_coords=False)
            loncrd = cube.coord('longitude', dim_coords=False)

    return latcrd, loncrd


def _get_custom_diag_defns(diag_defn):
    """
    Create definitions for each of the diagnostics required by the custom
    diagnostic defined in ``diag_defn``. Assumes that the attributes on the
    custom diagnostic definition - such as stream and region - are applicable
    to each of the actual diagnostics from which it is derived."
    """

    # Store the custom class and formula on the diagnostic definition as
    # they are needed later on in the _compute_custom_diagnostic() method.
    if diag_defn.formula:
        if diag_defn.class_path:
            diag_defn.cls = get_class_object_from_class_path(diag_defn.class_path)
        else:
            diag_defn.cls = SimpleDerivedDiagnostic
        stashcodes = None
    elif diag_defn.var_name in CUSTOM_DIAGNOSTICS:
        custom_diag = CUSTOM_DIAGNOSTICS[diag_defn.var_name]
        diag_defn.cls = custom_diag['class']
        diag_defn.formula = custom_diag.get('formula')   # usually None
        stashcodes = custom_diag.get('stashcodes')
    else:
        msg = "Invalid definition for custom diagnostic '%s'." % diag_defn.var_name
        _logger.error(msg)
        raise AppConfigError(msg)

    # If prerequisite STASH codes are not explicitly defined then determine
    # from the diagnostic's formula, if one is specified.
    if diag_defn.formula and not stashcodes:
        proc = diag_defn.cls(diag_defn.formula)
        stashcodes = [x for x in filter(is_msi_stash_code, proc.formula_vars)]

    _logger.debug("Prerequisite STASH codes: %s", ', '.join(stashcodes))
    source_diags = []
    for stashcode in stashcodes:
        ddefn = NamespacePlus(**vars(diag_defn))
        ddefn.stashcode = stashcode
        ddefn.var_name = stashcode
        if not ddefn.lbproc: ddefn.lbproc = 128
        if not ddefn.lbtim: ddefn.lbtim = CAL_LBTIM_MAP.get(diag_defn.calendar, 122)
        source_diags.append(ddefn)

    return source_diags


def _get_mass_time_extent(model_defn, diag_defn):
    """
    Return the date-time extent of the MASS data collection associated with the
    specified model and diagnostic.
    """
    # NEMO and CICE diagnostics require special treatment.
    if diag_defn.model_name == afterburner.MODEL_NEMO:
        return _get_mass_time_extent_nemo(model_defn, diag_defn)
    if diag_defn.model_name == afterburner.MODEL_CICE:
        return _get_mass_time_extent_cice(model_defn, diag_defn)

    # Remainder of this function assumes we're dealing with a UM diagnostic.

    collection = mass_collection_from_stream(diag_defn.model_name, diag_defn.stream)
    stashcodes = [diag_defn.stashcode] if diag_defn.stashcode else None
    mass_start_dts, mass_end_dts = moose.query_time_extent(model_defn.name,
        collection, data_class=model_defn.data_class, stashcodes=stashcodes,
        levels=diag_defn.surface_lblevs)

    if not (mass_start_dts and mass_end_dts):
        msg = "Start and end times undefined for {0}/{1}.".format(model_defn.name,
            diag_defn.var_name)
        raise DataStoreError(msg)

    return DateTimeRange(mass_start_dts, mass_end_dts)


def _get_mass_time_extent_nemo(model_defn, diag_defn):
    """
    Return the date-time extent of the MASS data collection associated with the
    specified NEMO model and diagnostic.
    """
    collection = mass_collection_from_stream(diag_defn.model_name, diag_defn.stream)
    files = moose.list_struct_files(model_defn.name, collection,
        data_class=model_defn.data_class)
    if not files:
        msg = "No files found in MASS data collection {0}/{1}.".format(model_defn.name,
            collection)
        _logger.debug(msg)
        raise DataStoreError(msg)

    files.sort()
    first_file = os.path.basename(files[0])
    last_file = os.path.basename(files[-1])
    _logger.debug("First and last files:\n%s / %s", first_file, last_file)

    # Decode date-time info from first and last files.
    mass_start_dts, _dummy = decode_dates_from_nemo_filename(first_file)
    _dummy, mass_end_dts = decode_dates_from_nemo_filename(last_file)

    if not (mass_start_dts and mass_end_dts):
        msg = "Unable to query start and end times for MASS collection {0}/{1}.".format(
            model_defn.name, collection)
        raise DataStoreError(msg)

    return DateTimeRange(mass_start_dts, mass_end_dts)


def _get_mass_time_extent_cice(model_defn, diag_defn):
    """
    Return the date-time extent of the MASS data collection associated with the
    specified CICE model and diagnostic.

    Note that postproc_vn1-style CICE filenames provide limited date-time
    information and, as such, should be considered as unreliable determinants
    of the extent of available data.
    """
    collection = mass_collection_from_stream(diag_defn.model_name, diag_defn.stream)
    files = moose.list_struct_files(model_defn.name, collection,
        data_class=model_defn.data_class)
    if not files:
        msg = "No files found in MASS data collection {0}/{1}.".format(model_defn.name,
            collection)
        _logger.debug(msg)
        raise DataStoreError(msg)

    files.sort()
    first_file = os.path.basename(files[0])
    last_file = os.path.basename(files[-1])
    _logger.debug("First and last files:\n%s / %s", first_file, last_file)

    # Decode date-time info from first and last files.
    # NB: some CICE filenames do not include start date-time values.
    mass_start_dts, _dummy = decode_dates_from_cice_filename(first_file)
    _dummy, mass_end_dts = decode_dates_from_cice_filename(last_file)

    if not (mass_start_dts and mass_end_dts):
        msg = "Unable to query start and end times for MASS collection {0}/{1}.".format(
            model_defn.name, collection)
        raise DataStoreError(msg)

    return DateTimeRange(mass_start_dts, mass_end_dts)


def _get_mass_fetch_extent(model_defn, diag_defn, diag_dtrange, mass_dtrange):
    """
    Return the date-time extent of the fileset to retrieve from MASS. Return
    None if files in MASS already exist on disk (as determined from diag_dtrange).
    """

    mass_start_pdt, mass_end_pdt = mass_dtrange.start_pdt, mass_dtrange.end_pdt
    t1_start_pdt = t1_end_pdt = None

    # See if start and/or end dates were defined for the current model.
    if model_defn.start_date:
        model_start_pdt = pdt_from_date_string(model_defn.start_date)
        if pdt_compare(model_start_pdt, 'gt', mass_start_pdt):
            mass_start_pdt = model_start_pdt
        #_logger.debug("Model start date: %s", model_defn.start_date)

    if model_defn.end_date:
        model_end_pdt = pdt_from_date_string(model_defn.end_date)
        if pdt_compare(model_end_pdt, 'lt', mass_end_pdt):
            mass_end_pdt = model_end_pdt
        #_logger.debug("Model end date: %s", model_defn.end_date)

    # Compare time extent of data held in MASS and netCDF cache.
    if diag_dtrange is None:
        # Set T1 time extent so as to get all files from MASS, or at least
        # for particular model start/end dates, if defined by user.
        t1_start_pdt = mass_start_pdt
        t1_end_pdt = mass_end_pdt

    elif pdt_compare(mass_end_pdt, 'gt', diag_dtrange.end_pdt):
        t1_end_pdt = mass_end_pdt
        if pdt_compare(mass_start_pdt, 'gt', diag_dtrange.end_pdt):
            # No overlap - set start time to earliest date in MASS.
            t1_start_pdt = mass_start_pdt
        else:
            # Partial overlap - set start time to latest date in netCDF cache.
            t1_start_pdt = diag_dtrange.end_pdt

    # Quit early if no date-time range was defined or if the start time is
    # equal to or later than the end time (a pathological case, perhaps?)
    if not t1_start_pdt or pdt_compare(t1_start_pdt, 'ge', t1_end_pdt):
        return None

    # Zero out time elements on the basis that they are not really needed for
    # retrieving apy/apm data from MASS.
    t1_start_pdt.hour = t1_start_pdt.minute = t1_start_pdt.second = 0
    t1_end_pdt.hour = t1_end_pdt.minute = t1_end_pdt.second = 0

    # Increment end date by 1 meaning period (a year or a month) in order to
    # catch the final T1-T2 period.
    if diag_defn.stream == 'apy':
        t1_end_pdt.year += 1
    else:
        t1_end_pdt.month += 1
        if t1_end_pdt.month == 13:
            t1_end_pdt.month = 1
            t1_end_pdt.year += 1

    # Convert PDTs to date-time strings.
    t1_start_dts = pdt_to_date_string(t1_start_pdt)
    t1_end_dts = pdt_to_date_string(t1_end_pdt)

    return DateTimeRange(t1_start_dts, t1_end_dts)


def _get_cached_data_extent(model_defn, diag_defn, diag_dtrange):
    """
    Determine the date-time extent of the new model data to load from on-disk
    cache, taking into account the time span of any statistical data residing
    in the netCDF data cache. Possible return values are as follows:

    - a DateTimeRange object if both start and end times can be determined
    - an Iris time constraint if only one of the limits can be determined
    - None if no time constraints should be applied (i.e. load all data files)
    - False if user-defined start/end dates have been specified and statistical
      data for that time period exists in the nc data cache.
    """

    time_cons = None
    start_date = start_pdt = end_date = None
    model_start_pdt = model_end_pdt = None

    # See if start and/or end dates were defined for the current model.
    if model_defn.start_date:
        model_start_pdt = pdt_from_date_string(model_defn.start_date)
    if model_defn.end_date:
        model_end_pdt = pdt_from_date_string(model_defn.end_date)

    # Set start time to the latest date in nc data cache, if defined via the
    # diag_dtrange argument.
    if diag_dtrange:
        # create constraint to select data > latest date in nc data cache
        start_date, start_pdt = diag_dtrange.end, diag_dtrange.end_pdt
        time_cons = iris.Constraint(time=lambda cell: cell.point > diag_dtrange.end_ncdt)
        _logger.debug("Constraining on times > %s...", start_date)

    # Set start time to user-defined model start date, if specified, and no
    # data is present in nc cache.
    if model_start_pdt:
        if not start_pdt:
            # create constraint to select data >= user-defined model start date
            start_date, start_pdt = model_defn.start_date, model_start_pdt
            start_ncdt = pdt_to_nc_datetime(model_start_pdt, diag_defn.calendar)
            time_cons = iris.Constraint(time=lambda cell: cell.point >= start_ncdt)
            _logger.debug("Constraining on times >= %s...", start_date)

    # Set end time to user-defined model end date, if specified.
    if model_end_pdt:
        if start_pdt and pdt_compare(model_end_pdt, 'le', start_pdt):
            # existing nc data extends beyond end time so return sentinel value
            return False
        # select data <= model end date
        end_date = model_defn.end_date
        end_ncdt = pdt_to_nc_datetime(model_end_pdt, diag_defn.calendar)
        cons = iris.Constraint(time=lambda cell: cell.point <= end_ncdt)
        if time_cons:
            time_cons &= cons
        else:
            time_cons = cons
        _logger.debug("Constraining on times <= %s...", end_date)

    # If both start and end dates are defined then return a DateTimeRange object.
    if start_date and end_date:
        return DateTimeRange(start_date, end_date)
    else:
        return time_cons


def _set_model_symbology(model_defn):
    """
    Set symbology properties (line style, marker symbol, colours, etc) to use
    for plotting time-series data for the model specified by model_defn.

    The correspondence between CMM2 app config properties and matplotlib
    graphic properties is as follows:

    line_style => linestyle
    line_width => linewidth
    line_colour => color

    marker_style => marker
    marker_size => markersize
    marker_face_colour => markerfacecolour
    marker_edge_colour => markeredgecolour

    transparency => alpha
    """

    bad_col_msg = "Unrecognised matplotlib colour definition: '{}'"

    # Set line style, width and colour.
    if not model_defn.line_style:
        model_defn.line_style = DEFAULT_SYMBOLOGY['line_style']
    elif model_defn.line_style.lower() == 'none':
        model_defn.line_style = ''

    if not model_defn.line_width:
        model_defn.line_width = DEFAULT_SYMBOLOGY['line_width']
    else:
        model_defn.line_width = float(model_defn.line_width)

    if not model_defn.line_colour:
        model_defn.line_colour = DEFAULT_SYMBOLOGY['line_colour']
    if model_defn.line_colour == 'auto':
        model_defn.line_colour = 'black'
    if not mplc.is_color_like(model_defn.line_colour):
        raise AppConfigError(bad_col_msg.format(model_defn.line_colour))

    # Set marker style, size and colour.
    if not model_defn.marker_style:
        model_defn.marker_style = DEFAULT_SYMBOLOGY['marker_style']
    elif model_defn.marker_style.lower() == 'none':
        model_defn.marker_style = ''

    if not model_defn.marker_size:
        model_defn.marker_size = DEFAULT_SYMBOLOGY['marker_size']
    else:
        model_defn.marker_size = float(model_defn.marker_size)

    if not model_defn.marker_face_colour:
        model_defn.marker_face_colour = DEFAULT_SYMBOLOGY['marker_face_colour']
    if model_defn.marker_face_colour == 'auto':
        # if undefined then make equal to line colour
        model_defn.marker_face_colour = model_defn.line_colour
    if not mplc.is_color_like(model_defn.marker_face_colour):
        raise AppConfigError(bad_col_msg.format(model_defn.marker_face_colour))

    if not model_defn.marker_edge_colour:
        model_defn.marker_edge_colour = DEFAULT_SYMBOLOGY['marker_edge_colour']
    if model_defn.marker_edge_colour == 'auto':
        # if undefined then make equal to face colour
        model_defn.marker_edge_colour = model_defn.marker_face_colour
    if not mplc.is_color_like(model_defn.marker_edge_colour):
        raise AppConfigError(bad_col_msg.format(model_defn.marker_edge_colour))

    # Set alpha (transparency) level.
    if not model_defn.transparency:
        model_defn.transparency = DEFAULT_SYMBOLOGY['transparency']
    else:
        model_defn.transparency = float(model_defn.transparency)


def _utc_now_string():
    """Returns a string containing the current UTC time in ISO 8601 format."""
    utcnow = datetime.datetime.utcnow().replace(microsecond=0)
    return utcnow.isoformat() + 'Z'


def _promote_scalar_time_coord(cubes):
    """
    For each input cube, promote a scalar time coordinate, if present, to a full
    dimension coordinate.
    """
    for ic, cube in enumerate(cubes):
        if is_scalar_coord(cube, 'time'):
            cubes[ic] = iris.util.new_axis(cube, 'time')
        elif _is_aux_time_coord(cube):
            iris.util.promote_aux_coord_to_dim_coord(cube, 'time')


def _is_aux_time_coord(cube, coord_name='time'):
    """
    Check to see if a cube's time coordinate is a non-scalar auxiliary coordinate,
    one that is associated with a single dimension of the cube. Some model data
    in netCDF format produces cubes that have the time coordinate defined in this
    way.
    """
    return bool(coord_name in [c.name() for c in cube.coords(dim_coords=False)]
        and len(cube.coord_dims(coord_name)) == 1)


def _offset_time_origin(tunits, offset_in_years):
    """
    Adjust the year component of the time origin associated with the time units
    defined in tunits by adding the specified offset. Pass in a negative offset
    to adjust the origin backwards.
    """
    parts = tunits.origin.split()
    if len(parts) == 3: parts.append('')
    units, since, date, time = parts
    yy, mm, dd = date.split('-')
    yy = str(int(yy) + offset_in_years)
    date = '-'.join([yy, mm, dd])
    origin = ' '.join([units, since, date, time])
    return cf_units.Unit(origin.strip(), calendar=tunits.calendar)


def _interval_type_as_iris_boolean_flags(interval_type):
    """
    Convert a numeric interval type (e.g. 'leftclosed') to boolean values suitable
    for passing, via the min_inclusive and max_inclusive arguments, to  Iris'
    cube.intersection() method. If the interval_type value is not recognised then
    a left-closed interval is assumed.
    """
    if interval_type == INTERVAL_OPEN:
        return (False, False)
    elif interval_type == INTERVAL_LEFTOPEN:
        return (False, True)
    elif interval_type == INTERVAL_CLOSED:
        return (True, True)
    else:
        return (True, False)
