# (C) British Crown Copyright 2016-2019, Met Office
#
# See the LICENSE.TXT file included with the Afterburner
# software distribution for full license details.
"""
Provides an implementation of the Afterburner ModelMonitor application, which
enables scientists to monitor selected model diagnostics as they are being
generated by a climate simulation.

In its present form the ModelMonitor application has a number of limitations,
as follows:

* Only plots UM diagnostics, either from an apy or apm stream (but not both).
* Only plots 2D fields, or a specified level of a 3D field.
* All input diagnostics are assumed to have the same calendar type which, by
  default is '360_day'.
* If source model data is to be retrieved from the MASS data archive, then the
  runtime environment must have access to the MOOSE command-line interface.

Refer to the :class:`ModelMonitor` class for further information.
"""
from __future__ import (absolute_import, division, print_function, unicode_literals)
from six.moves import (filter, input, map, range, zip)
from six import integer_types, string_types, text_type

import os
import datetime
import cf_units

try:
    import cftime as cft
except ImportError:
    import netcdftime as cft

import jinja2
import matplotlib
matplotlib.use('Agg')   # directs output to file rather than screen
import matplotlib.pyplot as plt
import matplotlib.ticker as mtick

import iris
import iris.quickplot as qplt
try:
    from iris.util import equalise_attributes
except ImportError:
    from iris.experimental.equalise_cubes import equalise_attributes

import afterburner
from afterburner.io import moose
from afterburner.apps import AbstractApp
from afterburner.config import ConfigProvider
from afterburner.processors.diags import ToaRadiationBalance, NetHeatFluxIntoOcean
from afterburner.processors.diags.derived import SimpleDerivedDiagnostic
from afterburner.utils import (NamespacePlus, get_class_object_from_class_path,
    INTERVAL_OPEN, INTERVAL_LEFTOPEN, INTERVAL_LEFTCLOSED, INTERVAL_CLOSED)
from afterburner.utils.cubeutils import (is_scalar_coord, is_mean_of_all_time_steps,
    set_history_attribute, are_time_axes_equal)
from afterburner.utils.dateutils import (pdt_from_date_string, pdt_to_date_string,
    pdt_to_nc_datetime, pdt_compare, DateTimeRange)
from afterburner.utils.fileutils import expand_path
from afterburner.io.datastores import MassDataStore
from afterburner.io.datacaches import VAR_SPLIT_SCHEME, DataCache
from afterburner.metavar import UmMetaVariable
from afterburner.modelmeta import mass_collection_from_stream, is_msi_stash_code
from afterburner.exceptions import (DataProcessingError, DataStoreError,
    MooseUnavailableError, AppConfigError)

# MOOSE read-only flags.
MOOSE_READ = moose.MOOSE_LS | moose.MOOSE_MDLS | moose.MOOSE_GET | moose.MOOSE_SELECT

# IDs of namelists used in the application configuration file.
MODEL_NAMELIST = 'models'
DIAGS_NAMELIST = 'diags'

# Define the properties of supported custom diagnostics.
# Those below relate to the original prototype app. Where practicable, new
# custom diagnostics should use one of the new derived diagnostic classes.
CUSTOM_DIAGNOSTICS = {
    'toa_radiation_balance': {
        'stashcodes': ['m01s01i207', 'm01s01i208', 'm01s03i332'],
        'class': ToaRadiationBalance
    },
    'net_heat_flux_into_ocean': {
        'stashcodes': ['m01s01i203', 'm01s02i207', 'm01s03i228', 'm01s00i507', 'm01s03i232'],
        'class': NetHeatFluxIntoOcean
    },
}

# HTML template filename.
HTML_TEMPLATE = 'cmm_app_template.html'

# Names of recognised vertical coordinates.
LEVEL_COORD_NAMES = ['level', 'model_level', 'model_level_number',
    'soil_model_level_number',  'pseudo_level']

# Graph sorting options.
SORT_GRAPHS_BY_DIAG_NAME = 'diag_and_region_name'
SORT_GRAPHS_BY_ORDER_KEY = 'graph_order_key'


class ModelMonitor(AbstractApp):
    """
    Provides an implementation of the Climate Model Monitor (CMM) application,
    which enables scientists to monitor selected model diagnostics as they are
    being generated by a climate simulation.

    In normal operation the application retrieves UM annual-mean or monthly-mean
    data from the MASS archive, extracts the requested diagnostics, calculates
    the global or regional mean of the data, plots the resultant spatial-mean
    values as a time-series graph, and finally saves the spatial-mean data in
    netCDF format.

    At present the application only supports the calculation and plotting of
    UM diagnostics output to the annual-mean (apy) or monthly-mean (apm) streams.
    The target diagnostics must be either surface fields (e.g. 1.5m air temperature),
    or 3D fields from which a single, user-specified level is extracted.

    By default a global mean of the diagnostic data is calculated and plotted for
    for each time point. Alternatively, or in addition to the global mean, the
    regional mean of the data can be given similar treatment.
    """

    def __init__(self, arglist=None, **kwargs):
        """
        :param list arglist: List of raw options and/or arguments passed from the
            calling environment. Typically these will be unprocessed command-line
            arguments, e.g. ``['-f', 'foo', '--foe=fum', 'infile']``.
        """
        super(ModelMonitor, self).__init__(version='1.0.0b6', **kwargs)

        # Parse command-line arguments and read app configuration information
        # from a Rose configuration file.
        self._parse_args(arglist, desc="Climate Model Monitor: generates "
            "time-series graphs of climate model diagnostics.")
        self._parse_app_config()
        self._set_message_level()
        self.logger.info("Initialising ModelMonitor app...")

        # Extract general options from app config object.
        self.app_options = self._get_app_options()

        # Before proceeding, check access to MASS.
        self._check_mass_access()

        # Extract model definitions from app config object.
        self.model_defns = self._get_model_definitions()

        # Extract diag definitions from app config object.
        self.diag_defns = self._get_diag_definitions()

        # Initialise dictionary to store memory-cached source diagnostic data.
        self.model_data_memcache = {}

        # Initialise dictionary to store memory-cached target diagnostic data.
        self.smean_data_memcache = {}

        # Initialise model data cache.
        self._init_model_data_cache()

        # Initialise set object to record new files uploaded into cache.
        self.new_files_in_cache = set()

        # Plot size (w, h) in inches.
        self.plot_size = (9.5, 6.1)

    @property
    def cli_spec(self):
        """Defines the command-line interface specification for the application."""
        return [
            {'names': ['-c', '--config-file'], 'required': True,
                'help': 'Pathname of app configuration file'},
        ]

    def run(self):
        """Run the application."""
        self.logger.info("Running ModelMonitor app...")

        if not self._check_moose_status():
            return

        self._create_output_dirs()

        if self.app_options['clear_netcdf_cache']:
            self._clear_nc_cache()

        self._refresh_cached_diag_data()

        self._generate_plot_images()

        self._generate_html_page()

        self._teardown()

        self.logger.info("ModelMonitor app execution completed.")

    @property
    def model_names(self):
        """A list of names of all defined models, sorted by plot order."""
        names = [defn.name for defn in sorted(self.model_defns.values(),
            key=lambda sk: sk.plot_order)]
        return names

    @property
    def enabled_model_names(self):
        """A list of names of enabled models, sorted by plot order."""
        names = [defn.name for defn in sorted(self.model_defns.values(),
            key=lambda sk: sk.plot_order) if defn.enabled]
        return names

    @property
    def diag_names(self):
        """A sorted list of the names of all defined diagnostics."""
        return [self.diag_defns[k].name for k in self._sorted_diag_keys()]

    @property
    def enabled_diag_names(self):
        """A sorted list of the names of enabled diagnostics."""
        return [self.diag_defns[k].name for k in self._sorted_diag_keys(True)]

    @property
    def enabled_diag_keys(self):
        """A sorted list of (var_name, region_name) keys for enabled diagnostics."""
        return self._sorted_diag_keys(True)

    def _sorted_diag_keys(self, enabled_only=False):
        """
        Return a list of diagnostic keys sorted according to the value of the
        'sort_graphs_by' configuration option. If the enabled_only argument is
        set to true then only keys for enabled diagnostics are returned.
        """
        if self.app_options['sort_graphs_by'] == SORT_GRAPHS_BY_ORDER_KEY:
            keys = [k for k, defn in sorted(self.diag_defns.items(),
                    key=lambda kv: kv[1].graph_order)]

        else:    # SORT_GRAPHS_BY_DIAG_NAME
            keys = sorted(self.diag_defns)

        if enabled_only:
            keys = [k for k in keys if self.diag_defns[k].enabled]

        return keys

    def _get_app_options(self):
        """Read general app config options and store them in a dictionary."""

        section = 'general'
        app_opts = {}

        # directory locations
        output_dir = self.app_config.get_property(section, 'output_dir')
        output_dir = expand_path(output_dir)
        cache_dir = self.app_config.get_property(section, 'cache_dir')
        app_opts['cache_dir'] = expand_path(cache_dir)

        self.output_dir = output_dir
        self.nc_output_dir = os.path.join(output_dir, 'nc')
        self.img_output_dir = os.path.join(output_dir, 'images')
        self.html_output_dir = output_dir

        # cache management options
        app_opts['cache_type'] = self.app_config.get_property(section,
            'cache_type', default=VAR_SPLIT_SCHEME)
        app_opts['cache_readonly'] = self.app_config.get_bool_property(
            section, 'cache_readonly', default=False)
        app_opts['sync_with_mass'] = self.app_config.get_bool_property(
            section, 'sync_with_mass', default=True)
        app_opts['clear_model_cache'] = self.app_config.get_bool_property(
            section, 'clear_model_cache', default=False)
        app_opts['clear_netcdf_cache'] = self.app_config.get_bool_property(
            section, 'clear_netcdf_cache', default=False)

        if app_opts['cache_readonly']:
            # disable sync_with_mass option if read-only cache access selected
            app_opts['sync_with_mass'] = False

        if not app_opts['sync_with_mass']:
            # disable clear_model_cache if sync_with_mass option is disabled
            app_opts['clear_model_cache'] = False

        # default stream identifier, calendar, and reinit interval.
        app_opts['stream'] = self.app_config.get_property(section,
            'stream', default='apy')
        app_opts['calendar'] = self.app_config.get_property(section,
            'calendar', default='360_day')
        app_opts['reinit'] = self.app_config.get_int_property(section,
            'reinit', default=0)

        # treatment of region extent coordinates, e.g. closed, left-closed
        app_opts['treat_region_coords_as'] = self.app_config.get_property(
            section, 'treat_region_coords_as', default=INTERVAL_LEFTCLOSED)

        # graph annotation and sorting options
        for attname in ['include_stash', 'include_date', 'include_region']:
            app_opts[attname] = self.app_config.get_bool_property(section, attname)

        app_opts['legend_vis'] = self.app_config.get_property(section,
            'legend_vis', default='all')
        app_opts['legend_style'] = self.app_config.get_property(section,
            'legend_style', default='name')
        app_opts['sort_graphs_by'] = self.app_config.get_property(section,
            'sort_graphs_by', default=SORT_GRAPHS_BY_DIAG_NAME)

        # parallel processing options
        app_opts['num_processors'] = self.app_config.get_int_property(section,
            'num_processors', default=1)

        # ignoreable_attributes option
        attnames = self.app_config.get_property(section, 'ignoreable_attributes',
            default='um_version')
        if attnames:
            attnames = [s.strip() for s in attnames.split(',')]
        else:
            attnames = []
        app_opts['ignoreable_attributes'] = attnames

        return app_opts

    def _get_model_definitions(self):
        """
        Read all of the target model definitions from the MODEL_NAMELIST portion
        of the configuration file. Model definitions are stored as simple
        NamespacePlus objects.
        """
        default_reinit = self.app_options['reinit']

        model_defns = {}
        for mdict in self.app_config.iter_nl(MODEL_NAMELIST):
            mdict['enabled'] = _is_true(mdict.get('enabled', 't'))
            if not mdict.get('um_vn'): mdict['um_vn'] = '0'
            # TODO: add 'nemo_vn' as and when NEMO is supported

            mdefn = NamespacePlus(**mdict)
            mdefn.plot_order = int(mdefn.plot_order)
            if '/' in mdefn.name:
                mdefn.suite_id, mdefn.ens_id = mdefn.name.split('/')
            else:
                mdefn.suite_id = mdefn.name
                mdefn.ens_id = ''
            mdefn.runid = mdefn.suite_id.split('-')[-1]
            mdefn.reinit = int(mdefn.reinit or default_reinit)
            mdefn.time_offset = int(mdefn.time_offset or 0)

            model_defns[mdefn.name] = mdefn

        self.logger.debug("Read in %d model definitions.", len(model_defns))

        # For ensemble setups, check that all model definitions belong to
        # ensemble-based suites.
        msg = ''
        ems = [mdl.ens_id for mdl in model_defns.values() if mdl.enabled]
        if self._is_ensemble_data_cache():
            if not all(ems):
                msg = ("Ensemble-style model data cache has been selected but\n"
                    "1 or more configured models are NOT part of an ensemble.")
        elif any(ems):
            msg = ("1 or more configured models are part of an ensemble but\n"
                "an ensemble-style model data cache has NOT been selected.")
        if msg:
            self.logger.error(msg)
            raise AppConfigError(msg)

        return model_defns

    def _get_diag_definitions(self):
        """
        Read all of the target diagnostic definitions from the DIAGS_NAMELIST
        portion of the configuration file. Diagnostic definitions are stored as
        simple NamespacePlus objects. A potential upgrade would be to use
        Afterburner's MetaVariable classes.
        """
        default_stream = self.app_options['stream']
        default_calendar = self.app_options['calendar']

        diag_defns = {}
        for ddict in self.app_config.iter_nl(DIAGS_NAMELIST):
            ddict['enabled'] = _is_true(ddict.get('enabled', 't'))
            ddefn = NamespacePlus(**ddict)

            if not ddefn.stream: ddefn.stream = default_stream
            if not ddefn.calendar: ddefn.calendar = default_calendar
            if not ddefn.region_name: ddefn.region_name = 'Global'

            # Check that either the stashcode or the var_name option is defined.
            if not bool(ddefn.stashcode) ^ bool(ddefn.var_name):
                msg = ("Invalid definition for diagnostic '{0}'.\n"
                    "Either the 'stashcode' option or the 'var_name' option "
                    "must be defined (but not both).".format(ddefn._index))
                self.logger.error(msg)
                raise AppConfigError(msg)

            if not ddefn.var_name: ddefn.var_name = ddefn.stashcode
            key = (ddefn.var_name, ddefn.region_name)
            if not ddefn.name: ddefn.name = '/'.join(key)
            ddefn.is_custom = not is_msi_stash_code(ddefn.var_name)

            # Coerce lbproc/lbtim values from strings to integers.
            for att, val in [('lbproc', 128), ('lbtim', 122)]:
                x = getattr(ddefn, att, None)
                setattr(ddefn, att, int(x) if x else val)

            ddefn.graph_order = int(ddefn.graph_order or 0)

            diag_defns[key] = ddefn

        self.logger.debug("Read in %d diagnostic definitions.", len(diag_defns))

        return diag_defns

    def _get_model_line_style(self, model_defn):
        """
        Returns a dictionary of the line style properties (colour, style, marker)
        for the specified model.
        """
        line_spec = {}

        # Catch and convert 'none' values in style definitions.
        if not model_defn.line_colour:
            model_defn.line_colour = 'black'

        if not model_defn.line_style:
            model_defn.line_style = '-'
        elif model_defn.line_style.lower() == 'none':
            model_defn.line_style = ''

        if not model_defn.marker_style or model_defn.marker_style.lower() == 'none':
            model_defn.marker_style = ''

        line_spec['color'] = model_defn.line_colour
        line_spec['linestyle'] = model_defn.line_style
        line_spec['marker'] = model_defn.marker_style

        return line_spec

    def _get_legend_text(self, model_defn):
        """
        Generates the text string to appear in the legend of a plot for the
        specified model.
        """
        legend_style = self.app_options['legend_style'].lower()

        if legend_style in ('name+description', 'name+label'):
            legend_text = model_defn.name + ' ' + (model_defn.label or '?undefined?')
        elif legend_style in ('description', 'label'):
            legend_text = model_defn.label or '?undefined?'
        else:
            legend_text = model_defn.name

        return legend_text

    def _get_custom_diag_defns(self, diag_defn):
        """
        Create definitions for each of the diagnostics required by the custom
        diagnostic defined in ``diag_defn``. Assumes that the attributes on the
        custom diagnostic definition - such as stream and region - are applicable
        to each of the actual diagnostics from which it is derived."
        """

        # Store the custom class and formula on the diagnostic definition as
        # they are needed later on in the _compute_custom_diagnostic() method.
        if diag_defn.formula:
            if diag_defn.class_path:
                diag_defn.cls = get_class_object_from_class_path(diag_defn.class_path)
            else:
                diag_defn.cls = SimpleDerivedDiagnostic
            stashcodes = None
        elif diag_defn.var_name in CUSTOM_DIAGNOSTICS:
            custom_diag = CUSTOM_DIAGNOSTICS[diag_defn.var_name]
            diag_defn.cls = custom_diag['class']
            diag_defn.formula = custom_diag.get('formula')   # usually None
            stashcodes = custom_diag.get('stashcodes')
        else:
            msg = "Invalid definition for custom diagnostic '%s'." % diag_defn.var_name
            self.logger.error(msg)
            raise AppConfigError(msg)

        # If prerequisite STASH codes are not explicitly defined then determine
        # from the diagnostic's formula, if one is specified.
        if diag_defn.formula and not stashcodes:
            proc = diag_defn.cls(diag_defn.formula)
            stashcodes = list(proc.formula_vars)

        self.logger.debug("Prerequisite STASH codes: %s", ', '.join(stashcodes))
        source_diags = []
        for stashcode in stashcodes:
            ddefn = NamespacePlus(**vars(diag_defn))
            ddefn.stashcode = stashcode
            ddefn.var_name = stashcode
            if not ddefn.lbproc: ddefn.lbproc = 128
            if not ddefn.lbtim: ddefn.lbtim = 122
            source_diags.append(ddefn)

        return source_diags

    def _check_mass_access(self):
        """If synchronisation with MASS is required then check its availability."""
        if self.app_options['sync_with_mass'] and not moose.has_moose_support():
            msg = ("The runtime environment does not appear to support\n"
                "the MOOSE command interface to the MASS data archive.")
            self.logger.error(msg)
            raise MooseUnavailableError(msg)

    def _check_moose_status(self):
        """If synchronisation with MASS is required then check MOOSE is available."""
        if (self.app_options['sync_with_mass'] and not
            moose.check_moose_commands_enabled(MOOSE_READ)):
            self.logger.warning("The MASS archive is not currently available for "
                "read operations.")
            return False
        else:
            return True

    def _is_ensemble_data_cache(self):
        """Return true if an ensemble-type data cache has been configured."""
        cache_type = self.app_options.get('cache_type', '').lower()
        return cache_type.startswith('ensemble')

    def _init_model_data_cache(self):
        """Initialise a data cache object based on the VarSplit scheme."""
        try:
            cache_type = self.app_options['cache_type']
            cache_dir = self.app_options['cache_dir']
            cache_readonly = self.app_options['cache_readonly']
            data_store = MassDataStore()
            self.logger.info("Initialising %s data cache...", cache_type)
            self.data_cache = DataCache.create_cache(cache_type, data_store,
                cache_dir, read_only=cache_readonly)
        except Exception as exc:
            self.logger.error("Problem trying to initialise data cache rooted "
                "at directory:\n%s.", cache_dir)
            self.logger.error(str(exc))
            raise

    def _create_output_dirs(self):
        """Create any required output directories."""

        for opdir in [self.output_dir, self.nc_output_dir, self.img_output_dir]:
            if not os.path.exists(opdir):
                try:
                    os.makedirs(opdir)
                    self.logger.info("Created output directory: %s", opdir)
                except OSError:
                    self.logger.error("Problem trying to create output directory: %s",
                        opdir)
                    raise

            # Directory exists so do a write permission check.
            elif not os.access(opdir, os.W_OK):
                msg = "You do not have write permission to directory: %s" % opdir
                self.logger.error(msg)
                raise OSError(msg)

    def _clear_nc_cache(self):
        """
        Clears the netCDF data cache of any spatial-mean data files. Currently
        this entails a brute force removal of all files with a .nc extension.
        """
        self.logger.info("Clearing netCDF data cache...")
        cwd = os.getcwd()
        os.chdir(self.nc_output_dir)
        ndel = nerr = 0
        for fn in os.listdir('.'):
            if fn.endswith('.nc'):
                try:
                    os.remove(fn)
                    ndel += 1
                except OSError:
                    nerr += 1
        os.chdir(cwd)
        if ndel:
            self.logger.info("Deleted %s .nc files.", ndel)
        if nerr:
            self.logger.info("Unable to delete %s .nc files.", nerr)

    def _refresh_cached_diag_data(self):
        """Refresh the cached netCDF files with the latest meaned model data."""

        for model_name in self.enabled_model_names:
            model_defn = self.model_defns[model_name]
            self.model_data_memcache = {}
            self.logger.info("\nRefreshing data caches for model %s...", model_name)

            for diag_key in self.enabled_diag_keys:
                diag_defn = self.diag_defns[diag_key]

                # Load existing spatial-mean data, if any, from netCDF cache.
                diag_data, diag_dtrange = self._load_cached_diag_data(model_defn,
                    diag_defn)
                self.smean_data_memcache[(model_name,) + diag_key] = diag_data

                # Check to see if this is a custom diagnostic.
                if diag_defn.is_custom:
                    diag_list = self._get_custom_diag_defns(diag_defn)
                else:
                    diag_list = [diag_defn]

                # If required, synchronise model data files from MASS.
                model_data = iris.cube.CubeList()
                if self.app_options['sync_with_mass']:
                    for diag in diag_list:
                        cubes = self._update_model_data_cache(model_defn,
                            diag, diag_dtrange)
                        if cubes:
                            _promote_scalar_time_coord(cubes)
                            model_data.extend(cubes)

                # Otherwise load model data from files in disk cache.
                else:
                    for diag in diag_list:
                        cubes = self._load_latest_model_data(model_defn,
                            diag, diag_dtrange)
                        if cubes:
                            _promote_scalar_time_coord(cubes)
                            model_data.extend(cubes)

                if not model_data: continue

                try:
                    self.logger.debug("Model data cubes:\n%s", str(model_data))

                    # For a custom diagnostic, replace the input model data with
                    # the computed target diagnostic data.
                    if diag_defn.is_custom:
                        model_data = self._extract_custom_data_subset(model_data,
                            model_defn, diag_list)
                        model_data = self._compute_custom_diagnostic(model_data,
                            diag_defn)
                        # result cube should be first element of returned cubelist
                        model_data = model_data[0]

                    # Extract required model data subset into a single cube.
                    else:
                        model_data = self._extract_model_data_subset(model_data,
                            model_defn, diag_defn)

                    # Compute the spatial-mean of the model data.
                    smean_data = self._compute_spatial_mean(model_data,
                        model_defn, diag_defn)

                    # Concatenate with existing spatial-mean data.
                    if diag_data:
                        smean_data = self._concat_mean_data(diag_data, smean_data)

                    # Update netCDF data files.
                    self._update_diag_data_cache(smean_data, model_defn, diag_defn)

                    # Add spatial-mean data for this diagnostic to in-memory cache.
                    self.smean_data_memcache[(model_name,) + diag_key] = smean_data

                except Exception as exc:
                    # Catch and report errors trying to compute the spatial mean.
                    # This will prevent the app exiting and ensure that the user
                    # should at least see some useful diagnostic output.
                    self.logger.error("Problem trying to generate spatial mean "
                        "for diagnostic %s/%s", model_name, diag_defn.name)
                    self.logger.error(str(exc))

    def _load_cached_diag_data(self, model_defn, diag_defn):
        """Load spatial-mean model data from cached netCDF files."""

        self.logger.info("\nLoading cached netCDF data for %s/%s...",
            model_defn.name, diag_defn.name)

        ncfile = _make_netcdf_filename(model_defn, diag_defn)
        ncpath = os.path.join(self.nc_output_dir, ncfile)

        diag_data = dt_range = None
        if os.path.exists(ncpath):
            try:
                diag_data = iris.load_cube(ncpath)
                dt_range = _get_datetime_range(diag_data, use_bounds=True)
                self.logger.debug("Cube summary:\n%s", diag_data.summary(shorten=True))
                self.logger.debug("Time extent of data in netCDF cache: %s to %s",
                    dt_range.start, dt_range.end)
            except (IOError, iris.exceptions.IrisError) as exc:
                self.logger.warning("Unable to load data from file: %s", ncpath)
                self.logger.warning(str(exc))
        else:
            self.logger.debug("No cached netCDF data found.")

        return diag_data, dt_range

    def _update_model_data_cache(self, model_defn, diag_defn, diag_dtrange):
        """Update the on-disk model data cache."""

        self.logger.info("Updating cached model data for %s/%s...",
            model_defn.name, diag_defn.var_name)

        # Check to see if data for this model/diagnostic combination has already
        # been updated.
        memcache = self.model_data_memcache.setdefault(diag_defn.var_name, {})
        visited = memcache.setdefault('visited', False)

        # Get time extent of data currently held in MASS for the current
        # diagnostic.
        if visited:
            mass_start_pdt = memcache['mass_start_pdt']
            mass_end_pdt = memcache['mass_end_pdt']
        else:
            try:
                mass_start_dts, mass_end_dts = _get_mass_time_extent(model_defn,
                    diag_defn)
                mass_start_pdt = pdt_from_date_string(mass_start_dts)
                mass_end_pdt = pdt_from_date_string(mass_end_dts)
                self.logger.debug("Time extent (T1-T1) of data in MASS: %s to %s",
                    mass_start_dts, mass_end_dts)
                memcache['mass_start_pdt'] = mass_start_pdt
                memcache['mass_end_pdt'] = mass_end_pdt
            except DataStoreError:
                self.logger.warning("Problem querying time extent of diagnostic "
                    "%s/%s in MASS.", model_defn.name, diag_defn.var_name)
                return None

        # See if start and/or end dates were defined for the current model.
        model_start_pdt = model_end_pdt = None
        if model_defn.start_date:
            model_start_pdt = pdt_from_date_string(model_defn.start_date)
            if pdt_compare(model_start_pdt, 'gt', mass_start_pdt):
                mass_start_pdt = model_start_pdt
            self.logger.debug("Model start date: %s", model_defn.start_date)
        if model_defn.end_date:
            model_end_pdt = pdt_from_date_string(model_defn.end_date)
            if pdt_compare(model_end_pdt, 'lt', mass_end_pdt):
                mass_end_pdt = model_end_pdt
            self.logger.debug("Model end date: %s", model_defn.end_date)

        # Compare time extent of data held in MASS and netCDF cache.
        t1_start_pdt = t1_end_pdt = None
        if diag_dtrange is None:
            # Set T1 time extent so as to get all files from MASS, or at least
            # for particular model start/end dates, if defined by user.
            t1_start_pdt = model_start_pdt or mass_start_pdt
            t1_end_pdt = model_end_pdt or mass_end_pdt
        elif mass_end_pdt > diag_dtrange.end_dt:
            # Set T1 time extent so as to get all files from MASS later than
            # latest date in netCDF cache.
            t1_start_pdt = pdt_from_date_string(str(diag_dtrange.end_dt))
            t1_end_pdt = mass_end_pdt

        # Fetch any required files into the model data cache.
        cubes = None
        if t1_start_pdt:
            # Zero out time elements.
            t1_start_pdt.hour = t1_start_pdt.minute = t1_start_pdt.second = 0
            t1_end_pdt.hour = t1_end_pdt.minute = t1_end_pdt.second = 0

            # Increment end date by 1 meaning period (year or month) in order to
            # catch the final T1-T2 period.
            if diag_defn.stream == 'apy':
                t1_end_pdt.year += 1
            else:
                t1_end_pdt.month += 1
                if t1_end_pdt.month == 13:
                    t1_end_pdt.month = 1
                    t1_end_pdt.year += 1

            # Convert to date-time strings.
            t1_start_dts = pdt_to_date_string(t1_start_pdt)
            t1_end_dts = pdt_to_date_string(t1_end_pdt)

            self.logger.info("Fetching model data files for period %s to %s...",
                t1_start_dts, t1_end_dts)

            # Create a meta-variable object from model + diag definitions.
            metavar = UmMetaVariable(model_defn.um_vn, model_defn.suite_id,
                realization_id=model_defn.ens_id,
                stream_id=diag_defn.stream, stash_code=diag_defn.stashcode,
                lbproc=diag_defn.lbproc, lbtim=diag_defn.lbtim,
                time_range=(t1_start_dts, t1_end_dts), calendar=diag_defn.calendar,
                reinit=model_defn.reinit)

            try:
                # Get a list of actual data files on disk before the refresh.
                files_before = set(self.data_cache.get_filepaths([metavar]))

                # Fetch any new files.
                self.data_cache.fetch_files([metavar])

                # Get a list of data files on disk after the refresh.
                files_after = set(self.data_cache.get_filepaths([metavar]))
                new_files = files_after - files_before
                if new_files: self.new_files_in_cache.update(new_files)
                self.logger.debug("Number of new files retrieved: %d", len(new_files))

                # Load data from any new files.
                futures = afterburner.compare_iris_version('2', 'lt') and \
                    {'cell_datetime_objects': True} or {}
                with iris.FUTURE.context(**futures):
                    cubes = self.data_cache.load_data([metavar], minimal_data=True)
                    cubes = self._normalise_cube_attributes(cubes, do_concat=True)

                self.logger.debug("Cubes loaded from MASS:\n%s", str(cubes))
            except DataStoreError:
                self.logger.warning("Problem fetching data files for %s/%s.",
                    model_defn.name, diag_defn.var_name)

        else:
            self.logger.info("Model data cache is up to date.")

        # Mark the current model/diagnostic combination as visited so that we
        # don't make repeated retrievals of the same files.
        memcache['visited'] = True
        #memcache['data'] = cubes

        return cubes

    def _load_latest_model_data(self, model_defn, diag_defn, diag_dtrange):
        """Load latest model data from the data cache."""
        self.logger.info("Loading latest model data for %s/%s...", model_defn.name,
            diag_defn.var_name)

        # See if start and/or end dates were defined for the current model.
        model_start_pdt = model_end_pdt = None
        if model_defn.start_date:
            model_start_pdt = pdt_from_date_string(model_defn.start_date)
        if model_defn.end_date:
            model_end_pdt = pdt_from_date_string(model_defn.end_date)

        # Set start time to the later of (i) latest date in diagnostic data, or
        # (ii) user-defined model start date.
        time_cons = time_range = None
        start_date = start_pdt = end_date = None
        if diag_dtrange:
            if model_start_pdt and model_start_pdt > diag_dtrange.end_dt:
                # select data >= model start date
                start_date, start_pdt = model_defn.start_date, model_start_pdt
                time_cons = iris.Constraint(time=lambda cell: cell.point >= model_start_pdt)
                self.logger.debug("Constraining on times >= %s...", start_date)
            else:
                # select data > last diagnostic data date
                start_date, start_pdt = diag_dtrange.end, diag_dtrange.end_pdt
                time_cons = iris.Constraint(time=lambda cell: cell.point > diag_dtrange.end_pdt)
                self.logger.debug("Constraining on times > %s...", start_date)

        elif model_start_pdt:
            # select data >= model start date
            start_date, start_pdt = model_defn.start_date, model_start_pdt
            time_cons = iris.Constraint(time=lambda cell: cell.point >= model_start_pdt)
            self.logger.debug("Constraining on times >= %s...", start_date)

        # Set end time to model end date, if defined.
        if model_end_pdt:
            if start_pdt and pdt_compare(model_end_pdt, 'le', start_pdt):
                # existing data extends beyond end time so return empty cubelist
                self.logger.debug("Cached model data covers user-defined date range.")
                return iris.cube.CubeList()
            # select data <= model end date
            end_date = model_defn.end_date
            cons = iris.Constraint(time=lambda cell: cell.point <= model_end_pdt)
            if time_cons:
                time_cons &= cons
            else:
                time_cons = cons
            self.logger.debug("Constraining on times <= %s...", end_date)

        if start_date and end_date:
            time_range = (start_date, end_date)
            time_cons = None

        # Create a meta-variable object from model + diag definitions.
        metavar = UmMetaVariable(model_defn.um_vn, model_defn.suite_id,
            realization_id=model_defn.ens_id,
            stream_id=diag_defn.stream, stash_code=diag_defn.stashcode,
            lbproc=diag_defn.lbproc, lbtim=diag_defn.lbtim,
            time_range=time_range, calendar=diag_defn.calendar,
            reinit=model_defn.reinit)

        # Load model data into a cubelist.
        futures = afterburner.compare_iris_version('2', 'lt') and \
            {'cell_datetime_objects': True} or {}
        with iris.FUTURE.context(**futures):
            cubes = self.data_cache.load_data([metavar], constraints=time_cons,
                minimal_data=True)
            cubes = self._normalise_cube_attributes(cubes, do_concat=True)

        self.logger.debug("Cubes loaded from model data cache:\n%s", str(cubes))

        return cubes

    def _normalise_cube_attributes(self, cubes, do_concat=False):
        """
        Normalise selected cube attributes in order to avoid later cube merge or
        concatenation errors. The cubes argument should normally relate to one
        diagnostic/variable, though this is not essential.
        """
        if not self.app_options.get('ignoreable_attributes'):
            return cubes

        for cube in cubes:
            for att_name in self.app_options['ignoreable_attributes']:
                if att_name in cube.attributes:
                    del cube.attributes[att_name]

        if do_concat and len(cubes) > 1:
            cubes = cubes.concatenate()

        return cubes

    def _compute_custom_diagnostic(self, model_data, diag_defn):
        """
        Compute the custom diagnostic defined by diag_defn using the supplied
        model source data.
        """
        self.logger.info("Computing custom diagnostic %s...", diag_defn.var_name)

        # Check that all source cubes have equal time axes.
        if not are_time_axes_equal(model_data):
            msg = "Source cubes for custom diagnostic '%s' have unequal time axes." \
                % diag_defn.var_name
            self.logger.error(msg)
            raise DataProcessingError(msg)

        metadata = {}
        for att_name in ['standard_name', 'long_name', 'var_name', 'units']:
            value = getattr(diag_defn, att_name, None)
            if value: metadata[att_name] = value

        # Instantiate the diagnostic processor class, then run it.
        if diag_defn.formula:
            diag_proc = diag_defn.cls(diag_defn.formula, result_metadata=metadata)
        else:
            diag_proc = diag_defn.cls(result_metadata=metadata)

        cubes = diag_proc.run(model_data)
        if cubes:
            self.logger.debug("Cube summary:\n%s", cubes[0].summary(shorten=True))

        return cubes

    def _extract_custom_data_subset(self, cubes, model_defn, diag_defns):
        """
        Extract the required subset of model data needed as input to the custom
        diagnostic generation task. The returned cubelist should be a stripped
        down version of the input cubelist, i.e. with any diurnal mean fields
        and unneeded coordinates removed, and subsetted to a particular vertical
        level, if one has been specified as part of the diagnostic definition.
        """
        cube_subset = iris.cube.CubeList()

        for diag_defn in diag_defns:
            diag_cubes = cubes.extract(iris.AttributeConstraint(STASH=diag_defn.stashcode))
            cube = self._extract_model_data_subset(diag_cubes, model_defn, diag_defn)
            cube_subset.append(cube)

        return cube_subset

    def _extract_model_data_subset(self, cubes, model_defn, diag_defn):
        """Extract the required subset of model data for the specified diagnostic"""
        self.logger.info("Extracting model data subset for %s/%s...",
            model_defn.name, diag_defn.name)

        # Extract cubes which correspond to data averaged over all time steps.
        if len(cubes) > 1:
            mean_constraint = iris.Constraint(cube_func=is_mean_of_all_time_steps)
            tmp_cubes = cubes.extract(mean_constraint)
            if tmp_cubes and len(tmp_cubes) == 1:
                model_data = tmp_cubes[0]
            else:
                raise DataProcessingError("Error trying to extract mean data "
                    "(over all time steps) from the following cubes:\n" +
                    str(cubes))
        else:
            model_data = cubes[0]

        # If a vertical level coordinate was defined for the current diagnostic
        # then extract it.
        if diag_defn.level:
            self.logger.info("Extracting vertical level %s...", diag_defn.level)
            model_data = _extract_vertical_level(model_data, diag_defn)
        self.logger.debug("Cube summary:\n%s", model_data.summary(shorten=True))

        # Remove unwanted coordinates.
        removed = _remove_redundant_coords(model_data)
        if removed:
            self.logger.debug("Removed the following coordinates:\n%s", removed)

        return model_data

    def _compute_spatial_mean(self, model_data, model_defn, diag_defn):
        """Compute the global/regional mean for the specified diagnostic"""
        self.logger.info("Calculating spatial-mean data for %s/%s...",
            model_defn.name, diag_defn.name)

        # Compute spatial mean of new data, possibly for a regional subset.
        if diag_defn.region_name.lower() == 'global':
            region_coords = None
        else:
            region_coords = list(map(float, diag_defn.region_extent.split(',')))

        mean_data = _calc_spatial_mean(model_data, region=region_coords,
            interval_type=self.app_options['treat_region_coords_as'])
        self.logger.debug("Cube summary:\n%s", mean_data.summary(shorten=True))

        return mean_data

    def _concat_mean_data(self, old_mean_data, new_mean_data):
        """Concatenate the existing (on-disk) and new spatial-mean data cubes."""
        self.logger.info("Concatenating spatial-mean data...")

        mean_cubes = iris.cube.CubeList([old_mean_data, new_mean_data])
        contiguous = _make_data_contiguous(mean_cubes)

        # Adjust cube metadata so that they are amenable to concatenation.
        prepared = _prep_data_for_concatenation(contiguous)

        try:
            concatenated = prepared.concatenate_cube()
        except iris.exceptions.ConcatenateError as exc:
            error_message = "Problem trying to concatenate the following cubes:\n"
            for cube in prepared:
                error_message += cube.summary(shorten=True) + '\n'
            self.logger.error(error_message)
            self.logger.error(str(exc))
            raise

        self.logger.debug("Cube summary:\n%s", concatenated.summary(shorten=True))

        return concatenated

    def _update_diag_data_cache(self, mean_data, model_defn, diag_defn):
        """Update the on-disk, netCDF-based data cache."""
        self.logger.info("Updating netCDF data cache for %s/%s...",
            model_defn.name, diag_defn.name)

        mean_data.attributes['region'] = diag_defn.region_name
        set_history_attribute(mean_data, "Spatial-mean data generated by the "
            "Climate Model Monitor application.")

        ncfile = _make_netcdf_filename(model_defn, diag_defn)
        ncpath = os.path.join(self.nc_output_dir, ncfile)
        tmppath = os.path.splitext(ncpath)[0] + '.tmp'

        try:
            futures = afterburner.compare_iris_version('2', 'lt') and \
                {'netcdf_no_unlimited': True} or {}
            with iris.FUTURE.context(**futures):
                iris.save(mean_data, tmppath, saver=iris.fileformats.netcdf.save)
            if os.path.exists(ncpath):
                # not strictly necessary as the rename command should do this
                os.remove(ncpath)
            os.rename(tmppath, ncpath)
            self.logger.info("Spatial-mean data saved to netCDF file: %s", ncpath)
        except (OSError, iris.exceptions.IrisError) as exc:
            self.logger.warning("Unable to save spatial-mean data to file: %s", ncpath)
            self.logger.warning(str(exc))
        finally:
            if os.path.exists(tmppath): os.remove(tmppath)

    def _generate_plot_images(self):
        """Generate time-series plots for each model-diagnostic combination."""
        self.logger.info("\nGenerating time-series plots...")
        legvis = self.app_options['legend_vis'].lower()

        for ndiag, key in enumerate(self.enabled_diag_keys):
            diag_defn = self.diag_defns[key]
            self.logger.info("Generating plot image for diagnostic %s...",
                diag_defn.name)

            fig, axes = plt.subplots()
            fig.set_size_inches(self.plot_size)

            # See if we need to add a legend to the current plot.
            if legvis == 'all' or (legvis == 'first' and ndiag == 0):
                legend = True
            else:
                legend = False

            dt_range = self._plot_data_series(axes, diag_defn, add_legend=legend)
            self._plot_annotation(axes, diag_defn, dt_range)

            # Save the plot image to file.
            try:
                imgfile = _make_image_filename(diag_defn)
                imgpath = os.path.join(self.img_output_dir, imgfile)
                plt.savefig(imgpath)
            except IOError as exc:
                self.logger.warning("Unable to save plot for diagnostic %s "
                    "to file:\n%s", diag_defn.name, imgpath)
                self.logger.warning(str(exc))

            plt.close(fig)

        # Create the plot legend in a separate image file.
        if legvis == 'extra':
            self._plot_legend()

    def _plot_data_series(self, axes, diag_defn, add_legend=False):
        """Plot the data series for a diagnostic for all enabled models."""
        plotted_something = False
        title_text = ''
        dt_range = None

        for model_name in self.enabled_model_names:
            model_defn = self.model_defns[model_name]
            smean_data = self.smean_data_memcache.get((model_name, diag_defn.var_name,
                diag_defn.region_name))
            if not smean_data: continue

            # Update overall date-time range.
            dtr = _get_datetime_range(smean_data)
            if dtr:
                if dt_range:
                    if pdt_compare(dtr.start_pdt, 'lt', dt_range.start_pdt):
                        dt_range.start = dtr.start
                    if pdt_compare(dtr.end_pdt, 'lt', dt_range.end_pdt):
                        dt_range.end = dtr.end
                else:
                    dt_range = dtr

            try:
                linestyle_args = self._get_model_line_style(model_defn)
                label = self._get_legend_text(model_defn)
                tcoord = smean_data.coord('time')
                if model_defn.time_offset:
                    tunits = _offset_time_origin(tcoord.units, model_defn.time_offset)
                    tcoord = iris.coords.DimCoord(points=tcoord.points,
                        standard_name='time', units=tunits)
                    #tcoord.guess_bounds()
                qplt.plot(tcoord, smean_data, label=label, **linestyle_args)
                plotted_something = True
                title_text = smean_data.name()
            except ValueError as exc:
                self.logger.warning("Unable to generate plot for model %s, "
                    "diagnostic %s.", model_name, diag_defn.name)
                self.logger.warning(str(exc))

        # Update time axis label.
        axes.set_xlabel('Model Time')

        if plotted_something:
            plt.grid(True)
            if add_legend:
                plt.legend(loc=1, fontsize=10, framealpha=0.5, fancybox=True)
            if self.app_options['include_stash']:
                title_text = title_text.replace('_', ' ').title()
                title_text += ' (' + diag_defn.var_name + ')'
                plt.title(title_text)
        else:
            plt.title('Diagnostic Name: ' + diag_defn.var_name)
            axes.text(0.5, 0.5, 'No Data Available', verticalalignment='center',
                horizontalalignment='center', transform=axes.transAxes,
                fontsize=14)

        # Return the overall time range spanned by the current diagnostic.
        return dt_range

    def _plot_annotation(self, axes, diag_defn, dt_range=None):
        """Add annotation to the time-series plot for the specified diagnostic."""
        if self.app_options['include_date']:
            time_str = 'Graph generated at: ' + _utc_now_string().replace('T', ' ')
            axes.text(0.98, 0.05, time_str, verticalalignment='center',
                horizontalalignment='right', transform=axes.transAxes,
                color='darkgray', fontsize=10)

        # Add a label for the geographical region and, if set, vertical level.
        if diag_defn.region_name.lower() == 'global':
            region_text = 'Global Mean'
        else:
            region_text = diag_defn.region_name.title()

        if self.app_options['include_region'] and diag_defn.region_extent:
            coords = diag_defn.region_extent.split(',')
            region_text += " (Lat:{1},{3}; Long:{0},{2})".format(*coords)

        if diag_defn.level:
            region_text += '\nLevel: ' + diag_defn.level

        axes.text(0.02, 0.95, region_text, horizontalalignment='left',
            verticalalignment='center', transform=axes.transAxes, fontsize=10)

        # set the y-axis (diagnostic data value) limits
        if diag_defn.ymin and diag_defn.ymax:
            ymin = float(diag_defn.ymin)
            ymax = float(diag_defn.ymax)
            axes.set_ylim(ymin, ymax)
            # if the limits are large or small then use scientific notation
            if ymax < 0.01 or ymax >= 10000:
                axes.yaxis.set_major_formatter(mtick.FormatStrFormatter('%.2e'))

        # set the x-axis (time coordinate) limits
        if diag_defn.xmin or diag_defn.xmax:
            xmin = xmax = None

            if diag_defn.xmin:
                xmin = int(diag_defn.xmin)
            elif dt_range:
                xmin = dt_range.start_pdt.year

            if diag_defn.xmax:
                xmax = int(diag_defn.xmax)
            elif dt_range:
                xmax = dt_range.end_pdt.year

            if not None in (xmin, xmax):
                try:
                    from nc_time_axis import CalendarDateTime
                    start_date = cft.datetime(xmin, 1, 1)
                    end_date = cft.datetime(xmax, 1, 1)
                    axes.set_xlim(CalendarDateTime(start_date, diag_defn.calendar),
                        CalendarDateTime(end_date, diag_defn.calendar))
                except (ImportError, ValueError, AttributeError) as exc:
                    self.logger.debug(str(exc))
                    start_date = datetime.datetime(xmin, 1, 1)
                    end_date = datetime.datetime(xmax, 1, 1)
                    axes.set_xlim(start_date, end_date)

    def _plot_legend(self):
        """Create the plot legend as a separate figure."""
        self.logger.info("Generating plot legend as separate figure...")

        fig, ax = plt.subplots()
        fig.set_size_inches(self.plot_size)
        plt.title('Key To Plotted Climate Models')

        try:
            ax.set_facecolor('none')
        except AttributeError:
            ax.set_axis_bgcolor('none')   # deprecated at MPL v2.0

        ax.get_xaxis().set_visible(0)
        ax.get_yaxis().set_visible(0)

        for model_name in self.enabled_model_names:
            model_defn = self.model_defns[model_name]
            linestyle_args = self._get_model_line_style(model_defn)
            label = self._get_legend_text(model_defn)
            plt.plot([0], [0], label=label, **linestyle_args)

        plt.legend(loc=2, ncol=2, fontsize=10, fancybox=True)

        try:
            imgfile = 'cmm_plot_legend.png'
            imgpath = os.path.join(self.img_output_dir, imgfile)
            plt.savefig(imgpath)
        except IOError as exc:
            self.logger.warning("Unable to save plot legend image to file:\n%s", imgpath)
            self.logger.warning(str(exc))

        plt.close(fig)

    def _generate_html_page(self):
        """Generate an HTML page containing the plots."""
        self.logger.info("\nGenerating HTML page...")

        # Load the jinja-based HTML template for the CMM app.
        ab_config = ConfigProvider()
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(ab_config.template_dir))
        template = env.get_template(HTML_TEMPLATE)

        # Create the HTML output file.
        try:
            htmlpath = os.path.join(self.html_output_dir, 'cmm.html')
            fh = open(htmlpath, 'w')
        except IOError as exc:
            self.logger.error("Unable to create HTML output file: " + htmlpath)
            raise

        try:
            # Build a list of image file paths
            imgdir = os.path.basename(self.img_output_dir)
            image_paths = []
            for key in self.enabled_diag_keys:
                diag_defn = self.diag_defns[key]
                imgfile = _make_image_filename(diag_defn)
                image_paths.append(os.path.join(imgdir, imgfile))

            # Append separate legend image file if it was created.
            if self.app_options['legend_vis'].lower() == 'extra':
                image_paths.append(os.path.join(imgdir, 'cmm_plot_legend.png'))

            # Convert the list of filenames into a Nx2 array.
            first_in_row = True
            table = []
            for item in image_paths:
                if first_in_row:
                    row = [item]
                else:
                    row.append(item)
                    table.append(row)
                first_in_row = not first_in_row

            if not first_in_row:
                row.append(None)
                table.append(row)

            # process the template and write to file
            vn, rev = afterburner.__version__, afterburner.__version_info__[4]
            timestamp = _utc_now_string().replace('T', ' ')
            fh.write(template.render(table=table, version=vn, rev=rev,
                timestamp=timestamp))

        except Exception as exc:
            self.logger.error("Problem trying to generate HTML page.")
            self.logger.error(str(exc))
            raise

        finally:
            fh.close()

    def _teardown(self):
        """Perform teardown operations."""

        if self.app_options.get('clear_model_cache') and self.new_files_in_cache:
            nfiles = len(self.new_files_in_cache)
            self.logger.info("Deleting %d files from model data cache...", nfiles)
            for fpath in self.new_files_in_cache:
                try:
                    os.remove(fpath)
                except OSError:
                    self.logger.warning("Unable to delete file: " + fpath)


def _make_netcdf_filename(model_defn, diag_defn, sep='_', ext='.nc'):
    """
    Returns the name of a NetCDF file based upon runid, stream, variable name,
    level, and region name, e.g. '<runid>_<stream>_<stash>_<region>.nc'. Extra
    tokens are included in the case of ensemble runs and level-specific fields.
    """
    region = diag_defn.region_name or 'global'
    region = ''.join(x for x in region.lower() if x.isalnum())

    runid = model_defn.runid
    if model_defn.ens_id:
        runid += '_' + model_defn.ens_id

    if diag_defn.level:
        level = 'lev' + diag_defn.level
        fn = sep.join([runid, diag_defn.stream, diag_defn.var_name,
            level, region]) + ext
    else:
        fn = sep.join([runid, diag_defn.stream, diag_defn.var_name,
            region]) + ext

    return fn


def _make_image_filename(diag_defn, sep='_', ext='.png'):
    """
    Returns the name of an image file based upon stream, variable name,
    level, and region name, i.e. '<stream>_<stash>_<region>.png'.
    """
    region = diag_defn.region_name or 'global'
    region = ''.join(x for x in region.lower() if x.isalnum())
    if diag_defn.level:
        level = 'lev' + diag_defn.level
        fn = sep.join([diag_defn.stream, diag_defn.var_name, level, region]) + ext
    else:
        fn = sep.join([diag_defn.stream, diag_defn.var_name, region]) + ext
    return fn

# TODO: implement as an Afterburner processor class?
def _calc_spatial_mean(cube, region=None, interval_type=None):
    """
    Calculates the area-weighted spatial mean of the passed-in cube. By default
    the global mean is calculated. If a region is defined then the mean is
    calculated over the corresponding lat/long subset of the input cube.

    :param iris.cube.Cube cube: The cube for which to calculate the spatial mean.
    :param list region: If specified, defines the geographical region over which
        to calculate the area-weighted mean. The region should be a list of
        lat/long coordinates in the order: [min-long, min-lat, max-long, max-lat]
    :returns: A scalar cube containing the spatial mean.
    """
    # Ensure that lat/long coordinates have bounds.
    for coord_name in ['latitude', 'longitude']:
        coord = cube.coord(coord_name)
        if not coord.has_bounds(): coord.guess_bounds()

    if region:
        minlon, minlat, maxlon, maxlat = region[:]
        mininc, maxinc = _interval_type_as_iris_boolean_flags(interval_type)
        cube = cube.intersection(latitude=(minlat, maxlat, mininc, maxinc),
            longitude=(minlon, maxlon, mininc, maxinc), ignore_bounds=True)

    # Calculate grid weights.
    grid_weights = iris.analysis.cartography.area_weights(cube)

    # Calculate the spatial mean.
    mean_cube = cube.collapsed(['latitude', 'longitude'], iris.analysis.MEAN,
        weights=grid_weights)

    return mean_cube


def _extract_vertical_level(cube, diag_defn):
    """
    Extract a single vertical level from ``cube``. The level to extract is
    specified by the ``diag_defn.level`` attribute.
    """
    try:
        level = int(diag_defn.level)
    except ValueError:
        level = float(diag_defn.level)

    level_coord = None
    try:
        # First search for a dimension coordinate marked as the Z axis.
        level_coord = cube.coord(axis='Z', dim_coords=True)
    except iris.exceptions.CoordinateNotFoundError:
        # Otherwise search for a coordinate with a recognised level name.
        for coord in cube.coords():
            if coord.name() in LEVEL_COORD_NAMES:
                level_coord = coord
                break

    if level_coord:
        level_constraint = iris.Constraint(**{level_coord.name(): level})
        return cube.extract(level_constraint)
    else:
        msg = "Unable to determine vertical coordinate for cube:\n{0}.".format(
            cube.summary(shorten=True))
        raise iris.exceptions.CoordinateNotFoundError(msg)


def _remove_redundant_coords(cube):
    """
    Remove non-essential coordinates from ``cube``. This should prevent a number
    of subsequent cube concatenation issues, plus reduce output file size. Note,
    however, that scalar coordinates, such as level number in the case of a
    vertical subset, are also removed.
    """
    removed_coord_names = []

    # First remove any auxiliary coordinates.
    for coord in cube.coords(dim_coords=False):
        removed_coord_names.append(coord.name())
        cube.remove_coord(coord)

    # Then remove any dimension coordinates not in the following list.
    dim_coords_to_keep = ['time', 'latitude', 'longitude']
    for coord in cube.coords(dim_coords=True):
        if coord.standard_name not in dim_coords_to_keep:
            removed_coord_names.append(coord.name())
            cube.remove_coord(coord)

    return removed_coord_names


def _make_data_contiguous(cubelist):
    """
    Checks that all of the cubes in a cubelist are time contiguous. If one isn't
    then that cube is split into two at the break and the original cube is
    replaced in the cubelist by the two new ones. This function is then called
    recursively to check the new cubelist.
    """
    all_contiguous = True
    new_list = iris.cube.CubeList()

    for cube in cubelist:
        time_coord = cube.coord('time')
        if not time_coord.has_bounds():
            time_coord.guess_bounds()  # bounds will be contiguous by definition
            new_list.append(cube)
        elif time_coord.is_contiguous():
            new_list.append(cube)
        else:
            all_contiguous = False
            bounds = time_coord.bounds
            bi = 1
            while bi < len(bounds) and \
                  iris.util.approx_equal(bounds[bi,0], bounds[bi-1,1]):
                bi += 1
            if bi < len(bounds):
                bval = bounds[bi-1,1]
                earlier_constraint = iris.Constraint(time=lambda c, x=bval: c <= x)
                later_constraint = iris.Constraint(time=lambda c, x=bval: c > x)
                new_list.append(cube.extract(earlier_constraint))
                new_list.append(cube.extract(later_constraint))

    if not all_contiguous:
        new_list = _make_data_contiguous(new_list)

    return new_list


def _prep_data_for_concatenation(cubelist):
    """
    Equalises the metadata in a cubelist to prepare the cubes for concatenation.
    It loops through the cubes in the cubelist and makes sure that variable and
    coordinate names are set correctly, that the cube is a vector rather than
    scalar along the time coordinate, and that any coordinates that won't
    concatenate have been removed.
    """
    # It's necessary to make a new list and add the cubes to this because if
    # the time axis is scalar then a new cube is created, which won't
    # then appear in the list
    new_list = iris.cube.CubeList()
    equalise_attributes(cubelist)
    for cube in cubelist:
        # read the data into memory or else the concatenation may not work
        _touch_data = cube.data

        # make sure that var_name is set
        if not cube.var_name:
            if cube.standard_name:
                cube.var_name = cube.standard_name
            elif 'STASH' in cube.attributes:
                cube.var_name = str(cube.attributes['STASH'])
            else:
                # replace any spaces with underscores
                cube.var_name = cube.name().replace(' ', '_').lower()

        # If the time axis is scalar then promote it to a dimension coordinate.
        if is_scalar_coord(cube, 'time'):
            cube = iris.util.new_axis(cube, 'time')

        # Make sure that the coordinate names are set correctly. Cubes loaded
        # from NetCDF will have coordinate names in unicode and so make sure
        # all are.
        for coord in cube.coords():
            try:
                coord.standard_name = text_type(coord.standard_name or '')
            except ValueError:
                coord.standard_name = text_type('')
            coord.long_name = text_type(coord.long_name or '')
            if coord.var_name:
                coord.var_name = text_type(coord.var_name)
            else:
                coord.var_name = coord.standard_name

        # make sure that the circular flag is consistent on longitude
        cube.coord('longitude').circular = False

        # add the cube to the new cube list
        new_list.append(cube)

    return new_list


# TODO: implement as from_cube() method on DateTimeRange class?
def _get_datetime_range(cube, use_bounds=False):
    """
    Get the date-time range covered by the time coordinates or cell bounds
    of the specified cube.
    """
    if not cube: return None

    tcoord = cube.coord('time')
    tunits = tcoord.units

    if use_bounds:
        start_dt = tunits.num2date(tcoord.bounds[0,0])
        end_dt = tunits.num2date(tcoord.bounds[-1,1])
    else:
        start_dt = tunits.num2date(tcoord.points[0])
        end_dt = tunits.num2date(tcoord.points[-1])

    dtr = DateTimeRange.from_datetime(start_dt, end_dt, calendar=tunits.calendar)

    # Include datetime attributes in case client code needs to compare with
    # PDT objects.
    dtr.start_dt = start_dt
    dtr.end_dt = end_dt

    return dtr


def _get_mass_time_extent(model_defn, diag_defn):
    """
    Return the date-time extent of the MASS data collection associated with the
    specified model and diagnostic.
    """
    collection = mass_collection_from_stream(afterburner.MODEL_UM, diag_defn.stream)
    if model_defn.ens_id:
        dclass = 'ens'
    else:
        dclass = 'crum'

    mass_start_dts, mass_end_dts = moose.query_time_extent(model_defn.name,
        collection, data_class=dclass, stashcodes=[diag_defn.stashcode])

    if not (mass_start_dts and mass_end_dts):
        msg = "Start and end times undefined for {0}/{1}.".format(model_defn.name,
            diag_defn.var_name)
        raise DataStoreError(msg)

    return mass_start_dts, mass_end_dts


def _utc_now_string():
    """Returns a string containing the current UTC time in ISO 8601 format."""
    utcnow = datetime.datetime.utcnow().replace(microsecond=0)
    return utcnow.isoformat() + 'Z'


def _is_true(value):
    """Test to see if ``value`` is true-valued."""
    if not value:
        return False
    elif value is True:
        return True
    elif isinstance(value, string_types):
        true_values = ['t', 'true', '.true.', 'on', 'y', 'yes', '1']
        if value.lower() in true_values:
            return True
    elif isinstance(value, integer_types):
        if value == 1: return True

    return False


def _promote_scalar_time_coord(cubes):
    """
    For each input cube, promote a scalar time coordinate, if present, to a full
    dimension coordinate.
    """
    for ic, cube in enumerate(cubes):
        if is_scalar_coord(cube, 'time'):
            cubes[ic] = iris.util.new_axis(cube, 'time')


def _increment_date(date, calendar='standard', inc_in_secs=60):
    """Increment date (as an ISO 8601 string) by the specified number of seconds."""
    u = cf_units.Unit('hours since 1970-01-01', calendar=calendar)
    pdt = pdt_from_date_string(date, default=0)
    datenum = u.date2num(pdt_to_nc_datetime(pdt))
    datenum += inc_in_secs/3600.0
    datetime = u.num2date(datenum)
    return datetime.strftime('%Y-%m-%dT%H:%M:%S')


def _offset_time_origin(tunits, offset_in_years):
    """
    Adjust the year component of the time origin associated with the time units
    defined in tunits by adding the specified offset. Pass in a negative offset
    to adjust the origin backwards.
    """
    parts = tunits.origin.split()
    if len(parts) == 3: parts.append('')
    units, since, date, time = parts
    yy, mm, dd = date.split('-')
    yy = str(int(yy) + offset_in_years)
    date = '-'.join([yy, mm, dd])
    origin = ' '.join([units, since, date, time])
    return cf_units.Unit(origin.strip(), calendar=tunits.calendar)


def _interval_type_as_iris_boolean_flags(interval_type):
    """
    Convert a numeric interval type (e.g. 'leftclosed') to boolean values suitable
    for passing, via the min_inclusive and max_inclusive arguments, to  Iris'
    cube.intersection() method. If the interval_type value is not recognised then
    a left-closed interval is assumed.
    """
    if interval_type == INTERVAL_OPEN:
        return (False, False)
    elif interval_type == INTERVAL_LEFTOPEN:
        return (False, True)
    elif interval_type == INTERVAL_CLOSED:
        return (True, True)
    else:
        return (True, False)
