# (C) British Crown Copyright 2021, Met Office
#
# See the LICENSE.TXT file included with the Afterburner
# software distribution for full license details.
"""
This module implements the Afterburner InlineRegridder app, an application for
regridding climate model data as it is being generated by a climate simulation.

Refer to the :class:`InlineRegridder` class documentation for further information.
"""
from __future__ import (absolute_import, division, print_function)
from six.moves import (filter, input, map, range, zip)

import os
import ast
import pprint
import logging
from glob import glob

import cf_units as cfu
import numpy.ma as ma

import iris
import iris.analysis
from iris.exceptions import IrisError

from afterburner.apps import AbstractApp
from afterburner.exceptions import AppConfigError, DataProcessingError
from afterburner.filename_providers import TemplateDrivenFilenameProvider
from afterburner.modelmeta import cf_cell_method_from_lbproc, is_msi_stash_code
from afterburner.io import NetcdfFileWriter
from afterburner.utils import (is_true, get_class_object_from_class_path,
    get_cylc_task_work_dir, get_cylc_variables, NamespacePlus)
from afterburner.utils.cubeutils import (set_history_attribute,
    make_calendar_type_cube_func, make_cell_method_cube_func)
from afterburner.utils.fileutils import expand_path
from afterburner.utils.textutils import decode_string_value

try:
    from functools import lru_cache
except ImportError:
    from afterburner.utils import lru_cache

# IDs of namelists used in the application configuration file.
GRIDS_NAMELIST = 'grids'
REGRIDDERS_NAMELIST = 'regridders'
DIAGS_NAMELIST = 'diagnostics'
DEFAULT_DIAGNOSTIC_NAME = '_defaults_'

# Dictionary of mappings from model calendar type to hourly-mean LBTIM value.
CAL_LBTIM_MAP = {
    cfu.CALENDAR_STANDARD: 121,
    cfu.CALENDAR_GREGORIAN: 121,
    cfu.CALENDAR_PROLEPTIC_GREGORIAN: 121,
    cfu.CALENDAR_360_DAY: 122,
    cfu.CALENDAR_365_DAY: 124,
}

# Default filename templates for all supported input file formats.
# The {dotstream} token represents a stream name with a '.' character at index
# location 1 (0-based), e.g. 'a.pm'.
FILENAME_TEMPLATES = {
    'ff': '{runid}{dotstream}*',
    'pp': '{runid}{dotstream}*.pp',
    'nc': '{runid}{realm}_{frequency}*.nc',
}

# Create a logger object.
_logger = logging.getLogger(__name__)

# Dictionary for caching regridding functions.
_cached_regridders = dict()


class InlineRegridder(AbstractApp):
    """
    Implements the Afterburner InlineRegridder app, an application for regridding
    climate model diagnostics as they are being generated by a climate simulation.
    The app will typically be configured to regrid a selected subset of diagnostics
    at the end of each model integration step (or some other user-defined cycle
    point).

    The source model data can be in any file format recognised by Iris, e.g.
    UM fieldsfile, UM PP, or netCDF format. The regridded data is output, by
    default, in netCDF-4 classic format.

    At present, the supported regridding schemes are those recognized by
    `Iris <https://scitools-iris.readthedocs.io/en/latest/userguide/interpolation_and_regridding.html>`_.
    In addition to performing the regridding of model data, the app may also be
    configured to remove a rim of a user-defined width, or apply a land-sea mask,
    both as precursor steps to the main regridding task.

    The app is configured via an INI-style text file conforming to Rose's
    extended INI file format. For more information, see the
    :doc:`app documentation </rose_apps/inline_regridder/guide>`

    Model diagnostics to be regridded are identified either by STASH code or by
    CF standard name. Extra attributes (e.g. LBPROC OR LBTIM in the case of UM PP
    data) may be required to distinguish between like-named diagnostics that are
    output at different meaning periods.

    If the app is run under the control of a Rose/cylc suite then the following
    environment variables are referenced by the app:

    * CYLC_SUITE_NAME
    * CYLC_TASK_NAME
    * CYLC_TASK_WORK_DIR
    """

    def __init__(self, arglist=None, **kwargs):
        """
        :param list arglist: List of raw options and/or arguments passed from the
            calling environment. Typically these will be unprocessed command-line
            arguments, e.g. ``['-f', 'foo', '--foe=fum', 'infile']``.
        """
        super(InlineRegridder, self).__init__(version='1.0.0b1', **kwargs)

        # Parse command-line arguments and, if one was specified, a Rose
        # configuration file.
        self._parse_args(arglist, desc="Inline Regridder: regrids climate model data.")
        self._parse_app_config()

        self._set_message_level()
        self.logger.info("Initialising the InlineRegridder app...")

        # Record any CYLC_* environment variables that have been defined by the
        # calling environment.
        self.cylc = get_cylc_variables()

        # A FilenameProvider object which yields the names of input files for a
        # given diagnostic.
        self.input_filename_provider = None

        # A FilenameProvider object which yields the names of output files for a
        # given diagnostic.
        self.output_filename_provider = None

        # Read general options from app config object.
        self.gen_options = self._get_general_options()

        # Read model options from app config object.
        self.model_options = self._get_model_options()

        # Read netCDF saver options from app config object and instantiate a
        # netCDF file writer object.
        netcdf_options = self._get_netcdf_saver_options()
        self.netcdf_writer = NetcdfFileWriter(**vars(netcdf_options))

        # Read target grid definitions from app config object.
        self.grid_defns = self._get_grid_definitions()

        # Read regridder definitions from app config object.
        self.regridder_defns = self._get_regridder_definitions()

        # Read default diagnostic options from app config object.
        self.diag_defaults = self._get_diag_defaults()

        # Read diag definitions from app config object.
        self.diag_defns = self._get_diag_definitions()

    @property
    def cli_spec(self):
        """Defines the command-line interface specification for the application."""
        return [
            {'names': ['-c', '--config-file'],
                'help': 'Pathname of app configuration file'},
        ]

    def run(self, *args, **kwargs):
        """Runs the InlineRegridder application."""

        self.logger.info("Running the InlineRegridder app...")

        # Create any required directories.
        self._create_directories()

        # Create any required filename provider objects.
        self._create_filename_providers()

        for stream in _get_stream_list(self.diag_defns):
            self._process_stream(stream)

        self.logger.info("\nInlineRegridder app execution completed.")

    def _sorted_diag_keys(self, enabled_only=False):
        """
        Return a list of lexically sorted diagnostic keys.

        :param bool enabled_only: If this argument evaluates to true then only
            keys for enabled diagnostics are returned.
        :returns: A list of the diagnostic keys defined in the app config file.
        """
        keys = sorted(self.diag_defns)
        if enabled_only:
            keys = [k for k in keys if self.diag_defns[k].enabled]
        return keys

    def _get_general_options(self, section='general'):
        """
        Read general config options from the app config file.

        :returns: An afterburner.utils.NamespacePlus object whose attributes
            record options defined under the [general] section of the app config
            file.
        """

        try:
            ddict = self.app_config.section_to_dict(section)
            ddict = {k: decode_string_value(v) for k, v in ddict.items()}
            gen_opts = NamespacePlus(**ddict)
        except ValueError:
            raise AppConfigError("Unable to find a section named '{0}' in the "
                "app config file".format(section))

        # Action to take when processing errors are encountered during regridding.
        if gen_opts.abort_on_error is None:
            gen_opts.abort_on_error = True

        # Whether or not to use cached regridder objects.
        if gen_opts.use_cached_regridders is None:
            gen_opts.use_cached_regridders = True

        # Input file format and filename template.
        gen_opts.input_file_format = ifmt = gen_opts.input_file_format or 'pp'
        if not gen_opts.input_filename_template:
            gen_opts.input_filename_template = FILENAME_TEMPLATES.get(ifmt,
                FILENAME_TEMPLATES['pp'])

        # Output file format and save options.
        gen_opts.output_file_format = gen_opts.output_file_format or 'nc'

        # Model input and output directories.
        gen_opts.model_data_dir = expand_path(gen_opts.model_data_dir or '$DATAM')
        if gen_opts.output_dir:
            gen_opts.output_dir = expand_path(gen_opts.output_dir)
        else:
            gen_opts.output_dir = os.path.join(gen_opts.model_data_dir, 'regridded')

        # Datetime string format.
        gen_opts.datetime_format = gen_opts.datetime_format or '%Y%m%d'

        return gen_opts

    def _get_model_options(self):
        """
        Read config options for various climate models and store them in a
        dictionary of afterburner.utils.NamespacePlus objects. Note, however,
        that only UM options currently get used.

        :returns: A dictionary of afterburner.utils.NamespacePlus objects, one
            for each of the following models: um, nemo, cice, jules.
        """

        model_opts = {}

        for section in ['um', 'nemo', 'cice', 'jules']:
            try:
                model_name = section.upper()
                mdict = self.app_config.section_to_dict(section)
                mdict = {k: decode_string_value(v) for k, v in mdict.items()}
                opts = NamespacePlus(**mdict)
                opts.namelist_id = section

                # If the sentinel_file_ext option is defined, make sure it starts
                # with a '.' character.
                if opts.sentinel_file_ext and opts.sentinel_file_ext[0] != '.':
                    opts.sentinel_file_ext = '.' + opts.sentinel_file_ext

                model_opts[model_name] = opts
            except Exception:
                pass

        return model_opts

    def _get_netcdf_saver_options(self, section='netcdf_saver'):
        """
        Read netCDF saver options from the app config file.

        :returns: An afterburner.utils.NamespacePlus object whose attributes
            record options defined under the [netcdf_saver] section of the app
            config file.
        """

        try:
            ddict = self.app_config.section_to_dict(section)
            netcdf_opts = NamespacePlus(**ddict)
        except ValueError:
            raise AppConfigError("Unable to find a section named '{0}' in the "
                "app config file".format(section))

        if netcdf_opts.overwrite:
            netcdf_opts.overwrite = is_true(netcdf_opts.overwrite)

        if netcdf_opts.append:
            netcdf_opts.append = is_true(netcdf_opts.append)

        if netcdf_opts.unlimited_dimensions:
            netcdf_opts.unlimited_dimensions = [s.strip() for s in
                netcdf_opts.unlimited_dimensions.split(',')]

        if netcdf_opts.zlib:
            netcdf_opts.zlib = is_true(netcdf_opts.zlib)

        if netcdf_opts.shuffle:
            netcdf_opts.shuffle = is_true(netcdf_opts.shuffle)

        if netcdf_opts.fletcher32:
            netcdf_opts.fletcher32 = is_true(netcdf_opts.fletcher32)

        if netcdf_opts.contiguous:
            netcdf_opts.contiguous = is_true(netcdf_opts.contiguous)

        if netcdf_opts.complevel:
            netcdf_opts.complevel = int(netcdf_opts.complevel)

        if netcdf_opts.least_significant_digit:
            netcdf_opts.least_significant_digit = int(netcdf_opts.least_significant_digit)

        self.logger.debug("NetCDF saver options:\n%s", pprint.pformat(
            vars(netcdf_opts)))

        return netcdf_opts

    def _get_grid_definitions(self):
        """
        Read all of the target grid definitions from the GRIDS_NAMELIST portion
        of the configuration file.

        :returns: A dictionary of afterburner.utils.NamespacePlus objects, each
            one specifying a grid definition.
        """

        nerrors = 0
        grid_defns = {}

        for ddict in self.app_config.iter_nl(GRIDS_NAMELIST):
            idx = ddict.pop('_index')
            ddict = {k: decode_string_value(v) for k, v in ddict.items()}
            gdefn = NamespacePlus(**ddict)
            gdefn.namelist_id = idx

            # Check that the referenced grid file exists.
            gdefn.file_path = expand_path(gdefn.file_path)
            if not os.path.isfile(gdefn.file_path):
                msg = "Grid file {0} does not exist.".format(gdefn.file_path)
                self.logger.warning(msg)

            grid_defns[gdefn.namelist_id] = gdefn

        if nerrors:
            msg = "{0} error(s) encountered in grid definitions.".format(nerrors)
            self.logger.error(msg)
            raise AppConfigError(msg)

        self.logger.debug("Read in %d grid definitions, as named below:\n\t%s",
            len(grid_defns), '\n\t'.join(grid_defns.keys()))

        return grid_defns

    def _get_regridder_definitions(self):
        """
        Read all of the regridder definitions from the REGRIDDERS_NAMELIST
        portion of the configuration file.

        :returns: A dictionary of afterburner.utils.NamespacePlus objects, each
            one specifying a regridder definition.
        """

        nerrors = 0
        regridder_defns = {}

        for ddict in self.app_config.iter_nl(REGRIDDERS_NAMELIST):
            idx = ddict.pop('_index')
            ddict = {k: decode_string_value(v) for k, v in ddict.items()}
            gdefn = NamespacePlus(**ddict)
            gdefn.namelist_id = idx

            # Check that the target_grid option references a valid grid id.
            if gdefn.target_grid not in self.grid_defns:
                msg = ("Regridder definition '{0}' references target grid with "
                    "unrecognised identifier: '{1}'.".format(idx, gdefn.target_grid))
                self.logger.error(msg)
                nerrors += 1

            # Parse any arguments that are to be passed to the scheme object.
            if gdefn.scheme_args:
                arglist = gdefn.scheme_args.strip('()')
                args = {k.strip(): ast.literal_eval(v) for k, v in [s.split('=')
                    for s in arglist.split(',')]}
                self.logger.debug("Regridding scheme arguments:\n%s", args)
            else:
                args = {}
            gdefn.scheme_args_dict = args

            regridder_defns[gdefn.namelist_id] = gdefn

        if nerrors:
            msg = "{0} error(s) encountered in regridder definitions.".format(nerrors)
            self.logger.error(msg)
            raise AppConfigError(msg)

        self.logger.debug("Read in %d regridder definitions, as named below:\n\t%s",
            len(regridder_defns), '\n\t'.join(regridder_defns.keys()))

        return regridder_defns

    def _get_diag_defaults(self):
        """
        Read default options from the diagnostic named '_defaults_' in the
        DIAGS_NAMELIST portion of the configuration file.

        :returns: An afterburner.utils.NamespacePlus object whose attributes
            record diagnostic default settings.
        """

        diag_defs = None
        for ddict in self.app_config.iter_nl(DIAGS_NAMELIST):
            if ddict.get('_index', '') == DEFAULT_DIAGNOSTIC_NAME:
                idx = ddict.pop('_index')
                ddict = {k: decode_string_value(v) for k, v in ddict.items()}
                diag_defs = NamespacePlus(**ddict)
                diag_defs.namelist_id = idx
                break

        if not diag_defs:
            raise AppConfigError("Unable to find a diagnostic named '{0}' "
                "in the app config file.".format(DEFAULT_DIAGNOSTIC_NAME))

        if diag_defs.enabled is None:
            diag_defs.enabled = True

        # Assign default values for model and simulation attributes.
        if not diag_defs.model_name:
            diag_defs.model_name = 'UM'
        if not diag_defs.suite_name:
            diag_defs.suite_name = self.cylc.suite_name or 'UNDEF'
            self.logger.debug("Suite name is %s", diag_defs.suite_name)
        if not diag_defs.streams:
            diag_defs.streams = 'apy'

        # Create aliases of key attributes for use in metavariables and filename
        # templates.
        diag_defs.model = diag_defs.model_name
        diag_defs.suite = diag_defs.suite_id = diag_defs.suite_name
        diag_defs.runid = diag_defs.suite_name.split('-')[-1]
        diag_defs.realization_id = diag_defs.realization

        # Assign default values for lbtim, lbproc, calendar, and reinit interval.
        if not diag_defs.calendar:
            diag_defs.calendar = cfu.CALENDAR_360_DAY
        lbtim_from_cal = CAL_LBTIM_MAP.get(diag_defs.calendar, 122)
        if diag_defs.lbtim is None:
            diag_defs.lbtim = lbtim_from_cal
        if diag_defs.lbproc is None:
            diag_defs.lbproc = 128
        if diag_defs.reinit is None:
            diag_defs.reinit = 0

        # Default values for land-sea mask and rim.
        diag_defs.apply_lsm = is_true(diag_defs.apply_lsm)
        diag_defs.remove_rim = is_true(diag_defs.remove_rim)
        if diag_defs.rim_width is None:
            diag_defs.rim_width = 0

        self.logger.debug("Default diagnostic options:\n%s", pprint.pformat(
            vars(diag_defs)))

        return diag_defs

    def _get_diag_definitions(self):
        """
        Read all of the target diagnostic definitions from the DIAGS_NAMELIST
        portion of the configuration file.

        :returns: A dictionary of afterburner.utils.NamespacePlus objects, each
            one specifying a diagnostic definition.
        """

        defaults = self.diag_defaults
        diag_defns = {}
        nerrors = 0

        for ddict in self.app_config.iter_nl(DIAGS_NAMELIST):
            if ddict.get('_index', '') == DEFAULT_DIAGNOSTIC_NAME:
                continue
            idx = ddict.pop('_index')
            ddict = {k: decode_string_value(v) for k, v in ddict.items()}
            ddefn = NamespacePlus(**ddict)
            ddefn.key = idx
            ddefn.namelist_id = idx

            # Copy over diagnostic default values for any undefined attributes.
            if defaults:
                for att in defaults.iter_names():
                    if getattr(ddefn, att, None) is None:
                        setattr(ddefn, att, getattr(defaults, att))

            # Check that the regridder option references a valid regridder id.
            if ddefn.regridder not in self.regridder_defns:
                msg = ("Diagnostic definition '{0}' references regridder with "
                    "unrecognised identifier: '{1}'.".format(idx, ddefn.regridder))
                self.logger.error(msg)
                nerrors += 1

            diag_defns[ddefn.key] = ddefn

        if nerrors:
            msg = "{0} error(s) encountered in diagnostic definitions.".format(nerrors)
            self.logger.error(msg)
            raise AppConfigError(msg)

        self.logger.debug("Read in %d diagnostic definitions, as named below:\n\t%s",
            len(diag_defns), '\n\t'.join(diag_defns.keys()))

        return diag_defns

    def _augment_diag_definition(self, diag_defn, stream):
        """
        Augment a diagnostic definition with attributes taken from the current
        context and stream.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param str stream: The name of the current stream.
        """
        diag_defn.stream = diag_defn.stream_id = stream
        diag_defn.realm = stream[0]
        diag_defn.dotstream = stream[0] + '.' + stream[1:]

        if is_msi_stash_code(diag_defn.var_id or ''):
            diag_defn.stash_code = diag_defn.var_id

        diag_defn.cycle_point = self.cylc.task_cycle_point or 'R1'

    def _create_filename_providers(self):
        """Create any filename provider objects required by the app."""

        # Create input filename provider based on the user-defined template string.
        # Any env variables in the template are expanded on a one-time basis.
        templ = os.path.expandvars(self.gen_options.input_filename_template)
        self.input_filename_provider = TemplateDrivenFilenameProvider(templ)

        # Create output filename provider based on the user-defined template string.
        # Any env variables in the template are expanded on a one-time basis.
        templ = os.path.expandvars(self.gen_options.output_filename_template)
        self.output_filename_provider = TemplateDrivenFilenameProvider(templ)

    def _create_directories(self):
        """Create any directories required by the app."""

        for dirpath in [self.gen_options.output_dir]:
            if not os.path.exists(dirpath):
                try:
                    os.makedirs(dirpath)
                    self.logger.info("Created directory: %s", dirpath)
                except OSError:
                    self.logger.error("Unable to create directory: %s", dirpath)
                    raise

    def _process_stream(self, stream):
        """
        Process all diagnostics that have been enabled for the specified stream.
        If no data can be found for the stream in the configured model data
        directory then the function emits an info message and returns.

        :params str stream: The name of the stream to process.
        """

        text = "Processing {0} stream diagnostics".format(stream)
        self.logger.info('\n'+text)
        self.logger.info('='*len(text))

        # Obtain a list of all enabled diagnostic keys for the current stream.
        diag_keys = _get_stream_diag_list(stream, self.diag_defns, enabled_only=True)
        if not diag_keys:
            self.logger.info("No diagnostics enabled for this stream")
            return

        # Load data for all diagnostics required from the current stream.
        try:
            stream_data = self._load_stream_data(stream, diag_keys)
        except (IOError, IrisError) as exc:
            self.logger.error(str(exc))
            if self.gen_options.abort_on_error:
                self.logger.info("Error encountered. Aborting further processing.")
                raise
            else:
                self.logger.info("Error encountered. Skipping to next stream.")
                return

        if not stream_data:
            self.logger.info("No model data found for this stream.")
            return

        # Loop over all target diagnostics.
        for diag_key in diag_keys:
            diag_defn = self.diag_defns[diag_key].copy()
            self._augment_diag_definition(diag_defn, stream)

            try:
                self._process_diagnostic(stream_data, diag_defn)

            except (IOError, ValueError, AppConfigError, DataProcessingError) as exc:
                self.logger.error(str(exc))
                if self.gen_options.abort_on_error:
                    self.logger.info("Error encountered. Aborting further processing.")
                    raise
                else:
                    self.logger.info("Error encountered. Skipping to next diagnostic.")

        self.logger.info("Completed processing stream %s.", stream)

    def _process_diagnostic(self, stream_data, diag_defn):
        """
        Process the specified diagnostic. If no data can be found for the diagnostic
        in the stream_data cubelist then the function emits an info message and
        returns.

        :param iris.cube.CubeList stream_data: A cubelist containing multiple
            diagnostics from a single model output stream.
        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            defining the diagnostic to process.
        """

        text = "Regridding diagnostic {0}...".format(diag_defn.key)
        self.logger.info('\n'+text)
        self.logger.info('-'*len(text))

        diag_cube = self._extract_diag_data(stream_data, diag_defn)
        if not diag_cube:
            self.logger.info("No model data found for this diagnostic.")
            return
        self.logger.info("Input cube: %s", diag_cube.summary(shorten=True))

        # Attach start and end date strings to the diag_defn object in case they
        # are needed by a filename provider.
        diag_defn.data_start_date, diag_defn.data_end_date = _get_datetime_extent(
            diag_cube, dtformat=self.gen_options.datetime_format)
        self.logger.debug("Data start & end dates: %s, %s", diag_defn.data_start_date,
            diag_defn.data_end_date)

        if diag_defn.remove_rim:
            diag_cube = _remove_rim(diag_cube, diag_defn.rim_width)
            self.logger.info("Input cube (w/o rim): %s", diag_cube.summary(shorten=True))

        if diag_defn.apply_lsm:
            self._apply_landsea_mask(diag_defn, diag_cube)

        result_cube = self._regrid_data(diag_defn, diag_cube)

        self._update_metadata(diag_defn, result_cube)

        self._save_data(diag_defn, result_cube)

        self.logger.info("Completed diagnostic.")

    def _load_stream_data(self, stream, diag_keys):
        """
        Load data for all enabled diagnostics for the specified stream. This is
        done to avoid repeatedly reading the same data files for different
        diagnostics.

        :param str stream: The name of the stream for which to load model data.
        :param list diag_keys: A list of diagnostic keys.
        :returns: An Iris cubelist containing model data for the specified
            stream-diagnostic combination. The cubelist will be empty if no data
            could be loaded.
        """

        self.logger.info("Loading model data for stream %s...", stream)
        cubes = iris.cube.CubeList()

        data_dir = self.gen_options.model_data_dir
        if not os.path.isdir(data_dir):
            self.logger.warning("Data directory does not exist: %s", data_dir)
            return cubes

        # Create a list of Iris load constraints, one for each target diagnostic.
        constraints = []
        var_ids = set(self.diag_defns[k].var_id for k in diag_keys)
        for var_id in var_ids:
            if is_msi_stash_code(var_id):
                constraints.append(iris.AttributeConstraint(STASH=var_id))
            else:
                constraints.append(iris.Constraint(name=var_id))

        # Take a copy of the first diagnostic definition and augment it with realm,
        # stream and dotstream attributes based upon the current stream.
        diag_defn = self.diag_defns[diag_keys[0]].copy()
        self._augment_diag_definition(diag_defn, stream)

        # Obtain a list of names of potential input files and, if running as
        # a cylc task, filter the list by sentinel files (if configured).
        filenames = self.input_filename_provider.get_filenames(diag_defn)
        self.logger.debug("Candidate model data input filenames:\n%s", filenames)

        if self.cylc.is_active:
            filenames = self._filter_by_sentinel_files(diag_defn, filenames)
            self.logger.debug("Filtered model data filenames:\n%s", filenames)

        if not filenames:
            self.logger.warning("No matching filenames found for this stream.")
            return cubes

        filepaths = [os.path.join(data_dir, f) for f in filenames]
#        self.logger.debug("Candidate filepaths:\n%s", filepaths)
        cubes = iris.load(filepaths, constraints)
#        self.logger.debug("Loaded cubes:\n%s", cubes)

        return cubes

    def _extract_diag_data(self, stream_data, diag_defn):
        """
        Extract model data for the specified diagnostic from the stream_data
        cubelist. Ordinarily the diag_defn object should result in a single cube
        being extracted and returned. If no data could be found then the function
        returns None. Otherwise, if multiple cubes are found, then the function
        raises a DataProcessingError exception.

        :param iris.cube.CubeList stream_data: A cubelist containing multiple
            diagnostics from a single model output stream.
        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            defining the diagnostic whose data is to be extracted.
        :returns: An Iris cube containing data for the specified diagnostic, or
            None if no data was found.
        """

        fmt = self.gen_options.input_file_format.lower()
        if fmt in ['ff', 'pp']:
            cubes = _extract_pp_model_data(stream_data, diag_defn)
        elif fmt in ['nc', 'netcdf']:
            cubes = _extract_nc_model_data(stream_data, diag_defn)
        else:
            raise AppConfigError("Unrecognised input file format: " + fmt)

        ncubes = len(cubes)
        if ncubes == 0:
            return None
        elif ncubes == 1:
            cube = cubes[0]
        else:
            msg = "Expected a single cube, but got {0}.".format(len(cubes))
            self.logger.error(msg)
            self.logger.error("Loaded cubes are as follows:\n%s", str(cubes))
            raise DataProcessingError(msg)

        # Check that we have coordinate bounds - needed during regridding.
        lat = cube.coord('latitude')
        if not lat.has_bounds():
            lat.guess_bounds()
        lon = cube.coord('longitude')
        if not lon.has_bounds():
            lon.guess_bounds()

        return cube

    def _apply_landsea_mask(self, diag_defn, diag_cube):
        """
        Apply a land-sea mask to a cube. The cube is modified in situ.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube diag_cube: The cube of diagnostic data to which
            the land-sea mask is to be applied.
        """

        self.logger.info("Applying land-sea mask...")

        # Load the land-sea mask data.
        if not diag_defn.lsm_grid:
            msg = "No land-sea mask grid defined for current diagnostic."
            self.logger.error(msg)
            raise AppConfigError(msg)
        grid_defn = self.grid_defns[diag_defn.lsm_grid]
        lsm_cube = _load_grid_data(grid_defn.file_path, grid_defn.var_id)
        self.logger.debug("LSM cube: %s", lsm_cube.summary(shorten=True))

        # If a rim has been, or will be, removed from the original diagnostic
        # data then we assume it should also be removed from the land-sea mask.
        if diag_defn.remove_rim:
            lsm_cube = _remove_rim(lsm_cube, diag_defn.rim_width)
            self.logger.debug("LSM cube (w/o rim): %s", lsm_cube.summary(shorten=True))

        # Check that the cubes of diagnostic data and LSM data have the same shape.
        if lsm_cube.shape != diag_cube.shape[-2:]:
            raise DataProcessingError("Shape of land-sea mask grid {0}\n"
                "does not match that of diagnostic {1}.".format(lsm_cube.shape,
                diag_cube.shape[-2:]))

        # Obtain a handle to the LSM data array and, if it isn't already masked,
        # mask elements whose array value is equal to 0.
        lsm_data = lsm_cube.data
        if not ma.is_masked(lsm_data):
            lsm_data = ma.masked_where(lsm_data == 0, lsm_data)

        diag_cube.data = diag_cube.data * lsm_data

    def _regrid_data(self, diag_defn, diag_cube):
        """
        Regrid the specified cube of diagnostic data using the scheme defined in
        the app config file. This function checks for, and if necessary adds, a
        regridder object to a cache of such objects.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube diag_cube: The cube of diagnostic data to be
            regridded.
        :returns: An Iris cube containing the regridded data.
        """

        # Obtain a handle to the associated regridder definition.
        regridder_defn = self.regridder_defns[diag_defn.regridder]

        # Load the target grid.
        grid_defn = self.grid_defns[regridder_defn.target_grid]
        target_grid = _load_grid_data(grid_defn.file_path, grid_defn.var_id)

        if self.gen_options.use_cached_regridders:
            # Obtain the required regridder object from cache.
            regridder = _get_regridder(regridder_defn, diag_cube, target_grid)
        else:
            # Create a regridder object from the diag and target grid cubes.
            self.logger.info("Creating regridder object from data cube to "
                " target grid '%s'...", regridder_defn.target_grid)
            scheme_class = get_class_object_from_class_path(regridder_defn.scheme)
            scheme = scheme_class(**regridder_defn.scheme_args_dict)
            regridder = scheme.regridder(diag_cube, target_grid)

        # Regrid the source data.
        self.logger.info("Regridding model data using '%s' regridder...",
            diag_defn.regridder)
        cube = regridder(diag_cube)
        self.logger.debug("Regridded cube: %s", cube.summary(shorten=True))

        # Set the cube's history attribute.
        history = ("Data generated by the InlineRegridder app using the {0} "
            "regridding scheme.".format(regridder_defn.scheme))
        set_history_attribute(cube, history)

        return cube

    def _update_metadata(self, diag_defn, cube):
        """
        Attach any desirable extra attributes to the cube.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube cube: The cube whose metadata is to be updated.
        """

        self.logger.info("Updating metadata...")

        # Add standard_name, long_name, var_name, and units attributes to the
        # cube if they are attached to diag_defn.
        for attname in ['standard_name', 'long_name', 'var_name', 'units']:
            attval = getattr(diag_defn, attname, None)
            if attval is not None:
                setattr(cube, attname, attval)

    def _save_data(self, diag_defn, cube):
        """
        Save a cube of regridded model data to the output directory defined in
        the app config file. The name of the output file is generated from the
        attributes attached to the diag_defn object.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube cube: The cube to save.
        """

        filename = _make_output_filename(diag_defn, self.output_filename_provider)
        filepath = os.path.join(self.gen_options.output_dir, filename)
        self.logger.info("Saving regridded data to file %s...", filename)

        # The template-supplied filename might include '/' characters so the
        # following code attempts to create any intermediate subdirectories.
        dirpath = os.path.dirname(filepath)
        if not os.path.exists(dirpath):
            os.makedirs(dirpath)

        self.netcdf_writer.run(cube, filepath)

    def _filter_by_sentinel_files(self, diag_defn, filenames):
        """
        Takes a list of input filenames and filters out those files which do
        not have an associated sentinel file, e.g. one with a .arch extension
        in the case of UM model output.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param list filenames: The list of filenames to filter.
        :returns: A list, possibly empty, of filtered filenames.
        """
        self.logger.debug("Filtering input files against sentinel files...")

        # Obtain the sentinel filename extension and the name of the cylc task
        # associated with the model that has generated the diagnostic
        model_opts = self.model_options[diag_defn.model_name]
        sentinel_file_ext = model_opts.sentinel_file_ext
        task_name = model_opts.cylc_task_name
        task_work_dir = get_cylc_task_work_dir(task_name)

        if not (sentinel_file_ext and task_work_dir and os.path.isdir(task_work_dir)):
            return filenames

        # Extract the input filename extension, if one is included in the
        # associated filename template.
        ipext = ''
        for ext in ['.ff', '.pp', '.nc']:
            if self.gen_options.input_filename_template.endswith(ext):
                ipext = ext

        # Obtain a list of sentinel files. If none found then no match against
        # input data files is possible so return an empty list.
        cwd = os.getcwd()
        os.chdir(task_work_dir)
        sentinel_files = glob('*'+sentinel_file_ext)
        os.chdir(cwd)
        if not sentinel_files:
            return []

        # Replace sentinel file extension (e.g. '.arch') with the input filename
        # extension (which might be blank).
        sentinel_files = {os.path.splitext(f)[0]+ipext for f in sentinel_files}

        # Expand the (possibly wildcarded) names of any/all input data files.
        os.chdir(self.gen_options.model_data_dir)
        input_files = set()
        for fname in filenames:
            files = glob(fname)
            input_files.update(files)
        os.chdir(cwd)

        # Remove any input files that do not have a matching sentinel file.
        # This is achieved by finding the intersection between the set of input
        # files and the set of sentinel files.
        input_files.intersection_update(sentinel_files)

        return list(input_files)


def _get_stream_list(diag_defns):
    """
    Determine the full list of target streams for which diagnostics have been
    specified. The returned list is sorted alphabetically.

    :param dict diag_defns: A dictionary of diagnostic definition objects.
    :returns: A list of unique stream names.
    """
    stream_list = set()

    for ddefn in diag_defns.values():
        streams = ddefn.streams   # or ddefn.stream
        if streams:
            stream_list.update([s.strip() for s in streams.split(',')])

    return sorted(stream_list)


def _get_stream_diag_list(stream, diag_defns, enabled_only=False):
    """
    Determine the full list of diagnostic keys associated with the specified
    stream.

    :param str stream: The name of the stream.
    :param dict diag_defns: A dictionary of diagnostic definition objects.
    :param bool enabled_only: If this argument evaluates to true then only
        user-enabled diagnostics are included in the returned list.
    :returns: A list of keys to diagnostic definitions.
    """
    diag_list = []

    for ddefn in diag_defns.values():
        if enabled_only and not ddefn.enabled:
            continue
        streams = ddefn.streams   # or ddefn.stream
        if streams and stream in [s.strip() for s in streams.split(',')]:
            diag_list.append(ddefn.key)

    return sorted(diag_list)


def _get_datetime_extent(cube, dtformat='%Y%m%d'):
    """
    Return the datetime extent covered by the specified cube. If the time coordinate
    contains cell bounds then those are used to determine the extent. Otherwise
    the actual time points are used.

    The time coordinate is identified by the name 'time' or, if that fails, an
    axis label of 'T'.

    :param iris.cube.Cube cube: The Iris cube to inspect.
    :param str dtformat: The format to use for constructing the returned datetime
        strings. The default format is '%Y%m%d'.
    :returns: A 2-tuple of strings (start_date, end_date).
    """

    tcoord = None
    for kwds in [{'name_or_coord': 'time'}, {'axis': 'T'}]:
        try:
            tcoords = cube.coords(**kwds)
            if tcoords:
                tcoord = tcoords[0]  # assumes first/only coord is the best one to use
                break
        except iris.exceptions.CoordinateNotFoundError:
            continue

    if not tcoord:
        raise DataProcessingError("Unable to find a time coordinate in cube of "
            "input diagnostic data.")

    if tcoord.has_bounds():
        start_dt = tcoord.units.num2date(tcoord.bounds[0,0])
        end_dt = tcoord.units.num2date(tcoord.bounds[-1,1])
    else:
        start_dt = tcoord.units.num2date(tcoord.points[0])
        end_dt = tcoord.units.num2date(tcoord.points[-1])

    return start_dt.strftime(dtformat), end_dt.strftime(dtformat)


def _get_regridder(regridder_defn, source_grid, target_grid):
    """
    Get a regridder object capable of regridding data from source grid to
    target_grid. If a suitable regridder already exists in the cache of such
    objects then it is returned. Otherwise a new regridder object is created,
    added to the cache, and returned.

    :param afterburner.utils.NamespacePlus regridder_defn: A regridder definition
        object.
    :param iris.cube.Cube source_grid: A cube that encapsulates the definition
        of the source grid.
    :param iris.cube.Cube target_grid: A cube that encapsulates the definition
        of the target grid.
    :returns: An iris regridder (callable) object.
    """

    # Obtain hash values for the source and target grids.
    source_grid_hash = _calc_2d_grid_hash(source_grid)
    target_grid_hash = _calc_2d_grid_hash(target_grid)

    # Create a cache key based on the grid hash values plus any scheme args.
    scheme_args = regridder_defn.scheme_args or 'none'
    key = (source_grid_hash, target_grid_hash, scheme_args)

    # If the required regridder object doesn't exist in the cache, create it and
    # add it to the cache.
    if key not in _cached_regridders:
        scheme_class = get_class_object_from_class_path(regridder_defn.scheme)
        scheme = scheme_class(**regridder_defn.scheme_args_dict)
        regridder = scheme.regridder(source_grid, target_grid)
        _cached_regridders[key] = regridder
        _logger.info("Added regridder object to cache.")

    return _cached_regridders[key]


def _extract_pp_model_data(stream_data, diag_defn):
    """
    Extract model data for the specified diagnostic from the stream_data cubelist.

    :param iris.cube.CubeList stream_data: A cubelist representing multiple
        diagnostics from a single PP-type model output stream.
    :param afterburner.utils.NamespacePlus diag_defn: A namespace object defining
        the diagnostic to extract from the stream_data cubelist.
    :returns: A cubelist containing all cubes that match the given diagnostic.
    """

    # Add a STASH code or name constraint.
    if is_msi_stash_code(diag_defn.var_id):
        constraints = iris.AttributeConstraint(STASH=diag_defn.var_id)
    else:
        constraints = iris.Constraint(name=diag_defn.var_id)

    # Add a cell method constraint if required.
    if diag_defn.lbproc is not None:
        method = cf_cell_method_from_lbproc(diag_defn.lbproc)
        if diag_defn.lbtim:
            nhours = diag_defn.lbtim // 100
            interval = "{} hour".format(nhours)
        else:
            interval = None
        constraints &= iris.Constraint(
            cube_func=make_cell_method_cube_func(method, 'time', interval=interval))

    cubes = stream_data.extract(constraints)

    return cubes


def _extract_nc_model_data(stream_data, diag_defn):
    """
    Extract model data for the specified diagnostic from the stream_data cubelist.

    :param iris.cube.CubeList stream_data: A cubelist representing multiple
        diagnostics from a single netCDF-type model output stream.
    :param afterburner.utils.NamespacePlus diag_defn: A namespace object defining
        the diagnostic to extract from the stream_data cubelist.
    :returns: A cubelist containing all cubes that match the given diagnostic.
    """

    constraints = iris.Constraint(name=diag_defn.var_id)
    # or, to constrain explicitly on the cube.var_name attribute
    # constraints = iris.Constraint(cube_func=lambda c: c.var_name == diag_defn.var_id)

    # Add a cell method constraint if required.
    if diag_defn.lbproc is not None:
        method = cf_cell_method_from_lbproc(diag_defn.lbproc)
        if diag_defn.lbtim:
            nhours = diag_defn.lbtim // 100
            interval = "{} hour".format(nhours)
        else:
            interval = None
        constraints &= iris.Constraint(
            cube_func=make_cell_method_cube_func(method, 'time', interval=interval))

    # Add a calendar constraint if needed.
    if diag_defn.calendar:
        constraints &= iris.Constraint(
            cube_func=make_calendar_type_cube_func(diag_defn.calendar))

    cubes = stream_data.extract(constraints)

    return cubes


@lru_cache(maxsize=32)
def _load_grid_data(file_path, var_id):
    """
    Caching function which reads a gridded variable from the specified file,
    which could be in UM fieldsfile, PP or netCDF format. If the variable has
    previously been read and cached then it is returned, thus avoiding the
    overhead of reloading the data.

    :param str file_path: The pathname to the file containing a gridded variable
        in some Iris-supported format.
    :param str var_id: The identifier of the variable to load. This can be a CF
        standard name, a variable name, or an msi-style STASH code.
    :returns: A cube containing the gridded variable.
    """

    grid_data = None

    try:
        _logger.info("Loading grid data from file %s...", file_path)

        # First try loading cubes based on CF standard name.
        name_constraint = iris.Constraint(name=var_id)
        cubes = iris.load(file_path, name_constraint)

        # If no cubes found then try loading cubes based on STASH code.
        if not len(cubes):
            stash_constraint = iris.AttributeConstraint(STASH=var_id)
            cubes = iris.load(file_path, stash_constraint)

        if len(cubes) != 1:
            msg = ("Expected a single cube of gridded data, but found {0}.\n"
                "Please check the contents of file {1}".format(len(cubes), file_path))
            _logger.error(msg)
            raise DataProcessingError(msg)
        else:
            grid_data = cubes[0]
            lat = grid_data.coord('latitude')
            if not lat.has_bounds():
                lat.guess_bounds()
            lon = grid_data.coord('longitude')
            if not lon.has_bounds():
                lon.guess_bounds()

    except (IOError, OSError, iris.exceptions.IrisError) as exc:
        _logger.error(str(exc))
        raise

    return grid_data


def _make_output_filename(diag_defn, filename_provider, ext=''):
    """
    Construct the name of the output file to use for the diagnostic defined
    by the diag_defn argument. If the constructed filename doesn't include
    an extension then the one defined by the ext argument is used.

    :param afterburner.utils.NamespacePlus diag_defn: A namespace object
        containing the diagnostic definition.
    :param FilenameProvider filename_provider: The filename provider object that
        will be used to generate a filename from the diag_defn object.
    :param str ext: The extension, if any, to append to the filename string.
    :returns: The generated filename.
    """

    errmsg = "Error trying to construct output filename for diagnostic " + diag_defn.key

    try:
        filenames = filename_provider.get_filenames(diag_defn)
    except:
        _logger.error(errmsg)
        raise DataProcessingError(errmsg)

    if filenames:
        filename = filenames[0]
        if ext:
            filename += ext
    else:
        _logger.error(errmsg)
        raise DataProcessingError(errmsg)

    return filename


@lru_cache(maxsize=32)
def _remove_rim(cube, rim_width):
    """
    Caching function which removes a rim with the specified width from each side
    of a cube. If the cube has previously been de-rimmed then the cached version
    is returned.

    :param iris.cube.Cube: The cube from which to remove a rim.
    :param int rim_width: The width of the rim.
    :returns: An Iris cube containing a de-rimmed version of the input cube.
    """

    if rim_width <= 0:
        return cube

    _logger.info("Removing size-%d rim from cube of %s...", rim_width, cube.name())

    # Since we can't know in advance which dimensions are associated with
    # longitude/x and latitude/y, we'll search by axis label and use the
    # cube.subset() function to trim the length of both axes.
    xcoord = cube.coord(axis='X', dim_coords=True)
    ycoord = cube.coord(axis='Y', dim_coords=True)

    rrcube = cube.subset(xcoord[rim_width:-rim_width])
    rrcube = rrcube.subset(ycoord[rim_width:-rim_width])

    return rrcube


def _calc_2d_grid_hash(cube):
    """
    Calculate a hash value for the 2D horizontal grid associated with the passed
    in cube. The hash value is computed by passing a tuple containing the following
    attributes, for both coordinate axes, to the hash() builtin function:
    (axis name, unit name, coord system string representation, axis length,
    first coordinate value, last coordinate value).

    :param iris.cube.Cube: A cube containing X and Y axes from which to compute
        a hash value.
    :returns: An integer hash value.
    """

    xcoord = cube.coord(axis='X', dim_coords=True)
    ycoord = cube.coord(axis='Y', dim_coords=True)

    xprops = (xcoord.name(), xcoord.units.origin, str(xcoord.coord_system),
        xcoord.shape[0], xcoord.points[0], xcoord.points[-1])

    yprops = (ycoord.name(), ycoord.units.origin, str(ycoord.coord_system),
        ycoord.shape[0], ycoord.points[0], ycoord.points[-1])

    return hash(xprops + yprops)
