# (C) British Crown Copyright 2019-2020, Met Office
#
# See the LICENSE.TXT file included with the Afterburner
# software distribution for full license details.
"""
Provides an implementation of the ASoV Calculator application, which enables
scientists to calculate scales-of-variability histogram data for chosen model
diagnostics as they are being generated by a climate simulation.

Refer to the :class:`AsovCalculator` class documentation for further information.
"""
from __future__ import (absolute_import, division, print_function)
from six.moves import (filter, input, map, range, zip)
from six import string_types

import os
import re
import math
import logging

import numpy as np
import cf_units as cfu

import iris
import iris.util
import iris.coords
import iris.analysis
from iris.exceptions import CoordinateNotFoundError

from afterburner.apps.multi_stream_template import MultiStreamTemplateApp
from afterburner.coords import CoordRange
from afterburner.processors.diags import HistogramMaker
from afterburner.exceptions import (AppConfigError, DataProcessingError)
from afterburner.utils import NamespacePlus
from afterburner.utils.cubeutils import (extract_lat_long_region, is_scalar_coord,
    compare_cubes)
from afterburner.utils.dateutils import DateTimeRange
from afterburner.utils.textutils import decode_string_value

# IDs of namelists used in the application configuration file.
DIAGS_NAMELIST = 'diagnostics'
DEFAULT_DIAGNOSTIC_NAME = '_defaults_'

# Dictionary of mappings from model calendar type to hourly-mean LBTIM value.
CAL_LBTIM_MAP = {
    cfu.CALENDAR_STANDARD: 121,
    cfu.CALENDAR_GREGORIAN: 121,
    cfu.CALENDAR_PROLEPTIC_GREGORIAN: 121,
    cfu.CALENDAR_360_DAY: 122,
    cfu.CALENDAR_365_DAY: 124,
}

# Default filename templates for all supported input file formats.
# The {dotstream} token represents a stream name with a '.' character at index
# location 1 (0-based), e.g. 'a.pm'.
FILENAME_TEMPLATES = {
    'ff': '{runid}{dotstream}*',
    'pp': '{runid}{dotstream}*.pp',
}

# Create a logger object.
_logger = logging.getLogger(__name__)


class AsovCalculator(MultiStreamTemplateApp):
    """
    This class implements the ASoV Calculator application. ASoV is short for
    Analysing Scales of Variability. The original methodology was (and is)
    applied to precipitation data but the method can probably be used with any
    suitable scalar model diagnostics/quantities.

    The ASoV Calculator application is primarily designed to be incorporated
    into climate model suites. It can, however, be run in stand-alone mode
    against existing data files on disk.

    In either mode, the app currently reads diagnostic data from model files in
    UM fieldsfile or PP format. All required input files are assumed to reside
    within a single directory. This is the typical situation in the case of
    output from the Met Office Unified Model (where the directory is often named
    'History_Data').

    The actual ASoV frequency (or histogram) data is calculated by counting the
    number of data values which fall within user-defined bins. In Iris terms
    this is a collapse operation over a specified dimension of an input cube
    using the iris.analysis.COUNT aggregation method. By default the collapse
    operation is applied over the time dimension, but this is user-configurable.

    The resulting cubes of ASoV data are saved in netCDF format. In normal mode
    of operation the app adds the data for the current cycle point to a file of
    cumulative data, i.e. containing all data calculated to date. A scalar time
    coordinate in the cumulative file details the *overall* time period spanned
    by the contained data. Ordinarily the data for the current cycle point is not
    saved to disk, but this can be enabled if required. An additional configuration
    option allows users to save the ASoV data for each cycle point to an ever-
    expanding time-series file. This facilitates, for example, the subsequent
    computation of cumulative values for arbitrary time periods.

    To avoid messily long output filenames - and the creation of large numbers
    of files in a single output directory - the ASoV Calculator app writes data
    to netCDF files stored in a hierarchy of subdirectories, the names of which
    are derived from the model input streams and diagnostics specified in the
    app config file. The hierarchy of subdirectories adheres to the following
    scheme::

        <output_dir>
          <stream_id>
            <var_id>
              <region_name>
                cp1.nc     # all filenames are user-configurable
                cp2.nc
                cumulative.nc
                timeseries.nc
                ...

    The pathname (absolute or relative) of ``output_dir`` is defined in the app
    config file; ``var_id`` is usually a UM STASH code, and ``region_name`` is,
    optionally, the name of the geographical region covered by the input data.
    If no region is specified by the user it defaults to 'global' (although for
    regional models the data wouldn't actually have global extent).

    .. warning:: The ASoV Calculator app is designed to be run as part of a
       cycling suite, and assumes that input datasets are supplied to the app in
       chronological order with no time gaps or overlaps. If a problem in the
       calling environment causes such gaps/overlaps then it may not be easy to
       spot them by inspection of the cumulative file (since it doesn't contain
       sufficient time information). If it is important to be able to detect
       such problems then it is recommended that you enable creation of the
       time-series file.
    """

    def __init__(self, arglist=None, **kwargs):
        """
        :param list arglist: List of raw options and/or arguments passed from the
            calling environment. Typically these will be unprocessed command-line
            arguments, e.g. ``['-f', 'foo', '--foe=fum', 'infile']``.
        """
        super(AsovCalculator, self).__init__(arglist=arglist, version='1.0.0b1',
            description="Analysing Scales of Variability (ASoV) Calculator",
            **kwargs)

    @property
    def cli_spec(self):
        """
        Defines the command-line interface specification for the application.
        This property should return a list of dictionary objects, each of which
        specifies a series of parameters to pass through to the add_argument()
        method of the argparse.ArgumentParser class.
        """
        cmd_args = [
            {'names': ['-c', '--config-file'],
                'help': 'Pathname of the app configuration file'},
            {'names': ['--abort-on-error'], 'action': 'store_true',
                'help': 'Abort processing if an error is encountered'},
            {'names': ['-n', '--dry-run'], 'action': 'store_true',
                'help': 'Dry-run only: do not save results to output files'}
        ]

        return cmd_args

    def make_unique_diag_key(self, diag_defn):
        """Return a unique key for the specified diagnostic definition."""

        key = "{0}/lbproc={1}/lbtim={2}".format(diag_defn.var_id, diag_defn.lbproc,
            diag_defn.lbtim)
        if diag_defn.region_name:
            key += '/' +  diag_defn.region_name_token

        return key

    def make_unique_source_key(self, diag_defn):
        """
        Return a unique key for the data source associated with the specified
        diagnostic definition.
        """
        return MultiStreamTemplateApp.make_unique_source_key(self, diag_defn)

    def get_general_options(self, section=None):
        """
        Read general config options from the app config file.

        :returns: An afterburner.utils.NamespacePlus object whose attributes
            record options defined under the [general] section of the app config
            file.
        """

        gen_opts = MultiStreamTemplateApp.get_general_options(self, section=section)

        # Input file format and filename template.
        # These options are now read by the MultiStreamTemplateApp base class.

        # Output file format and various templates.
        # These options are now read by the MultiStreamTemplateApp base class.

        # Datetime string format.
        gen_opts.datetime_format = gen_opts.datetime_format or '%Y%m%d'

        # Default region name to use when one isn't explicitly defined.
        gen_opts.default_region_name = gen_opts.default_region_name or 'global'

        return gen_opts

    def get_diag_definitions(self, namelist=None):
        """
        Read all of the target diagnostic definitions from the DIAGS_NAMELIST
        portion of the configuration file.

        :returns: A dictionary of afterburner.utils.NamespacePlus objects, each
            one specifying a diagnostic definition.
        """

        diag_defns = {}
        namelist = namelist or DIAGS_NAMELIST
        defaults = self.diag_defaults
        nerrors = 0

        for ddict in self.app_config.iter_nl(namelist):
            if ddict.get('_index', '') == DEFAULT_DIAGNOSTIC_NAME: continue
            idx = ddict.pop('_index')
            ddict = {k: decode_string_value(v) for k, v in ddict.items()}
            ddefn = NamespacePlus(**ddict)
            ddefn.namelist_id = idx

            # Copy over diagnostic default values for any undefined attributes.
            if defaults:
                for att in defaults.iter_names():
                    if getattr(ddefn, att, None) is None:
                        setattr(ddefn, att, getattr(defaults, att))

            # Decode the bin array from the bin_values property.
            try:
                ddefn.bin_array = _decode_bin_array_from_string(ddefn.bin_values)
            except:
                msg = ("({0}) Error trying to parse the numeric list or expression\n"
                "defined in the bin_values option.".format(ddefn.namelist_id))
                self.logger.error(msg)
                nerrors += 1

            # Check region definition and set region name if required.
            if ddefn.region_extent:
                if isinstance(ddefn.region_extent, string_types):
                    ddefn.region_extent = [float(s) for s in ddefn.region_extent.split(',')]
                else:
                    ddefn.region_extent = [float(s) for s in ddefn.region_extent]
                if len(ddefn.region_extent) != 4:
                    msg = ("({0}) A region extent must comprise four coordinates\n"
                        "(min-long, min-lat, max-long, max-lat). Got: {1}".format(
                        ddefn.namelist_id, ddefn.region_extent))
                    self.logger.error(msg)
                    nerrors += 1
                    continue
                if not ddefn.region_name:
                    msg = ("({0}) A region name must must also be specified when\n"
                        "a region extent is defined.".format(ddefn.namelist_id))
                    self.logger.error(msg)
                    nerrors += 1
                    continue
            else:
                ddefn.region_name = self.general_opts.default_region_name

            # Tokenise the region name for use in directory or file names.
            ddefn.region_name_token = ddefn.region_name.replace(' ', '-').lower()

            # Assign a unique key for the current diagnostic.
            key = self.make_unique_diag_key(ddefn)

            # Check for uniqueness of diagnostic key.
            if key in diag_defns:
                msg = ("Diagnostic definition '{0}' with key '{1}' is not unique.\n"
                    "Please check app config file for duplicate definitions.".format(
                    idx, key))
                self.logger.error(msg)
                continue

            ddefn.key = key
            diag_defns[key] = ddefn

        if nerrors:
            msg = "{0} error(s) encountered in diagnostic definitions.".format(nerrors)
            self.logger.error(msg)
            raise AppConfigError(msg)

        self.logger.debug("Diagnostic names: %s", ', '.join(sorted(diag_defns)))

        return diag_defns

    def process_diagnostic(self, stream_data, diag_defn):
        """
        Process the specified diagnostic.

        :param cubelist stream_data: A cubelist containing multiple diagnostics
            from a single model output stream.
        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            defining the diagnostic to process.
        """

        text = "Processing diagnostic {0}".format(diag_defn.key)
        self.logger.info('\n'+text)
        self.logger.info('-'*len(text))

        diag_cubes = self.extract_diag_data(stream_data, diag_defn)
        if not diag_cubes:
            self.logger.info("No model data found for this diagnostic.")
            return

        diag_cube = diag_cubes[0]
        self.logger.info("Input model data: %s", diag_cube.summary(shorten=True))

        # Attach start and end date strings (derived from the input data) to the
        # diag_defn object in case they are needed by a filename provider.
        dtr = DateTimeRange.from_cube(diag_cube, use_bounds=True)
        fmt = self.general_opts.datetime_format
        diag_defn.data_start_date = dtr.start_ncdt.strftime(fmt)
        diag_defn.data_end_date = dtr.end_ncdt.strftime(fmt)
        self.logger.debug("Data start & end dates: %s, %s", diag_defn.data_start_date,
            diag_defn.data_end_date)

        if diag_defn.region_extent:
            diag_cube = self._extract_region(diag_defn, diag_cube)

        self._convert_units(diag_defn, diag_cube)

        asov_cube = self._calc_asov_data(diag_defn, diag_cube)

        self._update_metadata(diag_defn, asov_cube)

        if self.general_opts.save_cyclepoint_file:
            self._save_asov_data(diag_defn, asov_cube)

        if self.general_opts.update_cumulative_file:
            self._update_cumulative_file(diag_defn, asov_cube)

        if self.general_opts.update_timeseries_file:
            self._update_timeseries_file(diag_defn, asov_cube)

        self.logger.info("Completed diagnostic.\n")

    @property
    def on_load_callback(self):
        """
        The callback function, if any, to pass to ``iris.load()`` calls within the
        various ``load_stream_data*()`` functions. If no callback is needed then
        return None, this being the default.
        """

        if self.data_reader_opts.input_file_format.lower() in ['ff', 'pp']:
            return _on_load_pp_callback
        else:
            return None

    def _extract_region(self, diag_defn, diag_cube):
        """
        If the user has specified a geographical region for a given diagnostic,
        extract the corresponding spatial subset from the input cube.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube cube: The cube from which to extract a regional
            subset.
        :returns: An Iris cube containing data for the geographical region
            defined on the supplied diagnostic definition.
        """

        minlon, minlat, maxlon, maxlat = diag_defn.region_extent[:]
        self.logger.info("Extracting data for region (%s,%s) -> (%s,%s) ...",
            minlon, minlat, maxlon, maxlat)

        lat_extent = CoordRange([minlat, maxlat], leftclosed=True)
        lon_extent = CoordRange([minlon, maxlon], leftclosed=True)

        try:
            region_cube = extract_lat_long_region(diag_cube, lat_extent, lon_extent)
        except CoordinateNotFoundError as exc:
            self.logger.error(str(exc))
            raise DataProcessingError("Error trying to extract regional subset.")

        if region_cube is None:
            msg = "No data found within specified geographic region."
            self.logger.error(msg)
            raise DataProcessingError(msg)

        self.logger.debug("Regional cube: %s", region_cube.summary(shorten=True))

        return region_cube

    def _convert_units(self, diag_defn, diag_cube):
        """
        If the user has specified target units for a given diagnostic, attempt
        to apply a unit conversion using the cube.convert_units method. Note
        that the unit conversion is applied in situ to the supplied cube.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube cube: The cube whose units are to be converted.
        """

        if not diag_defn.target_units: return

        try:
            self.logger.info("Converting units from '%s' to '%s' ...",
                diag_cube.units, diag_defn.target_units)
            diag_cube.convert_units(diag_defn.target_units)
        except ValueError as exc:
            self.logger.error(str(exc))
            raise DataProcessingError("Error trying to convert units.")

    def _calc_asov_data(self, diag_defn, diag_cube):
        """
        Generate ASoV frequency data for the specified cube.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube diag_cube: The cube of diagnostic data for which
            to generate ASoV values.
        :returns: An Iris cube containing the calculated ASoV data.
        """

        self.logger.info("Calculating ASoV frequency data ...")

        # Create and run an instance of the HistogramMaker class.
        try:
            proc = HistogramMaker(fix_cell_methods=True)
            cubes = proc.run(diag_cube, bin_array=diag_defn.bin_array)
            cube = cubes[0]
        except Exception as exc:
            self.logger.error(str(exc))
            raise DataProcessingError("Error trying to calculate ASoV data.")

        return cube

    def _save_asov_data(self, diag_defn, asov_cube):
        """
        Save a cube of ASoV frequency data to the output directory defined in
        the app config file. The name of the output file is generated from the
        attributes attached to the diag_defn object.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube asov_cube: The cube of ASoV data to save.
        """

        filename = self.make_output_filename(diag_defn, self.per_cp_filename_provider)
        filepath = os.path.join(self.data_writer_opts.output_dir, filename)
        self.logger.info("Saving ASoV frequency data to file %s ...", filename)

        if self.dry_run: return

        # Construct output directory path and make sure that it exists.
        dirpath = os.path.dirname(filepath)
        if not os.path.exists(dirpath): os.makedirs(dirpath)

        self.data_writer.run(asov_cube, filepath)

    def _update_metadata(self, diag_defn, asov_cube):
        """
        Attach any desirable extra attributes to the cube of ASoV data.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube asov_cube: The cube whose metadata is to be updated.
        """

        # Remove any unwanted PP attributes as they can disrupt save operations.
        pp_att_names = ['lbproc', 'lbtim']
        for att_name in pp_att_names:
            _ = asov_cube.attributes.pop(att_name, None)

        # Set the cube's history attribute, though without a timestamp since
        # that will complicate subsequent concatenation of data over time.
        asov_cube.attributes['history'] = ("ASoV frequency data generated by "
            "the Afterburner AsovCalculator app.")

    def _update_cumulative_file(self, diag_defn, asov_cube):
        """
        Add or append the supplied cube of ASoV data to the cumulative file for
        the given diagnostic. The serialised variable does not include a first
        class time dimension, and although the overall time period is captured
        in a scalar time coordinate, this provides no guarantee that data for
        all time instants encompassed by the period are included in the cumulative
        total.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube asov_cube: The cube of ASoV data to append.
        """

        # Construct output file path.
        filename = self.make_output_filename(diag_defn, self.all_cp_filename_provider)
        filepath = os.path.join(self.data_writer_opts.output_dir, filename)
        self.logger.info("Adding ASoV data to cumulative file %s ...", filename)

        if self.dry_run: return

        # Construct output directory path and make sure that it exists.
        dirpath = os.path.dirname(filepath)
        if not os.path.exists(dirpath): os.makedirs(dirpath)

        if os.path.exists(filepath):
            old_cube = iris.load_cube(filepath)
            if old_cube.shape != asov_cube.shape:
                msg = ("Shape of existing data on disk ({0}) does not match shape "
                    "of calculated data ({1}).".format(old_cube.shape, asov_cube.shape))
                raise DataProcessingError(msg)
            old_cube.data += asov_cube.data
            old_tcoord = old_cube.coord('time')
            new_tcoord = asov_cube.coord('time')
            tmin = min(old_tcoord.bounds.min(), new_tcoord.bounds.min())
            tmax = max(old_tcoord.bounds.max(), new_tcoord.bounds.max())
            new_tcoord = old_tcoord.copy(points=np.array([(tmin+tmax)//2]),
                bounds=np.array([tmin, tmax]))
            old_cube.replace_coord(new_tcoord)
            self.logger.debug("extended time axis: %s", new_tcoord)
            self.data_writer.run(old_cube, filepath, overwrite=True)
        else:
            self.logger.debug("initial time axis: %s", asov_cube.coord('time'))
            self.data_writer.run(asov_cube, filepath)

    def _update_timeseries_file(self, diag_defn, asov_cube):
        """
        Add or append the supplied cube of ASoV data to the time-series file for
        the given diagnostic. As the filename implies, the serialised variable
        includes a leading time dimension.

        :param afterburner.utils.NamespacePlus diag_defn: A namespace object
            containing the diagnostic definition.
        :param iris.cube.Cube asov_cube: The cube of ASoV data to append.
        """

        # If required, promote the time coordinate from a scalar to a full
        # dimension coordinate.
        if is_scalar_coord(asov_cube, 'time'):
            asov_cube = iris.util.new_axis(asov_cube, 'time')

        # Construct output file path.
        filename = self.make_output_filename(diag_defn, self.tseries_filename_provider)
        filepath = os.path.join(self.data_writer_opts.output_dir, filename)
        self.logger.info("Adding ASoV data to time-series file %s ...", filename)

        if self.dry_run: return

        # Construct output directory path and make sure that it exists.
        dirpath = os.path.dirname(filepath)
        if not os.path.exists(dirpath): os.makedirs(dirpath)

        if self.cli_args.debug and os.path.exists(filepath):
            old_cube = iris.load_cube(filepath)
            result = compare_cubes(old_cube, asov_cube)
            if not result:
                self.logger.warning("Old and new ASoV cubes exhibit metadata "
                    "conflicts and therefore may not concatenate.")

        self.data_writer.run(asov_cube, filepath, append=True, equalise_attrs=True)


def _decode_bin_array_from_string(textstr):
    """
    Decode an array of (lower) bin values from a text string. The array may be
    specified either as a sequence of comma-separated values, or as an arbitrary
    Python expression. In the latter case the expression will be evaluated using
    the eval() builtin function. If the expression contains numpy functions then
    these should be identified using the prefix 'np.' rather than 'numpy.'. For
    example, use "np.linspace(0, 100, 5)" instead of "numpy.linspace(0, 100, 5)".

    :param str textstr: A string containing either a comma-separated list of
        numbers or an arbitrary Python expression to evaluate.
    :returns: An array of bin values, the values representing the lower bound of
        each bin.
    """

    if re.search(r'[a-zA-Z]{2,}', textstr):
        bin_array = eval(textstr, {'math': math, 'numpy': np, 'np': np})
    else:
        bin_array = np.fromstring(textstr, sep=',')

    return bin_array


def _on_load_pp_callback(cube, field, filename):
    """
    Callback function for attaching selected PP header items to ``cube``.
    """

    target_pp_att_names = ['lbproc', 'lbtim']
    for att_name in target_pp_att_names:
        att_val = getattr(field, att_name)
        if att_val is not None:
            # use numpy scalars if attributes will be written to netcdf file
            cube.attributes[att_name] = att_val  # np.int(att_val)

    if cube.attributes.get('lbproc') == 0 and not cube.cell_methods:
        cm = iris.coords.CellMethod('point', coords=['time'])
        cube.cell_methods = (cm,)


# Following function no longer required, but preserved here for reference purposes.
#
# def _make_default_precip_bins():
#     """
#     Create an array of default bin values for the precipitation flux diagnostic.
#     The array values are equivalent to those used in the ASoP-Spectral codebase,
#     for which see https://github.com/nick-klingaman/ASoP.
#     """
#
#     sqterm = np.square(np.log(120.0) - np.log(0.005))
#     pbin = np.exp(np.log(0.005) + np.sqrt(np.linspace(0, 98, 99) * (sqterm / 59.0)))
#     return np.concatenate([np.zeros(1), pbin])
