# (C) British Crown Copyright 2016-2023, Met Office
#
# See the LICENSE.TXT file included with the Afterburner
# software distribution for full license details.
"""
A collection of software tools for processing multidimensional data from climate models.

The afterburner package is a collection of Python modules for working with
multidimensional geophysical data generated by numerical climate models. In
large part it provides an extra layer of functionality on top of the Iris
toolkit (see https://github.com/SciTools/iris). It also includes partial
integration with the Rose framework (see https://github.com/metomi/rose).

The Afterburner project home page can be found at https://code.metoffice.gov.uk/trac/afterburner,
from where registered users may download Afterburner software artifacts.
Comprehensive documentation for the afterburner Python package can be viewed at
https://code.metoffice.gov.uk/doc/afterburner/current/html/index.html

The main piece of functionality enabled during initialisation of the afterburner
package is a logger object named 'afterburner' which, by default, sends messages
with a severity of ERROR or higher to the standard error stream, and messages
with a lesser severity (WARNING or lower) to standard output.

If the AFTERBURNER_LOG_LEVEL environment variable is defined then its value is
used to set the default level for the 'afterburner' logger obect. If the value
is not a recognised level name (such as 'INFO') or an integer then it is ignored,
in which case a default log level of WARNING is assigned.

Afterburner classes and functions which need to handle log messages should
normally define their logger object(s) in such a way as to inherit the settings
of the 'afterburner' logger. The simplest way to do this, and one that is
adequate for most scenarios, is to use the ``__name__`` attribute::

   import logging
   logger = logging.getLogger(__name__)

Refer to the ``_init_logger()`` function documentation for more information.
"""
from __future__ import (absolute_import, division, print_function)
from six.moves import (filter, input, map, range, zip)
from six import string_types

import os
import sys
import logging
import logging.config
import operator

from pkg_resources import parse_version
from afterburner.modelmeta import MODEL_UM, MODEL_NEMO, MODEL_CICE, KNOWN_MODELS

# Version metadata.
_svn_rev = '$Rev: 1561 $'
__version_info__ = (1, 3, 3, '', int(_svn_rev[5:-1] or 0))
__version__ = '{}.{}.{}{}'.format(*__version_info__[:4])

#: Default log level.
DEFAULT_LOG_LEVEL = logging.WARNING

#: Default format to use for log messages.
DEFAULT_LOG_FORMAT = "%(levelname)s: [%(module)s] %(message)s"

#: Default format to use for date-time strings in log messages.
DEFAULT_LOG_DATE_FORMAT = "%Y-%m-%d %H:%M:%S"

#: Lowest log level for messages sent to standard error: usually ERROR or WARNING.
STDERR_MIN_LOG_LEVEL = logging.ERROR

# Name of the logger object used to log Afterburner package or app invocations.
PACKAGE_LOGGER_NAME = 'afterburner_invoc'


def compare_iris_version(vn, op='eq'):
    """
    Compare the version of Iris against the version number defined in ``vn``
    using the specified comparison operator.

    :param str vn: The version against which to compare the Iris version.
    :param str op: The comparison operator. One of 'eq', 'ne', 'lt', 'le', 'gt',
        or 'ge'.
    """
    import iris
    oper = getattr(operator, op, None) or operator.eq
    return oper(parse_version(iris.__version__), parse_version(vn))


def set_log_level(logger, level):
    """
    Set the level of a logger object. Level can either be one of the level names
    recognised by the `logging <https://docs.python.org/2/library/logging.html>`_
    module (e.g. 'INFO'), or else a numeric level. Level names are converted to
    upper case if necessary. A numeric level can be defined as a regular integer,
    e.g. 10, or as a decimal string, e.g. '10'.

    If an invalid level name is specified, then the logger object is left
    unmodified.

    :param object logger: The logger object whose level is to be set.
    :param str/int level: The desired log level, defined as one of the familiar
        level names ('INFO', 'WARNING', 'ERROR', etc), or as an integer.
    """
    try:
        # For integer values, try setting the log level directly.
        logger.setLevel(int(level))
    except ValueError:
        # Else try decoding standard level names.
        if isinstance(level, string_types):
            level = getattr(logging, level.upper(), None)
            if level is not None: logger.setLevel(level)


class _CustomFormatter(logging.Formatter):
    """
    A custom log formatter class which defines level-specific message formats.
    The docstring for the :func:`_init_logger` function provides information
    about the default configuration of the afterburner logger object.
    """
    # Define formats for those log levels which differ from the default format.
    _detailed_format = "%(levelname)s: [%(module)s.%(funcName)s:%(lineno)s] %(message)s"
    _msg_formats = {
        logging.INFO: "%(message)s",
        logging.ERROR: _detailed_format,
        logging.CRITICAL: _detailed_format,
    }

    def __init__(self, fmt=None, datefmt=None):
        logging.Formatter.__init__(self, fmt=fmt, datefmt=datefmt)
        self.default_fmt = fmt
        # Dictionary for caching level-specific formatter objects.
        self._formatters = {}

    def format(self, record):
        # Override the base class method by retrieving a level-specific formatter
        # object from a cache, and invoking that object's format method. There
        # are more elegant ways to do this at Python 3.x, though clearly these
        # would not be back-compatible with Python 2.x.
        fmttr = self._get_formatter(record)
        return fmttr.format(record)

    def _get_formatter(self, record):
        # Return a formatter object to use with the specified record. The
        # formatter is cached for subsequent use with records of the same level.
        fmttr = self._formatters.get(record.levelno)
        if not fmttr:
            fmt = self._msg_formats.get(record.levelno, self.default_fmt)
            fmttr = logging.Formatter(fmt=fmt, datefmt=self.datefmt)
            self._formatters[record.levelno] = fmttr
        return fmttr


class _LogLevelLTFilter(logging.Filter):
    """
    Filter for use in selecting all log messages whose level is *less than* the
    value specified via the ``level`` argument.
    """
    def __init__(self, level):
        super(_LogLevelLTFilter, self).__init__()
        self.level = level

    def filter(self, record):
        return record.levelno < self.level


def _init_logger():
    """
    Initialises a logger object for use by afterburner modules.

    Afterburner modules can access the logger object created by this function
    by calling either logging.getLogger(__name__), which is the recommended idiom,
    or logging.getLogger('afterburner').

    The logger can be configured via a site-specific or user-specific configuration
    file, both of which are optional. If neither file exists then, by default,
    the logger is set up so as to print messages with a severity <= WARNING to
    standard output, and with severity >= ERROR to standard error. The default
    message format is "log-level: message (module-name)"

    If used, the site configuration file should be located at
    $AFTERBURNER_HOME_DIR/etc/logging.conf

    If used, the user configuration file should be located at
    $HOME/.config/afterburner/logging.conf

    It should not normally be necessary to configure Afterburner logging using
    the aforementioned files. If it is found necessary then details of the file
    format can be found at https://docs.python.org/2/library/logging.config.html.
    """

    # Initialise default logger and formatter objects.
    logger = logging.getLogger(__name__)
    logger.setLevel(DEFAULT_LOG_LEVEL)
    level = os.environ.get('AFTERBURNER_LOG_LEVEL')
    if level: set_log_level(logger, level)
    fmtr = _CustomFormatter(fmt=DEFAULT_LOG_FORMAT, datefmt=DEFAULT_LOG_DATE_FORMAT)

    # Create a log handler to direct messages to stdout.
    console_out = logging.StreamHandler(stream=sys.stdout)
    console_out.setLevel(logging.DEBUG)
    console_out.setFormatter(fmtr)
    console_out.addFilter(_LogLevelLTFilter(STDERR_MIN_LOG_LEVEL))

    # Create a log handler to direct messages to stderr.
    console_err = logging.StreamHandler(stream=sys.stderr)
    console_err.setLevel(STDERR_MIN_LOG_LEVEL)
    console_err.setFormatter(fmtr)

    # TODO: Add a RotatingFileHandler which writes to ~/logs/afterburner.log

    # Add handlers to logger object.
    logger.addHandler(console_out)
    logger.addHandler(console_err)
    logger.propagate = False

    # Look for a site configuration file at $AFTERBURNER_HOME_DIR/etc/logging.conf
    # Assumes that the 'etc' directory has the same parent directory as the 'lib'
    # directory which contains, via intermediate directories, the current module.
    site_config_file = ''
    fpath = os.path.abspath(__file__)
    index = fpath.find('lib')
    if index > 0:
        site_config_file = os.path.join(fpath[:index-1], 'etc', 'logging.conf')
    if site_config_file and os.path.isfile(site_config_file):
        try:
            logging.config.fileConfig(site_config_file, disable_existing_loggers=False)
            logger.debug("Loaded site log configuration file: %s", site_config_file)
        except:
            logger.error("Problem trying to load site log configuration file:\n%s",
                site_config_file)

    # Look for a user configuration file at $HOME/.config/afterburner/logging.conf
    home_dir = os.path.expandvars(os.environ.get('HOME', ''))
    user_config_file = os.path.join(home_dir, '.config', 'afterburner', 'logging.conf')
    if os.path.isfile(user_config_file):
        try:
            logging.config.fileConfig(user_config_file, disable_existing_loggers=False)
            logger.debug("Loaded user log configuration file: %s", user_config_file)
        except:
            logger.error("Problem trying to load user log configuration file:\n%s",
                user_config_file)

    return logger


def _init_package_logger():
    """
    Initialise a socket-based logger object for the purposes of logging package
    or app invocations to a TCP-based log server listening on a particular host
    and port address. It is assumed that such a TCP server has previously been
    started, typically by privileged user. If not then the logging attempt fails
    silently.

    The host name and port number are obtained either from the site configuration
    file, or from the environment variables named TCP_LOG_SERVER_HOST and
    TCP_LOG_SERVER_PORT. The environment variables take precedence.

    In the case of specification via a site config file the relevant options must
    be defined under a section named ``tcp_log_server``, as follows::

        [tcp_log_server]
        host=<hostname>
        port=<port-number>

    If an error is encountered creating the logger then the function returns None.
    """

    host = port = pkg_lgr = None
    logger = logging.getLogger(__name__)

    try:
        import package_logger
    except ImportError:
        logger.warning("Unable to create a package logger: the package_logger "
            "module could not be imported.")
        return None

    # See if the log level has been defined via the AFTERBURNER_PKG_LOG_LEVEL
    # variable. If it's set to 0 then return without creating a package logger.
    try:
        level = int(os.environ.get('AFTERBURNER_PKG_LOG_LEVEL', logging.INFO))
        if level == 0:
            return None
    except (TypeError, ValueError):
        level = logging.INFO

    try:
        from afterburner.config import ConfigProvider
        cfg = ConfigProvider()
        # Check for host name and port settings in the afterburner site config file.
        host = cfg.get_config_option('tcp_log_server', 'host', default=None)
        port = cfg.get_config_option('tcp_log_server', 'port', default=None)
        # Check for host name and port settings in environment variables.
        host = os.environ.get('TCP_LOG_SERVER_HOST', host)
        port = os.environ.get('TCP_LOG_SERVER_PORT', port)
        if host:
            pkg_lgr = package_logger.create_package_logger(PACKAGE_LOGGER_NAME,
                'afterburner', host=host, port=port, level=level)
        else:
            logger.debug("Skipping package logger creation: no log server "
                "hostname was specified.")
    except package_logger.UncontactableServerError:
        logger.warning("Unable to detect a logging server at address %s:%s",
            host, port)
    except Exception as exc:
        logger.warning("Unable to create a package logger during afterburner "
            "package initialisation.")
        logger.warning(str(exc))

    return pkg_lgr


# Initialise a general-purpose logger object for use by afterburner subpackages
# and modules.
DEFAULT_LOGGER = _init_logger()

# Initialise a bespoke package logger and, if successful, log the current invocation
# of the afterburner package.
PACKAGE_LOGGER = _init_package_logger()
if PACKAGE_LOGGER:
    PACKAGE_LOGGER.info("Invoked the afterburner package")
