#!/usr/bin/env python
from flask import Flask, request
from flask_cors import cross_origin
import fasttext, re, ujson as json, os, numpy as np, click
from gunicorn.app.base import BaseApplication

class cFlask(Flask) :
    def __init__(self, *args) :
        super(cFlask, self).__init__(*args)
    def init_parser(self, model) :
        self.model_name = model,
        config = json.load(open(os.path.join(model, 'model.conf'), 'rt'))
        self.config.update(config)
        self.level_models = [fasttext.load_model(os.path.join(model, fname)) for fname in self.config['models']]
    def info(self) :
        return dict(NAME=self.model_name,
                    LEVELS=self.config['levels'],
                    POSSIBLE_CATEGORIES=sorted(self.config['combined_category']),
                    MULTIPLE_LABEL=self.config['multi_label'])

    def tokenize(self, sent, lower=True) :
        if lower :
            return [w for w in re.split(r"[^a-zA-Z]+", sent.lower()) if len(w) > 1]
        else :
            return [w for w in re.split(r"[^a-zA-Z]+", sent) if len(w) > 1]
    def query(self, qry) :
        tokens = self.tokenize(qry)
        res = []
        for m in self.level_models :
            r = m.predict(' '.join(tokens), k = -1)
            res.append(dict(zip(r[0], r[1].tolist())))
        paths = []
        for cat in self.config['combined_category'] :
            path = np.array([min(r.get(k, 0. if k else 0.5), 1.) for k, r in zip(cat, res)])
            prob = path.prod()**(1./len(path))
            p = dict(zip(self.config['levels'], [[c,p] for c, p in zip(cat, path)]))
            p['Pr'] = prob
            paths.append(p)
        if not self.config['multi_label'] :
            Pr_total = sum([ p['Pr'] for p in paths ])
            for p in paths :
                p['Pr'] /= Pr_total
            paths = [p for p in paths if p['Pr'] >= 0.1]
        else :
            paths = [p for p in paths if p['Pr'] >= 0.5]
        for path in paths :
            for k, v in path.items() :
                if k in self.config['levels'] :
                    v[0] = self.config['conversion'].get(v[0], '')
        return sorted(paths, key=lambda x:-x['Pr'])
app = cFlask(__name__)


@app.route("/", methods=['GET', 'POST'])
@cross_origin(allow_headers=['Content-Type'])
def info() :
    res = app.info()
    return json.dumps(res, indent=2, sort_keys=True)

@app.route("/query", methods=['GET', 'POST'])
@cross_origin(allow_headers=['Content-Type'])
def query() :
    qry = request.form.get('q', '') if request.method == 'POST' else request.args.get('q', '')
    paths = app.query(qry)
    return json.dumps(paths, indent=2, sort_keys=True)

@app.route("/batch_query", methods=['GET', 'POST'])
@cross_origin(allow_headers=['Content-Type'])
def batch_query() :
    result = {}
    qry = request.form.get('q', '') if request.method == 'POST' else request.args.get('q', '')
    for q in qry.split('\n') :
        res = app.query(q)
        result[q] = res
    return json.dumps(result, indent=2, sort_keys=True)


class Application(BaseApplication) :
    def __init__(self, app, options={}):
        self.options = options
        self.application = app
        super(Application, self).__init__()
    def load_config(self):
        config = { k: v for k, v in self.options.items() \
                   if k in self.cfg.settings and v is not None }
        for k, v in config.items() :
            self.cfg.set(k.lower(), v)
    def load(self):
        return self.application

@click.command(context_settings=dict(help_option_names=['-h', '--help']))
@click.option('-m', '--model', help='folder for the model generated by fastSrc_build', required=True)
@click.option('-w', '--workers', help='number of workers (processors) to be used. DEFAULT: 1', default=1)
@click.option('-b', '--bind', help='IP and port to be used for the API. DEFAULT: 0.0.0.0:4125', default='0.0.0.0:4125')
def main(model, workers, bind) :
    app.init_parser(model)
    app.query('human')
    Application(app, dict(bind=bind, workers=workers)).run()

if __name__ == '__main__' :
    main()
