"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnvironmentsClient = exports.EnvironmentNotFound = exports.EnvironmentAlreadyReallocated = exports.EnvironmentAlreadyCleaningError = exports.EnvironmentAlreadyDirtyError = exports.EnvironmentAlreadyInStatusError = exports.EnvironmentAlreadyInUseError = exports.EnvironmentAlreadyReleasedError = exports.EnvironmentAlreadyAcquiredError = exports.EnvironmentsError = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const ddb = require("@aws-sdk/client-dynamodb");
const dynamo_item_1 = require("./dynamo-item");
/**
 * Base error originating from actions on the environments table.
 */
class EnvironmentsError extends Error {
    constructor(account, region, message) {
        super(`Environment aws://${account}/${region}: ${message}`);
    }
}
exports.EnvironmentsError = EnvironmentsError;
/**
 * Error thrown when an environment is already acquired.
 */
class EnvironmentAlreadyAcquiredError extends EnvironmentsError {
    constructor(account, region) {
        super(account, region, 'already acquired');
    }
}
exports.EnvironmentAlreadyAcquiredError = EnvironmentAlreadyAcquiredError;
;
/**
 * Error thrown when an environment is already released.
 */
class EnvironmentAlreadyReleasedError extends EnvironmentsError {
    constructor(account, region) {
        super(account, region, 'already released');
    }
}
exports.EnvironmentAlreadyReleasedError = EnvironmentAlreadyReleasedError;
;
/**
 * Error thrown when an environment is already in-use.
 */
class EnvironmentAlreadyInUseError extends EnvironmentsError {
    constructor(account, region) {
        super(account, region, 'already in-use');
    }
}
exports.EnvironmentAlreadyInUseError = EnvironmentAlreadyInUseError;
;
/**
 * Error thrown when an environment is already in the target status.
 */
class EnvironmentAlreadyInStatusError extends EnvironmentsError {
    constructor(account, region, status) {
        super(account, region, `already ${status}`);
    }
}
exports.EnvironmentAlreadyInStatusError = EnvironmentAlreadyInStatusError;
;
/**
 * Error thrown when an environment is already dirty.
 */
class EnvironmentAlreadyDirtyError extends EnvironmentsError {
    constructor(account, region) {
        super(account, region, 'already dirty');
    }
}
exports.EnvironmentAlreadyDirtyError = EnvironmentAlreadyDirtyError;
/**
 * Error thrown when an environment is already cleaning.
 */
class EnvironmentAlreadyCleaningError extends EnvironmentsError {
    constructor(account, region) {
        super(account, region, 'already cleaning');
    }
}
exports.EnvironmentAlreadyCleaningError = EnvironmentAlreadyCleaningError;
/**
 * Error thrown when an environment is already reallocated.
 */
class EnvironmentAlreadyReallocated extends EnvironmentsError {
    constructor(account, region) {
        super(account, region, 'already reallocated');
    }
}
exports.EnvironmentAlreadyReallocated = EnvironmentAlreadyReallocated;
/**
 * Error thrown when an environment is not found.
 */
class EnvironmentNotFound extends EnvironmentsError {
    constructor(account, region) {
        super(account, region, 'not found');
    }
}
exports.EnvironmentNotFound = EnvironmentNotFound;
/**
 * Client for accessing the environments table at runtime.
 */
class EnvironmentsClient {
    constructor(tableName) {
        this.tableName = tableName;
        this.ddbClient = new ddb.DynamoDB({});
    }
    /**
     * Fetch a specific environment by account and region.
     * Will throw if it doesn't exists.
     */
    async get(account, region) {
        const response = await this.ddbClient.getItem({
            TableName: this.tableName,
            Key: {
                account: { S: account },
                region: { S: region },
            },
        });
        if (!response.Item) {
            throw new EnvironmentNotFound(account, region);
        }
        return {
            account: (0, dynamo_item_1.requiredValue)('account', response.Item),
            region: (0, dynamo_item_1.requiredValue)('region', response.Item),
            status: (0, dynamo_item_1.requiredValue)('status', response.Item),
            allocation: (0, dynamo_item_1.requiredValue)('allocation', response.Item),
        };
    }
    /**
     * Acquire an environment by inserting a new item into the table.
     * If the environment is already acquired, this will fail.
     */
    async acquire(allocationId, account, region) {
        try {
            await this.ddbClient.putItem({
                TableName: this.tableName,
                Item: {
                    account: { S: account },
                    region: { S: region },
                    status: { S: 'in-use' },
                    allocation: { S: allocationId },
                },
                // avoid attribute name collisions with reserved keywords.
                // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.ExpressionAttributeNames.html
                ExpressionAttributeNames: {
                    '#region': 'region',
                    '#account': 'account',
                },
                // ensures insertion.
                // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.ConditionExpressions.html#Expressions.ConditionExpressions.PreventingOverwrites
                ConditionExpression: 'attribute_not_exists(#account) AND attribute_not_exists(#region)',
            });
        }
        catch (e) {
            if (e instanceof ddb.ConditionalCheckFailedException) {
                throw new EnvironmentAlreadyAcquiredError(account, region);
            }
            throw e;
        }
    }
    /**
     * Release an environment by deleting an item from the table.
     * If the environment is already released, this will fail.
     */
    async release(allocationId, account, region) {
        try {
            await this.ddbClient.deleteItem({
                TableName: this.tableName,
                Key: {
                    account: { S: account },
                    region: { S: region },
                },
                // avoid attribute name collisions with reserved keywords.
                // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.ExpressionAttributeNames.html
                ExpressionAttributeNames: {
                    '#region': 'region',
                    '#account': 'account',
                    '#allocation': 'allocation',
                    '#status': 'status',
                },
                ExpressionAttributeValues: {
                    ':allocation_value': { S: allocationId },
                    // we shouldn't be releasing an environment if its still 'in-use'.
                    // it should be marked as either 'cleaning' or 'dirty' beforehand.
                    ':unexpected_status_value': { S: 'in-use' },
                },
                // ensures deletion.
                // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.ConditionExpressions.html#Expressions.ConditionExpressions.PreventingOverwrites
                ConditionExpression: 'attribute_exists(#account) AND attribute_exists(#region) AND #allocation = :allocation_value AND #status <> :unexpected_status_value',
                ReturnValuesOnConditionCheckFailure: 'ALL_OLD',
            });
        }
        catch (e) {
            if (e instanceof ddb.ConditionalCheckFailedException) {
                if (!e.Item) {
                    throw new EnvironmentAlreadyReleasedError(account, region);
                }
                const old_allocation = e.Item.allocation?.S;
                if (old_allocation && old_allocation !== allocationId) {
                    throw new EnvironmentAlreadyReallocated(account, region);
                }
                const old_status = e.Item.status?.S;
                if (old_status === 'in-use') {
                    throw new EnvironmentAlreadyInUseError(account, region);
                }
            }
            throw e;
        }
    }
    /**
     * Mark the environment status as 'cleaning'.
     * If the environment is already in a 'cleaning' status, this will fail.
     */
    async cleaning(allocationId, account, region) {
        try {
            await this.setStatus(allocationId, account, region, 'cleaning');
        }
        catch (e) {
            if (e instanceof EnvironmentAlreadyInStatusError) {
                throw new EnvironmentAlreadyCleaningError(account, region);
            }
            throw e;
        }
    }
    /**
     * Mark the environment status as 'dirty'.
     * If the environment is already in a 'dirty' status, this will fail.
     */
    async dirty(allocationId, account, region) {
        try {
            await this.setStatus(allocationId, account, region, 'dirty');
        }
        catch (e) {
            if (e instanceof EnvironmentAlreadyInStatusError) {
                throw new EnvironmentAlreadyDirtyError(account, region);
            }
            throw e;
        }
    }
    async setStatus(allocationId, account, region, status) {
        try {
            await this.ddbClient.updateItem({
                TableName: this.tableName,
                Key: {
                    account: { S: account },
                    region: { S: region },
                },
                // avoid attribute name collisions with reserved keywords.
                // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.ExpressionAttributeNames.html
                ExpressionAttributeNames: {
                    '#status': 'status',
                    '#allocation': 'allocation',
                },
                ExpressionAttributeValues: {
                    ':status_value': { S: status },
                    ':allocation_value': { S: allocationId },
                },
                UpdateExpression: 'SET #status = :status_value',
                // ensures update.
                // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/WorkingWithItems.html#WorkingWithItems.ConditionalUpdate
                ConditionExpression: '#allocation = :allocation_value AND #status <> :status_value',
                ReturnValuesOnConditionCheckFailure: 'ALL_OLD',
            });
        }
        catch (e) {
            if (e instanceof ddb.ConditionalCheckFailedException) {
                if (!e.Item) {
                    throw new EnvironmentAlreadyReleasedError(account, region);
                }
                const old_allocation = e.Item.allocation?.S;
                const old_status = e.Item.status?.S;
                if (old_allocation && old_allocation !== allocationId) {
                    throw new EnvironmentAlreadyReallocated(account, region);
                }
                if (old_status && old_status === status) {
                    throw new EnvironmentAlreadyInStatusError(account, region, old_status);
                }
            }
            throw e;
        }
    }
}
exports.EnvironmentsClient = EnvironmentsClient;
//# sourceMappingURL=data:application/json;base64,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