/**
 * Base error originating from actions on the environments table.
 */
export declare class EnvironmentsError extends Error {
    constructor(account: string, region: string, message: string);
}
/**
 * Error thrown when an environment is already acquired.
 */
export declare class EnvironmentAlreadyAcquiredError extends EnvironmentsError {
    constructor(account: string, region: string);
}
/**
 * Error thrown when an environment is already released.
 */
export declare class EnvironmentAlreadyReleasedError extends EnvironmentsError {
    constructor(account: string, region: string);
}
/**
 * Error thrown when an environment is already in-use.
 */
export declare class EnvironmentAlreadyInUseError extends EnvironmentsError {
    constructor(account: string, region: string);
}
/**
 * Error thrown when an environment is already in the target status.
 */
export declare class EnvironmentAlreadyInStatusError extends EnvironmentsError {
    constructor(account: string, region: string, status: string);
}
/**
 * Error thrown when an environment is already dirty.
 */
export declare class EnvironmentAlreadyDirtyError extends EnvironmentsError {
    constructor(account: string, region: string);
}
/**
 * Error thrown when an environment is already cleaning.
 */
export declare class EnvironmentAlreadyCleaningError extends EnvironmentsError {
    constructor(account: string, region: string);
}
/**
 * Error thrown when an environment is already reallocated.
 */
export declare class EnvironmentAlreadyReallocated extends EnvironmentsError {
    constructor(account: string, region: string);
}
/**
 * Error thrown when an environment is not found.
 */
export declare class EnvironmentNotFound extends EnvironmentsError {
    constructor(account: string, region: string);
}
/**
 * Represents an active environment, ie one that is currently allocated and exists
 * in the database.
 */
export interface ActiveEnvironment {
    /**
     * Account.
     */
    readonly account: string;
    /**
     * Region.
     */
    readonly region: string;
    /**
     * Status.
     */
    readonly status: string;
    /**
     * Allocation id.
     */
    readonly allocation: string;
}
/**
 * Client for accessing the environments table at runtime.
 */
export declare class EnvironmentsClient {
    private readonly tableName;
    private readonly ddbClient;
    constructor(tableName: string);
    /**
     * Fetch a specific environment by account and region.
     * Will throw if it doesn't exists.
     */
    get(account: string, region: string): Promise<ActiveEnvironment>;
    /**
     * Acquire an environment by inserting a new item into the table.
     * If the environment is already acquired, this will fail.
     */
    acquire(allocationId: string, account: string, region: string): Promise<void>;
    /**
     * Release an environment by deleting an item from the table.
     * If the environment is already released, this will fail.
     */
    release(allocationId: string, account: string, region: string): Promise<void>;
    /**
     * Mark the environment status as 'cleaning'.
     * If the environment is already in a 'cleaning' status, this will fail.
     */
    cleaning(allocationId: string, account: string, region: string): Promise<void>;
    /**
     * Mark the environment status as 'dirty'.
     * If the environment is already in a 'dirty' status, this will fail.
     */
    dirty(allocationId: string, account: string, region: string): Promise<void>;
    private setStatus;
}
