"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AllocationsClient = exports.AllocationNotFoundError = exports.InvalidInputError = exports.AllocationAlreadyEndedError = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const ddb = require("@aws-sdk/client-dynamodb");
const dynamo_item_1 = require("./dynamo-item");
/**
 * Error thrown if the allocation is already ended.
 *
 */
class AllocationAlreadyEndedError extends Error {
    constructor(id) {
        super(`Allocation ${id} is already ended`);
    }
}
exports.AllocationAlreadyEndedError = AllocationAlreadyEndedError;
/**
 * Error thrown if the input is invalid.
 */
class InvalidInputError extends Error {
    constructor(message) {
        super(`Invalid input: ${message}`);
    }
}
exports.InvalidInputError = InvalidInputError;
/**
 * Error thrown if the allocation is not found.
 */
class AllocationNotFoundError extends Error {
    constructor(id) {
        super(`Allocation ${id} not found`);
    }
}
exports.AllocationNotFoundError = AllocationNotFoundError;
/**
 * Client for accessing the allocations table at runtime.
 */
class AllocationsClient {
    constructor(tableName) {
        this.tableName = tableName;
        this.ddbClient = new ddb.DynamoDB({});
    }
    /**
     * Retrieve an allocation by id.
     */
    async get(id) {
        const response = await this.ddbClient.getItem({
            TableName: this.tableName,
            Key: {
                id: { S: id },
            },
        });
        if (!response.Item) {
            throw new AllocationNotFoundError(id);
        }
        return {
            account: (0, dynamo_item_1.requiredValue)('account', response.Item),
            region: (0, dynamo_item_1.requiredValue)('region', response.Item),
            pool: (0, dynamo_item_1.requiredValue)('pool', response.Item),
            start: (0, dynamo_item_1.requiredValue)('start', response.Item),
            requester: (0, dynamo_item_1.requiredValue)('requester', response.Item),
            id: (0, dynamo_item_1.requiredValue)('id', response.Item),
            // if an allocation is queried before it ended, these attributes
            // will be missing.
            end: (0, dynamo_item_1.optionalValue)('end', response.Item),
            outcome: (0, dynamo_item_1.optionalValue)('outcome', response.Item),
        };
    }
    /**
     * Start an allocation for a specific environment. Returns the allocation id.
     */
    async start(opts) {
        if (Buffer.byteLength(opts.requester) > 1024) {
            // this is user controlled so we impose a max length
            throw new InvalidInputError('requester must be less than 1024 bytes');
        }
        const nowSeconds = Math.floor(Date.now() / 1000);
        const sixMonthsSeconds = 26 * 7 * 24 * 60 * 60;
        await this.ddbClient.putItem({
            TableName: this.tableName,
            Item: {
                id: { S: opts.id },
                account: { S: opts.account },
                region: { S: opts.region },
                pool: { S: opts.pool },
                start: { S: new Date().toISOString() },
                requester: { S: opts.requester },
                ttl: { N: `${nowSeconds + sixMonthsSeconds}` },
            },
        });
    }
    /**
     * End the allocation. Throws if the allocation has already ended.
     */
    async end(opts) {
        if (Buffer.byteLength(opts.outcome) > 100) {
            // this is user controlled so we impose a max length
            throw new InvalidInputError('outcome must be less than 100 bytes');
        }
        try {
            const response = await this.ddbClient.updateItem({
                TableName: this.tableName,
                Key: {
                    id: { S: opts.id },
                },
                // avoid attribute name collisions with reserved keywords.
                // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.ExpressionAttributeNames.html
                ExpressionAttributeNames: {
                    '#end': 'end',
                    '#outcome': 'outcome',
                },
                ExpressionAttributeValues: {
                    ':end': { S: new Date().toISOString() },
                    ':outcome': { S: opts.outcome },
                },
                UpdateExpression: 'SET #end = :end, #outcome = :outcome',
                // ensures update.
                // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Expressions.ConditionExpressions.html#Expressions.ConditionExpressions.PreventingOverwrites
                ConditionExpression: 'attribute_not_exists(#end) AND attribute_not_exists(#outcome)',
                ReturnValues: 'ALL_NEW',
            });
            if (!response.Attributes) {
                throw new Error('Unable to retrieve item attributes');
            }
            return {
                account: (0, dynamo_item_1.requiredValue)('account', response.Attributes),
                region: (0, dynamo_item_1.requiredValue)('region', response.Attributes),
                pool: (0, dynamo_item_1.requiredValue)('pool', response.Attributes),
                start: (0, dynamo_item_1.requiredValue)('start', response.Attributes),
                end: (0, dynamo_item_1.requiredValue)('end', response.Attributes),
                requester: (0, dynamo_item_1.requiredValue)('requester', response.Attributes),
                id: (0, dynamo_item_1.requiredValue)('id', response.Attributes),
                outcome: (0, dynamo_item_1.requiredValue)('outcome', response.Attributes),
            };
        }
        catch (e) {
            if (e instanceof ddb.ConditionalCheckFailedException) {
                throw new AllocationAlreadyEndedError(opts.id);
            }
            throw e;
        }
    }
}
exports.AllocationsClient = AllocationsClient;
//# sourceMappingURL=data:application/json;base64,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