"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SchedulerClient = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const client_scheduler_1 = require("@aws-sdk/client-scheduler");
const envars = require("../envars");
/**
 * Scheduler client to create the EventBridge schedulers needed by the system.
 */
class SchedulerClient {
    constructor(props) {
        this.props = props;
        this.scheduler = new client_scheduler_1.Scheduler();
    }
    async scheduleAllocationTimeout(opts) {
        const prefix = `${SchedulerClient.TIMEOUT_EVENT_PREFIX}.aloc_`;
        await this.scheduleLambdaInvoke({
            name: `${prefix}${opts.allocationId}`,
            description: `Enforces timeout for allocation ${opts.allocationId}`,
            functionArn: opts.functionArn,
            at: opts.timeoutDate,
            payload: { allocationId: opts.allocationId },
            dlqArn: envars.Envars.required(envars.ALLOCATION_TIMEOUT_DLQ_ARN_ENV),
        });
    }
    async scheduleCleanupTimeout(opts) {
        const prefix = `${SchedulerClient.TIMEOUT_EVENT_PREFIX}.clean_`;
        await this.scheduleLambdaInvoke({
            name: `${prefix}${opts.allocationId}`,
            description: `Enforces cleanup for allocation ${opts.allocationId}`,
            functionArn: opts.functionArn,
            at: opts.timeoutDate,
            payload: { allocationId: opts.allocationId, account: opts.account, region: opts.region },
            dlqArn: envars.Envars.required(envars.CLEANUP_TIMEOUT_DLQ_ARN_ENV),
        });
    }
    scheduleExpression(date) {
        // slice to get rid of timezone and milliseconds.
        // see https://docs.aws.amazon.com/scheduler/latest/APIReference/API_CreateSchedule.html#scheduler-CreateSchedule-request-ScheduleExpression
        return `at(${date.toISOString().slice(0, 19)})`;
    }
    async scheduleLambdaInvoke(opts) {
        // https://docs.aws.amazon.com/scheduler/latest/APIReference/API_CreateSchedule.html#API_CreateSchedule_RequestSyntax
        if (opts.name.length >= 64) {
            throw new Error(`Scheduler name '${opts.name}' is too long. Must be less than 64 characters.`);
        }
        await this.scheduler.createSchedule({
            Name: opts.name,
            Description: opts.description,
            FlexibleTimeWindow: { Mode: 'OFF' },
            ScheduleExpression: this.scheduleExpression(opts.at),
            ScheduleExpressionTimezone: 'UTC',
            Target: {
                DeadLetterConfig: {
                    Arn: opts.dlqArn,
                },
                Arn: opts.functionArn,
                RoleArn: this.props.roleArn,
                Input: JSON.stringify(opts.payload),
            },
            ActionAfterCompletion: 'DELETE',
        });
    }
}
exports.SchedulerClient = SchedulerClient;
SchedulerClient.TIMEOUT_EVENT_PREFIX = 'atmosphere.timeout';
//# sourceMappingURL=data:application/json;base64,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