"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = handler;
exports.doHandler = doHandler;
const clients_1 = require("../../clients");
const logging_1 = require("../../logging");
const environments_client_1 = require("../../storage/environments.client");
const clients = clients_1.RuntimeClients.getOrCreate();
/**
 * Responsible for marking an environment as `dirty` in case the cleanup task (executed separately)
 * has timed out and didn't perform a successfull cleanup. When this happens, a human is notified in
 * order to attend to the environment and perform a manual cleanup. This ensures that the environment
 * doesn't remain in `cleaning` state forever.
 *
 * It is triggered on a fixed schedule via an EventBridge (one-time) schedule that is created upon deallocation.
 * Being a schedule, it may very well be triggered even if a cleanup was already performed successfully, or even
 * if the environment has already been reallocated to another requester; in that case, this function will simply
 * return and do nothing.
 *
 * Implementation
 * --------------
 *
 * 1. Update the `status` attribute in the `environments` table to `dirty`.
 *
 * > Note that we could have also configured the schedule itself to perform DB updates directly.
 * > However, custom exception handling is required in order to implement idempotancy, and
 * > that can only be done with a dedicated function.
 */
async function handler(event) {
    console.log('Event:', JSON.stringify(event, null, 2));
    const allocationId = event.allocationId;
    const allocation = await clients.allocations.get(event.allocationId);
    const log = new logging_1.Logger({ allocationId: allocationId, pool: allocation.pool, component: 'cleanup-timeout' });
    await doHandler(event, log);
}
async function doHandler(event, log) {
    try {
        log.info(`Marking environment 'aws://${event.account}/${event.region}' as dirty`);
        await clients.environments.dirty(event.allocationId, event.account, event.region);
        log.info('Done');
    }
    catch (e) {
        if (e instanceof environments_client_1.EnvironmentAlreadyReleasedError) {
            // happy path - the cleanup task succeeded and released the environment
            log.info(e.message);
            return;
        }
        if (e instanceof environments_client_1.EnvironmentAlreadyDirtyError) {
            // happy path - the cleanup task already marked it as dirty
            log.info(e.message);
            return;
        }
        if (e instanceof environments_client_1.EnvironmentAlreadyReallocated) {
            // semi happy path - the environment has already been reallocated by the time
            // this event got triggered. that's fine - it means it was already cleaned.
            log.info(e.message);
            return;
        }
        log.error(e);
        throw e;
    }
}
//# sourceMappingURL=data:application/json;base64,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