"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = handler;
exports.doHandler = doHandler;
// eslint-disable-next-line import/no-extraneous-dependencies
const client_lambda_1 = require("@aws-sdk/client-lambda");
const clients_1 = require("../../clients");
const envars_1 = require("../../envars");
const logging_1 = require("../../logging");
const clients = clients_1.RuntimeClients.getOrCreate();
/**
 * Responsible for forcefully releasing an environment in case its allocation period has
 * expired before the requester explicitly released it (for example if the requester process crashed).
 * This ensures a requester cannot hold an environment forever.
 *
 * It is triggered on a fixed schedule via an EventBridge (one-time) schedule that is created upon allocation.
 * Being a schedule, it may very well be triggered even if the allocation was already explicitly ended by the
 * requester; in that case, this function will simply return and do nothing.
 *
 * Implementation
 * --------------
 *
 * 1. Synchronously invoke the deallocate function with an outcome of `timeout`.
 *
 * > Note that we could have also configured the schedule itself to do so.
 * > However, this function gives us a place to perform additional perations
 * > on an allocation timeout.
*/
async function handler(event) {
    console.log('Event:', JSON.stringify(event, null, 2));
    const allocation = await clients.allocations.get(event.allocationId);
    const log = new logging_1.Logger({ allocationId: event.allocationId, pool: allocation.pool, component: 'allocation-timeout' });
    await doHandler(event, log);
}
async function doHandler(event, log) {
    try {
        const body = JSON.stringify({ outcome: 'timeout' });
        const lambda = new client_lambda_1.Lambda();
        const payload = JSON.stringify({ pathParameters: { id: event.allocationId }, body });
        const target = envars_1.Envars.required(envars_1.DEALLOCATE_FUNCTION_NAME_ENV);
        log.info(`Invoking ${target} with payload: ${payload}`);
        const response = await lambda.invoke({ FunctionName: target, InvocationType: 'RequestResponse', Payload: payload });
        const responsePayload = JSON.parse(response.Payload?.transformToString('utf-8') ?? '{}');
        if (responsePayload.statusCode !== 200) {
            throw new Error(`Unexpected response status code ${responsePayload.statusCode}: ${responsePayload.body}`);
        }
        log.info('Done');
    }
    catch (e) {
        log.error(e);
        throw e;
    }
}
//# sourceMappingURL=data:application/json;base64,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