"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = handler;
const clients_1 = require("../clients");
const envars = require("../envars");
const logging_1 = require("../logging");
const allocations_client_1 = require("../storage/allocations.client");
// an hour should plenty to clean one environment
const CLEANUP_TIMEOUT_SECONDS = 60 * 60;
class ProxyError extends Error {
    constructor(statusCode, message) {
        super(`${statusCode}: ${message}`);
        this.statusCode = statusCode;
        this.message = message;
    }
}
const clients = clients_1.RuntimeClients.getOrCreate();
async function handler(event) {
    console.log('Event:', JSON.stringify(event, null, 2));
    let allocationId, request, pool;
    try {
        [allocationId, request] = parseEvent(event);
        pool = (await clients.allocations.get(allocationId)).pool;
    }
    catch (e) {
        return failure(e);
    }
    const log = new logging_1.Logger({ allocationId, pool, component: 'deallocate' });
    return safeDoHandler(allocationId, request, log);
}
async function safeDoHandler(allocationId, request, log) {
    try {
        return await doHandler(allocationId, request, log);
    }
    catch (e) {
        log.error(e);
        return failure(e);
    }
}
async function doHandler(allocationId, request, log) {
    try {
        const cleanupTimeoutDate = new Date(Date.now() + 1000 * CLEANUP_TIMEOUT_SECONDS);
        log.info(`Ending allocation with outcome: ${request.outcome}`);
        const allocation = await endAllocation(allocationId, request.outcome);
        log.info(`Scheduling timeout for cleanup of environment 'aws://${allocation.account}/${allocation.region}' to ${cleanupTimeoutDate}`);
        await clients.scheduler.scheduleCleanupTimeout({
            allocationId: allocation.id,
            account: allocation.account,
            region: allocation.region,
            timeoutDate: cleanupTimeoutDate,
            functionArn: envars.Envars.required(envars.CLEANUP_TIMEOUT_FUNCTION_ARN_ENV),
        });
        log.info(`Starting cleanup of 'aws://${allocation.account}/${allocation.region}'`);
        await clients.environments.cleaning(allocationId, allocation.account, allocation.region);
        const taskInstanceArn = await clients.cleanup.start({ allocation, timeoutSeconds: CLEANUP_TIMEOUT_SECONDS });
        log.info(`Successfully started cleanup task: ${taskInstanceArn}`);
        return success();
    }
    catch (e) {
        if (e instanceof allocations_client_1.AllocationAlreadyEndedError) {
            // expected because deallocation can be requested either
            // by the timeout event or explicitly by the user.
            log.info(`Returning success because: ${e.message}`);
            return success();
        }
        throw e;
    }
}
function parseEvent(event) {
    const id = (event.pathParameters ?? {}).id;
    if (!id) {
        throw new ProxyError(400, 'Missing \'id\' path parameter');
    }
    if (!event.body) {
        throw new ProxyError(400, 'Request body not found');
    }
    const parsed = JSON.parse(event.body);
    if (!parsed.outcome) {
        throw new ProxyError(400, '\'outcome\' must be provided in the request body');
    }
    return [id, parsed];
}
async function endAllocation(id, outcome) {
    try {
        return await clients.allocations.end({ id, outcome });
    }
    catch (e) {
        if (e instanceof allocations_client_1.InvalidInputError) {
            throw new ProxyError(400, e.message);
        }
        throw e;
    }
}
function success() {
    return { statusCode: 200, body: JSON.stringify({}) };
}
function failure(e) {
    const statusCode = e instanceof ProxyError ? e.statusCode : 500;
    return { statusCode: statusCode, body: JSON.stringify({ message: e.message }) };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVhbGxvY2F0ZS5sYW1iZGEuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvZGVhbGxvY2F0ZS9kZWFsbG9jYXRlLmxhbWJkYS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQXFCQSwwQkFhQztBQWpDRCx3Q0FBNEM7QUFDNUMsb0NBQW9DO0FBQ3BDLHdDQUFvQztBQUNwQyxzRUFBMkc7QUFFM0csaURBQWlEO0FBQ2pELE1BQU0sdUJBQXVCLEdBQUcsRUFBRSxHQUFHLEVBQUUsQ0FBQztBQUV4QyxNQUFNLFVBQVcsU0FBUSxLQUFLO0lBQzVCLFlBQTRCLFVBQWtCLEVBQWtCLE9BQWU7UUFDN0UsS0FBSyxDQUFDLEdBQUcsVUFBVSxLQUFLLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFEVCxlQUFVLEdBQVYsVUFBVSxDQUFRO1FBQWtCLFlBQU8sR0FBUCxPQUFPLENBQVE7SUFFL0UsQ0FBQztDQUNGO0FBTUQsTUFBTSxPQUFPLEdBQUcsd0JBQWMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztBQUV0QyxLQUFLLFVBQVUsT0FBTyxDQUFDLEtBQTJCO0lBQ3ZELE9BQU8sQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBRXRELElBQUksWUFBWSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUM7SUFDaEMsSUFBSSxDQUFDO1FBQ0gsQ0FBQyxZQUFZLEVBQUUsT0FBTyxDQUFDLEdBQUcsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzVDLElBQUksR0FBRyxDQUFDLE1BQU0sT0FBTyxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7SUFDNUQsQ0FBQztJQUFDLE9BQU8sQ0FBTSxFQUFFLENBQUM7UUFDaEIsT0FBTyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDcEIsQ0FBQztJQUVELE1BQU0sR0FBRyxHQUFHLElBQUksZ0JBQU0sQ0FBQyxFQUFFLFlBQVksRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLFlBQVksRUFBRSxDQUFDLENBQUM7SUFDeEUsT0FBTyxhQUFhLENBQUMsWUFBWSxFQUFFLE9BQU8sRUFBRSxHQUFHLENBQUMsQ0FBQztBQUNuRCxDQUFDO0FBRUQsS0FBSyxVQUFVLGFBQWEsQ0FBQyxZQUFvQixFQUFFLE9BQTBCLEVBQUUsR0FBVztJQUN4RixJQUFJLENBQUM7UUFDSCxPQUFPLE1BQU0sU0FBUyxDQUFDLFlBQVksRUFBRSxPQUFPLEVBQUUsR0FBRyxDQUFDLENBQUM7SUFDckQsQ0FBQztJQUFDLE9BQU8sQ0FBTSxFQUFFLENBQUM7UUFDaEIsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNiLE9BQU8sT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3BCLENBQUM7QUFDSCxDQUFDO0FBRUQsS0FBSyxVQUFVLFNBQVMsQ0FBQyxZQUFvQixFQUFFLE9BQTBCLEVBQUUsR0FBVztJQUNwRixJQUFJLENBQUM7UUFFSCxNQUFNLGtCQUFrQixHQUFHLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBRyxJQUFJLEdBQUcsdUJBQXVCLENBQUMsQ0FBQztRQUVqRixHQUFHLENBQUMsSUFBSSxDQUFDLG1DQUFtQyxPQUFPLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUMvRCxNQUFNLFVBQVUsR0FBRyxNQUFNLGFBQWEsQ0FBQyxZQUFZLEVBQUUsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRXRFLEdBQUcsQ0FBQyxJQUFJLENBQUMsd0RBQXdELFVBQVUsQ0FBQyxPQUFPLElBQUksVUFBVSxDQUFDLE1BQU0sUUFBUSxrQkFBa0IsRUFBRSxDQUFDLENBQUM7UUFDdEksTUFBTSxPQUFPLENBQUMsU0FBUyxDQUFDLHNCQUFzQixDQUFDO1lBQzdDLFlBQVksRUFBRSxVQUFVLENBQUMsRUFBRTtZQUMzQixPQUFPLEVBQUUsVUFBVSxDQUFDLE9BQU87WUFDM0IsTUFBTSxFQUFFLFVBQVUsQ0FBQyxNQUFNO1lBQ3pCLFdBQVcsRUFBRSxrQkFBa0I7WUFDL0IsV0FBVyxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxnQ0FBZ0MsQ0FBQztTQUM3RSxDQUFDLENBQUM7UUFFSCxHQUFHLENBQUMsSUFBSSxDQUFDLDhCQUE4QixVQUFVLENBQUMsT0FBTyxJQUFJLFVBQVUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO1FBQ25GLE1BQU0sT0FBTyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUFFLFVBQVUsQ0FBQyxPQUFPLEVBQUUsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3pGLE1BQU0sZUFBZSxHQUFHLE1BQU0sT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRSxVQUFVLEVBQUUsY0FBYyxFQUFFLHVCQUF1QixFQUFFLENBQUMsQ0FBQztRQUU3RyxHQUFHLENBQUMsSUFBSSxDQUFDLHNDQUFzQyxlQUFlLEVBQUUsQ0FBQyxDQUFDO1FBRWxFLE9BQU8sT0FBTyxFQUFFLENBQUM7SUFDbkIsQ0FBQztJQUFDLE9BQU8sQ0FBTSxFQUFFLENBQUM7UUFFaEIsSUFBSSxDQUFDLFlBQVksZ0RBQTJCLEVBQUUsQ0FBQztZQUM3Qyx3REFBd0Q7WUFDeEQsa0RBQWtEO1lBQ2xELEdBQUcsQ0FBQyxJQUFJLENBQUMsOEJBQThCLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1lBQ3BELE9BQU8sT0FBTyxFQUFFLENBQUM7UUFDbkIsQ0FBQztRQUVELE1BQU0sQ0FBQyxDQUFDO0lBQ1YsQ0FBQztBQUNILENBQUM7QUFFRCxTQUFTLFVBQVUsQ0FBQyxLQUFVO0lBRTVCLE1BQU0sRUFBRSxHQUFHLENBQUMsS0FBSyxDQUFDLGNBQWMsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDM0MsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDO1FBQ1IsTUFBTSxJQUFJLFVBQVUsQ0FBQyxHQUFHLEVBQUUsK0JBQStCLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNoQixNQUFNLElBQUksVUFBVSxDQUFDLEdBQUcsRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN0QyxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ3BCLE1BQU0sSUFBSSxVQUFVLENBQUMsR0FBRyxFQUFFLGtEQUFrRCxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVELE9BQU8sQ0FBQyxFQUFFLEVBQUUsTUFBTSxDQUFDLENBQUM7QUFDdEIsQ0FBQztBQUVELEtBQUssVUFBVSxhQUFhLENBQUMsRUFBVSxFQUFFLE9BQWU7SUFDdEQsSUFBSSxDQUFDO1FBQ0gsT0FBTyxNQUFNLE9BQU8sQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsRUFBRSxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7SUFDeEQsQ0FBQztJQUFDLE9BQU8sQ0FBTSxFQUFFLENBQUM7UUFDaEIsSUFBSSxDQUFDLFlBQVksc0NBQWlCLEVBQUUsQ0FBQztZQUNuQyxNQUFNLElBQUksVUFBVSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDdkMsQ0FBQztRQUNELE1BQU0sQ0FBQyxDQUFDO0lBQ1YsQ0FBQztBQUNILENBQUM7QUFFRCxTQUFTLE9BQU87SUFDZCxPQUFPLEVBQUUsVUFBVSxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDO0FBQ3ZELENBQUM7QUFFRCxTQUFTLE9BQU8sQ0FBQyxDQUFNO0lBQ3JCLE1BQU0sVUFBVSxHQUFHLENBQUMsWUFBWSxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQztJQUNoRSxPQUFPLEVBQUUsVUFBVSxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUMsRUFBRSxDQUFDO0FBQ2xGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBUElHYXRld2F5UHJveHlFdmVudCwgQVBJR2F0ZXdheVByb3h5UmVzdWx0IH0gZnJvbSAnYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBSdW50aW1lQ2xpZW50cyB9IGZyb20gJy4uL2NsaWVudHMnO1xuaW1wb3J0ICogYXMgZW52YXJzIGZyb20gJy4uL2VudmFycyc7XG5pbXBvcnQgeyBMb2dnZXIgfSBmcm9tICcuLi9sb2dnaW5nJztcbmltcG9ydCB7IEFsbG9jYXRpb24sIEFsbG9jYXRpb25BbHJlYWR5RW5kZWRFcnJvciwgSW52YWxpZElucHV0RXJyb3IgfSBmcm9tICcuLi9zdG9yYWdlL2FsbG9jYXRpb25zLmNsaWVudCc7XG5cbi8vIGFuIGhvdXIgc2hvdWxkIHBsZW50eSB0byBjbGVhbiBvbmUgZW52aXJvbm1lbnRcbmNvbnN0IENMRUFOVVBfVElNRU9VVF9TRUNPTkRTID0gNjAgKiA2MDtcblxuY2xhc3MgUHJveHlFcnJvciBleHRlbmRzIEVycm9yIHtcbiAgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IHN0YXR1c0NvZGU6IG51bWJlciwgcHVibGljIHJlYWRvbmx5IG1lc3NhZ2U6IHN0cmluZykge1xuICAgIHN1cGVyKGAke3N0YXR1c0NvZGV9OiAke21lc3NhZ2V9YCk7XG4gIH1cbn1cblxuZXhwb3J0IGludGVyZmFjZSBEZWFsbG9jYXRlUmVxdWVzdCB7XG4gIHJlYWRvbmx5IG91dGNvbWU6IHN0cmluZztcbn1cblxuY29uc3QgY2xpZW50cyA9IFJ1bnRpbWVDbGllbnRzLmdldE9yQ3JlYXRlKCk7XG5cbmV4cG9ydCBhc3luYyBmdW5jdGlvbiBoYW5kbGVyKGV2ZW50OiBBUElHYXRld2F5UHJveHlFdmVudCk6IFByb21pc2U8QVBJR2F0ZXdheVByb3h5UmVzdWx0PiB7XG4gIGNvbnNvbGUubG9nKCdFdmVudDonLCBKU09OLnN0cmluZ2lmeShldmVudCwgbnVsbCwgMikpO1xuXG4gIGxldCBhbGxvY2F0aW9uSWQsIHJlcXVlc3QsIHBvb2w7XG4gIHRyeSB7XG4gICAgW2FsbG9jYXRpb25JZCwgcmVxdWVzdF0gPSBwYXJzZUV2ZW50KGV2ZW50KTtcbiAgICBwb29sID0gKGF3YWl0IGNsaWVudHMuYWxsb2NhdGlvbnMuZ2V0KGFsbG9jYXRpb25JZCkpLnBvb2w7XG4gIH0gY2F0Y2ggKGU6IGFueSkge1xuICAgIHJldHVybiBmYWlsdXJlKGUpO1xuICB9XG5cbiAgY29uc3QgbG9nID0gbmV3IExvZ2dlcih7IGFsbG9jYXRpb25JZCwgcG9vbCwgY29tcG9uZW50OiAnZGVhbGxvY2F0ZScgfSk7XG4gIHJldHVybiBzYWZlRG9IYW5kbGVyKGFsbG9jYXRpb25JZCwgcmVxdWVzdCwgbG9nKTtcbn1cblxuYXN5bmMgZnVuY3Rpb24gc2FmZURvSGFuZGxlcihhbGxvY2F0aW9uSWQ6IHN0cmluZywgcmVxdWVzdDogRGVhbGxvY2F0ZVJlcXVlc3QsIGxvZzogTG9nZ2VyKSB7XG4gIHRyeSB7XG4gICAgcmV0dXJuIGF3YWl0IGRvSGFuZGxlcihhbGxvY2F0aW9uSWQsIHJlcXVlc3QsIGxvZyk7XG4gIH0gY2F0Y2ggKGU6IGFueSkge1xuICAgIGxvZy5lcnJvcihlKTtcbiAgICByZXR1cm4gZmFpbHVyZShlKTtcbiAgfVxufVxuXG5hc3luYyBmdW5jdGlvbiBkb0hhbmRsZXIoYWxsb2NhdGlvbklkOiBzdHJpbmcsIHJlcXVlc3Q6IERlYWxsb2NhdGVSZXF1ZXN0LCBsb2c6IExvZ2dlcik6IFByb21pc2U8QVBJR2F0ZXdheVByb3h5UmVzdWx0PiB7XG4gIHRyeSB7XG5cbiAgICBjb25zdCBjbGVhbnVwVGltZW91dERhdGUgPSBuZXcgRGF0ZShEYXRlLm5vdygpICsgMTAwMCAqIENMRUFOVVBfVElNRU9VVF9TRUNPTkRTKTtcblxuICAgIGxvZy5pbmZvKGBFbmRpbmcgYWxsb2NhdGlvbiB3aXRoIG91dGNvbWU6ICR7cmVxdWVzdC5vdXRjb21lfWApO1xuICAgIGNvbnN0IGFsbG9jYXRpb24gPSBhd2FpdCBlbmRBbGxvY2F0aW9uKGFsbG9jYXRpb25JZCwgcmVxdWVzdC5vdXRjb21lKTtcblxuICAgIGxvZy5pbmZvKGBTY2hlZHVsaW5nIHRpbWVvdXQgZm9yIGNsZWFudXAgb2YgZW52aXJvbm1lbnQgJ2F3czovLyR7YWxsb2NhdGlvbi5hY2NvdW50fS8ke2FsbG9jYXRpb24ucmVnaW9ufScgdG8gJHtjbGVhbnVwVGltZW91dERhdGV9YCk7XG4gICAgYXdhaXQgY2xpZW50cy5zY2hlZHVsZXIuc2NoZWR1bGVDbGVhbnVwVGltZW91dCh7XG4gICAgICBhbGxvY2F0aW9uSWQ6IGFsbG9jYXRpb24uaWQsXG4gICAgICBhY2NvdW50OiBhbGxvY2F0aW9uLmFjY291bnQsXG4gICAgICByZWdpb246IGFsbG9jYXRpb24ucmVnaW9uLFxuICAgICAgdGltZW91dERhdGU6IGNsZWFudXBUaW1lb3V0RGF0ZSxcbiAgICAgIGZ1bmN0aW9uQXJuOiBlbnZhcnMuRW52YXJzLnJlcXVpcmVkKGVudmFycy5DTEVBTlVQX1RJTUVPVVRfRlVOQ1RJT05fQVJOX0VOViksXG4gICAgfSk7XG5cbiAgICBsb2cuaW5mbyhgU3RhcnRpbmcgY2xlYW51cCBvZiAnYXdzOi8vJHthbGxvY2F0aW9uLmFjY291bnR9LyR7YWxsb2NhdGlvbi5yZWdpb259J2ApO1xuICAgIGF3YWl0IGNsaWVudHMuZW52aXJvbm1lbnRzLmNsZWFuaW5nKGFsbG9jYXRpb25JZCwgYWxsb2NhdGlvbi5hY2NvdW50LCBhbGxvY2F0aW9uLnJlZ2lvbik7XG4gICAgY29uc3QgdGFza0luc3RhbmNlQXJuID0gYXdhaXQgY2xpZW50cy5jbGVhbnVwLnN0YXJ0KHsgYWxsb2NhdGlvbiwgdGltZW91dFNlY29uZHM6IENMRUFOVVBfVElNRU9VVF9TRUNPTkRTIH0pO1xuXG4gICAgbG9nLmluZm8oYFN1Y2Nlc3NmdWxseSBzdGFydGVkIGNsZWFudXAgdGFzazogJHt0YXNrSW5zdGFuY2VBcm59YCk7XG5cbiAgICByZXR1cm4gc3VjY2VzcygpO1xuICB9IGNhdGNoIChlOiBhbnkpIHtcblxuICAgIGlmIChlIGluc3RhbmNlb2YgQWxsb2NhdGlvbkFscmVhZHlFbmRlZEVycm9yKSB7XG4gICAgICAvLyBleHBlY3RlZCBiZWNhdXNlIGRlYWxsb2NhdGlvbiBjYW4gYmUgcmVxdWVzdGVkIGVpdGhlclxuICAgICAgLy8gYnkgdGhlIHRpbWVvdXQgZXZlbnQgb3IgZXhwbGljaXRseSBieSB0aGUgdXNlci5cbiAgICAgIGxvZy5pbmZvKGBSZXR1cm5pbmcgc3VjY2VzcyBiZWNhdXNlOiAke2UubWVzc2FnZX1gKTtcbiAgICAgIHJldHVybiBzdWNjZXNzKCk7XG4gICAgfVxuXG4gICAgdGhyb3cgZTtcbiAgfVxufVxuXG5mdW5jdGlvbiBwYXJzZUV2ZW50KGV2ZW50OiBhbnkpOiBbc3RyaW5nLCBEZWFsbG9jYXRlUmVxdWVzdF0ge1xuXG4gIGNvbnN0IGlkID0gKGV2ZW50LnBhdGhQYXJhbWV0ZXJzID8/IHt9KS5pZDtcbiAgaWYgKCFpZCkge1xuICAgIHRocm93IG5ldyBQcm94eUVycm9yKDQwMCwgJ01pc3NpbmcgXFwnaWRcXCcgcGF0aCBwYXJhbWV0ZXInKTtcbiAgfVxuXG4gIGlmICghZXZlbnQuYm9keSkge1xuICAgIHRocm93IG5ldyBQcm94eUVycm9yKDQwMCwgJ1JlcXVlc3QgYm9keSBub3QgZm91bmQnKTtcbiAgfVxuXG4gIGNvbnN0IHBhcnNlZCA9IEpTT04ucGFyc2UoZXZlbnQuYm9keSk7XG4gIGlmICghcGFyc2VkLm91dGNvbWUpIHtcbiAgICB0aHJvdyBuZXcgUHJveHlFcnJvcig0MDAsICdcXCdvdXRjb21lXFwnIG11c3QgYmUgcHJvdmlkZWQgaW4gdGhlIHJlcXVlc3QgYm9keScpO1xuICB9XG5cbiAgcmV0dXJuIFtpZCwgcGFyc2VkXTtcbn1cblxuYXN5bmMgZnVuY3Rpb24gZW5kQWxsb2NhdGlvbihpZDogc3RyaW5nLCBvdXRjb21lOiBzdHJpbmcpOiBQcm9taXNlPEFsbG9jYXRpb24+IHtcbiAgdHJ5IHtcbiAgICByZXR1cm4gYXdhaXQgY2xpZW50cy5hbGxvY2F0aW9ucy5lbmQoeyBpZCwgb3V0Y29tZSB9KTtcbiAgfSBjYXRjaCAoZTogYW55KSB7XG4gICAgaWYgKGUgaW5zdGFuY2VvZiBJbnZhbGlkSW5wdXRFcnJvcikge1xuICAgICAgdGhyb3cgbmV3IFByb3h5RXJyb3IoNDAwLCBlLm1lc3NhZ2UpO1xuICAgIH1cbiAgICB0aHJvdyBlO1xuICB9XG59XG5cbmZ1bmN0aW9uIHN1Y2Nlc3MoKSB7XG4gIHJldHVybiB7IHN0YXR1c0NvZGU6IDIwMCwgYm9keTogSlNPTi5zdHJpbmdpZnkoe30pIH07XG59XG5cbmZ1bmN0aW9uIGZhaWx1cmUoZTogYW55KSB7XG4gIGNvbnN0IHN0YXR1c0NvZGUgPSBlIGluc3RhbmNlb2YgUHJveHlFcnJvciA/IGUuc3RhdHVzQ29kZSA6IDUwMDtcbiAgcmV0dXJuIHsgc3RhdHVzQ29kZTogc3RhdHVzQ29kZSwgYm9keTogSlNPTi5zdHJpbmdpZnkoeyBtZXNzYWdlOiBlLm1lc3NhZ2UgfSkgfTtcbn0iXX0=