"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigurationClient = exports.EnvironmentNotFoundError = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const s3 = require("@aws-sdk/client-s3");
/**
 * Error thrown when an environment is not found.
 */
class EnvironmentNotFoundError extends Error {
    constructor(account, region) {
        super(`Environment aws://${account}/${region} not found`);
        this.account = account;
        this.region = region;
    }
}
exports.EnvironmentNotFoundError = EnvironmentNotFoundError;
/**
 * Client for accessing the configuration file at runtime.
 */
class ConfigurationClient {
    constructor(s3Location) {
        this.s3Location = s3Location;
        this.s3Client = new s3.S3();
    }
    /**
     * Retrieve environments belonging to a specific pool.
     */
    async listEnvironments(opts = {}) {
        return (await this.data).environments.filter(e => opts.pool ? e.pool === opts.pool : true);
    }
    /**
     * Retrieve a single environment based on account + region.
     */
    async getEnvironment(account, region) {
        const envs = (await this.data).environments.filter(e => e.account === account && e.region === region);
        if (envs.length === 0) {
            throw new EnvironmentNotFoundError(account, region);
        }
        if (envs.length > 1) {
            throw new Error(`Multiple environments found for aws://${account}/${region}`);
        }
        return envs[0];
    }
    // lazy async getter
    get data() {
        return (async () => {
            if (this._data) {
                return this._data;
            }
            this._data = await this.download();
            return this._data;
        })();
    }
    async download() {
        const response = await this.s3Client.getObject({
            Bucket: this.s3Location.bucket,
            Key: this.s3Location.key,
        });
        if (!response.Body) {
            throw new Error(`Configuration file (s3://${this.s3Location.bucket}/${this.s3Location.key}) is empty`);
        }
        return JSON.parse(await response.Body.transformToString('utf-8'));
    }
}
exports.ConfigurationClient = ConfigurationClient;
//# sourceMappingURL=data:application/json;base64,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