"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = handler;
const clients_1 = require("../clients");
const cleaner_1 = require("./cleaner");
const envars = require("../envars");
const logging_1 = require("../logging");
const environments_client_1 = require("../storage/environments.client");
const clients = clients_1.RuntimeClients.getOrCreate();
async function handler(req) {
    const allocation = await clients.allocations.get(req.allocationId);
    const log = new logging_1.Logger({ allocationId: req.allocationId, pool: allocation.pool, component: 'cleanup' });
    return doHandler(allocation, req.timeoutSeconds, log);
}
async function doHandler(allocation, timeoutSeconds, log) {
    const env = `aws://${allocation.account}/${allocation.region}`;
    try {
        log.info(`Fetching environment '${env}'`);
        const environment = await clients.configuration.getEnvironment(allocation.account, allocation.region);
        const cleaner = new cleaner_1.Cleaner(environment, log);
        log.info(`Starting cleanup of '${env}'`);
        await cleaner.clean(timeoutSeconds);
        log.info(`Successfully cleaned '${env}'`);
        log.info(`Releasing environment '${env}'`);
        await clients.environments.release(allocation.id, environment.account, environment.region);
        log.info(`Successfully released environment '${env}'`);
        log.info('Done!');
    }
    catch (e) {
        if (e instanceof environments_client_1.EnvironmentAlreadyInUseError) {
            // this is an illegal state transition and should never happen.
            // TODO - we should catch this state before cleaning any resources.
            log.error(e, `Failed releasing environment '${env}'`);
            throw e;
        }
        log.error(e);
        log.info(`Marking environment '${env}' as 'dirty'`);
        await clients.environments.dirty(allocation.id, allocation.account, allocation.region);
        log.info(`Successfully marked environment '${env}' as 'dirty'`);
        if (e instanceof cleaner_1.CleanerError) {
            log.info();
            log.info('>> Failed stacks errors report <<');
            for (const f of e.failedStacks) {
                log.info('');
                log.info(`----- Stack: ${f.name} -----`);
                log.info('');
                log.error(f.error);
            }
            return;
        }
        throw e;
    }
}
if (require.main !== module) {
    // do nothing - this file is just being imported
}
else {
    const allocationId = envars.Envars.required(envars.CLEANUP_TASK_ALLOCATION_ID);
    const timeoutSeconds = Number(envars.Envars.required(envars.CLEANUP_TASK_TIMEOUT_SECONDS));
    void handler({ allocationId, timeoutSeconds });
}
//# sourceMappingURL=data:application/json;base64,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