"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReposCleaner = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const client_cloudformation_1 = require("@aws-sdk/client-cloudformation");
// eslint-disable-next-line import/no-extraneous-dependencies
const client_ecr_1 = require("@aws-sdk/client-ecr");
class ReposCleaner {
    constructor(credentials, region, stack, log) {
        this.stack = stack;
        this.log = log;
        this.cfn = new client_cloudformation_1.CloudFormation({ credentials: credentials, region });
        this.ecr = new client_ecr_1.ECR({ credentials: credentials, region });
    }
    async clean() {
        // first collect all repositories in the stack
        const repos = await this.listRepos();
        // then force delete them (including images inside)
        for (const repo of repos) {
            await this.deleteRepo(repo);
        }
    }
    async listRepos() {
        this.log.info(`Collecting ECR repositories in stack ${this.stack.StackName}`);
        return ((await this.cfn.describeStackResources({ StackName: this.stack.StackName })).StackResources ?? [])
            .filter(r => r.ResourceType === 'AWS::ECR::Repository').map(r => r.PhysicalResourceId);
    }
    async deleteRepo(repositoryName) {
        this.log.info(`Deleting repository: ${repositoryName}`);
        try {
            await this.ecr.deleteRepository({ repositoryName: repositoryName, force: true });
        }
        catch (e) {
            if (e instanceof client_ecr_1.RepositoryNotFoundException) {
                this.log.info(`Repository ${repositoryName} does not exist. Skipping.`);
                return;
            }
            throw e;
        }
    }
}
exports.ReposCleaner = ReposCleaner;
//# sourceMappingURL=data:application/json;base64,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