"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cleaner = exports.CleanerError = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const client_cloudformation_1 = require("@aws-sdk/client-cloudformation");
// eslint-disable-next-line import/no-extraneous-dependencies
const credential_providers_1 = require("@aws-sdk/credential-providers");
const cleaner_buckets_1 = require("./cleaner.buckets");
const cleaner_repos_1 = require("./cleaner.repos");
/**
 * Error thrown when the cleanup process did not result in a clean environment.
 */
class CleanerError extends Error {
    constructor(failedStacks) {
        super('Some stacks failed to delete');
        this.failedStacks = failedStacks;
    }
}
exports.CleanerError = CleanerError;
class Cleaner {
    constructor(environment, log) {
        this.environment = environment;
        this.log = log;
        this.credentials = (0, credential_providers_1.fromTemporaryCredentials)({
            params: {
                RoleArn: this.environment.adminRoleArn,
                RoleSessionName: `atmosphere.cleanup.${this.environment.account}.${this.environment.region}`,
            },
        });
        this.cfn = new client_cloudformation_1.CloudFormation({ credentials: this.credentials, region: this.environment.region });
    }
    async clean(timeoutSeconds) {
        const timeoutDate = new Date(Date.now() + 1000 * timeoutSeconds);
        const stacks = await this.listStacks();
        const promises = stacks.map(async (s) => this.deleteStack(s, timeoutDate));
        const results = await Promise.all(promises);
        const failed = results.filter(r => r.error);
        if (failed.length > 0) {
            throw new CleanerError(failed);
        }
    }
    async listStacks() {
        const stacksPaginator = (0, client_cloudformation_1.paginateDescribeStacks)({ client: this.cfn, pageSize: 10 }, {});
        const stacks = [];
        for await (const page of stacksPaginator) {
            stacks.push(...page.Stacks ?? []);
        }
        return stacks;
    }
    async deleteStack(stack, timeoutDate) {
        try {
            // most commonly, stacks cannot be deleted because their buckets or ECR repositories
            // contain objects, and must be cleaned before attempting deletion.
            // so instead, we just forcefully remove them and hope it happens before CFN decides to fail
            // stack deletion. if this won't suffice, we can catch a DELETE_FAILED state and retry.
            const bucketsCleaner = new cleaner_buckets_1.BucketsCleaner(this.credentials, this.environment.region, stack, this.log);
            this.log.info(`Cleaning buckets in stack ${stack.StackName}`);
            await bucketsCleaner.clean({ timeoutDate });
            const reposCleaner = new cleaner_repos_1.ReposCleaner(this.credentials, this.environment.region, stack, this.log);
            this.log.info(`Cleaning repositories in stack ${stack.StackName}`);
            // not passing timeout date here because this operation does not need to wait for anything
            await reposCleaner.clean();
            if (stack.ParentId) {
                // for nested stacks, we stop here because deletion
                // itself will be done via the parent stack.
                return { name: stack.StackName };
            }
            if (stack.StackStatus !== 'DELETE_IN_PROGRESS') {
                this.log.info(`Disabling termination protection of stack ${stack.StackName}`);
                await this.cfn.send(new client_cloudformation_1.UpdateTerminationProtectionCommand({
                    StackName: stack.StackName,
                    EnableTerminationProtection: false,
                }));
                this.log.info(`Initiating stack deletion: ${stack.StackName} [Current Status: ${stack.StackStatus}]`);
                await this.cfn.send(new client_cloudformation_1.DeleteStackCommand({ StackName: stack.StackName, RoleARN: this.environment.adminRoleArn }));
            }
            const maxWaitSeconds = (timeoutDate.getTime() - Date.now()) / 1000;
            this.log.info(`Stack ${stack.StackName} deleting. Waiting ${maxWaitSeconds} seconds for completion`);
            await (0, client_cloudformation_1.waitUntilStackDeleteComplete)({ client: this.cfn, maxWaitTime: maxWaitSeconds, minDelay: 5, maxDelay: 5 }, { StackName: stack.StackName });
            this.log.info(`Stack ${stack.StackName} deleted.`);
            return { name: stack.StackName };
        }
        catch (e) {
            return { name: stack.StackName, error: e };
        }
    }
}
exports.Cleaner = Cleaner;
//# sourceMappingURL=data:application/json;base64,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