"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BucketsCleaner = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const client_cloudformation_1 = require("@aws-sdk/client-cloudformation");
// eslint-disable-next-line import/no-extraneous-dependencies
const client_s3_1 = require("@aws-sdk/client-s3");
class BucketsCleaner {
    constructor(credentials, region, stack, log) {
        this.stack = stack;
        this.log = log;
        this.cfn = new client_cloudformation_1.CloudFormation({ credentials: credentials, region });
        this.s3 = new client_s3_1.S3({ credentials: credentials, region });
    }
    async clean(opts) {
        // first collect all buckets in the stack
        const buckets = await this.listBuckets();
        // then empty them
        const emptyBuckets = buckets.map(b => this.emptyBucket(b, opts.timeoutDate));
        await Promise.all(emptyBuckets);
        // then delete then
        const deleteBuckets = buckets.map(b => this.deleteBucket(b, opts.timeoutDate));
        await Promise.all(deleteBuckets);
    }
    async listBuckets() {
        this.log.info(`Collecting buckets in stack ${this.stack.StackName}`);
        const resources = await this.cfn.describeStackResources({ StackName: this.stack.StackName });
        return (resources.StackResources ?? [])
            .filter(r => r.ResourceType === 'AWS::S3::Bucket').map(r => r.PhysicalResourceId);
    }
    async emptyBucket(bucketName, timeoutDate) {
        try {
            let isTruncated = true;
            let keyMarker = undefined;
            let versionIdMarker = undefined;
            const checkTimeout = () => {
                if (Date.now() > timeoutDate.getTime()) {
                    throw new Error(`Operation timed out. Timeout date: ${timeoutDate.toISOString()}`);
                }
            };
            this.log.info(`Starting to clean bucket: ${bucketName}`);
            while (isTruncated) {
                checkTimeout();
                const response = await this.s3.listObjectVersions({
                    Bucket: bucketName,
                    KeyMarker: keyMarker,
                    VersionIdMarker: versionIdMarker,
                });
                const versions = response.Versions ?? [];
                const deleteMarkers = response.DeleteMarkers ?? [];
                const objectsToDelete = [
                    ...versions.map((version) => ({
                        Key: version.Key,
                        VersionId: version.VersionId,
                    })),
                    ...deleteMarkers.map((marker) => ({
                        Key: marker.Key,
                        VersionId: marker.VersionId,
                    })),
                ];
                if (objectsToDelete.length === 0) {
                    this.log.info('Bucket is already empty.');
                    break;
                }
                if (objectsToDelete.length > 0) {
                    await this.s3.send(new client_s3_1.DeleteObjectsCommand({
                        Bucket: bucketName,
                        Delete: {
                            Objects: objectsToDelete,
                            Quiet: true,
                        },
                    }));
                    this.log.info(`Deleted ${objectsToDelete.length} objects.`);
                }
                isTruncated = response.IsTruncated ?? false;
                keyMarker = response.NextKeyMarker;
                versionIdMarker = response.NextVersionIdMarker;
            }
            this.log.info(`Bucket ${bucketName} has been emptied.`);
        }
        catch (e) {
            if (e instanceof client_s3_1.NoSuchBucket) {
                // can happen because cleanup may execute when the stack is already deleting
                this.log.info(`Bucket ${bucketName} does not exist. Skipping.`);
                return;
            }
            throw e;
        }
    }
    async deleteBucket(bucketName, timeoutDate) {
        this.log.info(`Deleting bucket: ${bucketName}`);
        try {
            await this.s3.deleteBucket({ Bucket: bucketName });
        }
        catch (e) {
            if (e instanceof client_s3_1.NoSuchBucket) {
                this.log.info(`Bucket ${bucketName} does not exist. Skipping.`);
                return;
            }
            throw e;
        }
        const maxWaitSeconds = (timeoutDate.getTime() - Date.now()) / 1000;
        this.log.info(`Bucket ${bucketName} deleting. Waiting ${maxWaitSeconds} seconds for completion`);
        await (0, client_s3_1.waitUntilBucketNotExists)({ client: this.s3, maxWaitTime: maxWaitSeconds, minDelay: 5, maxDelay: 5 }, { Bucket: bucketName });
    }
}
exports.BucketsCleaner = BucketsCleaner;
//# sourceMappingURL=data:application/json;base64,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