"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Endpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const certificates = require("aws-cdk-lib/aws-certificatemanager");
const iam = require("aws-cdk-lib/aws-iam");
const r53 = require("aws-cdk-lib/aws-route53");
const r53targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
class Endpoint extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-authorization-flow.html#apigateway-authorization-flow-resource-policy-only
        // see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-resource-policies-examples.html#apigateway-resource-policies-cross-account-example
        const principals = props.allowedPrincipals ?? [new iam.AnyPrincipal()];
        const effect = props.allowedPrincipals ? iam.Effect.ALLOW : iam.Effect.DENY;
        const resources = props.allowedPrincipals ? [
            'execute-api:/prod/POST/allocations',
            'execute-api:/prod/DELETE/allocations/{id}',
        ] : ['*'];
        // Create the API Gateway
        this.api = new apigateway.RestApi(this, 'Api', {
            description: 'RESTful endpoint for the Atmosphere service',
            restApiName: 'Atmosphere',
            policy: new iam.PolicyDocument({
                statements: [
                    new iam.PolicyStatement({
                        effect,
                        actions: ['execute-api:Invoke'],
                        principals,
                        resources,
                    }),
                ],
            }),
            endpointTypes: [apigateway.EndpointType.REGIONAL],
            disableExecuteApiEndpoint: props.hostedZone ? true : false,
        });
        if (props.hostedZone) {
            const certificate = new certificates.Certificate(this, 'Certificate', {
                domainName: props.hostedZone.zoneName,
                validation: certificates.CertificateValidation.fromDns(props.hostedZone),
            });
            const domainName = this.api.addDomainName('DomainName', {
                domainName: props.hostedZone.zoneName,
                certificate,
                endpointType: apigateway.EndpointType.REGIONAL,
                // will reject TLS 1.0
                securityPolicy: apigateway.SecurityPolicy.TLS_1_2,
            });
            new r53.ARecord(this, 'ARecord', {
                zone: props.hostedZone,
                target: r53.RecordTarget.fromAlias(new r53targets.ApiGatewayDomain(domainName)),
            });
        }
        // Create /allocations resource
        this.allocationsResource = this.api.root.addResource('allocations');
        // POST /allocations -> Allocation Lambda
        this.allocationsResource.addMethod('POST', new apigateway.LambdaIntegration(props.allocate.function, {
            proxy: true,
        }), { authorizationType: apigateway.AuthorizationType.IAM });
        // DELETE /allocations/{id} -> Deallocation Lambda
        this.allocationResource = this.allocationsResource.addResource('{id}');
        this.allocationResource.addMethod('DELETE', new apigateway.LambdaIntegration(props.deallocate.function, {
            proxy: true,
        }), { authorizationType: apigateway.AuthorizationType.IAM });
    }
}
exports.Endpoint = Endpoint;
_a = JSII_RTTI_SYMBOL_1;
Endpoint[_a] = { fqn: "@cdklabs/cdk-atmosphere-service.Endpoint", version: "0.0.4" };
//# sourceMappingURL=data:application/json;base64,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