"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = handler;
// eslint-disable-next-line import/no-extraneous-dependencies
const crypto = require("crypto");
const client_sts_1 = require("@aws-sdk/client-sts");
// eslint-disable-next-line import/no-extraneous-dependencies
const clients_1 = require("../clients");
const envars = require("../envars");
const logging_1 = require("../logging");
const allocations_client_1 = require("../storage/allocations.client");
const environments_client_1 = require("../storage/environments.client");
// this will be the duration of the credentials being passed
// the the caller. currently, it cannot be more than 1 hour because of role chaining.
// TODO - how can we avoid role chaining?
const ALLOCATION_DURATION_SECONDS = 60 * 60;
class ProxyError extends Error {
    constructor(statusCode, message) {
        super(`${statusCode}: ${message}`);
        this.statusCode = statusCode;
        this.message = message;
    }
}
const clients = clients_1.RuntimeClients.getOrCreate();
async function handler(event) {
    console.log('Event:', JSON.stringify(event, null, 2));
    const allocationId = crypto.randomUUID();
    let request;
    try {
        request = parseRequestBody(event.body);
    }
    catch (e) {
        return failure(e);
    }
    const log = new logging_1.Logger({ allocationId: allocationId, pool: request.pool, component: 'allocate' });
    return safeDoHandler(allocationId, request, log);
}
async function safeDoHandler(allocationId, request, log) {
    try {
        return await doHandler(allocationId, request, log);
    }
    catch (e) {
        log.error(e);
        return failure(e);
    }
}
async function doHandler(allocationId, request, log) {
    const timeoutDate = new Date(Date.now() + 1000 * ALLOCATION_DURATION_SECONDS);
    log.info(`Acquiring environment from pool '${request.pool}'`);
    const environment = await acquireEnvironment(allocationId, request.pool);
    log.info(`Starting allocation of 'aws://${environment.account}/${environment.region}'`);
    await startAllocation(allocationId, environment, request.requester);
    log.info(`Grabbing credentials to aws://${environment.account}/${environment.region} using role: ${environment.adminRoleArn}`);
    const credentials = await grabCredentials(allocationId, environment);
    log.info('Allocation started successfully');
    const response = { id: allocationId, environment, credentials };
    log.info(`Scheduling allocation timeout to ${timeoutDate}`);
    await clients.scheduler.scheduleAllocationTimeout({
        allocationId,
        timeoutDate,
        functionArn: envars.Envars.required(envars.ALLOCATION_TIMEOUT_FUNCTION_ARN_ENV),
    });
    log.info('Done');
    return success(response);
}
function parseRequestBody(body) {
    if (!body) {
        throw new ProxyError(400, 'Request body not found');
    }
    const parsed = JSON.parse(body);
    if (!parsed.pool) {
        throw new ProxyError(400, '\'pool\' must be provided in the request body');
    }
    if (!parsed.requester) {
        throw new ProxyError(400, '\'requester\' must be provided in the request body');
    }
    return parsed;
}
async function acquireEnvironment(allocaionId, pool) {
    const candidates = await clients.configuration.listEnvironments({ pool });
    console.log(`Found ${candidates.length} environments in pool '${pool}'`);
    for (const canditate of candidates) {
        try {
            console.log(`Acquiring environment 'aws://${canditate.account}/${canditate.region}'...`);
            await clients.environments.acquire(allocaionId, canditate.account, canditate.region);
            return canditate;
        }
        catch (e) {
            if (e instanceof environments_client_1.EnvironmentAlreadyAcquiredError) {
                console.log(`Environment 'aws://${canditate.account}/${canditate.region}' already acquired. Trying the next one.`);
                continue;
            }
            throw e;
        }
    }
    throw new ProxyError(423, `No environments available in pool '${pool}'`);
}
async function startAllocation(id, environment, requester) {
    try {
        await clients.allocations.start({
            id,
            account: environment.account,
            region: environment.region,
            pool: environment.pool,
            requester,
        });
    }
    catch (e) {
        if (e instanceof allocations_client_1.InvalidInputError) {
            throw new ProxyError(400, e.message);
        }
        throw e;
    }
}
async function grabCredentials(id, environment) {
    const sts = new client_sts_1.STS();
    const assumed = await sts.assumeRole({
        RoleArn: environment.adminRoleArn,
        RoleSessionName: `atmosphere.allocation.${id}`,
    });
    if (!assumed.Credentials) {
        throw new Error(`Assumed ${environment.adminRoleArn} role did not return credentials`);
    }
    if (!assumed.Credentials.AccessKeyId) {
        throw new Error(`Assumed ${environment.adminRoleArn} role did not return an access key id`);
    }
    if (!assumed.Credentials.SecretAccessKey) {
        throw new Error(`Assumed ${environment.adminRoleArn} role did not return a secret access key`);
    }
    if (!assumed.Credentials.SessionToken) {
        throw new Error(`Assumed ${environment.adminRoleArn} role did not return a session token`);
    }
    return {
        accessKeyId: assumed.Credentials.AccessKeyId,
        secretAccessKey: assumed.Credentials.SecretAccessKey,
        sessionToken: assumed.Credentials.SessionToken,
    };
}
function success(body) {
    return { statusCode: 200, body: JSON.stringify(body) };
}
function failure(e) {
    const statusCode = e instanceof ProxyError ? e.statusCode : 500;
    return { statusCode: statusCode, body: JSON.stringify({ message: e.message }) };
}
//# sourceMappingURL=data:application/json;base64,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