# Windows UTF-8 编码设置指南

## 问题描述

在Windows系统上运行Python脚本时，控制台默认使用GBK编码（代码页936），无法正确显示UTF-8字符，导致以下问题：
- Unicode字符显示为乱码（如 ✓ → ???）
- 错误信息显示异常
- 测试结果难以阅读

---

## ✅ 推荐解决方案

### 方案1：使用批处理脚本（最简单）

```bash
# 运行验证脚本
run_with_utf8.bat verify_stage5.py

# 运行所有测试
run_with_utf8.bat -m pytest tests/ -v
```

### 方案2：PowerShell（推荐开发环境）

**临时设置**：
```powershell
chcp 65001
uv run python verify_stage5.py
```

**永久设置**：
```powershell
# 1. 创建/编辑PowerShell配置文件
if (!(Test-Path -Path $PROFILE)) {
    New-Item -ItemType File -Path $PROFILE -Force
}
notepad $PROFILE

# 2. 在配置文件中添加以下内容：
[Console]::OutputEncoding = [System.Text.Encoding]::UTF8
$PSDefaultParameterValues['Out-File:Encoding'] = 'utf8'
Set-PSReadLineOption -EditMode Vi

# 3. 保存并重新打开PowerShell
```

### 方案3：修改代码（适用于所有脚本）

在每个Python脚本开头添加：

```python
#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys

# 修复Windows控制台编码问题
if sys.platform == 'win32':
    import ctypes
    try:
        ctypes.windll.kernel32.SetConsoleOutputCP(65001)
        ctypes.windll.kernel32.SetConsoleCP(65001)
    except Exception:
        pass

    if sys.stdout.encoding != 'utf-8':
        try:
            sys.stdout.reconfigure(encoding='utf-8')
            sys.stderr.reconfigure(encoding='utf-8')
        except Exception:
            pass
```

---

## 🔧 高级配置

### 系统级设置（需要管理员权限）

**注册表设置**：
```powershell
# 打开注册表编辑器
regedit

# 导航到：
HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\Nls\CodePage

# 修改 "OEMCP" 值为 "65001"
# 注意：这会影响整个系统，建议谨慎操作
```

### Windows Terminal设置

如果你使用Windows Terminal（新版Terminal），在设置中配置：

```json
{
    "profiles": {
        "defaults": {
            "fontFace": "Cascadia Mono",
            "fontSize": 14
        },
        "list": [
            {
                "guid": "{0caa0dad-35be-5f56-a8ff-afceeeaa6101}",
                "name": "命令提示符",
                "commandline": "cmd.exe /k chcp 65001",
                "hidden": false
            }
        ]
    }
}
```

### ConEmu设置

如果你使用ConEmu：
1. 打开设置 (Win + Alt + P)
2. 转到 Startup → Environment
3. 添加：`set PYTHONIOENCODING=utf-8`
4. 保存并重启

---

## 📝 IDE配置

### VS Code

在 `settings.json` 中添加：

```json
{
    "files.encoding": "utf8",
    "files.autoGuessEncoding": true,
    "terminal.integrated.defaultProfile.windows": "Command Prompt",
    "terminal.integrated.profiles.windows": {
        "Command Prompt": {
            "path": "cmd.exe",
            "args": ["/c", "chcp 65001"]
        }
    }
}
```

### PyCharm

1. 打开 Settings/Preferences
2. 转到 Editor → File Encodings
3. 设置：
   - Global Encoding: UTF-8
   - Project Encoding: UTF-8
   - Default encoding for properties files: UTF-8
4. 转到 Build, Execution, Deployment → Console
5. 设置：
   - Console encoding: UTF-8

---

## 🚀 快速开始

### 为当前项目设置UTF-8编码

**步骤1：运行批处理脚本**
```bash
run_with_utf8.bat verify_stage5.py
```

**步骤2：检查编码是否正确**
脚本会显示：
```
[INFO] 当前编码信息:
  - 标准输出编码: utf-8
  - 标准错误编码: utf-8
  - 文件系统编码: utf-8
  - 默认编码: utf-8
```

**步骤3：验证Unicode字符显示**
应该能看到正确的Unicode字符，如：
```
✓ 阶段5功能验证完成
[OK] 语义检索 (基于ChromaDB向量相似度)
[OK] 关键词检索 (基于SQLite FTS5)
```

---

## ⚠️ 常见问题

### Q1: 设置后仍然显示乱码？

**A**: 可能是字体不支持Unicode字符
**解决**: 更换为支持Unicode的字体，如：
- Consolas
- Cascadia Mono
- Fira Code

### Q2: chcp 65001 不起作用？

**A**: 旧版Windows CMD对UTF-8支持有限
**解决**: 使用PowerShell或Windows Terminal

### Q3: Python脚本仍然报错？

**A**: 脚本中可能硬编码了GBK编码
**解决**: 确保脚本开头有UTF-8编码声明：
```python
#!/usr/bin/env python
# -*- coding: utf-8 -*-
```

---

## 📚 推荐配置组合

### 开发环境（推荐）

1. **终端**: Windows Terminal + PowerShell
2. **编辑器**: VS Code
3. **配置**: 使用 `run_with_utf8.bat` 启动脚本
4. **Python环境**: uv (已配置)

### 简洁配置

如果你只需要运行验证脚本：

```powershell
# 1. 设置编码
chcp 65001

# 2. 运行脚本
uv run python verify_stage5.py
```

---

## 📊 测试验证

运行编码测试脚本：
```python
python fix_encoding.py
```

应该看到：
```
[INFO] 标准输出已设置为UTF-8编码
[INFO] 控制台代码页已设置为UTF-8 (65001)

[INFO] 当前编码信息:
  - 标准输出编码: utf-8
  - 标准错误编码: utf-8
  - 文件系统编码: utf-8
  - 默认编码: utf-8
```

---

## 🎯 总结

**最佳实践**：
1. 使用PowerShell或Windows Terminal
2. 设置 `chcp 65001`
3. 使用 `run_with_utf8.bat` 启动脚本
4. 确保Python脚本包含 `# -*- coding: utf-8 -*-` 声明

**推荐方案**: 使用 `run_with_utf8.bat` 脚本（已为项目创建）

---

**文档更新时间**: 2025-11-18
**适用范围**: Windows 10/11
