# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/04_inference.ipynb (unless otherwise specified).

__all__ = ['get_game_df', 'is_valid_starter', 'test_front_pad', 'game_df2tensor', 'load_drsa', 'make_altair_hist']

# Internal Cell
from .data import processing, pipeline
from .data.processing import add_pitcher_team, add_postouts, outs_per_inning, batters_faced, AL_teams
from .utils import load_model

import sqlite3
import pandas as pd
from pybaseball import statcast
import pickle
import torch
import torch.nn as nn
from drsa.model import DRSA
import altair as alt

# Cell

def get_game_df(db_path: str, year: int, game_pk: int):
    conn = sqlite3.connect(db_path)
    query = f"""select *
                from statcast_{year}
                where game_pk = {game_pk}"""

    # making sure year is in db
    cursor = conn.execute(f"select name from sqlite_master where type='table' and name='statcast_{year}'")
    if cursor.fetchone():
        df = pd.read_sql_query(query, conn)
    else:
        df = pd.DataFrame()
    conn.close()
    return df

# Cell

def is_valid_starter(db_path: str, year: int, game_pk: int, pitcher: int) -> bool:
    """
    df is the statcast data for a unique game_pk
    """
    # getting initial game_df
    game_df = get_game_df(db_path=db_path, year=year, game_pk=game_pk)

    # getting sorted (by at bat) df for a specific game
    game_df = game_df.sort_values("at_bat_number", ascending=True)

    # first pitcher for each team is throwing at min(at_bat_number)
    home_pitcher_first_ab = game_df.loc[(game_df["inning_topbot"]=="Top"), "at_bat_number"].min()
    home_team = game_df["home_team"].head(1).item()
    home_pitcher = game_df.loc[(game_df["at_bat_number"]==home_pitcher_first_ab), "pitcher"].head(1).item()

    away_pitcher_first_ab = game_df.loc[(game_df["inning_topbot"]=="Bot", "at_bat_number")].min()
    away_team = game_df["away_team"].head(1).item()
    away_pitcher = game_df.loc[(game_df["at_bat_number"]==away_pitcher_first_ab), "pitcher"].head(1).item()

    if pitcher not in (home_pitcher, away_pitcher):
        print(f"Pitcher {pitcher_id} was not a starter in game {game_pk}")

    # adding pitcher_team
    game_df.loc[:, "pitcher_team"] = game_df.apply(lambda row: add_pitcher_team(row), axis=1)

    # identifying the team of the pitcher of interest
    team = home_team if pitcher == home_pitcher else away_team

    # check for "openers"

    # adding postouts for entire game
    game_team_df = game_df.loc[(game_df["pitcher_team"]==team)]
    game_team_df = add_postouts(game_team_df)

    # subsetting to get pitches thrown by the starter
    game_team_pitcher_df = game_team_df.loc[(game_team_df["pitcher"]==pitcher)]

    # getting criteria to check if opener
    outs = game_team_pitcher_df.groupby(["inning"]).agg({"postouts": outs_per_inning}).sum().item()
    n_batters = batters_faced(game_team_pitcher_df["at_bat_number"])
    opener = outs < 7 or n_batters < 10

    # must not be opener, be from an AL team, and be playing in an AL stadium
    if not opener and (team in AL_teams) and (home_team in AL_teams):
        return True
    return False

# Cell

def test_front_pad(feature_engineered_df: pd.DataFrame) -> (torch.Tensor, int):
    X = torch.tensor(feature_engineered_df.values, dtype=torch.double).unsqueeze(0)
    batch_size, seq_len, n_features = X.shape
    diff = 36-seq_len
    X_padded = torch.cat([torch.zeros(1, diff, n_features), X.float()], dim=1)
    return X_padded, diff

# Cell

def game_df2tensor(game_df: pd.DataFrame, game_pk: int, pitcher: int, mappers, scaler) -> torch.tensor:
    # full data cleaning pipeline
    cleaned_df = processing.preliminary_clean(game_df, game_pk, pitcher)
    agged_df = processing.aggregate_at_bats(cleaned_df, pipeline.at_bat_aggs)
    feature_engineered_df = processing.feature_engineering(agged_df)
    feature_engineered_df = feature_engineered_df[pipeline.cols]

    # scaling new data (as tuned on training data)
    feature_engineered_df[pipeline.feature_cols] = scaler.transform(feature_engineered_df[pipeline.feature_cols])

    # replacing pitcher id with index of associated embedding
    feature_engineered_df["pitcher"] = [mappers["pitcher"].get(pitcher, 0) for pitcher in feature_engineered_df["pitcher"]]

    # subsetting to only get required cols
    feature_engineered_df = feature_engineered_df[["pitcher"] + pipeline.feature_cols]

    # padding as appropriate
    X_padded, diff = test_front_pad(feature_engineered_df)

    return X_padded, diff

# Cell

def load_drsa():
    # model hyperparameters
    n_pitchers = 270
    pitcher_emb_size = 20
    pitcher_embeddings = nn.Embedding(n_pitchers, pitcher_emb_size)
    ptp_embeddings = [pitcher_embeddings]

    # initializing model
    drsa = DRSA(n_features=17,
                hidden_dim=20,
                n_layers=5,
                embeddings=ptp_embeddings,
                output_size=1,
                LSTM_dropout=0.05,
                Linear_dropout=0.1)

    # loading up saved model
    load_model(drsa, f"../models/07-22-20_DRSA_2016_2017_2018_2019_loss_4.0985.pth")

    return drsa

# Cell

def make_altair_hist(preds: torch.tensor, diff: int):
    # preparing predictions
    preds_df = pd.DataFrame((preds[:, diff:, :]).squeeze().detach()).reset_index()
    preds_df.columns = ["at_bat_num", "pred"]
    preds_df["at_bat_num"] = preds_df["at_bat_num"] + 1
    preds_df["prediction"] = (preds_df["pred"] > 0.35).replace(True, "pulled").replace(False, "stayed")

    # making chart

    chart = alt.Chart(preds_df).mark_bar().encode(
                x='pred:Q',
                y="at_bat_num:O",
                tooltip=[alt.Tooltip('pred:Q', format='.4%'),
                         alt.Tooltip('at_bat_num:O')],
                color=alt.Color('prediction', scale=alt.Scale(domain=["stayed", "pulled"], range=["lightgrey", "lightblue"])),
            ).interactive()
    return chart