"""CLI commands for setup and configuration."""

import argparse
import os
import sys
from typing import Optional


def detect_framework_and_app() -> Optional[tuple[str, str, str]]:
    """
    Detect framework and app location.
    Returns (framework, file_path, app_variable) or None.
    """
    import glob

    # Search for app files at any depth in current directory
    app_patterns = ["app.py", "main.py", "server.py", "wsgi.py", "asgi.py"]
    common_vars = ["app", "application", "main", "server"]

    # Find all matching files recursively
    found_files = []
    for pattern in app_patterns:
        found_files.extend(glob.glob(f"**/{pattern}", recursive=True))

    # Sort by depth (shallowest first) and then by filename priority
    found_files.sort(key=lambda x: (x.count(os.sep), app_patterns.index(os.path.basename(x))))

    for file_path in found_files:
        try:
            with open(file_path, "r") as f:
                content = f.read()

            if "from fastapi import" in content or "import fastapi" in content:
                framework = "FastAPI"
            elif "from flask import" in content or "import flask" in content:
                framework = "Flask"
            else:
                continue

            for var_name in common_vars:
                if f"{var_name} = " in content:
                    return framework, file_path, var_name

        except Exception:
            continue

    return None


def generate_conftest_content(framework: str, file_path: str, app_variable: str) -> str:
    """Generate conftest.py content based on detected framework."""
    # Convert file path to import path (e.g., "src/main.py" -> "src.main")
    module_path = file_path.replace("/", ".").replace("\\", ".").replace(".py", "")

    return f'''"""conftest.py - Auto-generated by pytest-api-cov init"""

import pytest

# Import your {framework} app from anywhere in your project
from {module_path} import {app_variable}


@pytest.fixture
def app():
    """Provide the {framework} app for API coverage testing.

    You can import from any location - just change the import path above
    to match your project structure.
    """
    return {app_variable}


# The plugin will automatically create a 'coverage_client' fixture that uses your 'app' fixture
# You can use either:
# - def test_endpoint(app): ...  # Direct app access
# - def test_endpoint(coverage_client): ...  # Test client with API coverage tracking
#
# To wrap an existing custom fixture instead, specify the fixture name in pyproject.toml:
# [tool.pytest_api_cov]
# client_fixture_name = "my_custom_client"
#
# Example custom fixture:
# @pytest.fixture
# def my_custom_client(app):
#     client = app.test_client()  # Flask
#     # or client = TestClient(app)  # FastAPI
#     # Add custom setup here (auth headers, etc.)
#     return client
'''


def generate_pyproject_config(framework: str) -> str:
    """Generate pyproject.toml configuration section."""
    return """
# pytest-api-cov configuration
[tool.pytest_api_cov]
# Fail if coverage is below this percentage (optional)
# fail_under = 80.0

# Show different endpoint types in report
show_uncovered_endpoints = true
show_covered_endpoints = false
show_excluded_endpoints = false

# Exclude endpoints matching these patterns (optional)
# exclusion_patterns = [
#     "/health",
#     "/metrics",
#     "/docs",
# ]

# Save JSON report to file (optional)
# report_path = "api_coverage.json"

# Force Unicode symbols in terminal output (optional)
# force_sugar = true

# Wrap an existing custom test client fixture with coverage tracking (optional)
# client_fixture_name = "my_custom_client"
"""


def cmd_init() -> int:
    """Initialize pytest-api-cov setup in current directory."""
    print("🚀 pytest-api-cov Setup Wizard")
    print("=" * 40)

    detection_result = detect_framework_and_app()

    if detection_result:
        framework, file_path, app_variable = detection_result
        print(f"✅ Detected {framework} app in {file_path} (variable: {app_variable})")

        conftest_exists = os.path.exists("conftest.py")
        if conftest_exists:
            print("⚠️  conftest.py already exists")
            create_conftest = input("Do you want to overwrite it? (y/N): ").lower().startswith("y")
        else:
            create_conftest = True

        if create_conftest:
            conftest_content = generate_conftest_content(framework, file_path, app_variable)
            with open("conftest.py", "w") as f:
                f.write(conftest_content)
            print("✅ Created conftest.py")

        pyproject_exists = os.path.exists("pyproject.toml")
        if pyproject_exists:
            print("ℹ️  pyproject.toml already exists")
            print("Add this configuration to your pyproject.toml:")
            print(generate_pyproject_config(framework))
        else:
            create_pyproject = input("Create pyproject.toml with pytest-api-cov config? (Y/n): ").lower()
            if not create_pyproject.startswith("n"):
                pyproject_content = f"""[project]
name = "your-project"
version = "0.1.0"

{generate_pyproject_config(framework)}

[tool.pytest.ini_options]
testpaths = ["tests"]
"""
                with open("pyproject.toml", "w") as f:
                    f.write(pyproject_content)
                print("✅ Created pyproject.toml")

        print()
        print("🎉 Setup complete!")
        print()
        print("Next steps:")
        print("1. Write your tests using the 'coverage_client' fixture")
        print("2. Run: pytest --api-cov-report")
        print()
        print("Example test:")
        print("""
def test_root_endpoint(coverage_client):
    response = coverage_client.get("/")
    assert response.status_code == 200
""")

    else:
        print("❌ No FastAPI or Flask app detected in common locations")
        print()
        print("Please ensure you have one of these files with a Flask/FastAPI app:")
        print("• app.py, main.py, server.py, wsgi.py, or asgi.py")
        print("• Files can be in the current directory or any subdirectory")
        print("• The file must contain a variable named 'app', 'application', 'main', or 'server'")
        print()
        print("Example app.py:")
        print("""
from fastapi import FastAPI

app = FastAPI()

@app.get("/")
def read_root():
    return {"message": "Hello World"}
""")
        return 1

    return 0


def main() -> int:
    """Main CLI entry point."""
    parser = argparse.ArgumentParser(prog="pytest-api-cov", description="pytest API coverage plugin CLI tools")

    subparsers = parser.add_subparsers(dest="command", help="Available commands")

    subparsers.add_parser("init", help="Initialize pytest-api-cov setup")

    args = parser.parse_args()

    if args.command == "init":
        return cmd_init()
    else:
        parser.print_help()
        return 1


if __name__ == "__main__":
    sys.exit(main())
