#
# Copyright 2021 3liz
# Author: David Marteau
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Original parts are Copyright 2016 OSGeo Foundation,            
# represented by PyWPS Project Steering Committee,               
# and released under MIT license.                                
# Please consult PYWPS_LICENCE.txt for details
#

from .schema import E, OWS, WPS, NAMESPACES, XMLElement
from ..ogc import OGCTYPE
from ..traits import register_trait

import lxml.etree as etree


@register_trait
class BoundingBoxOutput:

    def describe_xml(self) -> XMLElement:
        doc = E.Output(
            OWS.Identifier(self.identifier),
            OWS.Title(self.title)
        )

        if self.abstract:
            doc.append(OWS.Abstract(self.abstract))

        for m in self.metadata:
            doc.append(m.describe_xml())

        bbox_data_doc = E.BoundingBoxOutput()
        doc.append(bbox_data_doc)

        default_doc = E.Default()
        default_doc.append(E.CRS(self.crss[0]))

        supported_doc = E.Supported()
        for c in self.crss:
            supported_doc.append(E.CRS(c))

        bbox_data_doc.append(default_doc)
        bbox_data_doc.append(supported_doc)

        return doc

    def execute_xml(self, context) -> XMLElement:
        doc = E.Output(
            OWS.Identifier(self.identifier),
            OWS.Title(self.title)
        )

        if self.abstract:
            doc.append(OWS.Abstract(self.abstract))

        bbox_data_doc = OWS.BoundingBox()

        bbox_data_doc.attrib['crs'] = self.crs
        bbox_data_doc.attrib['dimensions'] = str(self.dimensions)

        bbox_data_doc.append(OWS.LowerCorner('{0[0]} {0[1]}'.format(self.data)))
        bbox_data_doc.append(OWS.UpperCorner('{0[2]} {0[3]}'.format(self.data)))

        doc.append(bbox_data_doc)

        return doc


@register_trait
class ComplexOutput:

    def describe_xml(self) -> XMLElement:
        """Generate DescribeProcess element
        """
        default_format_el = self.supported_formats[0].describe_xml()
        supported_format_elements = [f.describe_xml() for f in self.supported_formats]

        doc = E.Output(
            OWS.Identifier(self.identifier),
            OWS.Title(self.title)
        )

        if self.abstract:
            doc.append(OWS.Abstract(self.abstract))

        for m in self.metadata:
            doc.append(m.describe_xml())

        doc.append(
            E.ComplexOutput(
                E.Default(default_format_el),
                E.Supported(*supported_format_elements)
            )
        )

        return doc

    def execute_xml_lineage(self) -> XMLElement:
        doc = WPS.Output(
            OWS.Identifier(self.identifier),
            OWS.Title(self.title)
        )

        if self.abstract:
            doc.append(OWS.Abstract(self.abstract))

        return doc

    def execute_xml(self, context) -> XMLElement:
        """Render Execute response XML node

        :return: node
        :rtype: ElementMaker
        """

        self.identifier

        node = None
        if self.as_reference:
            node = self._execute_xml_reference(context)
        else:
            node = self._execute_xml_data()

        doc = WPS.Output(
            OWS.Identifier(self.identifier),
            OWS.Title(self.title)
        )
        if self.abstract:
            doc.append(OWS.Abstract(self.abstract))
        doc.append(node)

        return doc

    def _execute_xml_reference(self, context) -> XMLElement:
        """Return Reference node
        """
        if self.url is None:
            raise ValueError("Missing url")

        doc = WPS.Reference()

        doc.attrib['href'] = context.resolve_store_url(self.url, as_output=True)
        if self.data_format:
            if self.data_format.mime_type:
                doc.attrib['mimeType'] = self.data_format.mime_type
            if self.data_format.encoding:
                doc.attrib['encoding'] = self.data_format.encoding
            if self.data_format.schema:
                doc.attrib['schema'] = self.data_format.schema
        return doc

    def _execute_xml_data(self) -> XMLElement:
        """Return Data node
        """
        doc = WPS.Data()

        complex_doc = WPS.ComplexData()

        if self.data is not None:
            try:
                data_doc = etree.parse(self.file)
                complex_doc.append(data_doc.getroot())
            except Exception:
                if isinstance(self.data, str):
                    complex_doc.text = self.data
                else:
                    complex_doc.text = etree.CDATA(self.base64)

        if self.data_format:
            if self.data_format.mime_type:
                complex_doc.attrib['mimeType'] = self.data_format.mime_type
            if self.data_format.encoding:
                complex_doc.attrib['encoding'] = self.data_format.encoding
            if self.data_format.schema:
                complex_doc.attrib['schema'] = self.data_format.schema
        doc.append(complex_doc)
        return doc
                       

@register_trait
class LiteralOutput:

    def describe_xml(self) -> XMLElement:
        doc = E.Output(
            OWS.Identifier(self.identifier),
            OWS.Title(self.title)
        )

        if self.abstract:
            doc.append(OWS.Abstract(self.abstract))

        for m in self.metadata:
            doc.append(m.describe_xml())

        literal_data_doc = E.LiteralOutput()

        if self.data_type:
            data_type = OWS.DataType(self.data_type)
            data_type.attrib['{%s}reference' % NAMESPACES['ows']] = OGCTYPE[self.data_type]
            literal_data_doc.append(data_type)

        if self.supported_uoms:
            default_uom_element = self.uom.describe_xml()
            supported_uom_elements = [u.describe_xml() for u in self.supported_uoms]

            literal_data_doc.append(
                E.UOMs(
                    E.Default(default_uom_element),
                    E.Supported(*supported_uom_elements)
                )
            )

        doc.append(literal_data_doc)

        return doc

    def execute_xml_lineage(self) -> XMLElement:
        doc = WPS.Output(
            OWS.Identifier(self.identifier),
            OWS.Title(self.title)
        )

        if self.abstract:
            doc.append(OWS.Abstract(self.abstract))

        return doc

    def execute_xml(self, context) -> XMLElement:
        doc = WPS.Output(
            OWS.Identifier(self.identifier),
            OWS.Title(self.title)
        )

        if self.abstract:
            doc.append(OWS.Abstract(self.abstract))

        data_doc = WPS.Data()

        literal_data_doc = WPS.LiteralData(str(self.data))
        literal_data_doc.attrib['dataType'] = OGCTYPE[self.data_type]
        if self.uom:
            literal_data_doc.attrib['uom'] = self.uom.ref
        data_doc.append(literal_data_doc)

        doc.append(data_doc)

        return doc
