# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pgwarehouse', 'pgwarehouse.tests']

package_data = \
{'': ['*'],
 'pgwarehouse.tests': ['data/*', 'output/*', 'output/my_orders_data/archive/*']}

install_requires = \
['clickhouse-driver>=0.2.5,<0.3.0',
 'psycopg2>=2.9.5,<3.0.0',
 'pytest>=7.2.2,<8.0.0',
 'python-dotenv>=1.0.0,<2.0.0',
 'pyyaml>=6.0,<7.0',
 'snowflake-connector-python>=3.0.1,<4.0.0',
 'tabulate>=0.9.0,<0.10.0',
 'testing-postgresql>=1.3.0,<2.0.0']

entry_points = \
{'console_scripts': ['pgwarehouse = pgwarehouse.__main__:main']}

setup_kwargs = {
    'name': 'pgwarehouse',
    'version': '0.1.2',
    'description': 'Esaily sync your Postgres database to your data warehouse.',
    'long_description': '# pgwarehouse - quickly sync Postgres data to your cloud warehouse\n\n## Introduction\n\nPostgres is an amazing, general purpose OLTP database. But it\'s not designed for heavy analytic (OLAP) usage. Analytic queries are much better served by a columnar store database like Snowflake or Clickhouse.\n\nThis package allows you to easily sync data from a Postgres database into a local or cloud data warehouse (currently Snowflake or Clickhouse). You can perform a one-time sync operation, or run periodic incremental syncs to keep your warehouse up to date.\n\n## Features\n\n* High performance by using `COPY` to move lots of data efficiently. `pgwarehouse` can easily sync hundreds of millions of rows of data (tens of GB) per hour.\n* Supports multiple update strategies for immutable or mutable tables.\n* Easy to configure and run.\n\n## Installation\n\n    pip install pgwarehouse\n\nNow you need to configure credentials for your **Postgres** source and the warehouse destination.\n\nYou can place Postgres credentials either in your config file or in your environment. If using the environment you need to set these variables:\n\n    PGHOST\n    PGDATABASE\n    PGUSER\n    PGPASSWORD\n    PGSCHEMA (defaults to \'public\')\n\n## Creating a config file\n\nRun this command to create a template config file:\n\n    pgwarehouse init\n\nThis will create a local `pgwarehouse_conf.yaml` file. Now you can edit your Postgres credentials in the `postgres` stanza of the config file:\n\n    postgres:\n        pghost: (defaults to $PGHOST)\n        pgdatabase: (defaults to $PGDATABASE\n        pguser: (defaults to $PGUSER)\n        pgpassword: (defaults to $PGPASSWORD)\n        pgschema: (defaults to \'public\')\n\n## Specifying the warehouse credentials\n\nAgain you can use the environment or the config file. Set these sets of vars in your env:\n\n    CLICKHOUSE_HOST\n    CLICKHOUSE_DATABASE\n    CLICKHOUSE_USER\n    CLICKHOUSE_PWD\n\nor\n\n    SNOWSQL_ACCOUNT\n    SNOWSQL_DATABASE\n    SNOWSQL_SCHEMA\n    SNOWSQL_WAREHOUSE\n    SNOWSQL_USER\n    SNOWSQL_PWD\n\n(The Snowflake parameters are the same as those for the [SnowSQL](https://docs.snowflake.com/en/user-guide/snowsql-start)\nCLI tool. The `SNOWSQL_ACCOUNT` value should be your "account identifier".)\n\nor set these values in the `warehouse` stanza in the config file:\n\n    warehouse:\n        backend: (clickhouse|snowflake)\n        clickhouse_host: \n        clickhouse_database: \n        clickhouse_user:\n        clickhouse_password:\n        --or--\n        snowsql_account:\n        snowsql_database:\n        snowsql_schema:\n        snowsql_warehouse:\n        snowsql_user:\n        snowsql_pwd:\n\n# Usage\n\nOnce the credentials are configured you can start syncing data. Start by listing tables from the Postgres database:\n\n    pgwarehouse list\n\nAnd you can see which tables exist so far in the warehouse:\n\n    pgwarehouse listwh\n    \nNow use `sync` to sync a table (eg. the \'users\' table):\n\n    pgwarehouse sync users\n\nData will be downloaded from the Postgres database into CSV files on the local machine, and then those files will be uploaded to the warehouse. Running `pgwarehouse listwh` will show the new table.\n\n## Updating a table\n\nAfter the initial sync has run, you can update the warehouse table with new records by running `sync` again:\n\n    pgwarehouse sync users\n\nSee [update strategies](#table-update-strategies) for different ways to update your table on each sync.\n\n## Syncing multiple tables\n\nThere are two ways to manage multiple tables. The first is just to pass `all` in place of the table name:\n\n    pgwarehouse sync all\n\nThis will attempt to sync ALL tables from Postgres into the warehouse. This could take a while!\n\nThe other way is to specify the `tables` list in the config file:\n\n    tables:\n        - users\n        - charges\n        - logs\n\nNow when you specify `sync all` the tool will use the list of tables specified in the config file.\n\n**Pro tip!** You can add the `max_records` settings to your `postgres` configuration to limit the number\nof records copied per table. This can be useful for testing the initial sync in case you have some\nlarge tables. Set this value to something reasonable (like 10000) and then try syncrhonizing all\ntables to make sure they copy properly. Once you have verified the tables in the warehouse then you\ncan remove this setting, drop any large tables, and then copy them in full (just run `sync all` again).\n\n## Table update strategies\n\n#### New Records Only (default)\nThe default update strategy is "new records only". This is done by selecting records with a greater value\nfor their primary id column than the greatest value currently in the warehouse. This strategy is simple\nand quick, but only works for monotonically incrementing primary keys, and only finds new records.\n\n#### Reload each time\nAnother supported strategy is "reload each time". This is the simplest strategy and we simply reload the\nentire table every time we sync. This strategy should be fine for small-ish tables (like <10m rows).\n\n#### Last Modified\nFinally, if your table has a `last modified` column then you can use the "all modifications strategy".\nIn this case all records with a `last modified` timestamp greater than the maximum value found in the\nwarehouse will be selected and "upserted" into the warehouse. Records that are already present\n(via matching the primary key) will be updated, and new records will be inserted.\n\n* The Snowflake backend uses the [MERGE](https://docs.snowflake.com/en/sql-reference/sql/merge) operation. \n* The Clickhouse backend uses `ALTER TABLE .. DELETE` to remove matching records and then `INSERT` to insert the new values.\n\n### What about deletes?\n\nThere is no simple way to capture deletes - you have to reload the entire table. A common pattern is\nto apply new records on a daily basis, and reload the entire table every week to remove deleted records.\n\n### What if my table has no primary key?\n\nAll the update strategies except "reload each time" require your table to have a primary key column.\n\n## Specifying update strategy at the command line\n\n    pgwarehouse sync <table>   (defaults to NEW RECORDS)\n    pgwarehouse sync <table> last_modified=<last modified column>   (MODIFIED RECORDS)\n    pgwarehouse reload <table> (reloads the whole table)\n\n## Specifying update strategy in the config file\n\nYou can configure the update strategy selectively for each table in the config file. To do so,\nspecify the table as a nested dictionary with options:\n\n    tables:\n        - accounts\n        - users:\n            reload: true\n        - orders:\n            last_modified: updated_at\n        - shoppers\n            last_modified: update_time\n            reload: sun\n        - original_orders:\n            skip: true\n\nIn this example:\n\n* `accounts` will have new records only applied at each sync\n* `users` will be reloaded completely on each sync\n* `orders` will have modified records (found by the \'updated_at\' column) applied on each sync\n* `shoppers` will have modified records applied on each sync, except for any sync\nwhich happens on Sunday, in which case the entire table will be reloaded.\n* `original_orders` will be skipped entirely\n\nThe `reload` argument can take 3 forms:\n\n    reload: true    - reload the table every sync\n    reload: [sun,mon,tue,wed,thur,fri]  - reload if the sync occurs on this day of the week\n    reload: 1-31    - reload if the sync occurs on this numeric day of the month (don\'t use 31!)\n\n## Scheduling regular data syncs\n\n`pgwarehouse` does not including any scheduling itself, you will need an external trigger like\n`cron`, [Heroku Scheduler](https://devcenter.heroku.com/articles/scheduler), or a K8s\n[CronJob](https://kubernetes.io/docs/tasks/job/automated-tasks-with-cron-jobs/).\n\nWhen running, the tool will need access to local storage - potentially a lot if you are synchronizing\nbig tables. But nothing needs to persist between sync runs (except the config file) - the tool \nonly relies on state it can query from Postgres or the warehouse.\n\n## Troubleshooting\n\nSometimes when you are testing things out it can be helpful to do the sync in two phases:\n1)download the data, 2)upload the data. You can use `extract` and `load` for this:\n\n    pgwarehouse extract <table>     - only downloads data\n    pgwarehouse load <table>        - loads the data into the warehouse\n\nWhen the `extract` process runs, its stores data in `./pgw_data/<table name>_data`. As\nfiles are uploaded they are moved into an `archive` subdirectory. When the **next sync**\nruns then this archive directory will be cleaned up. This allows you to go examine\nthe CSV downloaded data in case the upload fails for some reason. \n\n## Limitations\n\nColumn type mapping today is [very limited](https://github.com/scottpersinger/pgwarehouse/blob/a20dc316bbdbc78317cfdd796216a847919d8755/pgwarehouse/snowflake_backend.py). More esoteric column types like JSON or ARRAY are simply\nmapped as VARCHAR columns. Some of these types are supported in the warehouse and could be\nimplemented more accurately.\n\nComposite primary keys (using multiple columns) have limited support. Today they will only work\nwith the RELOAD strategy.\n\nNon-numeric primary key types (like UUIDs) probably won\'t work unless they have a good lexigraphic\nsort that supports a `>` where clause.\n\n\n\n',
    'author': 'Scott Persinger',
    'author_email': 'scottpersinger@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
