# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['autosig']

package_data = \
{'': ['*']}

install_requires = \
['attrs>=19.2.0', 'toolz>=0.9.0']

setup_kwargs = {
    'name': 'autosig',
    'version': '0.10.0',
    'description': 'Define signatures to create beautiful APIs',
    'long_description': "Introduction to autosig\n=======================\n\n\n.. image:: https://img.shields.io/pypi/v/autosig.svg\n        :target: https://pypi.python.org/pypi/autosig\n\n.. image:: https://img.shields.io/travis/piccolbo/autosig.svg\n        :target: https://travis-ci.org/piccolbo/autosig\n\n.. image:: https://codecov.io/gh/piccolbo/autosig/branch/master/graph/badge.svg\n        :target: https://codecov.io/gh/piccolbo/autosig\n\n.. image:: https://readthedocs.org/projects/autosig/badge/?version=latest\n        :target: https://autosig.readthedocs.io/en/latest/?badge=latest\n        :alt: Documentation Status\n\n\n.. image:: https://pyup.io/repos/github/piccolbo/autosig/shield.svg\n     :target: https://pyup.io/repos/github/piccolbo/autosig/\n     :alt: Updates\n\n.. image:: https://api.codeclimate.com/v1/badges/233681cf64a66ee9c50e/maintainability\n     :target: https://codeclimate.com/github/piccolbo/autosig/maintainability\n     :alt: Maintainability\n\n\nGo straight to the `documentation <https://autosig.readthedocs.io/en/latest/>`_. Install with ``pip install autosig``. Python 3 only.\n\nMotivation\n----------\n\nWhen I look at a great API I always observe a great level of consistency: similarly named and ordered arguments at a syntactic level; similar defaults, range of allowable values etc. on the semantic side. When looking at the code, one doesn't see these regularities represented very explicitly.\n\nImagine we are starting to develop a library with three entry points, ``map``, ``reduce`` and ``filter``::\n\n  from collections import Iterable\n\n\n  def map(function, iterable):\n      assert callable(function)\n      assert isinstance(iterable, Iterable)\n      return (function(x) for x in iterable)\n\n\n  def reduce(function, iterable):\n      total = next(iterable)\n      for x in iterable:\n          total = function(total, x)\n      return total\n\n\n  def filter(iterable, fun):\n      if not isinstance(iterable, Iterable):\n          iterable = [iterable]\n      if isinstance(fun, set):\n          fun = lambda x: x in fun\n      return (x for x in iterable if fun(x))\n\n\n\nBut this is hardly well crafted. The order and naming of arguments isn't consistent. One function checks its argument right away. The next doesn't. The third attempts certain conversions to try and work with arguments that are not iterables or functions. There are reasons to build strict or tolerant APIs, but it's unlikely that mixing the two within the same API is a good idea, unless it's done deliberately (for instance offering a strict and tolerant version of every function). It wouldn't be difficult to fix these problems in this small API but we would end up with duplicated logic that we need to keep aligned for the foreseeable future. Let's do it instead the ``autosig`` way::\n\n  from autosig import param, Signature, autosig, check\n  from collections import Iterable\n\n\n  def to_callable(x):\n      return (lambda y: y in x) if isinstance(x, set) else x\n\n\n  def to_iterable(x):\n      return x if isinstance(x, Iterable) else [x]\n\n\n  API_signature = Signature(\n      function=param(converter=to_callable, validator=callable),\n      iterable=param(converter=to_iterable, validator=Iterable))\n\n\n  @autosig(API_signature)\n  def map(function, iterable):\n      return (function(x) for x in iterable)\n\n\n  @autosig(API_signature)\n  def reduce(function, iterable):\n      total = next(iterable)\n      for x in iterable:\n          total = function(total, x)\n      return total\n\n\n  @autosig(API_signature)\n  def filter(function, iterable):\n      return (x for x in iterable if function(x))\n\n\nLet's go through it step by step. First we defined 2 simple conversion\nfunctions. This is a good first step independent of ``autosig``. Next we create\na signature object, with two parameters. These are intialized with objects that\ndefine the checking and conversion that need to be performed on those\nparameters, independent of which function is going to use that signature.\nA type works as a validator, as does any callable that returns `True` when a value is valid, returns `False` or raises an exception otherwise. Finally, we repeat\nthe definition of our three API function, attaching the signature just defined\nwith a decorator and then skipping all the checking and conversion logic and\ngoing straight to the meat of the function!\n\nAt the cost of a little code we have gained a lot:\n\n* Explicit definition of the desired API signature, in a single place --- DRY principle;\n* association of that signature with API functions, checked at load time --- no room for error;\n* uniform application of conversion and validation logic without repeating it;\n\n``autosig`` is the pro tool for the API designer! If you want to take a look at a real package that uses ``autosig``, check out `altair_recipes <https://github.com/piccolbo/altair_recipes>`_.\n\n\nFeatures\n--------\n\n* Define reusable parameters with defaults, conversion and validation logic, documentation, preferred position in the signature and whether keyword-only.\n* Define reusable return values with conversion and validation logic and documentation.\n* Define reusable signatures as ordered maps from names to parameters with optional return value definition.\n* Combine signatures to create complex ones on top of simple ones.\n* Decorate functions and methods with their signatures. Enforced at load time. Conversion and validation logic executed at call time.\n* Not hot about signatures? You can just use parameters as in::\n\n          @autosig\n          def reduce(function = param(...), iterable=param(...)):\n\n  for more free-form APIs.\n* Open source (BSD license)\n* Extensive property-based testing, excellent coverage\n\n\n\nCredits\n-------\n\nThis package is heavily based on `attrs <https://github.com/python-attrs/attrs>`_. While that may change in the future, for now it must be said this is a thin layer over that, with a bit of reflection sprinkled over. It is, I suppose, a quite original direction to take ``attrs`` into.\n",
    'author': 'Antonio Piccolboni',
    'author_email': 'antonio@piccolboni.info',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/piccolbo/autosig',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.5,<4.0',
}


setup(**setup_kwargs)
