"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.cantHaveMoreThan200Resources = exports.cantEditLSIKeySchema = exports.cantMutateMultipleGSIAtUpdateTime = exports.cantAddAndRemoveGSIAtSameTime = exports.cantEditGSIKeySchema = exports.cantAddLSILater = exports.cantEditKeySchema = exports.check = void 0;
const fs = require('fs-extra');
const path_1 = require("path");
const deep_diff_1 = require("deep-diff");
const fileUtils_1 = require("./fileUtils");
const errors_1 = require("../errors");
const __1 = require("..");
async function check(currentCloudBackendDir, buildDirectory, rootStackName = 'cloudformation-template.json') {
    const cloudBackendDirectoryExists = await fs.exists(currentCloudBackendDir);
    const buildDirectoryExists = await fs.exists(buildDirectory);
    const diffRules = [
        cantEditKeySchema,
        cantAddLSILater,
        cantEditGSIKeySchema,
        cantEditLSIKeySchema,
        cantAddAndRemoveGSIAtSameTime,
    ];
    const projectRules = [cantHaveMoreThan200Resources, cantMutateMultipleGSIAtUpdateTime];
    if (cloudBackendDirectoryExists && buildDirectoryExists) {
        const current = await loadDiffableProject(currentCloudBackendDir, rootStackName);
        const next = await loadDiffableProject(buildDirectory, rootStackName);
        const diffs = deep_diff_1.diff(current, next);
        if (diffs) {
            for (const diff of diffs) {
                for (const rule of diffRules) {
                    rule(diff, current, next);
                }
            }
            for (const projectRule of projectRules) {
                projectRule(diffs, current, next);
            }
        }
    }
}
exports.check = check;
function cantEditKeySchema(diff) {
    if (diff.kind === 'E' && diff.path.length === 8 && diff.path[5] === 'KeySchema') {
        const stackName = path_1.basename(diff.path[1], '.json');
        const tableName = diff.path[3];
        throw new errors_1.InvalidMigrationError(`Attempting to edit the key schema of the ${tableName} table in the ${stackName} stack. `, 'Adding a primary @key directive to an existing @model. ', 'Remove the @key directive or provide a name e.g @key(name: "ByStatus", fields: ["status"]).');
    }
}
exports.cantEditKeySchema = cantEditKeySchema;
function cantAddLSILater(diff) {
    if ((diff.kind === 'N' && diff.path.length === 6 && diff.path[5] === 'LocalSecondaryIndexes') ||
        (diff.kind === 'A' && diff.path.length === 6 && diff.path[5] === 'LocalSecondaryIndexes' && diff.item.kind === 'N')) {
        const stackName = path_1.basename(diff.path[1], '.json');
        const tableName = diff.path[3];
        throw new errors_1.InvalidMigrationError(`Attempting to add a local secondary index to the ${tableName} table in the ${stackName} stack. ` +
            'Local secondary indexes must be created when the table is created.', "Adding a @key directive where the first field in 'fields' is the same as the first field in the 'fields' of the primary @key.", "Change the first field in 'fields' such that a global secondary index is created or delete and recreate the model.");
    }
}
exports.cantAddLSILater = cantAddLSILater;
function cantEditGSIKeySchema(diff, currentBuild, nextBuild) {
    function throwError(indexName, stackName, tableName) {
        throw new errors_1.InvalidMigrationError(`Attempting to edit the global secondary index ${indexName} on the ${tableName} table in the ${stackName} stack. `, 'The key schema of a global secondary index cannot be changed after being deployed.', 'If using @key, first add a new @key, run `amplify push`, ' +
            'and then remove the old @key. If using @connection, first remove the @connection, run `amplify push`, ' +
            'and then add the new @connection with the new configuration.');
    }
    if ((diff.kind === 'E' && diff.path.length === 10 && diff.path[5] === 'GlobalSecondaryIndexes' && diff.path[7] === 'KeySchema') ||
        (diff.kind === 'A' && diff.path.length === 8 && diff.path[5] === 'GlobalSecondaryIndexes' && diff.path[7] === 'KeySchema')) {
        const pathToGSIs = diff.path.slice(0, 6);
        const oldIndexes = get(currentBuild, pathToGSIs);
        const newIndexes = get(nextBuild, pathToGSIs);
        const oldIndexesDiffable = keyBy(oldIndexes, 'IndexName');
        const newIndexesDiffable = keyBy(newIndexes, 'IndexName');
        const innerDiffs = deep_diff_1.diff(oldIndexesDiffable, newIndexesDiffable) || [];
        for (const innerDiff of innerDiffs) {
            if (innerDiff.kind === 'E' && innerDiff.path.length > 2 && innerDiff.path[1] === 'KeySchema') {
                const indexName = innerDiff.path[0];
                const stackName = path_1.basename(diff.path[1], '.json');
                const tableName = diff.path[3];
                throwError(indexName, stackName, tableName);
            }
            else if (innerDiff.kind === 'A' && innerDiff.path.length === 2 && innerDiff.path[1] === 'KeySchema') {
                const indexName = innerDiff.path[0];
                const stackName = path_1.basename(diff.path[1], '.json');
                const tableName = diff.path[3];
                throwError(indexName, stackName, tableName);
            }
        }
    }
}
exports.cantEditGSIKeySchema = cantEditGSIKeySchema;
function cantAddAndRemoveGSIAtSameTime(diff, currentBuild, nextBuild) {
    function throwError(stackName, tableName) {
        throw new errors_1.InvalidMigrationError(`Attempting to add and remove a global secondary index at the same time on the ${tableName} table in the ${stackName} stack. `, 'You may only change one global secondary index in a single CloudFormation stack update. ', 'If using @key, change one @key at a time. ' +
            'If using @connection, add the new @connection, run `amplify push`, ' +
            'and then remove the new @connection with the new configuration.');
    }
    if (diff.kind === 'E' &&
        diff.path.length > 6 &&
        diff.path[5] === 'GlobalSecondaryIndexes') {
        const pathToGSIs = diff.path.slice(0, 6);
        const oldIndexes = get(currentBuild, pathToGSIs);
        const newIndexes = get(nextBuild, pathToGSIs);
        const oldIndexesDiffable = keyBy(oldIndexes, 'IndexName');
        const newIndexesDiffable = keyBy(newIndexes, 'IndexName');
        const innerDiffs = deep_diff_1.diff(oldIndexesDiffable, newIndexesDiffable) || [];
        let sawDelete = false;
        let sawNew = false;
        for (const diff of innerDiffs) {
            if (diff.path.length === 1 && diff.kind === 'D') {
                sawDelete = true;
            }
            if (diff.path.length === 1 && diff.kind === 'N') {
                sawNew = true;
            }
        }
        if (sawDelete && sawNew) {
            const stackName = path_1.basename(diff.path[1], '.json');
            const tableName = diff.path[3];
            throwError(stackName, tableName);
        }
    }
}
exports.cantAddAndRemoveGSIAtSameTime = cantAddAndRemoveGSIAtSameTime;
function cantMutateMultipleGSIAtUpdateTime(diffs, currentBuild, nextBuild) {
    function throwError(stackName, tableName) {
        throw new errors_1.InvalidMigrationError(`Attempting to mutate more than 1 global secondary index at the same time on the ${tableName} table in the ${stackName} stack. `, 'You may only mutate one global secondary index in a single CloudFormation stack update. ', 'If using @key, include one @key at a time. ' +
            'If using @connection, just add one new @connection which is using @key, run `amplify push`, ');
    }
    if (diffs) {
        let gsiCount = 0;
        for (const diff of diffs) {
            if (diff.kind === 'A' &&
                diff.path.length >= 6 &&
                diff.path[5] === 'GlobalSecondaryIndexes') {
                if (diff.item.kind === 'N' || diff.item.kind === 'D') {
                    gsiCount += 1;
                }
                if (gsiCount > 1) {
                    const stackName = path_1.basename(diff.path[1], '.json');
                    const tableName = diff.path[3];
                    throwError(stackName, tableName);
                }
            }
        }
    }
}
exports.cantMutateMultipleGSIAtUpdateTime = cantMutateMultipleGSIAtUpdateTime;
function cantEditLSIKeySchema(diff, currentBuild, nextBuild) {
    if (diff.kind === 'E' &&
        diff.path.length === 10 &&
        diff.path[5] === 'LocalSecondaryIndexes' &&
        diff.path[7] === 'KeySchema') {
        const pathToGSIs = diff.path.slice(0, 6);
        const oldIndexes = get(currentBuild, pathToGSIs);
        const newIndexes = get(nextBuild, pathToGSIs);
        const oldIndexesDiffable = keyBy(oldIndexes, 'IndexName');
        const newIndexesDiffable = keyBy(newIndexes, 'IndexName');
        const innerDiffs = deep_diff_1.diff(oldIndexesDiffable, newIndexesDiffable) || [];
        for (const innerDiff of innerDiffs) {
            if (innerDiff.kind === 'E' && innerDiff.path.length > 2 && innerDiff.path[1] === 'KeySchema') {
                const indexName = innerDiff.path[0];
                const stackName = path_1.basename(diff.path[1], '.json');
                const tableName = diff.path[3];
                throw new errors_1.InvalidMigrationError(`Attempting to edit the local secondary index ${indexName} on the ${tableName} table in the ${stackName} stack. `, 'The key schema of a local secondary index cannot be changed after being deployed.', 'When enabling new access patterns you should: 1. Add a new @key 2. run amplify push ' +
                    '3. Verify the new access pattern and remove the old @key.');
            }
        }
    }
}
exports.cantEditLSIKeySchema = cantEditLSIKeySchema;
function cantHaveMoreThan200Resources(diffs, currentBuild, nextBuild) {
    const stackKeys = Object.keys(nextBuild.stacks);
    for (const stackName of stackKeys) {
        const stack = nextBuild.stacks[stackName];
        if (stack && stack.Resources && Object.keys(stack.Resources).length > 200) {
            throw new errors_1.InvalidMigrationError(`The ${stackName} stack defines more than 200 resources.`, 'CloudFormation templates may contain at most 200 resources.', 'If the stack is a custom stack, break the stack up into multiple files in stacks/. ' +
                'If the stack was generated, you have hit a limit and can use the StackMapping argument in ' +
                `${__1.TRANSFORM_CONFIG_FILE_NAME} to fine tune how resources are assigned to stacks.`);
        }
    }
}
exports.cantHaveMoreThan200Resources = cantHaveMoreThan200Resources;
function keyBy(objects, attr) {
    return objects.reduce((acc, obj) => ({
        ...acc,
        [obj[attr]]: obj,
    }), {});
}
async function loadDiffableProject(path, rootStackName) {
    const project = await fileUtils_1.readFromPath(path);
    const currentStacks = project.stacks || {};
    const diffableProject = {
        stacks: {},
        root: {},
    };
    for (const key of Object.keys(currentStacks)) {
        diffableProject.stacks[key] = JSON.parse(project.stacks[key]);
    }
    diffableProject.root = JSON.parse(project[rootStackName]);
    return diffableProject;
}
function get(obj, path) {
    let tmp = obj;
    for (const part of path) {
        tmp = tmp[part];
        if (!tmp) {
            return undefined;
        }
    }
    return tmp;
}
//# sourceMappingURL=sanity-check.js.map