import os
import re
from setuptools import setup, find_packages, find_namespace_packages
from setuptools.command.build_py import build_py as _build_py
import subprocess

def print_tree(startpath):
    for root, dirs, files in os.walk(startpath):
        level = root.replace(startpath, '').count(os.sep)
        indent = ' ' * 4 * (level)
        print(f"{indent}{os.path.basename(root)}/")
        subindent = ' ' * 4 * (level + 1)
        for f in files:
            print(f"{subindent}{f}")

def modify_proto_imports(file_path):
    """ modify absolute imports in generated protobuf files to relative imports """
    patterns = [
        # match import xxx_pb2 or import xxx_pb2 as alias
        (r'^import (\w+_pb2)( as \w+)?$', r'from . import \1\2'),
    ]
    
    with open(file_path, 'r', encoding='utf-8') as f:
        content = f.read()
    
    modified = False
    for pattern, replacement in patterns:
        new_content, count = re.subn(
            pattern, replacement, content,
            flags=re.MULTILINE
        )
        if count > 0:
            content = new_content
            modified = True
    
    if modified:
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write(content)

class build_py(_build_py):
    def run(self):
        # generate proto files to out_dir
        project_dir = os.path.abspath(os.path.dirname(__file__))
        proto_dir = os.path.join(project_dir, 'proto-public-api')
        out_dir = os.path.join(project_dir, 'hex_device', 'generated')

        # ensure out_dir exists
        os.makedirs(out_dir, exist_ok=True)

        # Check if proto files exist and compile them
        if os.path.exists(proto_dir) and os.path.isdir(proto_dir):
            try:
                # compile proto files
                proto_files = [f for f in os.listdir(proto_dir) if f.endswith('.proto')]
                if proto_files:
                    print(f"Found {len(proto_files)} proto files, compiling...")
                    for proto_file in proto_files:
                        proto_path = os.path.join(proto_dir, proto_file)
                        print(f"Compiling {proto_file}...")
                        subprocess.check_call([
                            'protoc',
                            '--python_out', out_dir,
                            '--proto_path', proto_dir,
                            proto_path
                        ])
                else:
                    print("Warning: No .proto files found in proto-public-api directory")
            except subprocess.CalledProcessError as e:
                print(f"Warning: Failed to compile proto files: {e}")
                print("This may be expected if protoc is not available or proto files are missing")
            except Exception as e:
                print(f"Warning: Error processing proto files: {e}")
        else:
            print("Warning: proto-public-api directory not found. This is expected when installing from PyPI.")
            print("Using pre-compiled proto files from hex_device/generated/")

        # generate __init__.py if it doesn't exist
        init_path = os.path.join(out_dir, '__init__.py')
        if not os.path.exists(init_path):
            with open(init_path, 'w') as f:
                f.write('# Auto-generated by setup.py\n')

        # Modify proto imports in generated files
        for root, _, files in os.walk(out_dir):
            for file in files:
                if file.endswith('_pb2.py'):
                    modify_proto_imports(os.path.join(root, file))

        # Ensure generated package is included
        generated_pkg = 'hex_device.generated'
        if generated_pkg not in self.packages:
            self.packages.append(generated_pkg)

        super().run()

setup(
    cmdclass={"build_py": build_py},
    include_package_data=True,
)