# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dictdatabase']

package_data = \
{'': ['*']}

install_requires = \
['orjson>=3.8.0,<4.0.0']

setup_kwargs = {
    'name': 'dictdatabase',
    'version': '2.1.0',
    'description': 'Easy-to-use database using dicts',
    'long_description': '![Logo](https://github.com/mkrd/DictDataBase/blob/master/assets/logo.png?raw=true)\n\n[![Downloads](https://pepy.tech/badge/dictdatabase)](https://pepy.tech/project/dictdatabase)\n[![Downloads](https://pepy.tech/badge/dictdatabase/month)](https://pepy.tech/project/dictdatabase)\n[![Downloads](https://pepy.tech/badge/dictdatabase/week)](https://pepy.tech/project/dictdatabase)\n![Tests](https://github.com/mkrd/DictDataBase/actions/workflows/test.yml/badge.svg)\n![Coverage](https://github.com/mkrd/DictDataBase/blob/master/assets/coverage.svg?raw=1)\n\nDictDataBase is a simple and fast database for handling json or compressed json as the underlying storage mechanism. Features:\n- **Multi threading and multi processing safe**. Multiple processes on the same machine can simultaneously read and write to dicts without losing data.\n- **ACID** compliant. Unlike TinyDB, it is suited for concurrent environments.\n- **No database server** required. Simply import DictDataBase in your project and use it.\n- **Compression**. Configure if the files should be stored as raw json or as json compressed with zlib.\n- **Fast**. A dict can be accessed partially without having to parse the entire file, making the read and writes very efficient.\n- **Tested** with over 400 test cases.\n\n### Why use DictDataBase\n- For example, have a webserver dispatch database reads and writes concurrently.\n- If spinning up a database server is an overkill for your application.\n    - But you need [ACID](https://en.wikipedia.org/wiki/ACID) guarantees.\n- You have a big database but only want to access single key-value pairs repeatedly. DictDataBase can do this efficiently and quickly.\n- Your use case is suited for working with json data, or you have to work with a lot of json data.\n\n### Why not DictDataBase\n- If your storage is slow.\n- If a relational database is better suited for your use case.\n\nInstall\n================================================================================\n\n```sh\npip install dictdatabase\n```\n\nConfiguration\n================================================================================\nThere are 5 configuration options:\n\n### Storage directory\nSet storage_directory to the path of the directory that will contain your json files:\n```python\nDDB.config.storage_directory = "./ddb_storage" # Default value\n```\n\n### Compression\nIf you want to use compressed files, set use_compression to `True`.\nThis will make the db files significantly smaller and might improve performance if your disk is slow.\nHowever, the files will not be human readable.\n```python\nDDB.config.use_compression = False # Default value\n```\n\n### Indentation\nSet the way how written json files should be indented. Behaves exactly like `json.dumps(indent=...)`.\nIt can be an `int` for the number of spaces, the tab character, or `None` if you don\'t want the files to be indented.\n```python\nDDB.config.indent = "\\t" # Default value\n```\n\n### Sort keys\nSpecify if you want the dict keys to be sorted when writing to a file. Behaves exactly like `json.dumps(sort_keys=...)`.\n```python\nDDB.config.sort_keys = True # Default value\n```\n\n### Use orjson\nYou can use the orjson encoder and decoder if you need to.\nThe standard library json module is sufficient most of the time.\nHowever, orjson is a lot more performant in virtually all cases.\n```python\nDDB.config.use_orjson = True # Default value\n```\n\nUsage\n================================================================================\n\nImport\n--------------------------------------------------------------------------------\n\n```python\nimport dictdatabase as DDB\n```\n\nCreate a file\n--------------------------------------------------------------------------------\nThis library is called DictDataBase, but you can actually use any json serializable object.\n```python\nuser_data_dict = {\n    "users": {\n        "Ben": { "age": 30, "job": "Software Engineer" },\n        "Sue": { "age": 21, "job": "Architect" },\n        "Joe": { "age": 50, "job": "Manager" }\n    },\n    "follows": [["Ben", "Sue"], ["Joe", "Ben"]]\n}\nDDB.at("users").create(user_data_dict)\n\n# There is now a file called users.json (or user_data.ddb if you use compression)\n# in your specified storage directory.\n```\n\nCheck if file or sub-key exists\n--------------------------------------------------------------------------------\n```python\nDDB.at("users").exists()  # True\nDDB.at("users", key="none").exists()  # False\n# Also works on nested keys\nDDB.at("users", key="Ben").exists()  # True\nDDB.at("users", key="Sam").exists()  # False\n```\n\nRead dicts\n--------------------------------------------------------------------------------\n\n```python\nd = DDB.at("users").read()\n# You now have a copy of the json file named "users"\nd == user_data_dict # True\n\n# Only partially read Joe\njoe = DDB.at("users", key="Joe").read()\njoe == user_data_dict["Joe"] # True\n```\nIt is also possible to only read a subset of keys based on a filter callback:\n\n```python\nDDB.at("numbers").create({"a", 1, "b", 2, "c": 3})\n\nabove_1 = DDB.at("numbers", where=lambda k, v: v > 1).read()\n>>> above_1 == {"b", 2, "c": 3}\n```\n\nWrite dicts\n--------------------------------------------------------------------------------\n\n```python\nwith DDB.at("users").session() as (session, users):\n    # You now have a copy of the json file users as the variable users\n    # Inside the with statement, the file of user_data will be locked, and no other\n    # processes will be able to interfere.\n    users["follows"].append(["Sue", "Ben"])\n    session.write()\n    # session.write() must be called to save the changes!\nprint(DDB.at("user_data").read()["follows"])\n>>> [["Ben", "Sue"], ["Joe", "Ben"], ["Sue", "Ben"]]\n```\n\nIf you do not call session.write(), changes will not be written to disk!\n\n\nPartial writing\n--------------------------------------------------------------------------------\nImagine you have a huge json file with many purchases.\nThe json file looks like this: `{<id>: <purchase>, <id>: <purchase>, ...}`.\nNormally, you would have to read and parse the entire file to get a specific key.\nAfter modifying the purchase, you would also have to serialize and write the\nentire file again. With DDB, you can do it more efficiently:\n```python\nwith DDB.at("purchases", key="3244").session() as (session, purchase):\n    purchase["status"] = "cancelled"\n    session.write()\n```\nAfterwards, the status is updated in the json file.\nHowever, DDB did only efficiently gather the one purchase with id 134425, parsed\nits value, and serialized that value alone before writing again. This is several\norders of magnitude faster than the naive approach when working with big files.\n\n\nFolders\n--------------------------------------------------------------------------------\n\nYou can also read and write to folders of files. Consider the same example as\nbefore, but now we have a folder called `purchases` that contains many files\n`<id>.json`. If you want to open a session or read a specific one, you can do:\n\n```python\nDDB.at("purchases/<id>").read()\n# Or equivalently:\nDDB.at("purchases", "<id>").read()\n```\n\nTo open a session or read all, do the following:\n```python\nDDB.at("purchases/*").read()\n# Or equivalently:\nDDB.at("purchases", "*").read()\n```\n\n### Select from folder\n\nIf you have a folder containing many json files, you can read them selectively\nbased on a function. The file is included if the provided function returns true\nwhen it get the file dict as input:\n\nTo open a session or read all, do the following:\n```python\nfor i in range(10):\n    DDB.at("folder", i).create({"a": i})\n# Now in the directory "folder", 10 files exist\nres = DDB.at("folder/*", where=lambda x: x["a"] > 7).read() # .session() also possible\nassert ress == {"8": {"a": 8}, "9": {"a": 9}} # True\n```\n\n\n\nPerformance\n================================================================================\n\nIn preliminary testing, DictDataBase showed promising performance.\n\n### SQLite vs DictDataBase\nIn each case, `16` parallel processes were spawned to perform `128` increments\nof a counter in `4` tables/files. SQLite achieves `2435 operations/s` while\nDictDataBase managed to achieve `3143 operations/s`.\n\n### More tests\nIt remains to be tested how DictDatabase performs in different scenarios, for\nexample when multiple processes want to perform full writes to one big file.\n\n\nAPI Reference\n================================================================================\n\n### `at(path) -> DDBMethodChooser:`\nSelect a file or folder to perform an operation on.\nIf you want to select a specific key in a file, use the `key` parameter,\ne.g. `DDB.at("file", key="subkey")`.\n\nIf you want to select an entire folder, use the `*` wildcard,\neg. `DDB.at("folder", "*")`, or `DDB.at("folder/*")`. You can also use\nthe `where` callback to select a subset of the file or folder.\n\nIf the callback returns `True`, the item will be selected. The callback\nneeds to accept a key and value as arguments.\n\nArgs:\n- `path`: The path to the file or folder. Can be a string, a\ncomma-separated list of strings, or a list.\n- `key`: The key to select from the file.\n- `where`: A function that takes a key and value and returns `True` if the\nkey should be selected.\n\nBeware: If you select a folder with the `*` wildcard, you can\'t use the `key`\nparameter.\nAlso, you cannot use the `key` and `where` parameters at the same time.\n\nDDBMethodChooser\n--------------------------------------------------------------------------------\n\n### `exists() -> bool:`\nCreate a new file with the given data as the content. If the file\nalready exists, a FileExistsError will be raised unless\n`force_overwrite` is set to True.\n\nArgs:\n- `data`: The data to write to the file. If not specified, an empty dict\nwill be written.\n- `force_overwrite`: If `True`, will overwrite the file if it already\nexists, defaults to False (optional).\n\n\n### `create(data=None, force_overwrite: bool = False):`\nIt creates a database file at the given path, and writes the given database to\nit\n:param db: The database to create. If not specified, an empty database is\ncreated.\n:param force_overwrite: If True, will overwrite the database if it already\nexists, defaults to False (optional).\n\n### `delete()`\nDelete the file at the selected path.\n\n### `read(self, as_type: T = None) -> dict | T | None:`\nReads a file or folder depending on previous `.at(...)` selection.\n\nArgs:\n- `as_type`: If provided, return the value as the given type.\nEg. as_type=str will return str(value).\n\n### `session(self, as_type: T = None) -> DDBSession[T]:`\nOpens a session to the selected file(s) or folder, depending on previous\n`.at(...)` selection. Inside the with block, you have exclusive access\nto the file(s) or folder.\nCall `session.write()` to write the data to the file(s) or folder.\n\nArgs:\n- `as_type`: If provided, cast the value to the given type.\nEg. as_type=str will return str(value).\n',
    'author': 'Marcel Kröker',
    'author_email': 'kroeker.marcel@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/mkrd/DictDataBase',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
