# AUTOGENERATED! DO NOT EDIT! File to edit: 00_core.ipynb (unless otherwise specified).

__all__ = ['DashComponentBase', 'DashFigureFactory', 'DashComponent', 'concat_docstring', 'DashApp']

# Cell

import sys
from abc import ABC
import inspect
import types
from importlib import import_module

import shortuuid
import oyaml as yaml

import dash
import jupyter_dash

import dash_bootstrap_components as dbc

# Cell
import dash_core_components as dcc
import dash_html_components as html

from dash.dependencies import Input, Output, State
from dash.exceptions import PreventUpdate

# Cell
class DashComponentBase(ABC):
    """Base class for all dash_oop_components classes.

    Stores parameter of child classes to attributes and ._stored_params.
    Proved .to_config(), to_yaml(), .from_config() and .from_yaml() methods
    """
    def __init__(self, no_store=None, no_attr=None, no_config=None, child_depth=3):
        """
        Args:
            no_store {list, bool}: either a list of parameters not to store or True, in which
                case no parameters gets stored.
            no_attr {list, bool}: either a list of parameter not to assign to attribute or True,
                in which case no parameters get assigned to attributes
            no_config {list, bool}: either a list of parameter not to store to ._stored_params
                or True, in which case no parameters get saved to ._stored_params
            child_depth (int): how deep the child is from which parameters will be read.
                Defaults to 3 (i.e. The child of the child of DashComponentBase)
        """
        self._store_child_params(no_store, no_attr, no_config, child_depth)

    def _store_child_params(self, no_store=None, no_attr=None, no_config=None, child_depth=3):
        """
        Args:
            no_store {list, bool}: either a list of parameters not to store or True, in which
                case no parameters gets stored.
            no_attr {list, bool}: either a list of parameter not to assign to attribute or True,
                in which case no parameters get assigned to attributes
            no_config {list, bool}: either a list of parameter not to store to ._stored_params
                or True, in which case no parameters get saved to ._stored_params
            child_depth (int): how deep the child is from which parameters will be read.
                Defaults to 3 (i.e. The child of the child of DashComponentBase)
        """

        if not hasattr(self, '_stored_params'):
            self._stored_params = {}

        child_frame = sys._getframe(child_depth)
        child_args = child_frame.f_code.co_varnames[1:child_frame.f_code.co_argcount]
        child_dict = {arg: child_frame.f_locals[arg] for arg in child_args}
        if 'kwargs' in child_frame.f_locals:
            child_dict['kwargs'] = child_frame.f_locals['kwargs']

        if isinstance(no_store, bool) and no_store:
            return
        else:
            if no_store is None: no_store = tuple()

        if isinstance(no_attr, bool) and no_attr: dont_attr = True
        else:
            if no_attr is None: no_attr = tuple()
            dont_attr = False

        if isinstance(no_config, bool) and no_param: dont_config = True
        else:
            if no_config is None: no_config= tuple()
            dont_config = False

        for name, value in child_dict.items():
            if name in {'dash_component', 'dash_figure_factory', 'dash_app'}:
                raise ValueError(f"Please do not use {name} as a parameter name, "
                                 "as this results in a confusing and hard to parse config.")
            if not dont_attr and name not in no_store and name not in no_attr:
                setattr(self, name, value)
            if not dont_config and name not in no_store and name not in no_config:
                self._stored_params[name] = value

    def to_config(self):
        """
        returns a dict with class name, module and params
        """
        return dict(dash_component=dict(
            name=self.__class__.__name__,
            module=self.__class__.__module__,
            params=self._stored_params))

    def to_yaml(self, filepath=None):
        """
        stores a yaml configuration to disk.

        If no filepath is given, returns a str of the yaml config.
        """
        yaml_config = self.to_config()
        if filepath is not None:
            yaml.dump(yaml_config, open(filepath, "w"))
            return
        return yaml.dump(yaml_config)

    def dump(self, filepath=None):
        """store the object to disk.

        Default serializer is pickle, however depending on file suffix,
        dill or joblib will be used."""
        filepath = str(filepath)
        if filepath.endswith(".pkl") or filepath.endswith(".pickle"):
            import pickle
            pickle.dump(self, open(filepath, "wb"))
        elif filepath.endswith(".dill"):
            import dill
            dill.dump(self, open(filepath, "wb"))
        elif str(filepath).endswith(".joblib"):
            import joblib
            joblib.dump(self, filepath)
        else:
            filepath = filepath + ".pkl"
            import pickle
            pickle.dump(self, open(filepath, "wb"))

    @classmethod
    def from_config(cls, config, **update_params):
        """
        Loads a dash_oop_component class from a configuration dict.

        Args:
            config (dict): configuration dict, generated from .to_config()
            update_params: a dict of parameters to be overridden by update_params

        Returns:
            Instance of the class defined in the config.
        """
        if 'dash_component' in config:
            config = config['dash_component']
        elif 'dash_figure_factory' in config:
            config = config['dash_figure_factory']
        elif 'dash_app' in config:
            config = config['dash_app']
        else:
            raise ValueError("I only know how to build dash_component, "
                             "dash_figure_factory and dash_app from config!", str(config))

        params = config['params']
        if not params: params = {}

        for k, v in update_params.items():
            if k in params:
                params[k] = v
            elif 'kwargs' in params:
                params['kwargs'][k]=v
            else:
                raise ValueError(f"This dash_oop_component does not take {k} as an argument, "
                                "nor does it take **kwargs!")

        for k, v in params.items():
            if isinstance(v, dict) and ('dash_figure_factory' in v or 'dash_component' in v):
                params[k] = DashComponentBase.from_config(v)

        component_class = getattr(import_module(config['module']), config['name'])
        if 'kwargs' in params:
            kwargs = params.pop('kwargs')
        else:
            kwargs = {}
        return component_class(**params, **kwargs)

    @classmethod
    def from_yaml(cls, yaml_filepath, **update_params):
        """
        Loads a dash_oop_component class from a yaml file.

        Args:
            yaml_filepath (str, Path): filepath of a .yaml file, generated from .to_yaml()
            update_params: a dict of parameters to be overridden by update_params

        Returns:
            Instance of the class defined in the yaml file.
        """
        config = yaml.safe_load(open(str(yaml_filepath), "r"))
        return cls.from_config(config, **update_params)

    @classmethod
    def from_file(cls, filepath):
        """Load a DashComponentBase from file. Depending on the suffix of the filepath
        will either load with pickle ('.pkl'), dill ('.dill') or joblib ('joblib').

        If no suffix given, will try with pickle (and try adding ''.pkl')

        Args:
            filepath {str, Path} the location of the stored component

        returns:
            DashComponentBase
        """
        filepath = str(filepath)
        if filepath.endswith(".pkl") or str(filepath).endswith(".pickle"):
            import pickle
            return pickle.load(open(filepath, "rb"))
        elif filepath.endswith(".dill"):
            import dill
            return dill.load(open(filepath, "rb"))
        elif filepath.endswith(".joblib"):
            import joblib
            return joblib.load(filepath)
        else:
            from pathlib import Path
            filepath = Path(filepath)
            if not filepath.exists():
                if (filepath.parent / (filepath.name + ".pkl")).exists():
                    filepath = filepath.parent / (filepath.name + ".pkl")
                else:
                    raise ValueError(f"Cannot find file: {str(filepath)}")
            import pickle
            return pickle.load(open(str(filepath), "rb"))


# Cell

class DashFigureFactory(DashComponentBase):
    """
    Helper class to store data for a dashboard and provide e.g. plotting functions.

    You should seperate the datastorage/plotting logic from the dashboard logic.
    All data/plotting logic goes into a DashFigureFactory.

    All dashboard logic goes into a DashComponent.

    Stores to config under key 'dash_figure_factory'
    """
    def __init__(self, no_store=None, no_attr=None, no_config=None):
        super().__init__(no_store=None, no_attr=None, no_config=None)

    def to_config(self):
        return dict(dash_figure_factory=dict(
            name=self.__class__.__name__,
            module=self.__class__.__module__,
            params=self._stored_params))



# Cell

class DashComponent(DashComponentBase):
    """DashComponent is a bundle of a dash layout and callbacks that
    can make use of DashFigureFactory objects.

    A DashComponent can have DashComponent subcomponents, that
    you register with register_components().

    DashComponents allow you to:

    1. Write clean, re-usable, composable code for your dashboard
    2. Store your dashboard to config files
    3. Load your dashboard from config files

    Each DashComponent should have a unique .name so that dash id's don't clash.
    If no name is given, DashComponent generates a unique uuid name. This allows
    for multiple instance of the same component type in a single layout.
    But remember to add `+self.name` to all id's.

    Important:
        define your callbacks in `_register_callbacks()` (note underscore!) and
        DashComponent will register callbacks of subcomponents in addition
        to _register_callbacks() when calling register_callbacks()
    """
    def __init__(self, title="Dash", name=None,
                 no_store=None, no_attr=None, no_config=None):
        """initialize the DashComponent

        Args:
            title (str, optional): Title of component. Defaults to "Dash".
            name (str, optional): unique name to add to Component elements.
                        If None then random uuid is generated to make sure
                        it's unique. Defaults to None.
        """
        super().__init__(no_store, no_attr, no_config)
        self._convert_ff_config_params()

        self.title = title
        self.name = name
        if self.name is None:
            self.name = str(shortuuid.ShortUUID().random(length=10))

        self._components = []

    def _convert_ff_config_params(self):
        """convert any DashFigureFactory in the ._stored_params dict to its config"""
        for k, v in self._stored_params.items():
            if isinstance(v, DashFigureFactory):
                self._stored_params[k] = self._stored_params[k].to_config()

    @staticmethod
    def make_hideable(element, hide=False):
        """helper function to optionally not display an element in a layout.

        Example:
            make_hideable(dbc.Col([cutoff.layout()]), hide=hide_cutoff)

        Args:
            hide(bool): wrap the element inside a hidden html.div. If the element
                        is a dbc.Col or a dbc.FormGroup, wrap element.children in
                        a hidden html.Div instead. Defaults to False.
        """
        if hide:
            if isinstance(element, dbc.Col) or isinstance(element, dbc.FormGroup):
                return html.Div(element.children, style=dict(display="none"))
            else:
                return html.Div(element, style=dict(display="none"))
        else:
            return element

    def register_components(self):
        """register subcomponents so that their callbacks will be registered

        Searches self.__dict__, finds all DashComponents and adds them to self._components
        """
        if not hasattr(self, '_components'):
            self._components = []
        for k, v in self.__dict__.items():
            if k != '_components' and isinstance(v, DashComponent) and v not in self._components:
                self._components.append(v)

    def layout(self):
        """layout to be defined by the particular ExplainerComponent instance.
        All element id's should append +self.name to make sure they are unique."""
        return None

    def _register_callbacks(self, app):
        """register callbacks specific to this ExplainerComponent."""
        pass

    def register_callbacks(self, app):
        """First register callbacks of all subcomponents, then call
        _register_callbacks(app)
        """
        self.register_components()
        for comp in self._components:
            comp.register_callbacks(app)
        self._register_callbacks(app)

# Cell

def concat_docstring(source=None):
    "Decorator: `__doc__` from `source` to __doc__"
    def _f(f):
        if isinstance(f, types.FunctionType):
            from_f = f
        else:
            from_f = f.__init__

        if isinstance(source, types.FunctionType):
            source_f = source
        elif source.__init__.__doc__ is not None:
            source_f = source.__init__
        else:
            source_f = source
        from_f.__doc__ = (
            str(from_f.__doc__) +
            "\n\n\-----------------------\n\n" +
            f"Docstring from {source.__name__}" +
            "\n\n" +
            str(source_f.__doc__))
        return f
    return _f

# Cell

class DashApp(DashComponentBase):
    """Wrapper class for dash apps. Assigns layout and callbacks from
    a DashComponent to a Dash app, and runs it.

    Can run both Dash and JupyterDash apps.

    """
    @concat_docstring(dash.Dash)
    def __init__(self, dashboard_component, port=8050, mode='dash', **kwargs):
        """

        Args:
            dashboard_component (DashComponent): component to be run
            port (int): port to run the server
            mode ({'dash', 'external', 'inline', 'jupyterlab'}): type of dash server to start
            kwargs: all kwargs will be passed down to dash.Dash. See below the docstring of dash.Dash

        Returns:
            DashApp: simply start .run() to start the dashboard
        """
        super().__init__(child_depth=2)
        self._stored_params['dashboard_component'] = dashboard_component.to_config()
        self.app = self._get_dash_app()

    def _get_dash_app(self):
        if self.mode == 'dash':
            app = dash.Dash(**self.kwargs)
        elif self.mode in {'inline', 'external', 'jupyterlab'}:
            app = jupyter_dash.JupyterDash(**self.kwargs)

        app.layout = self.dashboard_component.layout()
        self.dashboard_component.register_callbacks(app)
        app.title = self.dashboard_component.title
        return app

    def to_config(self):
        return dict(dash_app=dict(
            name=self.__class__.__name__,
            module=self.__class__.__module__,
            params=self._stored_params))

    def flask_server(self):
        """returns flask server inside self.app, for building wsgi apps"""
        return self.app.server

    def run(self, port=None):
        """Run the dash app"""
        self.app.run_server(port=port if port is not None else self.port)
