# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_genes.ipynb.

# %% auto 0
__all__ = ['filter_OGs', 'assign_homology', 'calculate_orthology_score', 'get_orthologs', 'get_orthologs_overlap']

# %% ../nbs/01_genes.ipynb 3
import os
from typing import Union

import anndata as ad
import numpy as np
import pandas as pd

# %% ../nbs/01_genes.ipynb 10
def filter_OGs(
    x: Union[
        list, str
    ],  # list of the EggNOG OGs or comma-separated string of EggNOG OGs.
    paralog: str = "Eukaryota",  # the level of the paralog OG
    ortholog: str = "Bilateria",  # the level of the ortholog OG
) -> list:  # the paralog OG and ortholog OG
    "Find the EggNOG OGs at the the paralog and ortholog level."
    paralog_OG = ""
    ortholog_OG = ""
    if isinstance(x, str):
        x = x.split(",")
    for s in x:
        if paralog in s:
            paralog_OG = s
        if ortholog in s:
            ortholog_OG = s
    return [paralog_OG, ortholog_OG]

# %% ../nbs/01_genes.ipynb 15
def assign_homology(
    species_OGs,  # the dataframe with the gene_id and the EggNOG OGs
    paralog: str = "Eukaryota",  # the level of the paralog OG
    ortholog: str = "Bilateria",  # the level of the ortholog OG)
) -> pd.DataFrame:  # the dataframe with the gene_id, paralog OG and ortholog OG
    "Get the taxonomy of the genes."
    homologs = (
        species_OGs["eggNOG_OGs"]
        .apply(filter_OGs, paralog=paralog, ortholog=ortholog)
        .to_list()
    )
    taxonomy = pd.DataFrame(homologs, columns=["paralog", "ortholog"])
    species_OGs = species_OGs.join(taxonomy)

    orthologs = pd.DataFrame(
        species_OGs.groupby("gene_id")["ortholog"].apply(np.unique).str[0]
    )
    paralogs = pd.DataFrame(
        species_OGs.groupby("gene_id")["paralog"].apply(np.unique).str[0]
    )

    result = orthologs.join(paralogs)
    result.replace(to_replace="", value=None, inplace=True)
    return result

# %% ../nbs/01_genes.ipynb 22
def compare_orthology(query, target):
    result = []
    for x in query:
        result.append(target == x)

    result = pd.DataFrame(result)
    result.index = query.index
    return result

# %% ../nbs/01_genes.ipynb 23
def calculate_orthology_score(
    query: pd.DataFrame,  # the dataframe with the gene_id, paralog OG and ortholog OG for the query species
    target: pd.DataFrame,
) -> pd.DataFrame:
    orthologs = compare_orthology(query["ortholog"], target["ortholog"])
    paralogs = compare_orthology(query["paralog"], target["paralog"])
    orthology_score = orthologs.astype(int) + paralogs.astype(int)
    return orthology_score

# %% ../nbs/01_genes.ipynb 29
def get_orthologs(
    genes: np.ndarray, orthology: pd.DataFrame, target: ad.AnnData, celltype_to: str
) -> np.ndarray:
    """
    Get orthologous and paralogous gene connections based on the given genes and orthology
    information.

    Parameters
    ----------
    genes : np.ndarray
        Array of gene names.
    orthology : pd.DataFrame
        Data frame representing the orthology information. The index should contain the query genes,
        the columns should overlap with the index of target.var, and the values should be 1 for
        paralogs and 2 for orthologs.
    target : ad.AnnData
        Target annotation data.
    celltype_to : str
        The target cell type. Must be a key in `target.uns["rank_genes_groups"]["names"]`.

    Returns
    -------
    np.ndarray
        Array of connections between genes, including orthologous and paralogous connections.
        Columns are (query, target, degree), where degree is 1 for paralogs and 2 for orthologs.
    """
    genes_in_table = np.intersect1d(genes, orthology.index)

    to_include = np.sum(orthology.loc[genes_in_table]) == 2
    orthologs = np.intersect1d(orthology.columns[to_include], target.var.index)

    to_include = np.sum(orthology.loc[genes_in_table]) == 1
    paralogs = np.intersect1d(orthology.columns[to_include], target.var.index)

    scores = pd.DataFrame(target.uns["rank_genes_groups"]["scores"])[celltype_to]
    names = pd.DataFrame(target.uns["rank_genes_groups"]["names"])[celltype_to]

    framed = pd.DataFrame(scores).set_index(names)
    significant = framed.loc[paralogs] > 0.01
    homologs = np.concatenate((orthologs, paralogs[significant[celltype_to]]))

    subset = orthology.loc[genes_in_table][homologs].melt(ignore_index=False)
    subset.reset_index(drop=False, inplace=True)

    connections = np.array(subset[subset["value"] > 0])

    not_in_table = np.setdiff1d(genes, orthology.index)
    if not_in_table.size > 0:
        unconnected = np.array([[g, None, 0] for g in not_in_table])
        return np.concatenate((connections, unconnected))
    else:
        return connections


def get_orthologs_overlap(genes1, genes2, query, target, orthology):
    """
    Returns a DataFrame of homologous gene pairs between two sets of genes based on their presence
    in an orthology table.

    Parameters
    ----------
    genes1 : numpy.ndarray
        A series of gene names.
    genes2 : numpy.ndarray
        A series of gene names.
    query : anndata.AnnData
        An AnnData object containing the query genes as indices of the `.var` slot.
    target : anndata.AnnData
        An AnnData object containing the target genes as indices of the `.var` slot.
    orthology : pandas.core.frame.DataFrame
        A DataFrame containing the orthology information.

    Returns
    -------
    connections : pandas.core.frame.DataFrame
        A DataFrame of homologous gene pairs and their degree of conservation. The array has
        three columns: 'query', 'target', and 'degree', where 'query' and 'target' are the gene
        names, and 'degree' is the degree of conservation, which can be either 1 or 2.
    """
    genes1_in_data = np.intersect1d(genes1, query.var.index)
    genes2_in_data = np.intersect1d(genes2, target.var.index)
    genes1_in_table = np.intersect1d(genes1_in_data, orthology.index)
    genes2_in_table = np.intersect1d(genes2_in_data, orthology.columns)

    connections = []

    subset = orthology.loc[genes1_in_table][genes2_in_table]
    _qo, _to = np.where(subset == 2)
    for q, t in zip(subset.index[_qo], subset.columns[_to]):
        connections.append([q, t, "2"])

    _qo, _to = np.where(subset == 1)
    for q, t in zip(subset.index[_qo], subset.columns[_to]):
        connections.append([q, t, "1"])

    return np.array(connections)
