# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/03_dotplot_util.ipynb.

# %% auto 0
__all__ = ['godsnot_102', 'map_array_to_color', 'map_to_colormap', 'unique_genes', 'highlight_cluster', 'prepare_dotplot',
           'label_pos', 'calculate_adjacency_matrix', 'gene_order', 'feature_colors', 'get_dot_size', 'get_dot_color',
           'plot_dot_legend', 'plot_colorbar_legend', 'make_dotplot', 'add_connections', 'plot_dotplot',
           'add_homology_context']

# %% ../nbs/03_dotplot_util.ipynb 5
from typing import Any, Dict, List, Optional, Tuple, Union

import matplotlib as mpl
import matplotlib.pyplot as plt
import numpy as np
import pandas as pd
from matplotlib import cm, colors
from matplotlib.cm import ScalarMappable
from matplotlib.colors import Colormap, Normalize
from matplotlib.patches import ConnectionPatch
from numpy.random import default_rng
from scipy.sparse.csgraph import connected_components

try:
    from matplotlib.colors import TwoSlopeNorm as DivNorm
except ImportError:
    # matplotlib<3.2
    from matplotlib.colors import DivergingNorm as DivNorm

from . import util

# %% ../nbs/03_dotplot_util.ipynb 6
def map_array_to_color(
    x: np.ndarray, palette: colors.Colormap, xmax: Union[float, None] = None
) -> np.ndarray:
    """
    Map an array of values to a color palette.

    Parameters
    ----------
    x : np.ndarray
        The array to map.
    palette : colors.Colormap
        The color map to use. Should be a matplotlib colormap object.
    xmax : Union[float, None], optional
        The maximum value to use for normalization. Should be >= np.max(x), and will use np.max(x) if set to `None` (default: None).

    Returns
    -------
    np.ndarray
        Array of RGBA values with a shape of ``x.shape + (4, )``.
    """
    if xmax is None:
        xmax = np.max(x)
    color = palette(x / xmax)
    return color


def map_to_colormap(
    x: np.ndarray,
    cmap: Union[str, colors.Colormap] = "magma_r",
    vmin: float = 0,
    vmax: Union[float, None] = None,
) -> np.ndarray:
    """
    Map an array of values to a color palette.

    Parameters
    ----------
    x : np.ndarray
        The array to map.
    cmap : Union[str, colors.Colormap], optional
        The color map to use. Should be a matplotlib colormap object or a string with the name of a matplotlib colormap (default: "magma_r").
    vmin : float, optional
        The value to obtain the minimum color in the colormap. Should be <= np.min(x) to avoid truncation (default: 0).
    vmax : Union[float, None], optional
        The value to obtain the maximum color in the colormap. Should be >= np.max(x), and will use np.max(x) if set to `None` (default: None).

    Returns
    -------
    np.ndarray
        Array of RGBA values with a shape of ``x.shape + (4, )``.
    """
    if vmax is None:
        vmax = np.max(x)
    norm = mpl.colors.Normalize(vmin=vmin, vmax=vmax)
    m = cm.ScalarMappable(norm=norm, cmap=cmap)
    return m.to_rgba(x)


def unique_genes(connections: np.ndarray) -> np.ndarray:
    """
    Extract the unique gene names from an array of connections.

    Parameters
    ----------
    connections : np.ndarray
        The array of connections. Columns should be (query_genes, target_genes, connection_strength). The last column is optional.

    Returns
    -------
    np.ndarray
        The array of unique genes.
    """
    real = connections != np.array(None)
    return np.unique(connections[real])


def _check_colornorm(vmin=None, vmax=None, vcenter=None, norm=None):
    # from ScanPy: https://github.com/scverse/scanpy/blob/9c6996f374370a7d50768c634e76e50173d22839/scanpy/plotting/_utils.py#L1201
    if norm is not None:
        if (vmin is not None) or (vmax is not None) or (vcenter is not None):
            raise ValueError("Passing both norm and vmin/vmax/vcenter is not allowed.")
    else:
        if vcenter is not None:
            norm = DivNorm(vmin=vmin, vmax=vmax, vcenter=vcenter)
        else:
            norm = Normalize(vmin=vmin, vmax=vmax)

    return norm


def highlight_cluster(
    clusters: np.ndarray,
    cluster: Union[str, None] = None,
    bg: Union[str, tuple] = "black",
    hl: Union[str, tuple] = "red",
) -> np.ndarray:
    """
    Highlight a cluster in a list of clusters by setting the color of the cluster to `hl` and the color of the rest to `bg`.

    Parameters
    ----------
    clusters : np.ndarray
        The array of cluster names.
    cluster : Union[str, None], optional
        The cluster to highlight (default: None).
    bg : Union[str, tuple], optional
        The background color. Accepts all matplotlib-compatible color formats (default: "black").
    hl : Union[str, tuple], optional
        The highlight color. Accepts all matplotlib-compatible color formats (default: "red").

    Returns
    -------
    np.ndarray
        The array of colors, with the same length as `clusters`.
    """
    clust_col = [bg] * len(clusters)
    if cluster is None or cluster == "":
        print("Invalid cluster name; returning background color.")
        return clust_col
    if cluster not in clusters:
        print("Cluster not found; returning background color.")
        return clust_col
    highlight = np.where(clusters == cluster)[0][0]
    clust_col[highlight] = hl
    return clust_col

# %% ../nbs/03_dotplot_util.ipynb 7
godsnot_102 = np.array(
    [
        "#FFFF00",
        "#1CE6FF",
        "#FF34FF",
        "#FF4A46",
        "#008941",
        "#006FA6",
        "#A30059",
        "#FFDBE5",
        "#7A4900",
        "#0000A6",
        "#63FFAC",
        "#B79762",
        "#004D43",
        "#8FB0FF",
        "#997D87",
        "#5A0007",
        "#809693",
        "#6A3A4C",
        "#1B4400",
        "#4FC601",
        "#3B5DFF",
        "#4A3B53",
        "#FF2F80",
        "#61615A",
        "#BA0900",
        "#6B7900",
        "#00C2A0",
        "#FFAA92",
        "#FF90C9",
        "#B903AA",
        "#D16100",
        "#DDEFFF",
        "#000035",
        "#7B4F4B",
        "#A1C299",
        "#300018",
        "#0AA6D8",
        "#013349",
        "#00846F",
        "#372101",
        "#FFB500",
        "#C2FFED",
        "#A079BF",
        "#CC0744",
        "#C0B9B2",
        "#C2FF99",
        "#001E09",
        "#00489C",
        "#6F0062",
        "#0CBD66",
        "#EEC3FF",
        "#456D75",
        "#B77B68",
        "#7A87A1",
        "#788D66",
        "#885578",
        "#FAD09F",
        "#FF8A9A",
        "#D157A0",
        "#BEC459",
        "#456648",
        "#0086ED",
        "#886F4C",
        "#34362D",
        "#B4A8BD",
        "#00A6AA",
        "#452C2C",
        "#636375",
        "#A3C8C9",
        "#FF913F",
        "#938A81",
        "#575329",
        "#00FECF",
        "#B05B6F",
        "#8CD0FF",
        "#3B9700",
        "#04F757",
        "#C8A1A1",
        "#1E6E00",
        "#7900D7",
        "#A77500",
        "#6367A9",
        "#A05837",
        "#6B002C",
        "#772600",
        "#D790FF",
        "#9B9700",
        "#549E79",
        "#FFF69F",
        "#201625",
        "#72418F",
        "#BC23FF",
        "#99ADC0",
        "#3A2465",
        "#922329",
        "#5B4534",
        "#FDE8DC",
        "#404E55",
        "#0089A3",
        "#CB7E98",
        "#A4E804",
        "#324E72",
    ]
)

# %% ../nbs/03_dotplot_util.ipynb 8
def prepare_dotplot(
    avg_expr: pd.DataFrame,
    perc_expr: pd.DataFrame,
    cmap: Union[str, Colormap] = "magma_r",
    vmin: float = 0,
    vmax: Union[float, None] = None,
    size_exponent: float = 1.5,
    dot_size: float = 200,
) -> Tuple[pd.DataFrame, pd.DataFrame, np.ndarray]:
    """
    Pivots average expression and percent expressed tables to make them dotplot-friendly.

    Parameters
    ----------
    avg_expr : pd.DataFrame
        Data frame that holds average expression for all genes and all clusters.
    perc_expr : pd.DataFrame
        Data frame that tracks the percentage of cells expressing each gene in every cluster.
    cmap : Union[str, Colormap], optional
        The Colormap instance or registered colormap name used to map scalar data to colors
        (default: "magma_r").
    vmin : float, optional
        Minimum average expression value to show (default: 0).
    vmax : Union[float, None], optional
        Maximum average expression value to show (default: maximum average expr. value).
    size_exponent : float, optional
        Dot size is computed as fraction ** size_exponent * dot_size (default: 1.5).
    dot_size : float, optional
        The size of the largest dot (default: 200).

    Returns
    -------
    Tuple[pd.DataFrame, pd.DataFrame, np.ndarray]
        A tuple containing the melted average expression data frame, the melted percentage
        expression data frame, and the array of RGBA-coded color values for the average expression
        in a cluster/gene combination, according to the input color map.
    """
    assert size_exponent > 0, "size_exponent must be > 0"
    assert dot_size > 0, "dot_size must be > 0"
    df_avg_expr = avg_expr.reset_index().melt("index")
    df_avg_expr.columns = ["row", "column", "value"]
    color = map_to_colormap(df_avg_expr["value"], cmap=cmap, vmin=vmin, vmax=vmax)

    df_perc_expr = perc_expr.reset_index().melt("index")
    df_perc_expr.columns = ["row", "column", "value"]
    df_perc_expr["value"] = df_perc_expr["value"] ** size_exponent * dot_size
    return df_avg_expr, df_perc_expr, color


def label_pos(
    display_coords: Dict[str, Tuple[float, float, float, float]],
    key: str,
    side: str = "left",
) -> Tuple[float, float]:
    """
    Get the edge coordinates of a label. Keep either the left or the right end of the word.

    Parameters
    ----------
    display_coords : Dict[str, Tuple[float, float, float, float]]
        A dictionary that holds the window extents of tick labels.
    key : str
        The label to retrieve; a gene name.
    side : str, optional
        One of "left" or "right"; depending on orientation will return the leftmost or rightmost
        position of the label (default: "left").

    Returns
    -------
    Tuple[float, float]
        A tuple containing the x and y coordinates of the label.
    """
    x0, y0, x1, y1 = display_coords[key].flatten()
    if side == "left":
        x = x1
    else:
        x = x0
    y = (y0 + y1) / 2
    return x, y


def calculate_adjacency_matrix(
    connections: np.ndarray, query_genes: List[Any], target_genes: List[Any]
) -> np.ndarray:
    """
    Calculate the adjacency matrix based on the given connections, query genes, and target genes.

    Parameters
    ----------
    connections : np.ndarray
        The 2D array representing the connections between genes. Each row contains two gene
        identifiers indicating a connection, and optionally the strength of that connection.
    query_genes : List[Any]
        A list of genes that act as queries.
    target_genes : List[Any]
        A list of genes that act as targets.

    Returns
    -------
    np.ndarray
        The adjacency matrix represented as a 2D numpy array. It has dimensions (query_G + target_G)
        x (query_G + target_G), where query_G and target_G are the lengths of query_genes and
        target_genes, respectively.
    """
    query_G = len(query_genes)
    target_G = len(target_genes)
    adjacency = np.zeros(
        (query_G, target_G), dtype=bool
    )  # not a true adjacency matrix; targetely the top right corner.
    query_connected = {gene: i for i, gene in enumerate(query_genes)}
    target_connected = {gene: i for i, gene in enumerate(target_genes)}

    if connections.shape[1] == 3:
        connections = connections[:, :2]
    for [g1, g2] in connections:
        if g1 is None or g2 is None:
            continue
        i1 = query_connected[g1]
        i2 = target_connected[g2]
        adjacency[i1, i2] = True

    full_adjacency = np.zeros((query_G + target_G, query_G + target_G), dtype=bool)
    full_adjacency[:query_G][:, query_G:] = adjacency
    full_adjacency[query_G:][:, :query_G] = adjacency.T
    return full_adjacency


def gene_order(
    full_adjacency: np.ndarray, components: np.ndarray, query_G: int
) -> Tuple[np.ndarray, np.ndarray]:
    """
    Calculate the order of genes based on the given full adjacency matrix and components. Highly
    connected genes are placed first, genes without any connections are randomly ordered in the
    bottom of the plot.

    Parameters
    ----------
    full_adjacency : np.ndarray
        The full adjacency matrix represented as a 2D numpy array.
    components : np.ndarray
        An array representing the components.
    query_G : int
        The number of query genes.

    Returns
    -------
    Tuple[np.ndarray, np.ndarray]
        A tuple containing the query gene order and the target gene order as numpy arrays.
    """
    comp, freq = np.unique(components, return_counts=True)
    degrees = np.sum(full_adjacency, axis=0)

    keep = freq > 1
    descending = np.argsort(-freq[freq > 1])

    no_comps = len(comp[freq > 1])
    for i, c in enumerate(comp[keep][descending]):
        # degrees[components == c] *= freq[keep][descending][i] * (no_comps - i)
        degrees[components == c] = no_comps - i

    query_degree = degrees[:query_G]
    target_degree = degrees[query_G:]

    query_order = np.argsort(-query_degree)
    target_order = np.argsort(-target_degree)
    return query_order, target_order


def feature_colors(
    components: np.ndarray, query_G: int, seed: int = 42
) -> Tuple[np.ndarray, np.ndarray]:
    """
    Assign colors to the components based on the given array of components.

    Parameters
    ----------
    components : np.ndarray
        The array of components.
    query_G : int
        The number of components for the query genes.
    seed : int, optional
        The seed value for the random number generator (default: 42).

    Returns
    -------
    Tuple[np.ndarray, np.ndarray]
        A tuple containing the colored components for query genes and target genes, respectively.
    """
    comp, freq = np.unique(components, return_counts=True)
    rng = default_rng(seed)
    no_comps = len(comp[freq > 1])
    if no_comps < 102:
        pick_colors = rng.choice(len(godsnot_102), size=no_comps, replace=False)
    else:
        pick_colors = rng.choice(len(godsnot_102), size=no_comps, replace=True)
    godsnot_keep = {}
    for i, component in enumerate(comp[freq > 1]):
        godsnot_keep[component] = godsnot_102[pick_colors][i]

    components_colored = ["black"] * len(components)
    for i, component in enumerate(components):
        if component in comp[freq > 1]:
            components_colored[i] = godsnot_keep[component]

    query_comp_color = np.array(components_colored[:query_G])
    target_comp_color = np.array(components_colored[query_G:])
    return query_comp_color, target_comp_color


def get_dot_size(
    query: pd.DataFrame,
    target: pd.DataFrame,
    query_clustering: str,
    target_clustering: str,
    query_genes: Optional[np.ndarray] = None,
    target_genes: Optional[np.ndarray] = None,
    query_gene_names: Optional[np.ndarray] = None,
    target_gene_names: Optional[np.ndarray] = None,
) -> Tuple[pd.DataFrame, pd.DataFrame]:
    """
    Calculate which percentage of cells in each cluster express each gene, and translate that to dot
    size for the dotplot.

    Parameters
    ----------
    query : pd.DataFrame
        The query dataset.
    target : pd.DataFrame
        The target dataset.
    query_clustering : str
        The .obs column name to use for the query dataset.
    target_clustering : str
        The .obs column name to use for the target dataset.
    query_genes : np.ndarray, optional
        Array of query genes to subset the data, if any. If None, use all genes (default: None).
    target_genes : np.ndarray, optional
        Array of target genes to subset the data, if any. If None, use all genes (default: None).
    query_gene_names : np.ndarray, optional
        Array of query gene names (default: None).
    target_gene_names : np.ndarray, optional
        Array of target gene names (default: None).

    Returns
    -------
    Tuple[pd.DataFrame, pd.DataFrame]
        A tuple containing the dot size values for query and target datasets, respectively.
    """

    query_cluster_size = query.obs[query_clustering].value_counts()
    target_cluster_size = target.obs[target_clustering].value_counts()

    query_perc_expr = (
        util.grouped_obs_present(query, query_clustering) / query_cluster_size
    )
    target_perc_expr = (
        util.grouped_obs_present(target, target_clustering) / target_cluster_size
    )

    # subset to only the genes we have
    if query_genes is not None and target_genes is not None:
        query_perc_expr = query_perc_expr.loc[query_genes].copy().T
        target_perc_expr = target_perc_expr.loc[target_genes].copy().T
    else:
        print("No genes supplied; returning all genes.")

    if query_gene_names is not None:
        query_perc_expr.columns = query.var[query_gene_names].loc[
            query_perc_expr.columns
        ]
    if target_gene_names is not None:
        target_perc_expr.columns = target.var[target_gene_names].loc[
            target_perc_expr.columns
        ]
    return query_perc_expr, target_perc_expr


def get_dot_color(
    query: pd.DataFrame,
    target: pd.DataFrame,
    query_clustering: str,
    target_clustering: str,
    query_genes: Optional[np.ndarray] = None,
    target_genes: Optional[np.ndarray] = None,
    query_gene_names: Optional[np.ndarray] = None,
    target_gene_names: Optional[np.ndarray] = None,
) -> Tuple[pd.DataFrame, pd.DataFrame]:
    """
    Calculate average expression in each cluster and translate that to dot color for the dotplot.

    Parameters
    ----------
    query : pd.DataFrame
        The query dataset.
    target : pd.DataFrame
        The target dataset.
    query_clustering : str
        The .obs column name to use for the query dataset.
    target_clustering : str
        The .obs column name to use for the target dataset.
    query_genes : np.ndarray, optional
        Array of query genes to subset the data, if any. If None, use all genes (default: None).
    target_genes : np.ndarray, optional
        Array of target genes to subset the data, if any. If None, use all genes (default: None).
    query_gene_names : np.ndarray, optional
        Array of query gene names (default: None).
    target_gene_names : np.ndarray, optional
        Array of target gene names (default: None).

    Returns
    -------
    Tuple[pd.DataFrame, pd.DataFrame]
        A tuple containing the dot color values for query and target datasets, respectively.
    """
    query_avg_expr = util.grouped_obs_mean(query, query_clustering)
    target_avg_expr = util.grouped_obs_mean(target, target_clustering)
    # subset to only the genes we have
    if query_genes is not None and target_genes is not None:
        query_avg_expr = query_avg_expr.loc[query_genes].copy().T
        target_avg_expr = target_avg_expr.loc[target_genes].copy().T
    else:
        print("No genes supplied; returning all genes.")

    if query_gene_names is not None:
        query_avg_expr.columns = query.var[query_gene_names].loc[query_avg_expr.columns]
    if target_gene_names is not None:
        target_avg_expr.columns = target.var[target_gene_names].loc[
            target_avg_expr.columns
        ]
    return query_avg_expr, target_avg_expr


def plot_dot_legend(
    dot_legend,
    size_exponent=1.5,
    dot_size=200,
):
    """
    Create the dotplot legend, explaining dot size.

    Parameters
    ----------
    dot_legend : matplotlib `axis`
        The subplot of the grid that contains the dotplot legend.
    size_exponent : float, optional
        The exponent to raise the fraction of cells in a group to, to get the dot size. The default
        is 1.5.
    dot_size : int, optional
        The size of the largest dot. The default is 200.
    """
    dot_legend.axis("equal")
    xticks = np.arange(5)
    dot_legend.scatter(
        xticks,
        np.zeros(5),
        s=np.arange(0.2, 1.1, 0.2) ** size_exponent * dot_size,
        c="gray",
        edgecolors="black",
    )
    dot_legend.set_xlim(-0.5, 4.5)
    dot_legend.yaxis.set_visible(False)
    dot_legend.spines[["left", "right", "top", "bottom"]].set_visible(False)
    dot_legend.xaxis.set_ticks(xticks)
    dot_legend.xaxis.set_ticklabels(["20", "40", "60", "80", "100"])
    dot_legend.set_title("Fraction of cells\nin group (%)", fontsize="small")


def _get_mappable(
    query_avg_expr: np.ndarray, target_avg_expr: np.ndarray, cmap: Colormap = "magma_r"
) -> ScalarMappable:
    """
    Get the ScalarMappable object for generating colors based on average expression values.

    Parameters
    ----------
    query_avg_expr : np.ndarray
        Array representing the average expression values of query genes.
    target_avg_expr : np.ndarray
        Array representing the average expression values of target genes.
    cmap : Colormap, optional
        The Colormap instance or registered colormap name used to map scalar data to colors
        (default: "magma_r").

    Returns
    -------
    ScalarMappable
        The ScalarMappable object for generating colors based on average expression values.
    """
    all_expr = np.concatenate(
        (query_avg_expr.values.flatten(), target_avg_expr.values.flatten())
    )
    max_expr = np.max(all_expr)
    norm = _check_colornorm(0, max_expr)
    return ScalarMappable(norm, cmap=cmap)


def plot_colorbar_legend(
    cbar_legend: plt.Axes,
    query_avg_expr: np.ndarray,
    target_avg_expr: np.ndarray,
    cmap: Colormap = "magma_r",
) -> None:
    """
    Plot the colorbar legend based on the average expression values of query and target genes.

    Parameters
    ----------
    cbar_legend : plt.Axes
        The Axes object representing the colorbar legend.
    query_avg_expr : np.ndarray
        Array representing the average expression values of query genes.
    target_avg_expr : np.ndarray
        Array representing the average expression values of target genes.
    cmap : Colormap, optional
        The Colormap instance or registered colormap name used to map scalar data to colors
        (default: "magma_r").
    """

    mappable = _get_mappable(query_avg_expr, target_avg_expr, cmap=cmap)
    mpl.colorbar.Colorbar(cbar_legend, mappable=mappable, orientation="horizontal")
    cbar_legend.yaxis.set_visible(False)
    cbar_legend.set_title("Mean expression\nin group", fontsize="small")
    cbar_legend.xaxis.set_tick_params(labelsize="small")


def make_dotplot(
    ax: plt.Axes,
    avg: np.ndarray,
    perc: np.ndarray,
    gene_names: List[str],
    species: str,
    clustering: str,
    clust_color: List[str],
    gene_color: List[str],
    side: str = "left",
) -> None:
    """
    Make a dotplot on the given Axes object based on the average and percentage expression values.

    Parameters
    ----------
    ax : plt.Axes
        The Axes object on which to create the dotplot.
    avg : np.ndarray
        The average expression values.
    perc : np.ndarray
        The percentage expression values.
    gene_names : List[str]
        The list of gene names.
    species : str
        The species name.
    clustering : str
        The clustering information.
    clust_color : List[str]
        The list of colors for clusters.
    gene_color : List[str]
        The list of colors for genes.
    side : str, optional
        The side to place the y-axis labels, either "left" or "right" (default: "left").
    """
    N, G = avg.shape
    df_avg, df_perc, color = prepare_dotplot(avg, perc)
    ax.scatter(
        df_avg["row"],
        df_avg["column"],
        c=color,
        s=df_perc["value"],
        edgecolors="black",
    )
    ax.axis("equal")
    ax.set_title(f"species: {species}, clustering: {clustering}")
    ax.xaxis.set_ticks(np.arange(N))
    ax.xaxis.set_ticklabels(avg.index, rotation=90)
    ax.yaxis.set_ticks(np.arange(G), labels=gene_names)
    if side == "left":
        ax.yaxis.tick_right()
    else:
        ax.yaxis.tick_left()
    ax.invert_yaxis()
    [t.set_color(clust_color[i]) for i, t in enumerate(ax.xaxis.get_ticklabels())]
    [t.set_color(gene_color[i]) for i, t in enumerate(ax.yaxis.get_ticklabels())]


def add_connections(
    fig: plt.Figure,
    connections: np.ndarray,
    query_gene_names: List[str],
    query_gene_colors: Dict[str, str],
    label_offset: float,
) -> None:
    """
    Add connections between genes to the given paired dotplot figure.

    Parameters
    ----------
    fig : plt.Figure
        The paired dotplot figure to which the connections will be added.
    connections : np.ndarray
        An array where each row contains two genes and (optionally) the strength of their
        connection.
    query_gene_names : List[str]
        The list of query gene names.
    query_gene_colors : Dict[str, str]
        The dictionary mapping query gene names to their colors.
    label_offset : float
        The offset for label positioning.

    Returns
    -------
    None
    """

    label_offset = np.array([label_offset, 0])
    left = fig.get_axes()[0]
    right = fig.get_axes()[1]

    left_labels = {}
    for label in left.get_yticklabels():
        bbox = label.get_window_extent()
        coords = fig.transFigure.inverted().transform(bbox)
        left_labels[label.get_text()] = coords + label_offset

    right_labels = {}
    for label in right.get_yticklabels():
        bbox = label.get_window_extent()
        coords = fig.transFigure.inverted().transform(bbox)
        right_labels[label.get_text()] = coords - label_offset

    query_index = {g: i for i, g in enumerate(query_gene_names)}
    for c in connections:
        linestyle = "dotted"
        if len(c) == 3:
            gene_from, gene_to, ls = c
            if ls == 0:
                linestyle = (0, (1, 5))
            elif ls == 1:
                linestyle = (0, (5, 5))
            elif ls == 2:
                linestyle = "solid"
        else:
            gene_from, gene_to = c
        if gene_from is None or gene_to is None:
            continue
        species_a = label_pos(left_labels, gene_from, side="left")
        species_b = label_pos(right_labels, gene_to, side="right")
        con = ConnectionPatch(
            xyA=species_a,
            xyB=species_b,
            coordsA="figure fraction",
            coordsB="figure fraction",
            color=query_gene_colors[query_index[gene_from]],
            linestyle=linestyle,
        )
        right.add_artist(con)


def plot_dotplot(
    query_avg_expr: np.ndarray,
    target_avg_expr: np.ndarray,
    query_perc_expr: np.ndarray,
    target_perc_expr: np.ndarray,
    query_genes: List[str],
    target_genes: List[str],
    connections: np.ndarray,
    query_cluster_colors: Dict[str, str],
    target_cluster_colors: Dict[str, str],
    query_gene_colors: Dict[str, str],
    target_gene_colors: Dict[str, str],
    query_species: str,
    target_species: str,
    x_offset: float = 1,
    y_offset: float = 0,
    grid_offset: int = 30,
    query_clustering: str = "leiden",
    target_clustering: str = "leiden",
    output: str = "./paired_dotplot.png",
    title: Union[None, str] = None,
    title_font_size: int = 16,
    center: bool = True,
) -> None:
    """
    Plot the paired dotplot based on the given data.

    Parameters
    ----------
    query_avg_expr : np.ndarray
        Array representing the average expression values of query genes.
    target_avg_expr : np.ndarray
        Array representing the average expression values of target genes.
    query_perc_expr : np.ndarray
        Array representing the percentage expression values of query genes.
    target_perc_expr : np.ndarray
        Array representing the percentage expression values of target genes.
    query_genes : List[str]
        List of query gene names.
    target_genes : List[str]
        List of target gene names.
    connections : np.ndarray
        An array where each row contains two genes and (optionally) the strength of their
        connection.
    query_cluster_colors : Dict[str, str]
        Dictionary mapping query cluster names to their colors.
    target_cluster_colors : Dict[str, str]
        Dictionary mapping target cluster names to their colors.
    query_gene_colors : Dict[str, str]
        Dictionary mapping query gene names to their colors.
    target_gene_colors : Dict[str, str]
        Dictionary mapping target gene names to their colors.
    query_species : str
        Species name of the query genes.
    target_species : str
        Species name of the target genes.
    x_offset : float, optional
        Offset for the x-axis (default: 1).
    y_offset : float, optional
        Offset for the y-axis (default: 0).
    grid_offset : int, optional
        Offset for the grid spacing (default: 30).
    query_clustering : str, optional
        Clustering method for the query genes (default: "leiden").
    target_clustering : str, optional
        Clustering method for the target genes (default: "leiden").
    output : str, optional
        Output file path for the plot (default: "./paired_dotplot.png").
    title : str, optional
        Title of the plot (default: None).
    title_font_size : int, optional
        Font size of the plot title (default: 16).
    center : bool, optional
        Whether to center the dotplots when the number of genes exceeds the maximum (default: True).

    Returns
    -------
    None
    """
    # make shortcuts for the number of clusters and number of genes
    query_N, query_G = query_avg_expr.shape
    target_N, target_G = target_avg_expr.shape

    # sometimes we end up overcorrecting when batch-producing images this way
    y_size = np.max([query_G, target_G]) / 2 + y_offset
    if y_size < 0:
        y_size = np.max([query_G, target_G]) / 2

    # create empty figure that scales with number of clusters and genes
    figsize = ((query_N + target_N) / 2 + x_offset, y_size)

    fig = plt.figure(figsize=figsize)
    max_genes = np.max([query_G, target_G])
    columns = query_N + target_N + grid_offset + 6
    ax = fig.add_gridspec(max_genes, columns)

    # add space for the left dotplot (query)
    left_start = 0
    if center and query_G > max_genes:
        left_start = (max_genes - query_G) // 2 - 1
    left_end = left_start + query_G
    left = fig.add_subplot(ax[left_start:left_end, 0:query_N])

    # add space for the right dotplot (target)
    right_start = 0
    if center and target_G > max_genes:
        right_start = (max_genes - target_G) // 2 - 1
    right_end = right_start + target_G

    after_blank = query_N + grid_offset
    before_legends = query_N + grid_offset + target_N
    right = fig.add_subplot(
        ax[
            right_start:right_end,
            after_blank:before_legends,
        ]
    )

    left.set_zorder(0)
    right.set_zorder(1)

    make_dotplot(
        left,
        query_avg_expr,
        query_perc_expr,
        query_genes,
        query_species,
        query_clustering,
        query_cluster_colors,
        query_gene_colors,
        side="left",
    )

    make_dotplot(
        right,
        target_avg_expr,
        target_perc_expr,
        target_genes,
        target_species,
        target_clustering,
        target_cluster_colors,
        target_gene_colors,
        side="right",
    )

    # saving the figure: don't forget the dpi option!
    if title is not None:
        fig.suptitle(title, fontsize=title_font_size)
    fig.savefig(output)

    # calculate label offset to make the links between genes more legible
    label_offset = 1 / (query_N + target_N + grid_offset * 2) / 3
    # draw appropriately colored lines between connected genes
    add_connections(fig, connections, query_genes, query_gene_colors, label_offset)

    dot_start = right_start + target_G // 3
    dot_legend = fig.add_subplot(ax[dot_start : (dot_start + 1), -5:])
    plot_dot_legend(dot_legend)

    cbar_start = right_start + 2 * target_G // 3
    cbar_legend = fig.add_subplot(ax[cbar_start : (cbar_start + 1), -3:])
    plot_colorbar_legend(cbar_legend, query_avg_expr, target_avg_expr)

    plt.savefig(output)


def add_homology_context(
    connections: np.ndarray, orthology: pd.DataFrame
) -> np.ndarray:
    """
    Add homology context to the given connections based on the orthology information.

    Parameters
    ----------
    connections : np.ndarray
        The connections between genes. The columns should be (query_gene, target_gene).
    orthology : pd.DataFrame
        The orthology information as a DataFrame.

    Returns
    -------
    np.ndarray
        The connections array with homology context added. The columns will be (query_gene,
        target_gene, connection_strength), and the values in connection_strength will depend on the
        content of the orthology DataFrame.
    """
    query_genes = [g for g in connections[:, 0] if g is not None]
    target_genes = [g for g in connections[:, 1] if g is not None]

    query_orthology = np.intersect1d(query_genes, orthology.index)
    target_orthology = np.intersect1d(target_genes, orthology.columns)

    subset = (
        orthology.loc[query_orthology][target_orthology]
        .melt(ignore_index=False)
        .reset_index(drop=False)
    )

    homologs = subset[subset["value"] > 0]
    homologs.columns = ["query_id", "target_id", "value"]
    query_no_homology = np.setdiff1d(query_genes, homologs["query_id"])
    target_no_homology = np.setdiff1d(target_genes, homologs["target_id"])

    query_side = np.concatenate(
        (
            homologs["query_id"].values,
            query_no_homology,
            [None] * len(target_no_homology),
        )
    )
    target_side = np.concatenate(
        (
            homologs["target_id"].values,
            [None] * len(query_no_homology),
            target_no_homology,
        )
    )
    orthology_info = np.concatenate(
        (
            homologs["value"].values,
            [0] * (len(query_no_homology) + len(target_no_homology)),
        )
    )

    connections = np.array([query_side, target_side, orthology_info]).T
    return connections
