# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['compressnets', 'compressnets.datafiles']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'compressnets',
    'version': '1.0.0',
    'description': 'Package for temporal network data compression',
    'long_description': '# compressnets\n\nCompressnets is a Python package designed to compress high-resolution temporal network data (eg. contact networks)\nto lower resolution, while maintaining important temporal structural features.\n\nCompressnets is designed for taking sequences of static networks (represented as adjacency matrices)\nand compressing them to a user-specified reduced number of adjacency matrices. For example, say you\nhave contact data at the resolution of 20 seconds, over the course of 24 hours, which can be represented\nas 4,320 adjacency matrices. \nYou might wish to compress the data from the original 4,320 adjacency matrices into the best 20 adjacency matrices\nthat best represent the temporal dynamics. The \n`compressnets` package can help you to progressively aggregate the data into the best 20 representative "snapshots" \n(single static network valid for a duration of time). \n\nPre-print with further details on the compression algorithm and theoretical framework is available [on the arXiv](https://arxiv.org/abs/2205.11566).\n\n## Most basic usage\nThis example is just to show how to use the package. In practice, 3 starting snapshots is unrealistic as there wouldn\'t\nbe a need to use an algorithm for compressing 3 snapshots into 2, so this example just demonstrates package usage in a simple way.\nSee below for a usage demo using a built-in sample network.\n\nTo use `compressnets`, install the package via the PyPi index via\n`pip install compressnets`.\n\nThe core elements of `compressnets` are the objects, `network.TemporalNetwork` and `network.Snapshot`,\nand the algorithm `compression.Compressor.compress(...)`.\n\nFollow along the following example in your own Python workspace with\n```\nfrom compressnets import compression, network\n```\n\nAs an example, create NumPy arrays to represent your adjacency matrices (or load them from data via your own code):\n```\nmatrix_1 = np.array([[0, 0, 1],[0, 0, 1],[1, 1, 0]])\nmatrix_2 = np.array([[0, 1, 1],[1, 0, 1],[1, 1, 0]])\nmatrix_3 = np.array([[0, 1, 0],[1, 0, 1],[0, 1, 0]])\n```\nNext, you will be creating `Snapshot` objects out of your arrays. A snapshot is simply an adjacency matrix\ncoupled with a start time, end time, and `beta` value, which will be used in the computation process for the\ncompression algorithm.\n#### Note: While the durations of each snapshot may vary in length from one another, the `end_time` of one snapshot should be the `start_time` of the next one. In other words, there should not be gaps in your snapshots.\n#### Note: It\'s also best practice to set the very first `start_time` as 0. The `to_snapshot_list()` helper function will do this automatically, but if you configure snapshots by hand, be sure to standardize to start at 0. \n\nIf the duration of all of your snapshots is uniform, you can use the `TemporalData` object\nas a helper object to quickly make a series of snapshots using your list of adjacency matrices, like this:\n```\ninfect_rate = 0.5\nyour_temporal_data = network.TemporalData([matrix_1, matrix_2, matrix_3], interval=1, beta=infect_rate)\nsnapshots = your_temporal_data.to_snapshot_list()\n```\n\nSee section below on choosing `beta`, but for this example, it can be any value between 0 and 1.\n#### Note: `beta` should be the same for all snapshots in the list.\n\n\nIf your snapshots have custom durations and are not all equal to the same interval, you can instead\ncreate your own list of `Snapshot` objects from your arrays, equipped with a consecutive start and end time for each,\n(instead of using the `TemporalData` helper):\n```\nsnapshots = [network.Snapshot(start_time=0, end_time=1, beta=infect_rate, A=matrix_1),\n             network.Snapshot(start_time=1, end_time=2, beta=infect_rate, A=matrix_2),\n             network.Snapshot(start_time=2, end_time=3, beta=infect_rate, A=matrix_3)]\n```\n\nEither way, once you have a list of `Snapshot` objects,\nthen create a `TemporalNetwork` object to contain all of your ordered snapshots:\n```\nyour_temporal_network = network.TemporalNetwork(snapshots)\n```\nUsing the algorithmic compression from our paper [link], you can compress\nthe temporal network into a desired number of compressed snapshots (in this example, 2), by calling on an instance of \nthe static `Compressor` class\n```\nyour_compressed_network_result = compression.Compressor.compress(your_temporal_network,\n                                                          compress_to=2,\n                                                          how=\'optimal\')\n``` \nwhich will return the new `TemporalNetwork` object, and also the total induced error from the snapshots that\nwere selected for compression. The elements can be accessed via a dictionary as\n```\nyour_compressed_network = your_compressed_network_result["compressed_network"]\ntotal_induced_error = your_compressed_network_result["error"]\n```\nTo compress your original network into an even division and aggregation of snapshots,\nnot using our algorithm, you can call `compress` and changing the `how` argument to `even`:\n```\nyour_even_compressed_network_result = compression.Compressor.compress(your_temporal_network,\n                                                          compress_to=2,\n                                                          how=\'even\')\neven_compressed_network = your_even_compressed_network_result["compressed_network"]                                                    \n``` \n\nFrom the resulting compressed `TemporalNetwork` objects, you can now access your snapshots as you\nwould with your original temporal network, by accessing the `snapshots` member via\n```\nyour_new_snapshots = your_compressed_network.snapshots\n```\nfrom which you can access each snapshot\'s new duration, adjacency matrix, start and end times.\n\n## Usage using demo network\n\nFor a more involved demo, make use of the `compressnets.demos` module to access a more complex\ntemporal network without having to create one yourself. Follow the code below in your\nown workspace to use a sample temporal network to compress it, and visualize a system of ODEs over\nthe compressed network vs. the original temporal solution. \n\n```\nfrom compressnets import compression, network, demos, solvers\n\ndemo_network = demos.Sample.get_sample_temporal_network()\ncompressed_optimal = compression.Compressor.compress(demo_network, compress_to=4, how=\'optimal\')["compressed_network"]\ncompressed_even = compression.Compressor.compress(demo_network, compress_to=4, how=\'even\')["compressed_network"]\n```\nNow you have the resulting compressed temporal networks for the optimal (algorithmic) method and from an even\naggregation method. \n\nTo visualize the new time boundaries of each aggregated snapshot, and compare a full\nSusceptible-Infected disease spread process against the fully temporal network, you can\nutilize the `compressnets.solvers` module to solve a system of ODEs and plot the resulting figure:\n\n```\n## Creating and solving a model with the original temporal network\n\nN = demo_network.snapshots[0].N\nbeta = demo_network.snapshots[0].beta\nmodel = solvers.TemporalSIModel({\'beta\': beta}, np.array([1/N for _ in range(N)]),\n                                demo_network.snapshots[demo_network.length-1].end_time,\n                                demo_network)\nsoln = model.solve_model()\nsmooth_soln = model.smooth_solution(soln)\nplt.plot(smooth_soln[0], smooth_soln[1], color=\'k\', label=\'Temporal solution\')\nplt.vlines(list(demo_network.get_time_network_map().keys()), ymin=0, ymax=N/3, ls=\'-\',\n          lw=0.5, alpha=1.0, color=\'k\')\n\n\n## Creating and solving a model with the algorithmically compressed temporal network\n\nN = compressed_optimal.snapshots[0].N\nbeta = compressed_optimal.snapshots[0].beta\nmodel = solvers.TemporalSIModel({\'beta\': beta}, np.array([1/N for _ in range(N)]),\n                                compressed_optimal.snapshots[compressed_optimal.length-1].end_time,\n                                compressed_optimal)\nsoln = model.solve_model()\nsmooth_soln = model.smooth_solution(soln)\nplt.plot(smooth_soln[0], smooth_soln[1], color=\'b\', label=\'Optimal compressed\')\nplt.vlines(list(compressed_optimal.get_time_network_map().keys()), ymin=N/3, ymax=2*N/3, ls=\'-\',\n          lw=0.5, alpha=1.0, color=\'b\')\n\n\n## Creating and solving a model with the evenly compressed temporal network\n\nN = compressed_even.snapshots[0].N\nbeta = compressed_even.snapshots[0].beta\nmodel = solvers.TemporalSIModel({\'beta\': beta}, np.array([1/N for _ in range(N)]),\n                                compressed_even.snapshots[compressed_even.length-1].end_time,\n                                compressed_even)\nsoln = model.solve_model()\nsmooth_soln = model.smooth_solution(soln)\nplt.plot(smooth_soln[0], smooth_soln[1], color=\'r\', label=\'Even compressed\')\nplt.vlines(list(compressed_even.get_time_network_map().keys()), ymin=2*N/3, ymax=N, ls=\'-\',\n          lw=0.5, alpha=1.0, color=\'r\')\n\nplt.ylabel(\'Number infected\')\nplt.xlabel(\'Time\')\nplt.legend()\nplt.show()\n```\n\n\n![](https://raw.githubusercontent.com/andrea-allen/compressnets/main/compressnets/datafiles/sample_fig.png)\n\n\nOutput figure from the sample code above using the provided demo temporal network.\nThe original temporal network has 50 snapshots and is compressed down to 4 snapshots.\nIn blue, you see the resulting temporal boundaries of the 4 snapshots compressed using our\nalgorithm. In red, you see the resulting temporal boundaries of the 4 snapshots compressed\ninto even-size aggregate matrices. The time series represent an SI epidemic process over the\n3 versions of the network.\n\n## Choosing the infect rate `beta`\nThe best `beta` value to choose for your series of networks is one that,\ngiven an SI process over the full sequence of temporal networks,\nwon\'t saturate (infect the entire network) too quickly, but that also provides\nrich enough dynamics that can be observed.\n\nOne way to test if a given `beta` value is appropriate is to use the `solvers` module.\nThe same module that is shown in the demo to assess how your compression\nperformed is also just as useful for a pre-analysis of your system.\n\nFor example, running just the top block of the demo above will give you a time series solution\non your full temporal network:\n```\nN = demo_network.snapshots[0].N\nbeta = YOUR_CHOSEN_VALUE\nmodel = solvers.TemporalSIModel({\'beta\': beta}, np.array([1/N for _ in range(N)]),\n                                demo_network.snapshots[demo_network.length-1].end_time,\n                                demo_network)\nsoln = model.solve_model()\nsmooth_soln = model.smooth_solution(soln)\nplt.plot(smooth_soln[0], smooth_soln[1], color=\'k\', label=\'Temporal solution\')\n```\n\nYou can try out a range of different `beta` values until you see a time series\nsolution that doesn\'t saturate too quickly and also doesn\'t die off.\nThe compression algorithm is robust to a range of values around your specified value,\nso no need to worry about a precise value of `beta`, just one that\'s in the appropriate range\nfor the epidemic dynamics to be observed.\n\n## Citation\n\nIf you use this package, please name your use of this package as well as the original \npaper on the framework, as \n\nAllen, Andrea J. and Moore, Cristopher and Hébert-Dufresne, Laurent.\nA network compression approach for quantifying the importance of temporal contact chronology.\nPreprint at https://arxiv.org/abs/2205.11566 (2022).\n\n\n\n',
    'author': 'Andrea Allen',
    'author_email': 'andrea2allen@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/andrea-allen/compressnets',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
