"use strict";
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const aws_codeguruprofiler_1 = require("@aws-cdk/aws-codeguruprofiler");
const ec2 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const _ = require("lodash");
const nodeunit_1 = require("nodeunit");
const lambda = require("../lib");
module.exports = nodeunit_1.testCase({
    'add incompatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const func = new lambda.Function(stack, 'myFunc', {
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            code,
        });
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [lambda.Runtime.NODEJS],
        });
        // THEN
        test.throws(() => func.addLayers(layer), /This lambda function uses a runtime that is incompatible with this layer/);
        test.done();
    },
    'add compatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const func = new lambda.Function(stack, 'myFunc', {
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            code,
        });
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [lambda.Runtime.PYTHON_3_7],
        });
        // THEN
        // should not throw
        func.addLayers(layer);
        test.done();
    },
    'add compatible layer for deep clone'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const runtime = lambda.Runtime.PYTHON_3_7;
        const func = new lambda.Function(stack, 'myFunc', {
            runtime,
            handler: 'index.handler',
            code,
        });
        const clone = _.cloneDeep(runtime);
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [clone],
        });
        // THEN
        // should not throw
        func.addLayers(layer);
        test.done();
    },
    'empty inline code is not allowed'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN/THEN
        test.throws(() => new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline(''),
        }), /Lambda inline code cannot be empty/);
        test.done();
    },
    'logGroup is correctly returned'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
        });
        const logGroup = fn.logGroup;
        test.ok(logGroup.logGroupName);
        test.ok(logGroup.logGroupArn);
        test.done();
    },
    'dlq is returned when provided by user'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriod: cdk.Duration.days(14),
        });
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            deadLetterQueue: dlQueue,
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueArn);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueName);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueUrl);
        test.done();
    },
    'dlq is returned when setup by cdk'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            deadLetterQueueEnabled: true,
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueArn);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueName);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueUrl);
        test.done();
    },
    'dlq is undefined when not setup'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok(deadLetterQueue === undefined);
        test.done();
    },
    'one and only one child LogRetention construct will be created'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            logRetention: logs.RetentionDays.FIVE_DAYS,
        });
        // Call logGroup a few times. If more than one instance of LogRetention was created,
        // the second call will fail on duplicate constructs.
        fn.logGroup;
        fn.logGroup;
        fn.logGroup;
        test.done();
    },
    'fails when inline code is specified on an incompatible runtime'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.PROVIDED,
            code: lambda.Code.fromInline('foo'),
        }), /Inline source not allowed for/);
        test.done();
    },
    'default function with CDK created Profiling Group'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            profiling: true,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeGuruProfiler::ProfilingGroup', {
            ProfilingGroupName: 'MyLambdaProfilingGroupC5B6CCD8',
            ComputePlatform: 'AWSLambda',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'codeguru-profiler:ConfigureAgent',
                            'codeguru-profiler:PostAgentProfile',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': ['MyLambdaProfilingGroupEC6DE32F', 'Arn'],
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            PolicyName: 'MyLambdaServiceRoleDefaultPolicy5BBC6F68',
            Roles: [
                {
                    Ref: 'MyLambdaServiceRole4539ECB6',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Environment: {
                Variables: {
                    AWS_CODEGURU_PROFILER_GROUP_ARN: { 'Fn::GetAtt': ['MyLambdaProfilingGroupEC6DE32F', 'Arn'] },
                    AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
                },
            },
        }));
        test.done();
    },
    'default function with client provided Profiling Group'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            profilingGroup: new aws_codeguruprofiler_1.ProfilingGroup(stack, 'ProfilingGroup'),
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'codeguru-profiler:ConfigureAgent',
                            'codeguru-profiler:PostAgentProfile',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': ['ProfilingGroup26979FD7', 'Arn'],
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            PolicyName: 'MyLambdaServiceRoleDefaultPolicy5BBC6F68',
            Roles: [
                {
                    Ref: 'MyLambdaServiceRole4539ECB6',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Environment: {
                Variables: {
                    AWS_CODEGURU_PROFILER_GROUP_ARN: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:', { Ref: 'AWS::Partition' }, ':codeguru-profiler:', { Ref: 'AWS::Region' },
                                ':', { Ref: 'AWS::AccountId' }, ':profilingGroup/', { Ref: 'ProfilingGroup26979FD7' },
                            ],
                        ],
                    },
                    AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
                },
            },
        }));
        test.done();
    },
    'default function with client provided Profiling Group but profiling set to false'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            profiling: false,
            profilingGroup: new aws_codeguruprofiler_1.ProfilingGroup(stack, 'ProfilingGroup'),
        });
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::IAM::Policy'));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Function', {
            Environment: {
                Variables: {
                    AWS_CODEGURU_PROFILER_GROUP_ARN: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:', { Ref: 'AWS::Partition' }, ':codeguru-profiler:', { Ref: 'AWS::Region' },
                                ':', { Ref: 'AWS::AccountId' }, ':profilingGroup/', { Ref: 'ProfilingGroup26979FD7' },
                            ],
                        ],
                    },
                    AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
                },
            },
        }));
        test.done();
    },
    'default function with profiling enabled and client provided env vars'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            profiling: true,
            environment: {
                AWS_CODEGURU_PROFILER_GROUP_ARN: 'profiler_group_arn',
                AWS_CODEGURU_PROFILER_ENABLED: 'yes',
            },
        }), /AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled/);
        test.done();
    },
    'default function with client provided Profiling Group and client provided env vars'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            profilingGroup: new aws_codeguruprofiler_1.ProfilingGroup(stack, 'ProfilingGroup'),
            environment: {
                AWS_CODEGURU_PROFILER_GROUP_ARN: 'profiler_group_arn',
                AWS_CODEGURU_PROFILER_ENABLED: 'yes',
            },
        }), /AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled/);
        test.done();
    },
    'multiple calls to latestVersion returns the same version'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version1 = fn.latestVersion;
        const version2 = fn.latestVersion;
        const expectedArn = {
            'Fn::Join': ['', [
                    { 'Fn::GetAtt': ['MyLambdaCCE802FB', 'Arn'] },
                    ':$LATEST',
                ]],
        };
        test.equal(version1, version2);
        test.deepEqual(stack.resolve(version1.functionArn), expectedArn);
        test.deepEqual(stack.resolve(version2.functionArn), expectedArn);
        test.done();
    },
    'currentVersion': {
        // see test.function-hash.ts for more coverage for this
        'logical id of version is based on the function hash'(test) {
            // GIVEN
            const stack1 = new cdk.Stack();
            const fn1 = new lambda.Function(stack1, 'MyFunction', {
                handler: 'foo',
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'handler.zip')),
                environment: {
                    FOO: 'bar',
                },
            });
            const stack2 = new cdk.Stack();
            const fn2 = new lambda.Function(stack2, 'MyFunction', {
                handler: 'foo',
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'handler.zip')),
                environment: {
                    FOO: 'bear',
                },
            });
            // WHEN
            new cdk.CfnOutput(stack1, 'CurrentVersionArn', {
                value: fn1.currentVersion.functionArn,
            });
            new cdk.CfnOutput(stack2, 'CurrentVersionArn', {
                value: fn2.currentVersion.functionArn,
            });
            // THEN
            const template1 = assert_1.SynthUtils.synthesize(stack1).template;
            const template2 = assert_1.SynthUtils.synthesize(stack2).template;
            // these functions are different in their configuration but the original
            // logical ID of the version would be the same unless the logical ID
            // includes the hash of function's configuration.
            test.notDeepEqual(template1.Outputs.CurrentVersionArn.Value, template2.Outputs.CurrentVersionArn.Value);
            test.done();
        },
    },
    'filesystem': {
        'mount efs filesystem'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'Vpc', {
                maxAzs: 3,
                natGateways: 1,
            });
            const fs = new efs.FileSystem(stack, 'Efs', {
                vpc,
            });
            const accessPoint = fs.addAccessPoint('AccessPoint');
            // WHEN
            new lambda.Function(stack, 'MyFunction', {
                handler: 'foo',
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'handler.zip')),
                filesystem: lambda.FileSystem.fromEfsAccessPoint(accessPoint, '/mnt/msg'),
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                FileSystemConfigs: [
                    {
                        Arn: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':elasticfilesystem:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':access-point/',
                                    {
                                        Ref: 'EfsAccessPointE419FED9',
                                    },
                                ],
                            ],
                        },
                        LocalMountPath: '/mnt/msg',
                    },
                ],
            }));
            test.done();
        },
    },
});
//# sourceMappingURL=data:application/json;base64,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