"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SingletonFunction = void 0;
const cdk = require("@aws-cdk/core");
const function_1 = require("./function");
const function_base_1 = require("./function-base");
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
class SingletonFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureLambda(props);
        this.permissionsNode = this.lambdaFunction.node;
        this.functionArn = this.lambdaFunction.functionArn;
        this.functionName = this.lambdaFunction.functionName;
        this.role = this.lambdaFunction.role;
        this.grantPrincipal = this.lambdaFunction.grantPrincipal;
        this.canCreatePermissions = true; // Doesn't matter, addPermission is overriden anyway
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * singleton Lambda function. A new version will be created every time the
     * function's configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.SingletonFunction`.
     */
    get currentVersion() {
        return this.lambdaFunction.currentVersion;
    }
    addPermission(name, permission) {
        return this.lambdaFunction.addPermission(name, permission);
    }
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled
     * as a singleton across the stack. Use this method instead to declare dependencies.
     */
    addDependency(...up) {
        this.lambdaFunction.node.addDependency(...up);
    }
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using
     * node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     */
    dependOn(down) {
        down.node.addDependency(this.lambdaFunction);
    }
    /** @internal */
    _checkEdgeCompatibility() {
        return this.lambdaFunction._checkEdgeCompatibility();
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * @internal
     */
    _functionNode() {
        return this.lambdaFunction.node;
    }
    ensureLambda(props) {
        const constructName = (props.lambdaPurpose || 'SingletonLambda') + slugify(props.uuid);
        const existing = cdk.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        return new function_1.Function(cdk.Stack.of(this), constructName, props);
    }
}
exports.SingletonFunction = SingletonFunction;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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