"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Runtime = exports.RuntimeFamily = void 0;
const core_1 = require("@aws-cdk/core");
var RuntimeFamily;
(function (RuntimeFamily) {
    RuntimeFamily[RuntimeFamily["NODEJS"] = 0] = "NODEJS";
    RuntimeFamily[RuntimeFamily["JAVA"] = 1] = "JAVA";
    RuntimeFamily[RuntimeFamily["PYTHON"] = 2] = "PYTHON";
    RuntimeFamily[RuntimeFamily["DOTNET_CORE"] = 3] = "DOTNET_CORE";
    RuntimeFamily[RuntimeFamily["GO"] = 4] = "GO";
    RuntimeFamily[RuntimeFamily["RUBY"] = 5] = "RUBY";
    RuntimeFamily[RuntimeFamily["OTHER"] = 6] = "OTHER";
})(RuntimeFamily = exports.RuntimeFamily || (exports.RuntimeFamily = {}));
/**
 * Lambda function runtime environment.
 *
 * If you need to use a runtime name that doesn't exist as a static member, you
 * can instantiate a `Runtime` object, e.g: `new Runtime('nodejs99.99')`.
 */
class Runtime {
    constructor(name, family, props = {}) {
        var _a;
        this.name = name;
        this.supportsInlineCode = !!props.supportsInlineCode;
        this.family = family;
        const imageName = (_a = props.bundlingDockerImage) !== null && _a !== void 0 ? _a : `amazon/aws-sam-cli-build-image-${name}`;
        this.bundlingDockerImage = core_1.BundlingDockerImage.fromRegistry(imageName);
        Runtime.ALL.push(this);
    }
    toString() {
        return this.name;
    }
    runtimeEquals(other) {
        return other.name === this.name &&
            other.family === this.family &&
            other.supportsInlineCode === this.supportsInlineCode;
    }
}
exports.Runtime = Runtime;
/** A list of all known `Runtime`'s. */
Runtime.ALL = new Array();
/**
 * The NodeJS runtime (nodejs)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS = new Runtime('nodejs', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 4.3 runtime (nodejs4.3)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_4_3 = new Runtime('nodejs4.3', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 6.10 runtime (nodejs6.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_6_10 = new Runtime('nodejs6.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 8.10 runtime (nodejs8.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_8_10 = new Runtime('nodejs8.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 10.x runtime (nodejs10.x)
 */
Runtime.NODEJS_10_X = new Runtime('nodejs10.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 12.x runtime (nodejs12.x)
 */
Runtime.NODEJS_12_X = new Runtime('nodejs12.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The Python 2.7 runtime (python2.7)
 */
Runtime.PYTHON_2_7 = new Runtime('python2.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.6 runtime (python3.6)
 */
Runtime.PYTHON_3_6 = new Runtime('python3.6', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.7 runtime (python3.7)
 */
Runtime.PYTHON_3_7 = new Runtime('python3.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.8 runtime (python3.8)
 */
Runtime.PYTHON_3_8 = new Runtime('python3.8', RuntimeFamily.PYTHON);
/**
 * The Java 8 runtime (java8)
 */
Runtime.JAVA_8 = new Runtime('java8', RuntimeFamily.JAVA);
/**
 * The Java 11 runtime (java11)
 */
Runtime.JAVA_11 = new Runtime('java11', RuntimeFamily.JAVA);
/**
 * The .NET Core 1.0 runtime (dotnetcore1.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_1 = new Runtime('dotnetcore1.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.0 runtime (dotnetcore2.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_2 = new Runtime('dotnetcore2.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.1 runtime (dotnetcore2.1)
 */
Runtime.DOTNET_CORE_2_1 = new Runtime('dotnetcore2.1', RuntimeFamily.DOTNET_CORE, {
    bundlingDockerImage: 'lambci/lambda:build-dotnetcore2.1',
});
/**
 * The .NET Core 3.1 runtime (dotnetcore3.1)
 */
Runtime.DOTNET_CORE_3_1 = new Runtime('dotnetcore3.1', RuntimeFamily.DOTNET_CORE, {
    bundlingDockerImage: 'lambci/lambda:build-dotnetcore3.1',
});
/**
 * The Go 1.x runtime (go1.x)
 */
Runtime.GO_1_X = new Runtime('go1.x', RuntimeFamily.GO, {
    bundlingDockerImage: 'lambci/lambda:build-go1.x',
});
/**
 * The Ruby 2.5 runtime (ruby2.5)
 */
Runtime.RUBY_2_5 = new Runtime('ruby2.5', RuntimeFamily.RUBY);
/**
 * The Ruby 2.7 runtime (ruby2.7)
 */
Runtime.RUBY_2_7 = new Runtime('ruby2.7', RuntimeFamily.RUBY);
/**
 * The custom provided runtime (provided)
 */
Runtime.PROVIDED = new Runtime('provided', RuntimeFamily.OTHER);
//# sourceMappingURL=data:application/json;base64,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